"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Billingconductor = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [billingconductor](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsbillingconductor.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Billingconductor extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate between one and 30 accounts to a billing group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/billingconductor/latest/APIReference/API_AssociateAccounts.html
     */
    toAssociateAccounts() {
        return this.to('AssociateAccounts');
    }
    /**
     * Grants permission to associate pricing rules
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/billingconductor/latest/APIReference/API_AssociatePricingRules.html
     */
    toAssociatePricingRules() {
        return this.to('AssociatePricingRules');
    }
    /**
     * Grants permission to batch associate resources to a percentage custom line item
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/billingconductor/latest/APIReference/API_BatchAssociateResourcesToCustomLineItem.html
     */
    toBatchAssociateResourcesToCustomLineItem() {
        return this.to('BatchAssociateResourcesToCustomLineItem');
    }
    /**
     * Grants permission to batch disassociate resources from a percentage custom line item
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/billingconductor/latest/APIReference/API_BatchDisassociateResourcesFromCustomLineItem.html
     */
    toBatchDisassociateResourcesFromCustomLineItem() {
        return this.to('BatchDisassociateResourcesFromCustomLineItem');
    }
    /**
     * Grants permission to create a billing group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/billingconductor/latest/APIReference/API_CreateBillingGroup.html
     */
    toCreateBillingGroup() {
        return this.to('CreateBillingGroup');
    }
    /**
     * Grants permission to create a custom line item
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/billingconductor/latest/APIReference/API_CreateCustomLineItem.html
     */
    toCreateCustomLineItem() {
        return this.to('CreateCustomLineItem');
    }
    /**
     * Grants permission to create a pricing plan
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/billingconductor/latest/APIReference/API_CreatePricingPlan.html
     */
    toCreatePricingPlan() {
        return this.to('CreatePricingPlan');
    }
    /**
     * Grants permission to create a pricing rule
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/billingconductor/latest/APIReference/API_CreatePricingRule.html
     */
    toCreatePricingRule() {
        return this.to('CreatePricingRule');
    }
    /**
     * Grants permission to delete a billing group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/billingconductor/latest/APIReference/API_DeleteBillingGroup.html
     */
    toDeleteBillingGroup() {
        return this.to('DeleteBillingGroup');
    }
    /**
     * Grants permission to delete a custom line item
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/billingconductor/latest/APIReference/API_DeleteCustomLineItem.html
     */
    toDeleteCustomLineItem() {
        return this.to('DeleteCustomLineItem');
    }
    /**
     * Grants permission to delete a pricing plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/billingconductor/latest/APIReference/API_DeletePricingPlan.html
     */
    toDeletePricingPlan() {
        return this.to('DeletePricingPlan');
    }
    /**
     * Grants permission to delete a pricing rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/billingconductor/latest/APIReference/API_DeletePricingRule.html
     */
    toDeletePricingRule() {
        return this.to('DeletePricingRule');
    }
    /**
     * Grants permission to detach between one and 30 accounts from a billing group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/billingconductor/latest/APIReference/API_DisassociateAccounts.html
     */
    toDisassociateAccounts() {
        return this.to('DisassociateAccounts');
    }
    /**
     * Grants permission to disassociate pricing rules
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/billingconductor/latest/APIReference/API_DisassociatePricingRules.html
     */
    toDisassociatePricingRules() {
        return this.to('DisassociatePricingRules');
    }
    /**
     * Grants permission to view the billing group cost report for the specified billing group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/billingconductor/latest/APIReference/API_GetBillingGroupCostReport.html
     */
    toGetBillingGroupCostReport() {
        return this.to('GetBillingGroupCostReport');
    }
    /**
     * Grants permission to list the linked accounts of the payer account for the given billing period while also providing the billing group the linked accounts belong to
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/billingconductor/latest/APIReference/API_ListAccountAssociations.html
     */
    toListAccountAssociations() {
        return this.to('ListAccountAssociations');
    }
    /**
     * Grants permission to view the billing group cost report
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/billingconductor/latest/APIReference/API_ListBillingGroupCostReports.html
     */
    toListBillingGroupCostReports() {
        return this.to('ListBillingGroupCostReports');
    }
    /**
     * Grants permission to view the details of billing groups
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/billingconductor/latest/APIReference/API_ListBillingGroups.html
     */
    toListBillingGroups() {
        return this.to('ListBillingGroups');
    }
    /**
     * Grants permission to view custom line item versions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/billingconductor/latest/APIReference/API_ListCustomLineItemVersions.html
     */
    toListCustomLineItemVersions() {
        return this.to('ListCustomLineItemVersions');
    }
    /**
     * Grants permission to view custom line item details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/billingconductor/latest/APIReference/API_ListCustomLineItems.html
     */
    toListCustomLineItems() {
        return this.to('ListCustomLineItems');
    }
    /**
     * Grants permission to view the pricing plans details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/billingconductor/latest/APIReference/API_ListPricingPlans.html
     */
    toListPricingPlans() {
        return this.to('ListPricingPlans');
    }
    /**
     * Grants permission to list pricing plans associated with a pricing rule
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/billingconductor/latest/APIReference/API_ListPricingPlansAssociatedWithPricingRule.html
     */
    toListPricingPlansAssociatedWithPricingRule() {
        return this.to('ListPricingPlansAssociatedWithPricingRule');
    }
    /**
     * Grants permission to view pricing rules details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/billingconductor/latest/APIReference/API_ListPricingRules.html
     */
    toListPricingRules() {
        return this.to('ListPricingRules');
    }
    /**
     * Grants permission to list pricing rules associated to a pricing plan
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/billingconductor/latest/APIReference/API_ListPricingRulesAssociatedToPricingPlan.html
     */
    toListPricingRulesAssociatedToPricingPlan() {
        return this.to('ListPricingRulesAssociatedToPricingPlan');
    }
    /**
     * Grants permission to list resources associated to a percentage custom line item
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/billingconductor/latest/APIReference/API_ListResourcesAssociatedToCustomLineItem.html
     */
    toListResourcesAssociatedToCustomLineItem() {
        return this.to('ListResourcesAssociatedToCustomLineItem');
    }
    /**
     * Grants permission to list tags of a resource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/billingconductor/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to tag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/billingconductor/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/billingconductor/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a billing group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/billingconductor/latest/APIReference/API_UpdateBillingGroup.html
     */
    toUpdateBillingGroup() {
        return this.to('UpdateBillingGroup');
    }
    /**
     * Grants permission to update a custom line item
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/billingconductor/latest/APIReference/API_UpdateCustomLineItem.html
     */
    toUpdateCustomLineItem() {
        return this.to('UpdateCustomLineItem');
    }
    /**
     * Grants permission to update a pricing plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/billingconductor/latest/APIReference/API_UpdatePricingPlan.html
     */
    toUpdatePricingPlan() {
        return this.to('UpdatePricingPlan');
    }
    /**
     * Grants permission to update a pricing rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/billingconductor/latest/APIReference/API_UpdatePricingRule.html
     */
    toUpdatePricingRule() {
        return this.to('UpdatePricingRule');
    }
    /**
     * Adds a resource of type billinggroup to the statement
     *
     * https://docs.aws.amazon.com/billingconductor/latest/userguide/understanding-abc.html
     *
     * @param billingGroupId - Identifier for the billingGroupId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBillinggroup(billingGroupId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:billingconductor::${account ?? this.defaultAccount}:billinggroup/${billingGroupId}`);
    }
    /**
     * Adds a resource of type pricingplan to the statement
     *
     * https://docs.aws.amazon.com/billingconductor/latest/userguide/understanding-abc.html
     *
     * @param pricingPlanId - Identifier for the pricingPlanId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPricingplan(pricingPlanId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:billingconductor::${account ?? this.defaultAccount}:pricingplan/${pricingPlanId}`);
    }
    /**
     * Adds a resource of type pricingrule to the statement
     *
     * https://docs.aws.amazon.com/billingconductor/latest/userguide/understanding-abc.html
     *
     * @param pricingRuleId - Identifier for the pricingRuleId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPricingrule(pricingRuleId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:billingconductor::${account ?? this.defaultAccount}:pricingrule/${pricingRuleId}`);
    }
    /**
     * Adds a resource of type customlineitem to the statement
     *
     * https://docs.aws.amazon.com/billingconductor/latest/userguide/understanding-abc.html
     *
     * @param customLineItemId - Identifier for the customLineItemId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCustomlineitem(customLineItemId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:billingconductor::${account ?? this.defaultAccount}:customlineitem/${customLineItemId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateBillingGroup()
     * - .toCreateCustomLineItem()
     * - .toCreatePricingPlan()
     * - .toCreatePricingRule()
     * - .toListTagsForResource()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - billinggroup
     * - pricingplan
     * - pricingrule
     * - customlineitem
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateBillingGroup()
     * - .toCreateCustomLineItem()
     * - .toCreatePricingPlan()
     * - .toCreatePricingRule()
     * - .toListTagsForResource()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [billingconductor](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsbillingconductor.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'billingconductor';
        this.accessLevelList = {
            Write: [
                'AssociateAccounts',
                'AssociatePricingRules',
                'BatchAssociateResourcesToCustomLineItem',
                'BatchDisassociateResourcesFromCustomLineItem',
                'CreateBillingGroup',
                'CreateCustomLineItem',
                'CreatePricingPlan',
                'CreatePricingRule',
                'DeleteBillingGroup',
                'DeleteCustomLineItem',
                'DeletePricingPlan',
                'DeletePricingRule',
                'DisassociateAccounts',
                'DisassociatePricingRules',
                'UpdateBillingGroup',
                'UpdateCustomLineItem',
                'UpdatePricingPlan',
                'UpdatePricingRule'
            ],
            Read: [
                'GetBillingGroupCostReport',
                'ListBillingGroupCostReports',
                'ListBillingGroups',
                'ListCustomLineItemVersions',
                'ListCustomLineItems',
                'ListPricingPlans',
                'ListPricingRules',
                'ListTagsForResource'
            ],
            List: [
                'ListAccountAssociations',
                'ListPricingPlansAssociatedWithPricingRule',
                'ListPricingRulesAssociatedToPricingPlan',
                'ListResourcesAssociatedToCustomLineItem'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Billingconductor = Billingconductor;
//# sourceMappingURL=data:application/json;base64,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