"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Budgets = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [budgets](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsbudgetservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Budgets extends shared_1.PolicyStatement {
    /**
     * Grants permission to configure a response that executes once your budget exceeds a specific budget threshold. Creating a budget action with tags also requires the 'budgets:TagResource' permission
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_budgets_CreateBudgetAction.html
     */
    toCreateBudgetAction() {
        return this.to('CreateBudgetAction');
    }
    /**
     * Grants permission to delete an action that is associated with a specific budget
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_budgets_DeleteBudgetAction.html
     */
    toDeleteBudgetAction() {
        return this.to('DeleteBudgetAction');
    }
    /**
     * Grants permission to retrieve the details of a specific budget action associated with a budget
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_budgets_DescribeBudgetAction.html
     */
    toDescribeBudgetAction() {
        return this.to('DescribeBudgetAction');
    }
    /**
     * Grants permission to retrieve a historical view of the budget actions statuses associated with a particular budget action. These status include statues such as 'Standby', 'Pending' and 'Executed'
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_budgets_DescribeBudgetActionHistories.html
     */
    toDescribeBudgetActionHistories() {
        return this.to('DescribeBudgetActionHistories');
    }
    /**
     * Grants permission to retrieve the details of all of the budget actions associated with your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_budgets_DescribeBudgetActionsForAccount.html
     */
    toDescribeBudgetActionsForAccount() {
        return this.to('DescribeBudgetActionsForAccount');
    }
    /**
     * Grants permission to retrieve the details of all of the budget actions associated with a budget
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_budgets_DescribeBudgetActionsForBudget.html
     */
    toDescribeBudgetActionsForBudget() {
        return this.to('DescribeBudgetActionsForBudget');
    }
    /**
     * Grants permission to initiate a pending budget action as well as reverse a previously executed budget action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_budgets_ExecuteBudgetAction.html
     */
    toExecuteBudgetAction() {
        return this.to('ExecuteBudgetAction');
    }
    /**
     * Grants permission to view resource tags for a budget or budget action
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_budgets_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to create and modify budgets, and edit budget details. Creating a budget with tags also requires the 'budgets:TagResource' permission
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_Operations_AWS_Budgets.html
     */
    toModifyBudget() {
        return this.to('ModifyBudget');
    }
    /**
     * Grants permission to apply resource tags to a budget or budget action. Also needed to create a budget or budget action with tags
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_budgets_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove resource tags from a budget or budget action
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_budgets_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update the details of a specific budget action associated with a budget
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_budgets_UpdateBudgetAction.html
     */
    toUpdateBudgetAction() {
        return this.to('UpdateBudgetAction');
    }
    /**
     * Grants permission to view budgets and budget details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_Operations_AWS_Budgets.html
     */
    toViewBudget() {
        return this.to('ViewBudget');
    }
    /**
     * Adds a resource of type budget to the statement
     *
     * https://docs.aws.amazon.com/cost-management/latest/userguide/budgets-managing-costs.html
     *
     * @param budgetName - Identifier for the budgetName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     */
    onBudget(budgetName, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:budgets::${account ?? this.defaultAccount}:budget/${budgetName}`);
    }
    /**
     * Adds a resource of type budgetAction to the statement
     *
     * https://docs.aws.amazon.com/cost-management/latest/userguide/budgets-controls.html
     *
     * @param budgetName - Identifier for the budgetName.
     * @param actionId - Identifier for the actionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     */
    onBudgetAction(budgetName, actionId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:budgets::${account ?? this.defaultAccount}:budget/${budgetName}/action/${actionId}`);
    }
    /**
     * Filters access based on the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateBudgetAction()
     * - .toTagResource()
     *
     * Applies to resource types:
     * - budget
     * - budgetAction
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toCreateBudgetAction()
     *
     * Applies to resource types:
     * - budget
     * - budgetAction
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateBudgetAction()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * Applies to resource types:
     * - budget
     * - budgetAction
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [budgets](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsbudgetservice.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'budgets';
        this.accessLevelList = {
            Write: [
                'CreateBudgetAction',
                'DeleteBudgetAction',
                'ExecuteBudgetAction',
                'ModifyBudget',
                'UpdateBudgetAction'
            ],
            Read: [
                'DescribeBudgetAction',
                'DescribeBudgetActionHistories',
                'DescribeBudgetActionsForAccount',
                'DescribeBudgetActionsForBudget',
                'ListTagsForResource',
                'ViewBudget'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Budgets = Budgets;
//# sourceMappingURL=data:application/json;base64,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