import { AccessLevelList } from '../../shared/access-level';
import { PolicyStatement, Operator } from '../../shared';
import { aws_iam as iam } from "aws-cdk-lib";
/**
 * Statement provider for service [cloudfront](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncloudfront.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class Cloudfront extends PolicyStatement {
    servicePrefix: string;
    /**
     * Grants permission to configure vended log delivery for a distribution
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/AWS-logs-and-resource-policy.html#AWS-logs-infrastructure-V2-service-specific
     */
    toAllowVendedLogDeliveryForResource(): this;
    /**
     * Grants permission to associate an alias to a CloudFront distribution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_AssociateAlias.html
     */
    toAssociateAlias(): this;
    /**
     * Grants permission to associate a distribution tenant with an AWS WAF web ACL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_AssociateDistributionTenantWebACL.html
     */
    toAssociateDistributionTenantWebACL(): this;
    /**
     * Grants permission to associate a distribution with an AWS WAF web ACL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_AssociateDistributionWebACL.html
     */
    toAssociateDistributionWebACL(): this;
    /**
     * Grants permission to copy an existing distribution and create a new web distribution
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - cloudfront:CopyDistribution
     * - cloudfront:CreateDistribution
     * - cloudfront:GetDistribution
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CopyDistribution.html
     */
    toCopyDistribution(): this;
    /**
     * Grants permission to create an Anycast static IP list
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateAnycastIpList.html
     */
    toCreateAnycastIpList(): this;
    /**
     * Grants permission to add a new cache policy to CloudFront
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateCachePolicy.html
     */
    toCreateCachePolicy(): this;
    /**
     * Grants permission to create a new CloudFront origin access identity
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateCloudFrontOriginAccessIdentity.html
     */
    toCreateCloudFrontOriginAccessIdentity(): this;
    /**
     * Grants permission to create a connection group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateConnectionGroup.html
     */
    toCreateConnectionGroup(): this;
    /**
     * Grants permission to add a new continuous-deployment policy to CloudFront
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateContinuousDeploymentPolicy.html
     */
    toCreateContinuousDeploymentPolicy(): this;
    /**
     * Grants permission to create a new web distribution
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - cloudfront:CreateConnectionGroup
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateDistribution.html
     */
    toCreateDistribution(): this;
    /**
     * Grants permission to create a distribution tenant
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateDistributionTenant.html
     */
    toCreateDistributionTenant(): this;
    /**
     * Grants permission to create a new field-level encryption configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateFieldLevelEncryptionConfig.html
     */
    toCreateFieldLevelEncryptionConfig(): this;
    /**
     * Grants permission to create a field-level encryption profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateFieldLevelEncryptionProfile.html
     */
    toCreateFieldLevelEncryptionProfile(): this;
    /**
     * Grants permission to create a CloudFront function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateFunction.html
     */
    toCreateFunction(): this;
    /**
     * Grants permission to create a new invalidation batch request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateInvalidation.html
     */
    toCreateInvalidation(): this;
    /**
     * Grants permission to create an invalidation for a distribution tenant
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateInvalidationForDistributionTenant.html
     */
    toCreateInvalidationForDistributionTenant(): this;
    /**
     * Grants permission to add a new key group to CloudFront
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateKeyGroup.html
     */
    toCreateKeyGroup(): this;
    /**
     * Grants permission to create a CloudFront KeyValueStore
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateKeyValueStore.html
     */
    toCreateKeyValueStore(): this;
    /**
     * Grants permission to enable additional CloudWatch metrics for the specified CloudFront distribution. The additional metrics incur an additional cost
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateMonitoringSubscription.html
     */
    toCreateMonitoringSubscription(): this;
    /**
     * Grants permission to create a new origin access control
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateOriginAccessControl.html
     */
    toCreateOriginAccessControl(): this;
    /**
     * Grants permission to add a new origin request policy to CloudFront
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateOriginRequestPolicy.html
     */
    toCreateOriginRequestPolicy(): this;
    /**
     * Grants permission to add a new public key to CloudFront
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreatePublicKey.html
     */
    toCreatePublicKey(): this;
    /**
     * Grants permission to create a real-time log configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateRealtimeLogConfig.html
     */
    toCreateRealtimeLogConfig(): this;
    /**
     * Grants permission to add a new response headers policy to CloudFront
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateResponseHeadersPolicy.html
     */
    toCreateResponseHeadersPolicy(): this;
    /**
     * Grants permission to create a new savings plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cf-api-permissions-ref.html
     */
    toCreateSavingsPlan(): this;
    /**
     * Grants permission to create a new RTMP distribution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateStreamingDistribution.html
     */
    toCreateStreamingDistribution(): this;
    /**
     * Grants permission to create a new RTMP distribution with tags
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateStreamingDistributionWithTags.html
     */
    toCreateStreamingDistributionWithTags(): this;
    /**
     * Grants permission to create a VPC origin
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateVpcOrigin.html
     */
    toCreateVpcOrigin(): this;
    /**
     * Grants permission to delete an Anycast static IP list
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteAnycastIpList.html
     */
    toDeleteAnycastIpList(): this;
    /**
     * Grants permission to delete a cache policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteCachePolicy.html
     */
    toDeleteCachePolicy(): this;
    /**
     * Grants permission to delete a CloudFront origin access identity
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteCloudFrontOriginAccessIdentity.html
     */
    toDeleteCloudFrontOriginAccessIdentity(): this;
    /**
     * Grants permission to delete a connection group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteConnectionGroup.html
     */
    toDeleteConnectionGroup(): this;
    /**
     * Grants permission to delete a continuous-deployment policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteContinuousDeploymentPolicy.html
     */
    toDeleteContinuousDeploymentPolicy(): this;
    /**
     * Grants permission to delete a web distribution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteDistribution.html
     */
    toDeleteDistribution(): this;
    /**
     * Grants permission to delete a distribution tenant
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteDistributionTenant.html
     */
    toDeleteDistributionTenant(): this;
    /**
     * Grants permission to delete a field-level encryption configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteFieldLevelEncryptionConfig.html
     */
    toDeleteFieldLevelEncryptionConfig(): this;
    /**
     * Grants permission to delete a field-level encryption profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteFieldLevelEncryptionProfile.html
     */
    toDeleteFieldLevelEncryptionProfile(): this;
    /**
     * Grants permission to delete a CloudFront function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteFunction.html
     */
    toDeleteFunction(): this;
    /**
     * Grants permission to delete a key group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteKeyGroup.html
     */
    toDeleteKeyGroup(): this;
    /**
     * Grants permission to delete a CloudFront KeyValueStore
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteKeyValueStore.html
     */
    toDeleteKeyValueStore(): this;
    /**
     * Grants permission to disable additional CloudWatch metrics for the specified CloudFront distribution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteMonitoringSubscription.html
     */
    toDeleteMonitoringSubscription(): this;
    /**
     * Grants permission to delete an origin access control
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteOriginAccessControl.html
     */
    toDeleteOriginAccessControl(): this;
    /**
     * Grants permission to delete an origin request policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteOriginRequestPolicy.html
     */
    toDeleteOriginRequestPolicy(): this;
    /**
     * Grants permission to delete a public key from CloudFront
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeletePublicKey.html
     */
    toDeletePublicKey(): this;
    /**
     * Grants permission to delete a real-time log configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteRealtimeLogConfig.html
     */
    toDeleteRealtimeLogConfig(): this;
    /**
     * Grants permission to delete a response headers policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteResponseHeadersPolicy.html
     */
    toDeleteResponseHeadersPolicy(): this;
    /**
     * Grants permission to delete an RTMP distribution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteStreamingDistribution.html
     */
    toDeleteStreamingDistribution(): this;
    /**
     * Grants permission to delete a VPC origin
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteVpcOrigin.html
     */
    toDeleteVpcOrigin(): this;
    /**
     * Grants permission to get a CloudFront function summary
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DescribeFunction.html
     */
    toDescribeFunction(): this;
    /**
     * Grants permission to get a CloudFront KeyValueStore summary
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DescribeKeyValueStore.html
     */
    toDescribeKeyValueStore(): this;
    /**
     * Grants permission to disassociate a distribution tenant from an AWS WAF web ACL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DisassociateDistributionTenantWebACL.html
     */
    toDisassociateDistributionTenantWebACL(): this;
    /**
     * Grants permission to disassociate a distribution from an AWS WAF web ACL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DisassociateDistributionWebACL.html
     */
    toDisassociateDistributionWebACL(): this;
    /**
     * Grants permission to get an Anycast static IP list
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetAnycastIpList.html
     */
    toGetAnycastIpList(): this;
    /**
     * Grants permission to get the cache policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetCachePolicy.html
     */
    toGetCachePolicy(): this;
    /**
     * Grants permission to get the cache policy configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetCachePolicyConfig.html
     */
    toGetCachePolicyConfig(): this;
    /**
     * Grants permission to get the information about a CloudFront origin access identity
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetCloudFrontOriginAccessIdentity.html
     */
    toGetCloudFrontOriginAccessIdentity(): this;
    /**
     * Grants permission to get the configuration information about a Cloudfront origin access identity
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetCloudFrontOriginAccessIdentityConfig.html
     */
    toGetCloudFrontOriginAccessIdentityConfig(): this;
    /**
     * Grants permission to get information about a connection group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetConnectionGroup.html
     */
    toGetConnectionGroup(): this;
    /**
     * Grants permission to get information about a connection group by the specified routing endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetConnectionGroupByRoutingEndpoint.html
     */
    toGetConnectionGroupByRoutingEndpoint(): this;
    /**
     * Grants permission to get the continuous-deployment policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetContinuousDeploymentPolicy.html
     */
    toGetContinuousDeploymentPolicy(): this;
    /**
     * Grants permission to get the continuous-deployment policy configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetContinuousDeploymentPolicyConfig.html
     */
    toGetContinuousDeploymentPolicyConfig(): this;
    /**
     * Grants permission to get the information about a web distribution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetDistribution.html
     */
    toGetDistribution(): this;
    /**
     * Grants permission to get the configuration information about a distribution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetDistributionConfig.html
     */
    toGetDistributionConfig(): this;
    /**
     * Grants permission to get information about a distribution tenant
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetDistributionTenant.html
     */
    toGetDistributionTenant(): this;
    /**
     * Grants permission to get information about a distribution tenant by the associated domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetDistributionTenantByDomain.html
     */
    toGetDistributionTenantByDomain(): this;
    /**
     * Grants permission to get the field-level encryption configuration information
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetFieldLevelEncryption.html
     */
    toGetFieldLevelEncryption(): this;
    /**
     * Grants permission to get the field-level encryption configuration information
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetFieldLevelEncryptionConfig.html
     */
    toGetFieldLevelEncryptionConfig(): this;
    /**
     * Grants permission to get the field-level encryption configuration information
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetFieldLevelEncryptionProfile.html
     */
    toGetFieldLevelEncryptionProfile(): this;
    /**
     * Grants permission to get the field-level encryption profile configuration information
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetFieldLevelEncryptionProfileConfig.html
     */
    toGetFieldLevelEncryptionProfileConfig(): this;
    /**
     * Grants permission to get a CloudFront function's code
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetFunction.html
     */
    toGetFunction(): this;
    /**
     * Grants permission to get the information about an invalidation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetInvalidation.html
     */
    toGetInvalidation(): this;
    /**
     * Grants permission to get information about an invalidation for a distribution tenant
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetInvalidationForDistributionTenant.html
     */
    toGetInvalidationForDistributionTenant(): this;
    /**
     * Grants permission to get a key group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetKeyGroup.html
     */
    toGetKeyGroup(): this;
    /**
     * Grants permission to get a key group configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetKeyGroupConfig.html
     */
    toGetKeyGroupConfig(): this;
    /**
     * Grants permission to get details about a CloudFront managed certificate
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetManagedCertificateDetails.html
     */
    toGetManagedCertificateDetails(): this;
    /**
     * Grants permission to get information about whether additional CloudWatch metrics are enabled for the specified CloudFront distribution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetMonitoringSubscription.html
     */
    toGetMonitoringSubscription(): this;
    /**
     * Grants permission to get the origin access control
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetOriginAccessControl.html
     */
    toGetOriginAccessControl(): this;
    /**
     * Grants permission to get the origin access control configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetOriginAccessControlConfig.html
     */
    toGetOriginAccessControlConfig(): this;
    /**
     * Grants permission to get the origin request policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetOriginRequestPolicy.html
     */
    toGetOriginRequestPolicy(): this;
    /**
     * Grants permission to get the origin request policy configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetOriginRequestPolicyConfig.html
     */
    toGetOriginRequestPolicyConfig(): this;
    /**
     * Grants permission to get the public key information
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetPublicKey.html
     */
    toGetPublicKey(): this;
    /**
     * Grants permission to get the public key configuration information
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetPublicKeyConfig.html
     */
    toGetPublicKeyConfig(): this;
    /**
     * Grants permission to get a real-time log configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetRealtimeLogConfig.html
     */
    toGetRealtimeLogConfig(): this;
    /**
     * Grants permission to get the response headers policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetResponseHeadersPolicy.html
     */
    toGetResponseHeadersPolicy(): this;
    /**
     * Grants permission to get the response headers policy configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetResponseHeadersPolicyConfig.html
     */
    toGetResponseHeadersPolicyConfig(): this;
    /**
     * Grants permission to get a savings plan
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cf-api-permissions-ref.html
     */
    toGetSavingsPlan(): this;
    /**
     * Grants permission to get the information about an RTMP distribution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetStreamingDistribution.html
     */
    toGetStreamingDistribution(): this;
    /**
     * Grants permission to get the configuration information about a streaming distribution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetStreamingDistributionConfig.html
     */
    toGetStreamingDistributionConfig(): this;
    /**
     * Grants permission to get the information about a VPC origin
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetVpcOrigin.html
     */
    toGetVpcOrigin(): this;
    /**
     * Grants permission to list your Anycast static IP lists
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListAnycastIpLists.html
     */
    toListAnycastIpLists(): this;
    /**
     * Grants permission to list all cache policies that have been created in CloudFront for this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListCachePolicies.html
     */
    toListCachePolicies(): this;
    /**
     * Grants permission to list your CloudFront origin access identities
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListCloudFrontOriginAccessIdentities.html
     */
    toListCloudFrontOriginAccessIdentities(): this;
    /**
     * Grants permission to list all aliases that conflict with the given alias in CloudFront
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListConflictingAliases.html
     */
    toListConflictingAliases(): this;
    /**
     * Grants permission to list the connection groups in your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListConnectionGroups.html
     */
    toListConnectionGroups(): this;
    /**
     * Grants permission to list all continuous-deployment policies in the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListContinuousDeploymentPolicies.html
     */
    toListContinuousDeploymentPolicies(): this;
    /**
     * Grants permission to list the distribution tenants in your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributionTenants.html
     */
    toListDistributionTenants(): this;
    /**
     * Grants permission to list the distribution tenants by the customization that you specify
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributionTenantsByCustomization.html
     */
    toListDistributionTenantsByCustomization(): this;
    /**
     * Grants permission to list the distributions associated with your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributions.html
     */
    toListDistributions(): this;
    /**
     * Grants permission to list the distributions in your account that are associated with the specified AnycastIpListId
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributionsByAnycastIpListId.html
     */
    toListDistributionsByAnycastIpListId(): this;
    /**
     * Grants permission to list distribution IDs for distributions that have a cache behavior that's associated with the specified cache policy
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributionsByCachePolicyId.html
     */
    toListDistributionsByCachePolicyId(): this;
    /**
     * Grants permission to list the distributions by the specified connection mode
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributionsByConnectionMode.html
     */
    toListDistributionsByConnectionMode(): this;
    /**
     * Grants permission to list distribution IDs for distributions that have a cache behavior that's associated with the specified key group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributionsByKeyGroup.html
     */
    toListDistributionsByKeyGroup(): this;
    /**
     * Grants permission to list the distributions associated a Lambda function
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cf-api-permissions-ref.html
     */
    toListDistributionsByLambdaFunction(): this;
    /**
     * Grants permission to list distribution IDs for distributions that have a cache behavior that's associated with the specified origin request policy
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributionsByOriginRequestPolicyId.html
     */
    toListDistributionsByOriginRequestPolicyId(): this;
    /**
     * Grants permission to get a list of distributions that have a cache behavior that's associated with the specified real-time log configuration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributionsByRealtimeLogConfig.html
     */
    toListDistributionsByRealtimeLogConfig(): this;
    /**
     * Grants permission to list distribution IDs for distributions that have a cache behavior that's associated with the specified response headers policy
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributionsByResponseHeadersPolicyId.html
     */
    toListDistributionsByResponseHeadersPolicyId(): this;
    /**
     * Grants permission to list IDs for distributions associated with the specified VPC origin
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributionsByVpcOriginId.html
     */
    toListDistributionsByVpcOriginId(): this;
    /**
     * Grants permission to list the distributions associated with your AWS account with given AWS WAF web ACL
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributionsByWebACLId.html
     */
    toListDistributionsByWebACLId(): this;
    /**
     * Grants permission to list domain conflicts for a specified domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDomainConflicts.html
     */
    toListDomainConflicts(): this;
    /**
     * Grants permission to list all field-level encryption configurations that have been created in CloudFront for this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListFieldLevelEncryptionConfigs.html
     */
    toListFieldLevelEncryptionConfigs(): this;
    /**
     * Grants permission to list all field-level encryption profiles that have been created in CloudFront for this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListFieldLevelEncryptionProfiles.html
     */
    toListFieldLevelEncryptionProfiles(): this;
    /**
     * Grants permission to get a list of CloudFront functions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListFunctions.html
     */
    toListFunctions(): this;
    /**
     * Grants permission to list your invalidation batches
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListInvalidations.html
     */
    toListInvalidations(): this;
    /**
     * Grants permission to list the invalidations for a distribution tenant
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListInvalidationsForDistributionTenant.html
     */
    toListInvalidationsForDistributionTenant(): this;
    /**
     * Grants permission to list all key groups that have been created in CloudFront for this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListKeyGroups.html
     */
    toListKeyGroups(): this;
    /**
     * Grants permission to get a list of CloudFront KeyValueStores
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListKeyValueStores.html
     */
    toListKeyValueStores(): this;
    /**
     * Grants permission to list all origin access controls in the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListOriginAccessControls.html
     */
    toListOriginAccessControls(): this;
    /**
     * Grants permission to list all origin request policies that have been created in CloudFront for this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListOriginRequestPolicies.html
     */
    toListOriginRequestPolicies(): this;
    /**
     * Grants permission to list all public keys that have been added to CloudFront for this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListPublicKeys.html
     */
    toListPublicKeys(): this;
    /**
     * Grants permission to list CloudFront rate cards for the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cf-api-permissions-ref.html
     */
    toListRateCards(): this;
    /**
     * Grants permission to get a list of real-time log configurations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListRealtimeLogConfigs.html
     */
    toListRealtimeLogConfigs(): this;
    /**
     * Grants permission to list all response headers policies that have been created in CloudFront for this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListResponseHeadersPolicies.html
     */
    toListResponseHeadersPolicies(): this;
    /**
     * Grants permission to list savings plans in the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cf-api-permissions-ref.html
     */
    toListSavingsPlans(): this;
    /**
     * Grants permission to list your RTMP distributions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListStreamingDistributions.html
     */
    toListStreamingDistributions(): this;
    /**
     * Grants permission to list tags for a CloudFront resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource(): this;
    /**
     * Grants permission to list CloudFront usage
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cf-api-permissions-ref.html
     */
    toListUsages(): this;
    /**
     * Grants permission to list VPC origins
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListVpcOrigins.html
     */
    toListVpcOrigins(): this;
    /**
     * Grants permission to publish a CloudFront function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_PublishFunction.html
     */
    toPublishFunction(): this;
    /**
     * Grants permission to add tags to a CloudFront resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_TagResource.html
     */
    toTagResource(): this;
    /**
     * Grants permission to test a CloudFront function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_TestFunction.html
     */
    toTestFunction(): this;
    /**
     * Grants permission to remove tags from a CloudFront resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UntagResource.html
     */
    toUntagResource(): this;
    /**
     * Grants permission to update a cache policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateCachePolicy.html
     */
    toUpdateCachePolicy(): this;
    /**
     * Grants permission to set the configuration for a CloudFront origin access identity
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateCloudFrontOriginAccessIdentity.html
     */
    toUpdateCloudFrontOriginAccessIdentity(): this;
    /**
     * Grants permission to update a connection group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateConnectionGroup.html
     */
    toUpdateConnectionGroup(): this;
    /**
     * Grants permission to update a continuous-deployment policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateContinuousDeploymentPolicy.html
     */
    toUpdateContinuousDeploymentPolicy(): this;
    /**
     * Grants permission to update the configuration for a web distribution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateDistribution.html
     */
    toUpdateDistribution(): this;
    /**
     * Grants permission to update a distribution tenant
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateDistributionTenant.html
     */
    toUpdateDistributionTenant(): this;
    /**
     * Grants permission to copy the configuration from a staging web distribution to its corresponding primary web distribution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateDistributionWithStagingConfig.html
     */
    toUpdateDistributionWithStagingConfig(): this;
    /**
     * Grants permission to update a domain association
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateDomainAssociation.html
     */
    toUpdateDomainAssociation(): this;
    /**
     * Grants permission to update a field-level encryption configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateFieldLevelEncryptionConfig.html
     */
    toUpdateFieldLevelEncryptionConfig(): this;
    /**
     * Grants permission to update a field-level encryption profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateFieldLevelEncryptionProfile.html
     */
    toUpdateFieldLevelEncryptionProfile(): this;
    /**
     * Grants permission to update a CloudFront function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateFunction.html
     */
    toUpdateFunction(): this;
    /**
     * Grants permission to update a key group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateKeyGroup.html
     */
    toUpdateKeyGroup(): this;
    /**
     * Grants permission to update a CloudFront KeyValueStore
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateKeyValueStore.html
     */
    toUpdateKeyValueStore(): this;
    /**
     * Grants permission to update an origin access control
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateOriginAccessControl.html
     */
    toUpdateOriginAccessControl(): this;
    /**
     * Grants permission to update an origin request policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateOriginRequestPolicy.html
     */
    toUpdateOriginRequestPolicy(): this;
    /**
     * Grants permission to update public key information
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdatePublicKey.html
     */
    toUpdatePublicKey(): this;
    /**
     * Grants permission to update a real-time log configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateRealtimeLogConfig.html
     */
    toUpdateRealtimeLogConfig(): this;
    /**
     * Grants permission to update a response headers policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateResponseHeadersPolicy.html
     */
    toUpdateResponseHeadersPolicy(): this;
    /**
     * Grants permission to update a savings plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cf-api-permissions-ref.html
     */
    toUpdateSavingsPlan(): this;
    /**
     * Grants permission to update the configuration for an RTMP distribution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateStreamingDistribution.html
     */
    toUpdateStreamingDistribution(): this;
    /**
     * Grants permission to update a VPC origin
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateVpcOrigin.html
     */
    toUpdateVpcOrigin(): this;
    /**
     * Grants permission to verify the DNS configuration for a specified domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_VerifyDnsConfiguration.html
     */
    toVerifyDnsConfiguration(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type distribution to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-working-with.html
     *
     * @param distributionId - Identifier for the distributionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDistribution(distributionId: string, account?: string, partition?: string): this;
    /**
     * Adds a resource of type streaming-distribution to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-working-with.html
     *
     * @param distributionId - Identifier for the distributionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStreamingDistribution(distributionId: string, account?: string, partition?: string): this;
    /**
     * Adds a resource of type origin-access-identity to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html#private-content-restricting-access-to-s3-overview
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onOriginAccessIdentity(id: string, account?: string, partition?: string): this;
    /**
     * Adds a resource of type field-level-encryption-config to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/field-level-encryption.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onFieldLevelEncryptionConfig(id: string, account?: string, partition?: string): this;
    /**
     * Adds a resource of type field-level-encryption-profile to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/field-level-encryption.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onFieldLevelEncryptionProfile(id: string, account?: string, partition?: string): this;
    /**
     * Adds a resource of type cache-policy to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/working-with-policies.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onCachePolicy(id: string, account?: string, partition?: string): this;
    /**
     * Adds a resource of type origin-request-policy to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/working-with-policies.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onOriginRequestPolicy(id: string, account?: string, partition?: string): this;
    /**
     * Adds a resource of type realtime-log-config to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/real-time-logs.html
     *
     * @param name - Identifier for the name.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onRealtimeLogConfig(name: string, account?: string, partition?: string): this;
    /**
     * Adds a resource of type function to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cloudfront-functions.html
     *
     * @param name - Identifier for the name.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onFunction(name: string, account?: string, partition?: string): this;
    /**
     * Adds a resource of type key-value-store to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/kvs-with-functions.html
     *
     * @param name - Identifier for the name.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onKeyValueStore(name: string, account?: string, partition?: string): this;
    /**
     * Adds a resource of type response-headers-policy to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/working-with-policies.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onResponseHeadersPolicy(id: string, account?: string, partition?: string): this;
    /**
     * Adds a resource of type origin-access-control to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/working-with-policies.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onOriginAccessControl(id: string, account?: string, partition?: string): this;
    /**
     * Adds a resource of type continuous-deployment-policy to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/working-with-policies.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onContinuousDeploymentPolicy(id: string, account?: string, partition?: string): this;
    /**
     * Adds a resource of type anycast-ip-list to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/request-static-ips.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAnycastIpList(id: string, account?: string, partition?: string): this;
    /**
     * Adds a resource of type vpcorigin to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-vpc-origins.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVpcorigin(id: string, account?: string, partition?: string): this;
    /**
     * Adds a resource of type distribution-tenant to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/API_DistributionTenant.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDistributionTenant(id: string, account?: string, partition?: string): this;
    /**
     * Adds a resource of type connection-group to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/API_ConnectionGroup.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConnectionGroup(id: string, account?: string, partition?: string): this;
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateAnycastIpList()
     * - .toCreateConnectionGroup()
     * - .toCreateDistributionTenant()
     * - .toCreateStreamingDistributionWithTags()
     * - .toCreateVpcOrigin()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - distribution
     * - streaming-distribution
     * - anycast-ip-list
     * - vpcorigin
     * - distribution-tenant
     * - connection-group
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateAnycastIpList()
     * - .toCreateConnectionGroup()
     * - .toCreateDistributionTenant()
     * - .toCreateStreamingDistributionWithTags()
     * - .toCreateVpcOrigin()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value: string | string[], operator?: Operator | string): this;
    /**
     * Statement provider for service [cloudfront](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncloudfront.html).
     *
     */
    constructor(props?: iam.PolicyStatementProps);
}
