"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cloudshell = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [cloudshell](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscloudshell.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Cloudshell extends shared_1.PolicyStatement {
    /**
     * Grants permission to approve a command sent by another AWS service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudshell/latest/userguide/sec-auth-with-identities.html#ApproveCommand
     */
    toApproveCommand() {
        return this.to('ApproveCommand');
    }
    /**
     * Grants permissions to create a CloudShell environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSecurityGroupIds()
     * - .ifSubnetIds()
     * - .ifVpcIds()
     *
     * https://docs.aws.amazon.com/cloudshell/latest/userguide/sec-auth-with-identities.html#CreateEnvironment
     */
    toCreateEnvironment() {
        return this.to('CreateEnvironment');
    }
    /**
     * Grants permissions to connect to a CloudShell environment from the AWS Management Console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudshell/latest/userguide/sec-auth-with-identities.html#CreateSession
     */
    toCreateSession() {
        return this.to('CreateSession');
    }
    /**
     * Grants permission to delete a CloudShell environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudshell/latest/userguide/sec-auth-with-identities.html#DeleteEnvironment
     */
    toDeleteEnvironment() {
        return this.to('DeleteEnvironment');
    }
    /**
     * Grants permission to return descriptions of existing user's environments
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudshell/latest/userguide/sec-auth-with-identities.html#DescribeEnvironments
     */
    toDescribeEnvironments() {
        return this.to('DescribeEnvironments');
    }
    /**
     * Grants permission to read a CloudShell environment status
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudshell/latest/userguide/sec-auth-with-identities.html#GetEnvironmentStatus
     */
    toGetEnvironmentStatus() {
        return this.to('GetEnvironmentStatus');
    }
    /**
     * Grants permissions to download files from a CloudShell environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudshell/latest/userguide/sec-auth-with-identities.html#GetFileDownloadUrls
     */
    toGetFileDownloadUrls() {
        return this.to('GetFileDownloadUrls');
    }
    /**
     * Grants permissions to upload files to a CloudShell environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudshell/latest/userguide/sec-auth-with-identities.html#GetFileUploadUrls
     */
    toGetFileUploadUrls() {
        return this.to('GetFileUploadUrls');
    }
    /**
     * Grants permissions to forward console credentials to the environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudshell/latest/userguide/sec-auth-with-identities.html#PutCredentials
     */
    toPutCredentials() {
        return this.to('PutCredentials');
    }
    /**
     * Grants permission to start a stopped CloudShell environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudshell/latest/userguide/sec-auth-with-identities.html#StartEnvironment
     */
    toStartEnvironment() {
        return this.to('StartEnvironment');
    }
    /**
     * Grants permission to stop a running CloudShell environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudshell/latest/userguide/sec-auth-with-identities.html#StopEnvironment
     */
    toStopEnvironment() {
        return this.to('StopEnvironment');
    }
    /**
     * Adds a resource of type Environment to the statement
     *
     * https://docs.aws.amazon.com/cloudshell/latest/userguide/sec-auth-with-identities.html#Environment
     *
     * @param environmentId - Identifier for the environmentId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onEnvironment(environmentId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudshell:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:environment/${environmentId}`);
    }
    /**
     * Filters access by security group ids. Available during CreateEnvironment operation
     *
     * https://docs.aws.amazon.com/cloudshell/latest/userguide/aws-cloudshell-vpc-permissions-1.html#vpc-condition-keys-examples-1
     *
     * Applies to actions:
     * - .toCreateEnvironment()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecurityGroupIds(value, operator) {
        return this.if(`SecurityGroupIds`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by subnet ids. Available during CreateEnvironment operation
     *
     * https://docs.aws.amazon.com/cloudshell/latest/userguide/aws-cloudshell-vpc-permissions-1.html#vpc-condition-keys-examples-1
     *
     * Applies to actions:
     * - .toCreateEnvironment()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSubnetIds(value, operator) {
        return this.if(`SubnetIds`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by vpc ids. Available during CreateEnvironment operation
     *
     * https://docs.aws.amazon.com/cloudshell/latest/userguide/aws-cloudshell-vpc-permissions-1.html#vpc-condition-keys-examples-1
     *
     * Applies to actions:
     * - .toCreateEnvironment()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpcIds(value, operator) {
        return this.if(`VpcIds`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [cloudshell](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscloudshell.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'cloudshell';
        this.accessLevelList = {
            Read: [
                'ApproveCommand',
                'GetEnvironmentStatus'
            ],
            Write: [
                'CreateEnvironment',
                'CreateSession',
                'DeleteEnvironment',
                'GetFileDownloadUrls',
                'GetFileUploadUrls',
                'PutCredentials',
                'StartEnvironment',
                'StopEnvironment'
            ],
            List: [
                'DescribeEnvironments'
            ]
        };
    }
}
exports.Cloudshell = Cloudshell;
//# sourceMappingURL=data:application/json;base64,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