"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cloudtrail = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [cloudtrail](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscloudtrail.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Cloudtrail extends shared_1.PolicyStatement {
    /**
     * Grants permission to add one or more tags to a trail, event data store, channel or dashboard, up to a limit of 50
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_AddTags.html
     */
    toAddTags() {
        return this.to('AddTags');
    }
    /**
     * Grants permission to cancel a running query
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_CancelQuery.html
     */
    toCancelQuery() {
        return this.to('CancelQuery');
    }
    /**
     * Grants permission to create a channel
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - cloudtrail:AddTags
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_CreateChannel.html
     */
    toCreateChannel() {
        return this.to('CreateChannel');
    }
    /**
     * Grants permission to create a dashboard
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - cloudtrail:AddTags
     * - cloudtrail:StartDashboardRefresh
     * - cloudtrail:StartQuery
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_CreateDashboard.html
     */
    toCreateDashboard() {
        return this.to('CreateDashboard');
    }
    /**
     * Grants permission to create an event data store
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - cloudtrail:AddTags
     * - iam:CreateServiceLinkedRole
     * - iam:GetRole
     * - kms:Decrypt
     * - kms:GenerateDataKey
     * - organizations:ListAWSServiceAccessForOrganization
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_CreateEventDataStore.html
     */
    toCreateEventDataStore() {
        return this.to('CreateEventDataStore');
    }
    /**
     * Grants permission to create a service-linked channel that specifies the settings for delivery of log data to an AWS service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/userguide/viewing-service-linked-channels.html#slc-service-events
     */
    toCreateServiceLinkedChannel() {
        return this.to('CreateServiceLinkedChannel');
    }
    /**
     * Grants permission to create a trail that specifies the settings for delivery of log data to an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - cloudtrail:AddTags
     * - iam:CreateServiceLinkedRole
     * - iam:GetRole
     * - organizations:ListAWSServiceAccessForOrganization
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_CreateTrail.html
     */
    toCreateTrail() {
        return this.to('CreateTrail');
    }
    /**
     * Grants permission to delete a channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_DeleteChannel.html
     */
    toDeleteChannel() {
        return this.to('DeleteChannel');
    }
    /**
     * Grants permission to delete a dashboard
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_DeleteDashboard.html
     */
    toDeleteDashboard() {
        return this.to('DeleteDashboard');
    }
    /**
     * Grants permission to delete an event data store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_DeleteEventDataStore.html
     */
    toDeleteEventDataStore() {
        return this.to('DeleteEventDataStore');
    }
    /**
     * Grants permission to delete a resource policy from the provided resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to delete a service-linked channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/userguide/viewing-service-linked-channels.html#slc-service-events
     */
    toDeleteServiceLinkedChannel() {
        return this.to('DeleteServiceLinkedChannel');
    }
    /**
     * Grants permission to delete a trail
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_DeleteTrail.html
     */
    toDeleteTrail() {
        return this.to('DeleteTrail');
    }
    /**
     * Grants permission to deregister an AWS Organizations member account as a delegated administrator
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - organizations:DeregisterDelegatedAdministrator
     * - organizations:ListAWSServiceAccessForOrganization
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_DeregisterOrganizationDelegatedAdmin.html
     */
    toDeregisterOrganizationDelegatedAdmin() {
        return this.to('DeregisterOrganizationDelegatedAdmin');
    }
    /**
     * Grants permission to list details for the query
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_DescribeQuery.html
     */
    toDescribeQuery() {
        return this.to('DescribeQuery');
    }
    /**
     * Grants permission to list settings for the trails associated with the current region for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_DescribeTrails.html
     */
    toDescribeTrails() {
        return this.to('DescribeTrails');
    }
    /**
     * Grants permission to disable federation of event data store data by using the AWS Glue Data Catalog
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - glue:DeleteDatabase
     * - glue:DeleteTable
     * - glue:PassConnection
     * - lakeformation:DeregisterResource
     * - lakeformation:RegisterResource
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_DisableFederation.html
     */
    toDisableFederation() {
        return this.to('DisableFederation');
    }
    /**
     * Grants permission to enable federation of event data store data by using the AWS Glue Data Catalog
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - glue:CreateDatabase
     * - glue:CreateTable
     * - iam:GetRole
     * - iam:PassRole
     * - lakeformation:DeregisterResource
     * - lakeformation:RegisterResource
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_EnableFederation.html
     */
    toEnableFederation() {
        return this.to('EnableFederation');
    }
    /**
     * Grants permission to generate a query for a specified event data store using the CloudTrail Lake query generator
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/userguide/lake-query-generator.html
     */
    toGenerateQuery() {
        return this.to('GenerateQuery');
    }
    /**
     * Grants permission to generate a results summary for specified queries using the CloudTrail natural language generator
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - cloudtrail:GetQueryResults
     * - kms:Decrypt
     * - kms:GenerateDataKey
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/userguide/query-results-summary.html
     */
    toGenerateQueryResultsSummary() {
        return this.to('GenerateQueryResultsSummary');
    }
    /**
     * Grants permission to return information about a specific channel
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_GetChannel.html
     */
    toGetChannel() {
        return this.to('GetChannel');
    }
    /**
     * Grants permission to list settings for the dashboard
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_GetDashboard.html
     */
    toGetDashboard() {
        return this.to('GetDashboard');
    }
    /**
     * Grants permission to list settings for the event data store
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_GetEventDataStore.html
     */
    toGetEventDataStore() {
        return this.to('GetEventDataStore');
    }
    /**
     * Grants permission to get data from an event data store by using the AWS Glue Data Catalog
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kms:Decrypt
     * - kms:GenerateDataKey
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/userguide/query-federation.html#query-federation-permissions
     */
    toGetEventDataStoreData() {
        return this.to('GetEventDataStoreData');
    }
    /**
     * Grants permission to list settings for event selectors configured for a trail
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_GetEventSelectors.html
     */
    toGetEventSelectors() {
        return this.to('GetEventSelectors');
    }
    /**
     * Grants permission to return information about a specific import
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_GetImport.html
     */
    toGetImport() {
        return this.to('GetImport');
    }
    /**
     * Grants permission to list CloudTrail Insights selectors that are configured for a trail or event data store
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_GetInsightSelectors.html
     */
    toGetInsightSelectors() {
        return this.to('GetInsightSelectors');
    }
    /**
     * Grants permission to fetch results of a complete query
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kms:Decrypt
     * - kms:GenerateDataKey
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_GetQueryResults.html
     */
    toGetQueryResults() {
        return this.to('GetQueryResults');
    }
    /**
     * Grants permission to get the resource policy attached to the provided resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_GetResourcePolicy.html
     */
    toGetResourcePolicy() {
        return this.to('GetResourcePolicy');
    }
    /**
     * Grants permission to list settings for the service-linked channel
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/userguide/viewing-service-linked-channels.html#slc-service-events
     */
    toGetServiceLinkedChannel() {
        return this.to('GetServiceLinkedChannel');
    }
    /**
     * Grants permission to list settings for the trail
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_GetTrail.html
     */
    toGetTrail() {
        return this.to('GetTrail');
    }
    /**
     * Grants permission to retrieve a JSON-formatted list of information about the specified trail
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_GetTrailStatus.html
     */
    toGetTrailStatus() {
        return this.to('GetTrailStatus');
    }
    /**
     * Grants permission to list the channels in the current account, and their source names
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_ListChannels.html
     */
    toListChannels() {
        return this.to('ListChannels');
    }
    /**
     * Grants permission to list dashboards associated with the current region for your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_ListDashboards.html
     */
    toListDashboards() {
        return this.to('ListDashboards');
    }
    /**
     * Grants permission to list event data stores associated with the current region for your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_ListEventDataStores.html
     */
    toListEventDataStores() {
        return this.to('ListEventDataStores');
    }
    /**
     * Grants permission to return a list of failures for the specified import
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_ListImportFailures.html
     */
    toListImportFailures() {
        return this.to('ListImportFailures');
    }
    /**
     * Grants permission to return information on all imports, or a select set of imports by ImportStatus or Destination
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_ListImports.html
     */
    toListImports() {
        return this.to('ListImports');
    }
    /**
     * Grants permission to list the public keys whose private keys were used to sign trail digest files within a specified time range
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_ListPublicKeys.html
     */
    toListPublicKeys() {
        return this.to('ListPublicKeys');
    }
    /**
     * Grants permission to list queries associated with an event data store
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_ListQueries.html
     */
    toListQueries() {
        return this.to('ListQueries');
    }
    /**
     * Grants permission to list service-linked channels associated with the current region for a specified account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/userguide/viewing-service-linked-channels.html#slc-service-events
     */
    toListServiceLinkedChannels() {
        return this.to('ListServiceLinkedChannels');
    }
    /**
     * Grants permission to list the tags for trails, event data stores, channels or dashboards in the current region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_ListTags.html
     */
    toListTags() {
        return this.to('ListTags');
    }
    /**
     * Grants permission to list trails associated with the current region for your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_ListTrails.html
     */
    toListTrails() {
        return this.to('ListTrails');
    }
    /**
     * Grants permission to look up and retrieve metric data for API activity events captured by CloudTrail that create, update, or delete resources in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_LookupEvents.html
     */
    toLookupEvents() {
        return this.to('LookupEvents');
    }
    /**
     * Grants permission to create and update event selectors for a trail
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_PutEventSelectors.html
     */
    toPutEventSelectors() {
        return this.to('PutEventSelectors');
    }
    /**
     * Grants permission to create and update CloudTrail Insights selectors for a trail or event data store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_PutInsightSelectors.html
     */
    toPutInsightSelectors() {
        return this.to('PutInsightSelectors');
    }
    /**
     * Grants permission to attach a resource policy to the provided resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_PutResourcePolicy.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to register an AWS Organizations member account as a delegated administrator
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - iam:GetRole
     * - organizations:ListAWSServiceAccessForOrganization
     * - organizations:RegisterDelegatedAdministrator
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_RegisterOrganizationDelegatedAdmin.html
     */
    toRegisterOrganizationDelegatedAdmin() {
        return this.to('RegisterOrganizationDelegatedAdmin');
    }
    /**
     * Grants permission to remove tags from a trail, event data store, channel or dashboard
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_RemoveTags.html
     */
    toRemoveTags() {
        return this.to('RemoveTags');
    }
    /**
     * Grants permission to restore an event data store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_RestoreEventDataStore.html
     */
    toRestoreEventDataStore() {
        return this.to('RestoreEventDataStore');
    }
    /**
     * Grants permission to perform semantic search for CloudTrail Lake sample queries
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/userguide/lake-console-queries.html
     */
    toSearchSampleQueries() {
        return this.to('SearchSampleQueries');
    }
    /**
     * Grants permission to start a refresh on the specified dashboard
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - cloudtrail:StartQuery
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_StartDashboardRefresh.html
     */
    toStartDashboardRefresh() {
        return this.to('StartDashboardRefresh');
    }
    /**
     * Grants permission to start ingestion on an event data store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_StartEventDataStoreIngestion.html
     */
    toStartEventDataStoreIngestion() {
        return this.to('StartEventDataStoreIngestion');
    }
    /**
     * Grants permission to start an import of logged trail events from a source S3 bucket to a destination event data store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_StartImport.html
     */
    toStartImport() {
        return this.to('StartImport');
    }
    /**
     * Grants permission to start the recording of AWS API calls and log file delivery for a trail
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_StartLogging.html
     */
    toStartLogging() {
        return this.to('StartLogging');
    }
    /**
     * Grants permission to start a new query on a specified event data store
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     * - kms:GenerateDataKey
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_StartQuery.html
     */
    toStartQuery() {
        return this.to('StartQuery');
    }
    /**
     * Grants permission to stop ingestion on an event data store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_StopEventDataStoreIngestion.html
     */
    toStopEventDataStoreIngestion() {
        return this.to('StopEventDataStoreIngestion');
    }
    /**
     * Grants permission to stop a specified import
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_StopImport.html
     */
    toStopImport() {
        return this.to('StopImport');
    }
    /**
     * Grants permission to stop the recording of AWS API calls and log file delivery for a trail
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_StopLogging.html
     */
    toStopLogging() {
        return this.to('StopLogging');
    }
    /**
     * Grants permission to update a channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_UpdateChannel.html
     */
    toUpdateChannel() {
        return this.to('UpdateChannel');
    }
    /**
     * Grants permission to update a dashboard
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - cloudtrail:StartDashboardRefresh
     * - cloudtrail:StartQuery
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_UpdateDashboard.html
     */
    toUpdateDashboard() {
        return this.to('UpdateDashboard');
    }
    /**
     * Grants permission to update an event data store
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - iam:GetRole
     * - kms:Decrypt
     * - kms:GenerateDataKey
     * - organizations:ListAWSServiceAccessForOrganization
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_UpdateEventDataStore.html
     */
    toUpdateEventDataStore() {
        return this.to('UpdateEventDataStore');
    }
    /**
     * Grants permission to update the service-linked channel settings for delivery of log data to an AWS service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/userguide/viewing-service-linked-channels.html#slc-service-events
     */
    toUpdateServiceLinkedChannel() {
        return this.to('UpdateServiceLinkedChannel');
    }
    /**
     * Grants permission to update the settings that specify delivery of log files
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - iam:GetRole
     * - organizations:ListAWSServiceAccessForOrganization
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_UpdateTrail.html
     */
    toUpdateTrail() {
        return this.to('UpdateTrail');
    }
    /**
     * Adds a resource of type trail to the statement
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/userguide/how-cloudtrail-works.html#how-cloudtrail-works-trails
     *
     * @param trailName - Identifier for the trailName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTrail(trailName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudtrail:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:trail/${trailName}`);
    }
    /**
     * Adds a resource of type eventdatastore to the statement
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/userguide/how-cloudtrail-works.html#how-cloudtrail-works-lake
     *
     * @param eventDataStoreId - Identifier for the eventDataStoreId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEventdatastore(eventDataStoreId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudtrail:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:eventdatastore/${eventDataStoreId}`);
    }
    /**
     * Adds a resource of type channel to the statement
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/userguide/how-cloudtrail-works.html#how-cloudtrail-works-channels
     *
     * @param channelId - Identifier for the channelId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onChannel(channelId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudtrail:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:channel/${channelId}`);
    }
    /**
     * Adds a resource of type dashboard to the statement
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/userguide/lake-dashboard.html
     *
     * @param dashboardName - Identifier for the dashboardName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDashboard(dashboardName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudtrail:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:dashboard/${dashboardName}`);
    }
    /**
     * Filters access by the tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toAddTags()
     * - .toCreateChannel()
     * - .toCreateDashboard()
     * - .toCreateEventDataStore()
     * - .toCreateTrail()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - trail
     * - eventdatastore
     * - channel
     * - dashboard
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys in a request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toAddTags()
     * - .toCreateChannel()
     * - .toCreateDashboard()
     * - .toCreateEventDataStore()
     * - .toCreateTrail()
     * - .toRemoveTags()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [cloudtrail](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscloudtrail.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'cloudtrail';
        this.accessLevelList = {
            Tagging: [
                'AddTags',
                'RemoveTags'
            ],
            Write: [
                'CancelQuery',
                'CreateChannel',
                'CreateDashboard',
                'CreateEventDataStore',
                'CreateServiceLinkedChannel',
                'CreateTrail',
                'DeleteChannel',
                'DeleteDashboard',
                'DeleteEventDataStore',
                'DeleteResourcePolicy',
                'DeleteServiceLinkedChannel',
                'DeleteTrail',
                'DeregisterOrganizationDelegatedAdmin',
                'DisableFederation',
                'EnableFederation',
                'GenerateQuery',
                'PutEventSelectors',
                'PutInsightSelectors',
                'PutResourcePolicy',
                'RegisterOrganizationDelegatedAdmin',
                'RestoreEventDataStore',
                'StartDashboardRefresh',
                'StartEventDataStoreIngestion',
                'StartImport',
                'StartLogging',
                'StartQuery',
                'StopEventDataStoreIngestion',
                'StopImport',
                'StopLogging',
                'UpdateChannel',
                'UpdateDashboard',
                'UpdateEventDataStore',
                'UpdateServiceLinkedChannel',
                'UpdateTrail'
            ],
            Read: [
                'DescribeQuery',
                'DescribeTrails',
                'GenerateQueryResultsSummary',
                'GetChannel',
                'GetDashboard',
                'GetEventDataStore',
                'GetEventDataStoreData',
                'GetEventSelectors',
                'GetImport',
                'GetInsightSelectors',
                'GetQueryResults',
                'GetResourcePolicy',
                'GetServiceLinkedChannel',
                'GetTrail',
                'GetTrailStatus',
                'ListImportFailures',
                'ListPublicKeys',
                'ListTags',
                'LookupEvents',
                'SearchSampleQueries'
            ],
            List: [
                'ListChannels',
                'ListDashboards',
                'ListEventDataStores',
                'ListImports',
                'ListQueries',
                'ListServiceLinkedChannels',
                'ListTrails'
            ]
        };
    }
}
exports.Cloudtrail = Cloudtrail;
//# sourceMappingURL=data:application/json;base64,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