"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Rum = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [rum](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscloudwatchrum.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Rum extends shared_1.PolicyStatement {
    /**
     * Grants permission to create rum metric definitions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_BatchCreateRumMetricDefinitions.html
     */
    toBatchCreateRumMetricDefinitions() {
        return this.to('BatchCreateRumMetricDefinitions');
    }
    /**
     * Grants permission to remove rum metric definitions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_BatchDeleteRumMetricDefinitions.html
     */
    toBatchDeleteRumMetricDefinitions() {
        return this.to('BatchDeleteRumMetricDefinitions');
    }
    /**
     * Grants permission to get rum metric definitions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_BatchGetRumMetricDefinitions.html
     */
    toBatchGetRumMetricDefinitions() {
        return this.to('BatchGetRumMetricDefinitions');
    }
    /**
     * Grants permission to create appMonitor metadata
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - iam:GetRole
     *
     * https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_CreateAppMonitor.html
     */
    toCreateAppMonitor() {
        return this.to('CreateAppMonitor');
    }
    /**
     * Grants permission to delete appMonitor metadata
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_DeleteAppMonitor.html
     */
    toDeleteAppMonitor() {
        return this.to('DeleteAppMonitor');
    }
    /**
     * Grants permission to delete a resource policy attached to an app monitor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to delete rum metrics destinations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_DeleteRumMetricsDestination.html
     */
    toDeleteRumMetricsDestination() {
        return this.to('DeleteRumMetricsDestination');
    }
    /**
     * Grants permission to get appMonitor metadata
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_GetAppMonitor.html
     */
    toGetAppMonitor() {
        return this.to('GetAppMonitor');
    }
    /**
     * Grants permission to get appMonitor data
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_GetAppMonitorData.html
     */
    toGetAppMonitorData() {
        return this.to('GetAppMonitorData');
    }
    /**
     * Grants permission to retrieve a resource policy attached to an app monitor
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_GetResourcePolicy.html
     */
    toGetResourcePolicy() {
        return this.to('GetResourcePolicy');
    }
    /**
     * Grants permission to list appMonitors metadata
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_ListAppMonitors.html
     */
    toListAppMonitors() {
        return this.to('ListAppMonitors');
    }
    /**
     * Grants permission to list rum metrics destinations
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_ListRumMetricsDestinations.html
     */
    toListRumMetricsDestinations() {
        return this.to('ListRumMetricsDestinations');
    }
    /**
     * Grants permission to list tags for resources
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to attach a resource policy to an app monitor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_PutResourcePolicy.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to put RUM events for appmonitor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_PutRumEvents.html
     */
    toPutRumEvents() {
        return this.to('PutRumEvents');
    }
    /**
     * Grants permission to put rum metrics destinations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_PutRumMetricsDestination.html
     */
    toPutRumMetricsDestination() {
        return this.to('PutRumMetricsDestination');
    }
    /**
     * Grants permission to tag resources
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag resources
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update appmonitor metadata
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - iam:GetRole
     *
     * https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_UpdateAppMonitor.html
     */
    toUpdateAppMonitor() {
        return this.to('UpdateAppMonitor');
    }
    /**
     * Grants permission to update rum metric definition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_UpdateRumMetricDefinition.html
     */
    toUpdateRumMetricDefinition() {
        return this.to('UpdateRumMetricDefinition');
    }
    /**
     * Adds a resource of type AppMonitorResource to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/API_AppMonitor.html
     *
     * @param name - Identifier for the name.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAppMonitorResource(name, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:rum:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:appmonitor/${name}`);
    }
    /**
     * Filters access by the tags that are passed the request on behalf of the IAM principal
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateAppMonitor()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource that make the request on behalf of the IAM principal
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html
     *
     * Applies to resource types:
     * - AppMonitorResource
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request on behalf of the IAM principal
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateAppMonitor()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [rum](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscloudwatchrum.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'rum';
        this.accessLevelList = {
            Write: [
                'BatchCreateRumMetricDefinitions',
                'BatchDeleteRumMetricDefinitions',
                'CreateAppMonitor',
                'DeleteAppMonitor',
                'DeleteResourcePolicy',
                'DeleteRumMetricsDestination',
                'PutResourcePolicy',
                'PutRumEvents',
                'PutRumMetricsDestination',
                'UpdateAppMonitor',
                'UpdateRumMetricDefinition'
            ],
            Read: [
                'BatchGetRumMetricDefinitions',
                'GetAppMonitor',
                'GetAppMonitorData',
                'GetResourcePolicy',
                'ListRumMetricsDestinations',
                'ListTagsForResource'
            ],
            List: [
                'ListAppMonitors'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Rum = Rum;
//# sourceMappingURL=data:application/json;base64,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