"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Codecatalyst = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [codecatalyst](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncodecatalyst.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Codecatalyst extends shared_1.PolicyStatement {
    /**
     * Grants permission to accept a request to connect this account to an Amazon CodeCatalyst space
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codecatalyst/latest/userguide/security-iam.html#permissions-reference-connections
     */
    toAcceptConnection() {
        return this.to('AcceptConnection');
    }
    /**
     * Grants permission to associate an IAM role to a connection
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/codecatalyst/latest/userguide/security-iam.html#permissions-reference-connections
     */
    toAssociateIamRoleToConnection() {
        return this.to('AssociateIamRoleToConnection');
    }
    /**
     * Grants permission to associate an IAM Identity Center application with an Amazon CodeCatalyst space
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/codecatalyst/latest/userguide/security-iam.html#permissions-reference-applications
     */
    toAssociateIdentityCenterApplicationToSpace() {
        return this.to('AssociateIdentityCenterApplicationToSpace');
    }
    /**
     * Grants permission to associate an identity with an IAM Identity Center application for an Amazon CodeCatalyst space
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/codecatalyst/latest/userguide/security-iam.html#permissions-reference-applications
     */
    toAssociateIdentityToIdentityCenterApplication() {
        return this.to('AssociateIdentityToIdentityCenterApplication');
    }
    /**
     * Grants permission to associate multiple identities with an IAM Identity Center application for an Amazon CodeCatalyst space
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/codecatalyst/latest/userguide/security-iam.html#permissions-reference-applications
     */
    toBatchAssociateIdentitiesToIdentityCenterApplication() {
        return this.to('BatchAssociateIdentitiesToIdentityCenterApplication');
    }
    /**
     * Grants permission to disassociate multiple identities from an IAM Identity Center application for an Amazon CodeCatalyst space
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/codecatalyst/latest/userguide/security-iam.html#permissions-reference-applications
     */
    toBatchDisassociateIdentitiesFromIdentityCenterApplication() {
        return this.to('BatchDisassociateIdentitiesFromIdentityCenterApplication');
    }
    /**
     * Grants permission to create an IAM Identity Center application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codecatalyst/latest/userguide/security-iam.html#permissions-reference-applications
     */
    toCreateIdentityCenterApplication() {
        return this.to('CreateIdentityCenterApplication');
    }
    /**
     * Grants permission to create an Amazon CodeCatalyst space
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codecatalyst/latest/userguide/security-iam.html#permissions-reference-applications
     */
    toCreateSpace() {
        return this.to('CreateSpace');
    }
    /**
     * Grants permission to create an administrator role assignment for a given Amazon CodeCatalyst space and IAM Identity Center application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/codecatalyst/latest/userguide/security-iam.html#permissions-reference-applications
     */
    toCreateSpaceAdminRoleAssignment() {
        return this.to('CreateSpaceAdminRoleAssignment');
    }
    /**
     * Grants permission to delete a connection
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/codecatalyst/latest/userguide/security-iam.html#permissions-reference-connections
     */
    toDeleteConnection() {
        return this.to('DeleteConnection');
    }
    /**
     * Grants permission to delete an IAM Identity Center application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/codecatalyst/latest/userguide/security-iam.html#permissions-reference-applications
     */
    toDeleteIdentityCenterApplication() {
        return this.to('DeleteIdentityCenterApplication');
    }
    /**
     * Grants permission to disassociate an IAM role from a connection
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/codecatalyst/latest/userguide/security-iam.html#permissions-reference-connections
     */
    toDisassociateIamRoleFromConnection() {
        return this.to('DisassociateIamRoleFromConnection');
    }
    /**
     * Grants permission to disassociate an IAM Identity Center application from an Amazon CodeCatalyst space
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/codecatalyst/latest/userguide/security-iam.html#permissions-reference-applications
     */
    toDisassociateIdentityCenterApplicationFromSpace() {
        return this.to('DisassociateIdentityCenterApplicationFromSpace');
    }
    /**
     * Grants permission to disassociate an identity from an IAM Identity Center application for an Amazon CodeCatalyst space
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/codecatalyst/latest/userguide/security-iam.html#permissions-reference-applications
     */
    toDisassociateIdentityFromIdentityCenterApplication() {
        return this.to('DisassociateIdentityFromIdentityCenterApplication');
    }
    /**
     * Grants permission to describe the billing authorization for a connection
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/codecatalyst/latest/userguide/security-iam.html#permissions-reference-connections
     */
    toGetBillingAuthorization() {
        return this.to('GetBillingAuthorization');
    }
    /**
     * Grants permission to get a connection
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/codecatalyst/latest/userguide/security-iam.html#permissions-reference-connections
     */
    toGetConnection() {
        return this.to('GetConnection');
    }
    /**
     * Grants permission to get information about an IAM Identity Center application
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/codecatalyst/latest/userguide/security-iam.html#permissions-reference-applications
     */
    toGetIdentityCenterApplication() {
        return this.to('GetIdentityCenterApplication');
    }
    /**
     * Grants permission to get a pending request to connect this account to an Amazon CodeCatalyst space
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecatalyst/latest/userguide/security-iam.html#permissions-reference-connections
     */
    toGetPendingConnection() {
        return this.to('GetPendingConnection');
    }
    /**
     * Grants permission to list connections that are not pending
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codecatalyst/latest/userguide/security-iam.html#permissions-reference-connections
     */
    toListConnections() {
        return this.to('ListConnections');
    }
    /**
     * Grants permission to list IAM roles associated with a connection
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/codecatalyst/latest/userguide/security-iam.html#permissions-reference-connections
     */
    toListIamRolesForConnection() {
        return this.to('ListIamRolesForConnection');
    }
    /**
     * Grants permission to view a list of all IAM Identity Center applications in the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codecatalyst/latest/userguide/security-iam.html#permissions-reference-applications
     */
    toListIdentityCenterApplications() {
        return this.to('ListIdentityCenterApplications');
    }
    /**
     * Grants permission to view a list of IAM Identity Center applications by Amazon CodeCatalyst space
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codecatalyst/latest/userguide/security-iam.html#permissions-reference-applications
     */
    toListIdentityCenterApplicationsForSpace() {
        return this.to('ListIdentityCenterApplicationsForSpace');
    }
    /**
     * Grants permission to view a list of Amazon CodeCatalyst spaces by IAM Identity Center application
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/codecatalyst/latest/userguide/security-iam.html#permissions-reference-applications
     */
    toListSpacesForIdentityCenterApplication() {
        return this.to('ListSpacesForIdentityCenterApplication');
    }
    /**
     * Grants permission to list tags for an Amazon CodeCatalyst resource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/codecatalyst/latest/userguide/security-iam.html#permissions-reference-connections
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to create or update the billing authorization for a connection
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/codecatalyst/latest/userguide/security-iam.html#permissions-reference-connections
     */
    toPutBillingAuthorization() {
        return this.to('PutBillingAuthorization');
    }
    /**
     * Grants permission to reject a request to connect this account to an Amazon CodeCatalyst space
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecatalyst/latest/userguide/security-iam.html#permissions-reference-connections
     */
    toRejectConnection() {
        return this.to('RejectConnection');
    }
    /**
     * Grants permission to synchronize an IAM Identity Center application with the backing identity store
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/codecatalyst/latest/userguide/security-iam.html#permissions-reference-applications
     */
    toSynchronizeIdentityCenterApplication() {
        return this.to('SynchronizeIdentityCenterApplication');
    }
    /**
     * Grants permission to tag an Amazon CodeCatalyst resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/codecatalyst/latest/userguide/security-iam.html#permissions-reference-connections
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag an Amazon CodeCatalyst resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/codecatalyst/latest/userguide/security-iam.html#permissions-reference-connections
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an IAM Identity Center application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/codecatalyst/latest/userguide/security-iam.html#permissions-reference-applications
     */
    toUpdateIdentityCenterApplication() {
        return this.to('UpdateIdentityCenterApplication');
    }
    /**
     * Adds a resource of type connections to the statement
     *
     * https://docs.aws.amazon.com/codecatalyst/latest/userguide/#
     *
     * @param connectionId - Identifier for the connectionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConnections(connectionId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codecatalyst:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:/connections/${connectionId}`);
    }
    /**
     * Adds a resource of type identity-center-applications to the statement
     *
     * https://docs.aws.amazon.com/codecatalyst/latest/userguide/#
     *
     * @param identityCenterApplicationId - Identifier for the identityCenterApplicationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIdentityCenterApplications(identityCenterApplicationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codecatalyst:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:/identity-center-applications/${identityCenterApplicationId}`);
    }
    /**
     * Adds a resource of type space to the statement
     *
     * https://docs.aws.amazon.com/codecatalyst/latest/userguide/#
     *
     * @param spaceId - Identifier for the spaceId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onSpace(spaceId, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codecatalyst:::space/${spaceId}`);
    }
    /**
     * Adds a resource of type project to the statement
     *
     * https://docs.aws.amazon.com/codecatalyst/latest/userguide/#
     *
     * @param spaceId - Identifier for the spaceId.
     * @param projectId - Identifier for the projectId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onProject(spaceId, projectId, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codecatalyst:::space/${spaceId}/project/${projectId}`);
    }
    /**
     * Filters access by a tag's key and value in a request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toAcceptConnection()
     * - .toCreateIdentityCenterApplication()
     * - .toCreateSpace()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toAssociateIamRoleToConnection()
     * - .toAssociateIdentityCenterApplicationToSpace()
     * - .toAssociateIdentityToIdentityCenterApplication()
     * - .toBatchAssociateIdentitiesToIdentityCenterApplication()
     * - .toBatchDisassociateIdentitiesFromIdentityCenterApplication()
     * - .toCreateSpaceAdminRoleAssignment()
     * - .toDeleteConnection()
     * - .toDeleteIdentityCenterApplication()
     * - .toDisassociateIamRoleFromConnection()
     * - .toDisassociateIdentityCenterApplicationFromSpace()
     * - .toDisassociateIdentityFromIdentityCenterApplication()
     * - .toGetBillingAuthorization()
     * - .toGetConnection()
     * - .toGetIdentityCenterApplication()
     * - .toListIamRolesForConnection()
     * - .toListSpacesForIdentityCenterApplication()
     * - .toListTagsForResource()
     * - .toPutBillingAuthorization()
     * - .toSynchronizeIdentityCenterApplication()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateIdentityCenterApplication()
     *
     * Applies to resource types:
     * - connections
     * - identity-center-applications
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys in a request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toAcceptConnection()
     * - .toCreateIdentityCenterApplication()
     * - .toCreateSpace()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [codecatalyst](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncodecatalyst.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'codecatalyst';
        this.accessLevelList = {
            Write: [
                'AcceptConnection',
                'AssociateIamRoleToConnection',
                'AssociateIdentityCenterApplicationToSpace',
                'AssociateIdentityToIdentityCenterApplication',
                'BatchAssociateIdentitiesToIdentityCenterApplication',
                'BatchDisassociateIdentitiesFromIdentityCenterApplication',
                'CreateIdentityCenterApplication',
                'CreateSpace',
                'CreateSpaceAdminRoleAssignment',
                'DeleteConnection',
                'DeleteIdentityCenterApplication',
                'DisassociateIamRoleFromConnection',
                'DisassociateIdentityCenterApplicationFromSpace',
                'DisassociateIdentityFromIdentityCenterApplication',
                'PutBillingAuthorization',
                'RejectConnection',
                'SynchronizeIdentityCenterApplication',
                'UpdateIdentityCenterApplication'
            ],
            Read: [
                'GetBillingAuthorization',
                'GetConnection',
                'GetIdentityCenterApplication',
                'GetPendingConnection',
                'ListTagsForResource'
            ],
            List: [
                'ListConnections',
                'ListIamRolesForConnection',
                'ListIdentityCenterApplications',
                'ListIdentityCenterApplicationsForSpace',
                'ListSpacesForIdentityCenterApplication'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Codecatalyst = Codecatalyst;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29kZWNhdGFseXN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiY29kZWNhdGFseXN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUNBLHlDQUF5RDtBQUd6RDs7OztHQUlHO0FBQ0gsTUFBYSxZQUFhLFNBQVEsd0JBQWU7SUFHL0M7Ozs7Ozs7Ozs7T0FVRztJQUNJLGtCQUFrQjtRQUN2QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0JBQWtCLENBQUMsQ0FBQztJQUNyQyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksOEJBQThCO1FBQ25DLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO0lBQ2pELENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSwyQ0FBMkM7UUFDaEQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDJDQUEyQyxDQUFDLENBQUM7SUFDOUQsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLDhDQUE4QztRQUNuRCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsOENBQThDLENBQUMsQ0FBQztJQUNqRSxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0kscURBQXFEO1FBQzFELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxREFBcUQsQ0FBQyxDQUFDO0lBQ3hFLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSwwREFBMEQ7UUFDL0QsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDBEQUEwRCxDQUFDLENBQUM7SUFDN0UsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxpQ0FBaUM7UUFDdEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGlDQUFpQyxDQUFDLENBQUM7SUFDcEQsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxhQUFhO1FBQ2xCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUNoQyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksZ0NBQWdDO1FBQ3JDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO0lBQ25ELENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxrQkFBa0I7UUFDdkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFrQixDQUFDLENBQUM7SUFDckMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLGlDQUFpQztRQUN0QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUNBQWlDLENBQUMsQ0FBQztJQUNwRCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksbUNBQW1DO1FBQ3hDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQ0FBbUMsQ0FBQyxDQUFDO0lBQ3RELENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxnREFBZ0Q7UUFDckQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdEQUFnRCxDQUFDLENBQUM7SUFDbkUsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLG1EQUFtRDtRQUN4RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbURBQW1ELENBQUMsQ0FBQztJQUN0RSxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0kseUJBQXlCO1FBQzlCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO0lBQzVDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxlQUFlO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksOEJBQThCO1FBQ25DLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO0lBQ2pELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxzQkFBc0I7UUFDM0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUM7SUFDekMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUJBQWlCLENBQUMsQ0FBQztJQUNwQyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksMkJBQTJCO1FBQ2hDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO0lBQzlDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQ0FBZ0M7UUFDckMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdDQUFnQyxDQUFDLENBQUM7SUFDbkQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdDQUF3QztRQUM3QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0NBQXdDLENBQUMsQ0FBQztJQUMzRCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksd0NBQXdDO1FBQzdDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx3Q0FBd0MsQ0FBQyxDQUFDO0lBQzNELENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLHlCQUF5QjtRQUM5QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMseUJBQXlCLENBQUMsQ0FBQztJQUM1QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxzQ0FBc0M7UUFDM0MsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHNDQUFzQyxDQUFDLENBQUM7SUFDekQsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksYUFBYTtRQUNsQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsYUFBYSxDQUFDLENBQUM7SUFDaEMsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxlQUFlO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksaUNBQWlDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxpQ0FBaUMsQ0FBQyxDQUFDO0lBQ3BELENBQUM7SUEyQ0Q7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksYUFBYSxDQUFDLFlBQW9CLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDOUYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsaUJBQWtCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxpQkFBa0IsWUFBYSxFQUFFLENBQUMsQ0FBQztJQUNsTCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksNEJBQTRCLENBQUMsMkJBQW1DLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDNUgsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsaUJBQWtCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxrQ0FBbUMsMkJBQTRCLEVBQUUsQ0FBQyxDQUFDO0lBQ2xOLENBQUM7SUFFRDs7Ozs7OztPQU9HO0lBQ0ksT0FBTyxDQUFDLE9BQWUsRUFBRSxTQUFrQjtRQUNoRCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQix5QkFBMEIsT0FBUSxFQUFFLENBQUMsQ0FBQztJQUNsRyxDQUFDO0lBRUQ7Ozs7Ozs7O09BUUc7SUFDSSxTQUFTLENBQUMsT0FBZSxFQUFFLFNBQWlCLEVBQUUsU0FBa0I7UUFDckUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIseUJBQTBCLE9BQVEsWUFBYSxTQUFVLEVBQUUsQ0FBQyxDQUFDO0lBQ3pILENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7T0FjRztJQUNJLGVBQWUsQ0FBQyxNQUFjLEVBQUUsS0FBd0IsRUFBRSxRQUE0QjtRQUMzRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0JBQW1CLE1BQU8sRUFBRSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDaEYsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FvQ0c7SUFDSSxnQkFBZ0IsQ0FBQyxNQUFjLEVBQUUsS0FBd0IsRUFBRSxRQUE0QjtRQUM1RixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW9CLE1BQU8sRUFBRSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDakYsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7OztPQWNHO0lBQ0ksWUFBWSxDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDeEUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2pFLENBQUM7SUFFRDs7O09BR0c7SUFDSCxZQUFZLEtBQWdDO1FBQzFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztRQXhsQlIsa0JBQWEsR0FBRyxjQUFjLENBQUM7UUFnYTVCLG9CQUFlLEdBQW9CO1lBQzNDLEtBQUssRUFBRTtnQkFDTCxrQkFBa0I7Z0JBQ2xCLDhCQUE4QjtnQkFDOUIsMkNBQTJDO2dCQUMzQyw4Q0FBOEM7Z0JBQzlDLHFEQUFxRDtnQkFDckQsMERBQTBEO2dCQUMxRCxpQ0FBaUM7Z0JBQ2pDLGFBQWE7Z0JBQ2IsZ0NBQWdDO2dCQUNoQyxrQkFBa0I7Z0JBQ2xCLGlDQUFpQztnQkFDakMsbUNBQW1DO2dCQUNuQyxnREFBZ0Q7Z0JBQ2hELG1EQUFtRDtnQkFDbkQseUJBQXlCO2dCQUN6QixrQkFBa0I7Z0JBQ2xCLHNDQUFzQztnQkFDdEMsaUNBQWlDO2FBQ2xDO1lBQ0QsSUFBSSxFQUFFO2dCQUNKLHlCQUF5QjtnQkFDekIsZUFBZTtnQkFDZiw4QkFBOEI7Z0JBQzlCLHNCQUFzQjtnQkFDdEIscUJBQXFCO2FBQ3RCO1lBQ0QsSUFBSSxFQUFFO2dCQUNKLGlCQUFpQjtnQkFDakIsMkJBQTJCO2dCQUMzQixnQ0FBZ0M7Z0JBQ2hDLHdDQUF3QztnQkFDeEMsd0NBQXdDO2FBQ3pDO1lBQ0QsT0FBTyxFQUFFO2dCQUNQLGFBQWE7Z0JBQ2IsZUFBZTthQUNoQjtTQUNGLENBQUM7SUFrSkYsQ0FBQztDQUNGO0FBM2xCRCxvQ0EybEJDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQWNjZXNzTGV2ZWxMaXN0IH0gZnJvbSAnLi4vLi4vc2hhcmVkL2FjY2Vzcy1sZXZlbCc7XG5pbXBvcnQgeyBQb2xpY3lTdGF0ZW1lbnQsIE9wZXJhdG9yIH0gZnJvbSAnLi4vLi4vc2hhcmVkJztcbmltcG9ydCB7IGF3c19pYW0gYXMgaWFtIH0gZnJvbSBcImF3cy1jZGstbGliXCI7XG5cbi8qKlxuICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFtjb2RlY2F0YWx5c3RdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXJ2aWNlLWF1dGhvcml6YXRpb24vbGF0ZXN0L3JlZmVyZW5jZS9saXN0X2FtYXpvbmNvZGVjYXRhbHlzdC5odG1sKS5cbiAqXG4gKiBAcGFyYW0gc2lkIFtTSURdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfc2lkLmh0bWwpIG9mIHRoZSBzdGF0ZW1lbnRcbiAqL1xuZXhwb3J0IGNsYXNzIENvZGVjYXRhbHlzdCBleHRlbmRzIFBvbGljeVN0YXRlbWVudCB7XG4gIHB1YmxpYyBzZXJ2aWNlUHJlZml4ID0gJ2NvZGVjYXRhbHlzdCc7XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFjY2VwdCBhIHJlcXVlc3QgdG8gY29ubmVjdCB0aGlzIGFjY291bnQgdG8gYW4gQW1hem9uIENvZGVDYXRhbHlzdCBzcGFjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlY2F0YWx5c3QvbGF0ZXN0L3VzZXJndWlkZS9zZWN1cml0eS1pYW0uaHRtbCNwZXJtaXNzaW9ucy1yZWZlcmVuY2UtY29ubmVjdGlvbnNcbiAgICovXG4gIHB1YmxpYyB0b0FjY2VwdENvbm5lY3Rpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0FjY2VwdENvbm5lY3Rpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBhc3NvY2lhdGUgYW4gSUFNIHJvbGUgdG8gYSBjb25uZWN0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIGlhbTpQYXNzUm9sZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29kZWNhdGFseXN0L2xhdGVzdC91c2VyZ3VpZGUvc2VjdXJpdHktaWFtLmh0bWwjcGVybWlzc2lvbnMtcmVmZXJlbmNlLWNvbm5lY3Rpb25zXG4gICAqL1xuICBwdWJsaWMgdG9Bc3NvY2lhdGVJYW1Sb2xlVG9Db25uZWN0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdBc3NvY2lhdGVJYW1Sb2xlVG9Db25uZWN0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYXNzb2NpYXRlIGFuIElBTSBJZGVudGl0eSBDZW50ZXIgYXBwbGljYXRpb24gd2l0aCBhbiBBbWF6b24gQ29kZUNhdGFseXN0IHNwYWNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlY2F0YWx5c3QvbGF0ZXN0L3VzZXJndWlkZS9zZWN1cml0eS1pYW0uaHRtbCNwZXJtaXNzaW9ucy1yZWZlcmVuY2UtYXBwbGljYXRpb25zXG4gICAqL1xuICBwdWJsaWMgdG9Bc3NvY2lhdGVJZGVudGl0eUNlbnRlckFwcGxpY2F0aW9uVG9TcGFjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQXNzb2NpYXRlSWRlbnRpdHlDZW50ZXJBcHBsaWNhdGlvblRvU3BhY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBhc3NvY2lhdGUgYW4gaWRlbnRpdHkgd2l0aCBhbiBJQU0gSWRlbnRpdHkgQ2VudGVyIGFwcGxpY2F0aW9uIGZvciBhbiBBbWF6b24gQ29kZUNhdGFseXN0IHNwYWNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlY2F0YWx5c3QvbGF0ZXN0L3VzZXJndWlkZS9zZWN1cml0eS1pYW0uaHRtbCNwZXJtaXNzaW9ucy1yZWZlcmVuY2UtYXBwbGljYXRpb25zXG4gICAqL1xuICBwdWJsaWMgdG9Bc3NvY2lhdGVJZGVudGl0eVRvSWRlbnRpdHlDZW50ZXJBcHBsaWNhdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQXNzb2NpYXRlSWRlbnRpdHlUb0lkZW50aXR5Q2VudGVyQXBwbGljYXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBhc3NvY2lhdGUgbXVsdGlwbGUgaWRlbnRpdGllcyB3aXRoIGFuIElBTSBJZGVudGl0eSBDZW50ZXIgYXBwbGljYXRpb24gZm9yIGFuIEFtYXpvbiBDb2RlQ2F0YWx5c3Qgc3BhY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZGVjYXRhbHlzdC9sYXRlc3QvdXNlcmd1aWRlL3NlY3VyaXR5LWlhbS5odG1sI3Blcm1pc3Npb25zLXJlZmVyZW5jZS1hcHBsaWNhdGlvbnNcbiAgICovXG4gIHB1YmxpYyB0b0JhdGNoQXNzb2NpYXRlSWRlbnRpdGllc1RvSWRlbnRpdHlDZW50ZXJBcHBsaWNhdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQmF0Y2hBc3NvY2lhdGVJZGVudGl0aWVzVG9JZGVudGl0eUNlbnRlckFwcGxpY2F0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGlzYXNzb2NpYXRlIG11bHRpcGxlIGlkZW50aXRpZXMgZnJvbSBhbiBJQU0gSWRlbnRpdHkgQ2VudGVyIGFwcGxpY2F0aW9uIGZvciBhbiBBbWF6b24gQ29kZUNhdGFseXN0IHNwYWNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlY2F0YWx5c3QvbGF0ZXN0L3VzZXJndWlkZS9zZWN1cml0eS1pYW0uaHRtbCNwZXJtaXNzaW9ucy1yZWZlcmVuY2UtYXBwbGljYXRpb25zXG4gICAqL1xuICBwdWJsaWMgdG9CYXRjaERpc2Fzc29jaWF0ZUlkZW50aXRpZXNGcm9tSWRlbnRpdHlDZW50ZXJBcHBsaWNhdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQmF0Y2hEaXNhc3NvY2lhdGVJZGVudGl0aWVzRnJvbUlkZW50aXR5Q2VudGVyQXBwbGljYXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYW4gSUFNIElkZW50aXR5IENlbnRlciBhcHBsaWNhdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlY2F0YWx5c3QvbGF0ZXN0L3VzZXJndWlkZS9zZWN1cml0eS1pYW0uaHRtbCNwZXJtaXNzaW9ucy1yZWZlcmVuY2UtYXBwbGljYXRpb25zXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVJZGVudGl0eUNlbnRlckFwcGxpY2F0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVJZGVudGl0eUNlbnRlckFwcGxpY2F0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGFuIEFtYXpvbiBDb2RlQ2F0YWx5c3Qgc3BhY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29kZWNhdGFseXN0L2xhdGVzdC91c2VyZ3VpZGUvc2VjdXJpdHktaWFtLmh0bWwjcGVybWlzc2lvbnMtcmVmZXJlbmNlLWFwcGxpY2F0aW9uc1xuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlU3BhY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZVNwYWNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGFuIGFkbWluaXN0cmF0b3Igcm9sZSBhc3NpZ25tZW50IGZvciBhIGdpdmVuIEFtYXpvbiBDb2RlQ2F0YWx5c3Qgc3BhY2UgYW5kIElBTSBJZGVudGl0eSBDZW50ZXIgYXBwbGljYXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZGVjYXRhbHlzdC9sYXRlc3QvdXNlcmd1aWRlL3NlY3VyaXR5LWlhbS5odG1sI3Blcm1pc3Npb25zLXJlZmVyZW5jZS1hcHBsaWNhdGlvbnNcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVNwYWNlQWRtaW5Sb2xlQXNzaWdubWVudCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlU3BhY2VBZG1pblJvbGVBc3NpZ25tZW50Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGEgY29ubmVjdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29kZWNhdGFseXN0L2xhdGVzdC91c2VyZ3VpZGUvc2VjdXJpdHktaWFtLmh0bWwjcGVybWlzc2lvbnMtcmVmZXJlbmNlLWNvbm5lY3Rpb25zXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVDb25uZWN0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVDb25uZWN0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGFuIElBTSBJZGVudGl0eSBDZW50ZXIgYXBwbGljYXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZGVjYXRhbHlzdC9sYXRlc3QvdXNlcmd1aWRlL3NlY3VyaXR5LWlhbS5odG1sI3Blcm1pc3Npb25zLXJlZmVyZW5jZS1hcHBsaWNhdGlvbnNcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUlkZW50aXR5Q2VudGVyQXBwbGljYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZUlkZW50aXR5Q2VudGVyQXBwbGljYXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkaXNhc3NvY2lhdGUgYW4gSUFNIHJvbGUgZnJvbSBhIGNvbm5lY3Rpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZGVjYXRhbHlzdC9sYXRlc3QvdXNlcmd1aWRlL3NlY3VyaXR5LWlhbS5odG1sI3Blcm1pc3Npb25zLXJlZmVyZW5jZS1jb25uZWN0aW9uc1xuICAgKi9cbiAgcHVibGljIHRvRGlzYXNzb2NpYXRlSWFtUm9sZUZyb21Db25uZWN0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEaXNhc3NvY2lhdGVJYW1Sb2xlRnJvbUNvbm5lY3Rpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkaXNhc3NvY2lhdGUgYW4gSUFNIElkZW50aXR5IENlbnRlciBhcHBsaWNhdGlvbiBmcm9tIGFuIEFtYXpvbiBDb2RlQ2F0YWx5c3Qgc3BhY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZGVjYXRhbHlzdC9sYXRlc3QvdXNlcmd1aWRlL3NlY3VyaXR5LWlhbS5odG1sI3Blcm1pc3Npb25zLXJlZmVyZW5jZS1hcHBsaWNhdGlvbnNcbiAgICovXG4gIHB1YmxpYyB0b0Rpc2Fzc29jaWF0ZUlkZW50aXR5Q2VudGVyQXBwbGljYXRpb25Gcm9tU3BhY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rpc2Fzc29jaWF0ZUlkZW50aXR5Q2VudGVyQXBwbGljYXRpb25Gcm9tU3BhY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkaXNhc3NvY2lhdGUgYW4gaWRlbnRpdHkgZnJvbSBhbiBJQU0gSWRlbnRpdHkgQ2VudGVyIGFwcGxpY2F0aW9uIGZvciBhbiBBbWF6b24gQ29kZUNhdGFseXN0IHNwYWNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlY2F0YWx5c3QvbGF0ZXN0L3VzZXJndWlkZS9zZWN1cml0eS1pYW0uaHRtbCNwZXJtaXNzaW9ucy1yZWZlcmVuY2UtYXBwbGljYXRpb25zXG4gICAqL1xuICBwdWJsaWMgdG9EaXNhc3NvY2lhdGVJZGVudGl0eUZyb21JZGVudGl0eUNlbnRlckFwcGxpY2F0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEaXNhc3NvY2lhdGVJZGVudGl0eUZyb21JZGVudGl0eUNlbnRlckFwcGxpY2F0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVzY3JpYmUgdGhlIGJpbGxpbmcgYXV0aG9yaXphdGlvbiBmb3IgYSBjb25uZWN0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZGVjYXRhbHlzdC9sYXRlc3QvdXNlcmd1aWRlL3NlY3VyaXR5LWlhbS5odG1sI3Blcm1pc3Npb25zLXJlZmVyZW5jZS1jb25uZWN0aW9uc1xuICAgKi9cbiAgcHVibGljIHRvR2V0QmlsbGluZ0F1dGhvcml6YXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldEJpbGxpbmdBdXRob3JpemF0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IGEgY29ubmVjdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlY2F0YWx5c3QvbGF0ZXN0L3VzZXJndWlkZS9zZWN1cml0eS1pYW0uaHRtbCNwZXJtaXNzaW9ucy1yZWZlcmVuY2UtY29ubmVjdGlvbnNcbiAgICovXG4gIHB1YmxpYyB0b0dldENvbm5lY3Rpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldENvbm5lY3Rpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgaW5mb3JtYXRpb24gYWJvdXQgYW4gSUFNIElkZW50aXR5IENlbnRlciBhcHBsaWNhdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlY2F0YWx5c3QvbGF0ZXN0L3VzZXJndWlkZS9zZWN1cml0eS1pYW0uaHRtbCNwZXJtaXNzaW9ucy1yZWZlcmVuY2UtYXBwbGljYXRpb25zXG4gICAqL1xuICBwdWJsaWMgdG9HZXRJZGVudGl0eUNlbnRlckFwcGxpY2F0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRJZGVudGl0eUNlbnRlckFwcGxpY2F0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IGEgcGVuZGluZyByZXF1ZXN0IHRvIGNvbm5lY3QgdGhpcyBhY2NvdW50IHRvIGFuIEFtYXpvbiBDb2RlQ2F0YWx5c3Qgc3BhY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlY2F0YWx5c3QvbGF0ZXN0L3VzZXJndWlkZS9zZWN1cml0eS1pYW0uaHRtbCNwZXJtaXNzaW9ucy1yZWZlcmVuY2UtY29ubmVjdGlvbnNcbiAgICovXG4gIHB1YmxpYyB0b0dldFBlbmRpbmdDb25uZWN0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRQZW5kaW5nQ29ubmVjdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgY29ubmVjdGlvbnMgdGhhdCBhcmUgbm90IHBlbmRpbmdcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlY2F0YWx5c3QvbGF0ZXN0L3VzZXJndWlkZS9zZWN1cml0eS1pYW0uaHRtbCNwZXJtaXNzaW9ucy1yZWZlcmVuY2UtY29ubmVjdGlvbnNcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RDb25uZWN0aW9ucygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdENvbm5lY3Rpb25zJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBJQU0gcm9sZXMgYXNzb2NpYXRlZCB3aXRoIGEgY29ubmVjdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlY2F0YWx5c3QvbGF0ZXN0L3VzZXJndWlkZS9zZWN1cml0eS1pYW0uaHRtbCNwZXJtaXNzaW9ucy1yZWZlcmVuY2UtY29ubmVjdGlvbnNcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RJYW1Sb2xlc0ZvckNvbm5lY3Rpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RJYW1Sb2xlc0ZvckNvbm5lY3Rpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB2aWV3IGEgbGlzdCBvZiBhbGwgSUFNIElkZW50aXR5IENlbnRlciBhcHBsaWNhdGlvbnMgaW4gdGhlIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlY2F0YWx5c3QvbGF0ZXN0L3VzZXJndWlkZS9zZWN1cml0eS1pYW0uaHRtbCNwZXJtaXNzaW9ucy1yZWZlcmVuY2UtYXBwbGljYXRpb25zXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0SWRlbnRpdHlDZW50ZXJBcHBsaWNhdGlvbnMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RJZGVudGl0eUNlbnRlckFwcGxpY2F0aW9ucycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHZpZXcgYSBsaXN0IG9mIElBTSBJZGVudGl0eSBDZW50ZXIgYXBwbGljYXRpb25zIGJ5IEFtYXpvbiBDb2RlQ2F0YWx5c3Qgc3BhY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlY2F0YWx5c3QvbGF0ZXN0L3VzZXJndWlkZS9zZWN1cml0eS1pYW0uaHRtbCNwZXJtaXNzaW9ucy1yZWZlcmVuY2UtYXBwbGljYXRpb25zXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0SWRlbnRpdHlDZW50ZXJBcHBsaWNhdGlvbnNGb3JTcGFjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdElkZW50aXR5Q2VudGVyQXBwbGljYXRpb25zRm9yU3BhY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB2aWV3IGEgbGlzdCBvZiBBbWF6b24gQ29kZUNhdGFseXN0IHNwYWNlcyBieSBJQU0gSWRlbnRpdHkgQ2VudGVyIGFwcGxpY2F0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZGVjYXRhbHlzdC9sYXRlc3QvdXNlcmd1aWRlL3NlY3VyaXR5LWlhbS5odG1sI3Blcm1pc3Npb25zLXJlZmVyZW5jZS1hcHBsaWNhdGlvbnNcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RTcGFjZXNGb3JJZGVudGl0eUNlbnRlckFwcGxpY2F0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0U3BhY2VzRm9ySWRlbnRpdHlDZW50ZXJBcHBsaWNhdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgdGFncyBmb3IgYW4gQW1hem9uIENvZGVDYXRhbHlzdCByZXNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlY2F0YWx5c3QvbGF0ZXN0L3VzZXJndWlkZS9zZWN1cml0eS1pYW0uaHRtbCNwZXJtaXNzaW9ucy1yZWZlcmVuY2UtY29ubmVjdGlvbnNcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RUYWdzRm9yUmVzb3VyY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RUYWdzRm9yUmVzb3VyY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgb3IgdXBkYXRlIHRoZSBiaWxsaW5nIGF1dGhvcml6YXRpb24gZm9yIGEgY29ubmVjdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29kZWNhdGFseXN0L2xhdGVzdC91c2VyZ3VpZGUvc2VjdXJpdHktaWFtLmh0bWwjcGVybWlzc2lvbnMtcmVmZXJlbmNlLWNvbm5lY3Rpb25zXG4gICAqL1xuICBwdWJsaWMgdG9QdXRCaWxsaW5nQXV0aG9yaXphdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnUHV0QmlsbGluZ0F1dGhvcml6YXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZWplY3QgYSByZXF1ZXN0IHRvIGNvbm5lY3QgdGhpcyBhY2NvdW50IHRvIGFuIEFtYXpvbiBDb2RlQ2F0YWx5c3Qgc3BhY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29kZWNhdGFseXN0L2xhdGVzdC91c2VyZ3VpZGUvc2VjdXJpdHktaWFtLmh0bWwjcGVybWlzc2lvbnMtcmVmZXJlbmNlLWNvbm5lY3Rpb25zXG4gICAqL1xuICBwdWJsaWMgdG9SZWplY3RDb25uZWN0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdSZWplY3RDb25uZWN0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc3luY2hyb25pemUgYW4gSUFNIElkZW50aXR5IENlbnRlciBhcHBsaWNhdGlvbiB3aXRoIHRoZSBiYWNraW5nIGlkZW50aXR5IHN0b3JlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlY2F0YWx5c3QvbGF0ZXN0L3VzZXJndWlkZS9zZWN1cml0eS1pYW0uaHRtbCNwZXJtaXNzaW9ucy1yZWZlcmVuY2UtYXBwbGljYXRpb25zXG4gICAqL1xuICBwdWJsaWMgdG9TeW5jaHJvbml6ZUlkZW50aXR5Q2VudGVyQXBwbGljYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1N5bmNocm9uaXplSWRlbnRpdHlDZW50ZXJBcHBsaWNhdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHRhZyBhbiBBbWF6b24gQ29kZUNhdGFseXN0IHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogVGFnZ2luZ1xuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZGVjYXRhbHlzdC9sYXRlc3QvdXNlcmd1aWRlL3NlY3VyaXR5LWlhbS5odG1sI3Blcm1pc3Npb25zLXJlZmVyZW5jZS1jb25uZWN0aW9uc1xuICAgKi9cbiAgcHVibGljIHRvVGFnUmVzb3VyY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1RhZ1Jlc291cmNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdW50YWcgYW4gQW1hem9uIENvZGVDYXRhbHlzdCByZXNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFRhZ2dpbmdcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlY2F0YWx5c3QvbGF0ZXN0L3VzZXJndWlkZS9zZWN1cml0eS1pYW0uaHRtbCNwZXJtaXNzaW9ucy1yZWZlcmVuY2UtY29ubmVjdGlvbnNcbiAgICovXG4gIHB1YmxpYyB0b1VudGFnUmVzb3VyY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VudGFnUmVzb3VyY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYW4gSUFNIElkZW50aXR5IENlbnRlciBhcHBsaWNhdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29kZWNhdGFseXN0L2xhdGVzdC91c2VyZ3VpZGUvc2VjdXJpdHktaWFtLmh0bWwjcGVybWlzc2lvbnMtcmVmZXJlbmNlLWFwcGxpY2F0aW9uc1xuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlSWRlbnRpdHlDZW50ZXJBcHBsaWNhdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlSWRlbnRpdHlDZW50ZXJBcHBsaWNhdGlvbicpO1xuICB9XG5cbiAgcHJvdGVjdGVkIGFjY2Vzc0xldmVsTGlzdDogQWNjZXNzTGV2ZWxMaXN0ID0ge1xuICAgIFdyaXRlOiBbXG4gICAgICAnQWNjZXB0Q29ubmVjdGlvbicsXG4gICAgICAnQXNzb2NpYXRlSWFtUm9sZVRvQ29ubmVjdGlvbicsXG4gICAgICAnQXNzb2NpYXRlSWRlbnRpdHlDZW50ZXJBcHBsaWNhdGlvblRvU3BhY2UnLFxuICAgICAgJ0Fzc29jaWF0ZUlkZW50aXR5VG9JZGVudGl0eUNlbnRlckFwcGxpY2F0aW9uJyxcbiAgICAgICdCYXRjaEFzc29jaWF0ZUlkZW50aXRpZXNUb0lkZW50aXR5Q2VudGVyQXBwbGljYXRpb24nLFxuICAgICAgJ0JhdGNoRGlzYXNzb2NpYXRlSWRlbnRpdGllc0Zyb21JZGVudGl0eUNlbnRlckFwcGxpY2F0aW9uJyxcbiAgICAgICdDcmVhdGVJZGVudGl0eUNlbnRlckFwcGxpY2F0aW9uJyxcbiAgICAgICdDcmVhdGVTcGFjZScsXG4gICAgICAnQ3JlYXRlU3BhY2VBZG1pblJvbGVBc3NpZ25tZW50JyxcbiAgICAgICdEZWxldGVDb25uZWN0aW9uJyxcbiAgICAgICdEZWxldGVJZGVudGl0eUNlbnRlckFwcGxpY2F0aW9uJyxcbiAgICAgICdEaXNhc3NvY2lhdGVJYW1Sb2xlRnJvbUNvbm5lY3Rpb24nLFxuICAgICAgJ0Rpc2Fzc29jaWF0ZUlkZW50aXR5Q2VudGVyQXBwbGljYXRpb25Gcm9tU3BhY2UnLFxuICAgICAgJ0Rpc2Fzc29jaWF0ZUlkZW50aXR5RnJvbUlkZW50aXR5Q2VudGVyQXBwbGljYXRpb24nLFxuICAgICAgJ1B1dEJpbGxpbmdBdXRob3JpemF0aW9uJyxcbiAgICAgICdSZWplY3RDb25uZWN0aW9uJyxcbiAgICAgICdTeW5jaHJvbml6ZUlkZW50aXR5Q2VudGVyQXBwbGljYXRpb24nLFxuICAgICAgJ1VwZGF0ZUlkZW50aXR5Q2VudGVyQXBwbGljYXRpb24nXG4gICAgXSxcbiAgICBSZWFkOiBbXG4gICAgICAnR2V0QmlsbGluZ0F1dGhvcml6YXRpb24nLFxuICAgICAgJ0dldENvbm5lY3Rpb24nLFxuICAgICAgJ0dldElkZW50aXR5Q2VudGVyQXBwbGljYXRpb24nLFxuICAgICAgJ0dldFBlbmRpbmdDb25uZWN0aW9uJyxcbiAgICAgICdMaXN0VGFnc0ZvclJlc291cmNlJ1xuICAgIF0sXG4gICAgTGlzdDogW1xuICAgICAgJ0xpc3RDb25uZWN0aW9ucycsXG4gICAgICAnTGlzdElhbVJvbGVzRm9yQ29ubmVjdGlvbicsXG4gICAgICAnTGlzdElkZW50aXR5Q2VudGVyQXBwbGljYXRpb25zJyxcbiAgICAgICdMaXN0SWRlbnRpdHlDZW50ZXJBcHBsaWNhdGlvbnNGb3JTcGFjZScsXG4gICAgICAnTGlzdFNwYWNlc0ZvcklkZW50aXR5Q2VudGVyQXBwbGljYXRpb24nXG4gICAgXSxcbiAgICBUYWdnaW5nOiBbXG4gICAgICAnVGFnUmVzb3VyY2UnLFxuICAgICAgJ1VudGFnUmVzb3VyY2UnXG4gICAgXVxuICB9O1xuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBjb25uZWN0aW9ucyB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlY2F0YWx5c3QvbGF0ZXN0L3VzZXJndWlkZS8jXG4gICAqXG4gICAqIEBwYXJhbSBjb25uZWN0aW9uSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgY29ubmVjdGlvbklkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25Db25uZWN0aW9ucyhjb25uZWN0aW9uSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTpjb2RlY2F0YWx5c3Q6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9Oi9jb25uZWN0aW9ucy8keyBjb25uZWN0aW9uSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGlkZW50aXR5LWNlbnRlci1hcHBsaWNhdGlvbnMgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29kZWNhdGFseXN0L2xhdGVzdC91c2VyZ3VpZGUvI1xuICAgKlxuICAgKiBAcGFyYW0gaWRlbnRpdHlDZW50ZXJBcHBsaWNhdGlvbklkIC0gSWRlbnRpZmllciBmb3IgdGhlIGlkZW50aXR5Q2VudGVyQXBwbGljYXRpb25JZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uSWRlbnRpdHlDZW50ZXJBcHBsaWNhdGlvbnMoaWRlbnRpdHlDZW50ZXJBcHBsaWNhdGlvbklkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06Y29kZWNhdGFseXN0OiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTovaWRlbnRpdHktY2VudGVyLWFwcGxpY2F0aW9ucy8keyBpZGVudGl0eUNlbnRlckFwcGxpY2F0aW9uSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHNwYWNlIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZGVjYXRhbHlzdC9sYXRlc3QvdXNlcmd1aWRlLyNcbiAgICpcbiAgICogQHBhcmFtIHNwYWNlSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgc3BhY2VJZC5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICovXG4gIHB1YmxpYyBvblNwYWNlKHNwYWNlSWQ6IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06Y29kZWNhdGFseXN0Ojo6c3BhY2UvJHsgc3BhY2VJZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgcHJvamVjdCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlY2F0YWx5c3QvbGF0ZXN0L3VzZXJndWlkZS8jXG4gICAqXG4gICAqIEBwYXJhbSBzcGFjZUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIHNwYWNlSWQuXG4gICAqIEBwYXJhbSBwcm9qZWN0SWQgLSBJZGVudGlmaWVyIGZvciB0aGUgcHJvamVjdElkLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKi9cbiAgcHVibGljIG9uUHJvamVjdChzcGFjZUlkOiBzdHJpbmcsIHByb2plY3RJZDogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTpjb2RlY2F0YWx5c3Q6OjpzcGFjZS8keyBzcGFjZUlkIH0vcHJvamVjdC8keyBwcm9qZWN0SWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IGEgdGFnJ3Mga2V5IGFuZCB2YWx1ZSBpbiBhIHJlcXVlc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXJlcXVlc3R0YWdcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0FjY2VwdENvbm5lY3Rpb24oKVxuICAgKiAtIC50b0NyZWF0ZUlkZW50aXR5Q2VudGVyQXBwbGljYXRpb24oKVxuICAgKiAtIC50b0NyZWF0ZVNwYWNlKClcbiAgICogLSAudG9UYWdSZXNvdXJjZSgpXG4gICAqXG4gICAqIEBwYXJhbSB0YWdLZXkgVGhlIHRhZyBrZXkgdG8gY2hlY2tcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NSZXF1ZXN0VGFnKHRhZ0tleTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYXdzOlJlcXVlc3RUYWcvJHsgdGFnS2V5IH1gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgcHJlc2VuY2Ugb2YgdGFnIGtleS12YWx1ZSBwYWlycyBpbiB0aGUgcmVxdWVzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtcmVzb3VyY2V0YWdcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0Fzc29jaWF0ZUlhbVJvbGVUb0Nvbm5lY3Rpb24oKVxuICAgKiAtIC50b0Fzc29jaWF0ZUlkZW50aXR5Q2VudGVyQXBwbGljYXRpb25Ub1NwYWNlKClcbiAgICogLSAudG9Bc3NvY2lhdGVJZGVudGl0eVRvSWRlbnRpdHlDZW50ZXJBcHBsaWNhdGlvbigpXG4gICAqIC0gLnRvQmF0Y2hBc3NvY2lhdGVJZGVudGl0aWVzVG9JZGVudGl0eUNlbnRlckFwcGxpY2F0aW9uKClcbiAgICogLSAudG9CYXRjaERpc2Fzc29jaWF0ZUlkZW50aXRpZXNGcm9tSWRlbnRpdHlDZW50ZXJBcHBsaWNhdGlvbigpXG4gICAqIC0gLnRvQ3JlYXRlU3BhY2VBZG1pblJvbGVBc3NpZ25tZW50KClcbiAgICogLSAudG9EZWxldGVDb25uZWN0aW9uKClcbiAgICogLSAudG9EZWxldGVJZGVudGl0eUNlbnRlckFwcGxpY2F0aW9uKClcbiAgICogLSAudG9EaXNhc3NvY2lhdGVJYW1Sb2xlRnJvbUNvbm5lY3Rpb24oKVxuICAgKiAtIC50b0Rpc2Fzc29jaWF0ZUlkZW50aXR5Q2VudGVyQXBwbGljYXRpb25Gcm9tU3BhY2UoKVxuICAgKiAtIC50b0Rpc2Fzc29jaWF0ZUlkZW50aXR5RnJvbUlkZW50aXR5Q2VudGVyQXBwbGljYXRpb24oKVxuICAgKiAtIC50b0dldEJpbGxpbmdBdXRob3JpemF0aW9uKClcbiAgICogLSAudG9HZXRDb25uZWN0aW9uKClcbiAgICogLSAudG9HZXRJZGVudGl0eUNlbnRlckFwcGxpY2F0aW9uKClcbiAgICogLSAudG9MaXN0SWFtUm9sZXNGb3JDb25uZWN0aW9uKClcbiAgICogLSAudG9MaXN0U3BhY2VzRm9ySWRlbnRpdHlDZW50ZXJBcHBsaWNhdGlvbigpXG4gICAqIC0gLnRvTGlzdFRhZ3NGb3JSZXNvdXJjZSgpXG4gICAqIC0gLnRvUHV0QmlsbGluZ0F1dGhvcml6YXRpb24oKVxuICAgKiAtIC50b1N5bmNocm9uaXplSWRlbnRpdHlDZW50ZXJBcHBsaWNhdGlvbigpXG4gICAqIC0gLnRvVGFnUmVzb3VyY2UoKVxuICAgKiAtIC50b1VudGFnUmVzb3VyY2UoKVxuICAgKiAtIC50b1VwZGF0ZUlkZW50aXR5Q2VudGVyQXBwbGljYXRpb24oKVxuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIGNvbm5lY3Rpb25zXG4gICAqIC0gaWRlbnRpdHktY2VudGVyLWFwcGxpY2F0aW9uc1xuICAgKlxuICAgKiBAcGFyYW0gdGFnS2V5IFRoZSB0YWcga2V5IHRvIGNoZWNrXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzUmVzb3VyY2VUYWcodGFnS2V5OiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6UmVzb3VyY2VUYWcvJHsgdGFnS2V5IH1gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgdGFnIGtleXMgaW4gYSByZXF1ZXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy10YWdrZXlzXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9BY2NlcHRDb25uZWN0aW9uKClcbiAgICogLSAudG9DcmVhdGVJZGVudGl0eUNlbnRlckFwcGxpY2F0aW9uKClcbiAgICogLSAudG9DcmVhdGVTcGFjZSgpXG4gICAqIC0gLnRvVGFnUmVzb3VyY2UoKVxuICAgKiAtIC50b1VudGFnUmVzb3VyY2UoKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1RhZ0tleXModmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpUYWdLZXlzYCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFtjb2RlY2F0YWx5c3RdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXJ2aWNlLWF1dGhvcml6YXRpb24vbGF0ZXN0L3JlZmVyZW5jZS9saXN0X2FtYXpvbmNvZGVjYXRhbHlzdC5odG1sKS5cbiAgICpcbiAgICovXG4gIGNvbnN0cnVjdG9yKHByb3BzPzogaWFtLlBvbGljeVN0YXRlbWVudFByb3BzKSB7XG4gICAgc3VwZXIocHJvcHMpO1xuICB9XG59XG4iXX0=