"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CognitoIdp = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [cognito-idp](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncognitouserpools.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class CognitoIdp extends shared_1.PolicyStatement {
    /**
     * Grants permission to add user attributes to the user pool schema
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AddCustomAttributes.html
     */
    toAddCustomAttributes() {
        return this.to('AddCustomAttributes');
    }
    /**
     * Grants permission to add any user to any group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminAddUserToGroup.html
     */
    toAdminAddUserToGroup() {
        return this.to('AdminAddUserToGroup');
    }
    /**
     * Grants permission to confirm any user's registration without a confirmation code
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminConfirmSignUp.html
     */
    toAdminConfirmSignUp() {
        return this.to('AdminConfirmSignUp');
    }
    /**
     * Grants permission to create new users and send welcome messages via email or SMS
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminCreateUser.html
     */
    toAdminCreateUser() {
        return this.to('AdminCreateUser');
    }
    /**
     * Grants permission to delete any user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminDeleteUser.html
     */
    toAdminDeleteUser() {
        return this.to('AdminDeleteUser');
    }
    /**
     * Grants permission to delete attributes from any user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminDeleteUserAttributes.html
     */
    toAdminDeleteUserAttributes() {
        return this.to('AdminDeleteUserAttributes');
    }
    /**
     * Grants permission to unlink any user pool user from a third-party identity provider (IdP) user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminDisableProviderForUser.html
     */
    toAdminDisableProviderForUser() {
        return this.to('AdminDisableProviderForUser');
    }
    /**
     * Grants permission to deactivate any user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminDisableUser.html
     */
    toAdminDisableUser() {
        return this.to('AdminDisableUser');
    }
    /**
     * Grants permission to activate any user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminEnableUser.html
     */
    toAdminEnableUser() {
        return this.to('AdminEnableUser');
    }
    /**
     * Grants permission to deregister any user's devices
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminForgetDevice.html
     */
    toAdminForgetDevice() {
        return this.to('AdminForgetDevice');
    }
    /**
     * Grants permission to get information about any user's devices
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminGetDevice.html
     */
    toAdminGetDevice() {
        return this.to('AdminGetDevice');
    }
    /**
     * Grants permission to look up any user by user name
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminGetUser.html
     */
    toAdminGetUser() {
        return this.to('AdminGetUser');
    }
    /**
     * Grants permission to authenticate any user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminInitiateAuth.html
     */
    toAdminInitiateAuth() {
        return this.to('AdminInitiateAuth');
    }
    /**
     * Grants permission to link any user pool user to a third-party IdP user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminLinkProviderForUser.html
     */
    toAdminLinkProviderForUser() {
        return this.to('AdminLinkProviderForUser');
    }
    /**
     * Grants permission to list any user's remembered devices
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminListDevices.html
     */
    toAdminListDevices() {
        return this.to('AdminListDevices');
    }
    /**
     * Grants permission to list the groups that any user belongs to
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminListGroupsForUser.html
     */
    toAdminListGroupsForUser() {
        return this.to('AdminListGroupsForUser');
    }
    /**
     * Grants permission to lists sign-in events for any user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminListUserAuthEvents.html
     */
    toAdminListUserAuthEvents() {
        return this.to('AdminListUserAuthEvents');
    }
    /**
     * Grants permission to remove any user from any group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminRemoveUserFromGroup.html
     */
    toAdminRemoveUserFromGroup() {
        return this.to('AdminRemoveUserFromGroup');
    }
    /**
     * Grants permission to reset any user's password
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminResetUserPassword.html
     */
    toAdminResetUserPassword() {
        return this.to('AdminResetUserPassword');
    }
    /**
     * Grants permission to respond to an authentication challenge during the authentication of any user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminRespondToAuthChallenge.html
     */
    toAdminRespondToAuthChallenge() {
        return this.to('AdminRespondToAuthChallenge');
    }
    /**
     * Grants permission to set any user's preferred MFA method
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminSetUserMFAPreference.html
     */
    toAdminSetUserMFAPreference() {
        return this.to('AdminSetUserMFAPreference');
    }
    /**
     * Grants permission to set any user's password
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminSetUserPassword.html
     */
    toAdminSetUserPassword() {
        return this.to('AdminSetUserPassword');
    }
    /**
     * Grants permission to set user settings for any user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminSetUserSettings.html
     */
    toAdminSetUserSettings() {
        return this.to('AdminSetUserSettings');
    }
    /**
     * Grants permission to update advanced security feedback for any user's authentication event
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminUpdateAuthEventFeedback.html
     */
    toAdminUpdateAuthEventFeedback() {
        return this.to('AdminUpdateAuthEventFeedback');
    }
    /**
     * Grants permission to update the status of any user's remembered devices
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminUpdateDeviceStatus.html
     */
    toAdminUpdateDeviceStatus() {
        return this.to('AdminUpdateDeviceStatus');
    }
    /**
     * Grants permission to updates any user's standard or custom attributes
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminUpdateUserAttributes.html
     */
    toAdminUpdateUserAttributes() {
        return this.to('AdminUpdateUserAttributes');
    }
    /**
     * Grants permission to sign out any user from all sessions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminUserGlobalSignOut.html
     */
    toAdminUserGlobalSignOut() {
        return this.to('AdminUserGlobalSignOut');
    }
    /**
     * Grants permission to return a unique generated shared secret key code for the user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AssociateSoftwareToken.html
     */
    toAssociateSoftwareToken() {
        return this.to('AssociateSoftwareToken');
    }
    /**
     * Grants permission to associate the user pool with an AWS WAF web ACL
     *
     * Access Level: Write
     */
    toAssociateWebACL() {
        return this.to('AssociateWebACL');
    }
    /**
     * Grants permission to change the password for a specified user in a user pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ChangePassword.html
     */
    toChangePassword() {
        return this.to('ChangePassword');
    }
    /**
     * Grants permission to confirm tracking of the device. This API call is the call that begins device tracking
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ConfirmDevice.html
     */
    toConfirmDevice() {
        return this.to('ConfirmDevice');
    }
    /**
     * Grants permission to allow a user to enter a confirmation code to reset a forgotten password
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ConfirmForgotPassword.html
     */
    toConfirmForgotPassword() {
        return this.to('ConfirmForgotPassword');
    }
    /**
     * Grants permission to confirm registration of a user and handles the existing alias from a previous user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ConfirmSignUp.html
     */
    toConfirmSignUp() {
        return this.to('ConfirmSignUp');
    }
    /**
     * Grants permission to create new user pool groups
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_CreateGroup.html
     */
    toCreateGroup() {
        return this.to('CreateGroup');
    }
    /**
     * Grants permission to add identity providers to user pools
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_CreateIdentityProvider.html
     */
    toCreateIdentityProvider() {
        return this.to('CreateIdentityProvider');
    }
    /**
     * Grants permission to create a branding settings for managed login and associate it with an app client
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_CreateManagedLoginBranding.html
     */
    toCreateManagedLoginBranding() {
        return this.to('CreateManagedLoginBranding');
    }
    /**
     * Grants permission to create and configure scopes for OAuth 2.0 resource servers
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_CreateResourceServer.html
     */
    toCreateResourceServer() {
        return this.to('CreateResourceServer');
    }
    /**
     * Grants permission to create user CSV import jobs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_CreateUserImportJob.html
     */
    toCreateUserImportJob() {
        return this.to('CreateUserImportJob');
    }
    /**
     * Grants permission to create and set password policy for user pools
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_CreateUserPool.html
     */
    toCreateUserPool() {
        return this.to('CreateUserPool');
    }
    /**
     * Grants permission to create user pool app clients
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_CreateUserPoolClient.html
     */
    toCreateUserPoolClient() {
        return this.to('CreateUserPoolClient');
    }
    /**
     * Grants permission to add user pool domains
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_CreateUserPoolDomain.html
     */
    toCreateUserPoolDomain() {
        return this.to('CreateUserPoolDomain');
    }
    /**
     * Grants permission to delete any empty user pool group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DeleteGroup.html
     */
    toDeleteGroup() {
        return this.to('DeleteGroup');
    }
    /**
     * Grants permission to delete any identity provider from user pools
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DeleteIdentityProvider.html
     */
    toDeleteIdentityProvider() {
        return this.to('DeleteIdentityProvider');
    }
    /**
     * Grants permission to delete the managed login branding style for any app client
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DeleteManagedLoginBranding.html
     */
    toDeleteManagedLoginBranding() {
        return this.to('DeleteManagedLoginBranding');
    }
    /**
     * Grants permission to delete any OAuth 2.0 resource server from user pools
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DeleteResourceServer.html
     */
    toDeleteResourceServer() {
        return this.to('DeleteResourceServer');
    }
    /**
     * Grants permission to allow a user to delete one's self
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DeleteUser.html
     */
    toDeleteUser() {
        return this.to('DeleteUser');
    }
    /**
     * Grants permission to delete the attributes for a user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DeleteUserAttributes.html
     */
    toDeleteUserAttributes() {
        return this.to('DeleteUserAttributes');
    }
    /**
     * Grants permission to delete user pools
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DeleteUserPool.html
     */
    toDeleteUserPool() {
        return this.to('DeleteUserPool');
    }
    /**
     * Grants permission to delete any user pool app client
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DeleteUserPoolClient.html
     */
    toDeleteUserPoolClient() {
        return this.to('DeleteUserPoolClient');
    }
    /**
     * Grants permission to delete any user pool domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DeleteUserPoolDomain.html
     */
    toDeleteUserPoolDomain() {
        return this.to('DeleteUserPoolDomain');
    }
    /**
     * Grants permission to describe any user pool identity provider
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DescribeIdentityProvider.html
     */
    toDescribeIdentityProvider() {
        return this.to('DescribeIdentityProvider');
    }
    /**
     * Grants permission to get the detailed information about the branding style of managed login
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DescribeManagedLoginBranding.html
     */
    toDescribeManagedLoginBranding() {
        return this.to('DescribeManagedLoginBranding');
    }
    /**
     * Grants permission to get the detailed information about the branding style of managed login associated with an appclient
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DescribeManagedLoginBrandingByClient.html
     */
    toDescribeManagedLoginBrandingByClient() {
        return this.to('DescribeManagedLoginBrandingByClient');
    }
    /**
     * Grants permission to describe any OAuth 2.0 resource server
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DescribeResourceServer.html
     */
    toDescribeResourceServer() {
        return this.to('DescribeResourceServer');
    }
    /**
     * Grants permission to describe the risk configuration settings of user pools and app clients
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DescribeRiskConfiguration.html
     */
    toDescribeRiskConfiguration() {
        return this.to('DescribeRiskConfiguration');
    }
    /**
     * Grants permission to describe any user import job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DescribeUserImportJob.html
     */
    toDescribeUserImportJob() {
        return this.to('DescribeUserImportJob');
    }
    /**
     * Grants permission to describe user pools
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DescribeUserPool.html
     */
    toDescribeUserPool() {
        return this.to('DescribeUserPool');
    }
    /**
     * Grants permission to describe any user pool app client
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DescribeUserPoolClient.html
     */
    toDescribeUserPoolClient() {
        return this.to('DescribeUserPoolClient');
    }
    /**
     * Grants permission to describe any user pool domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DescribeUserPoolDomain.html
     */
    toDescribeUserPoolDomain() {
        return this.to('DescribeUserPoolDomain');
    }
    /**
     * Grants permission to disassociate the user pool with an AWS WAF web ACL
     *
     * Access Level: Write
     */
    toDisassociateWebACL() {
        return this.to('DisassociateWebACL');
    }
    /**
     * Grants permission to forget the specified device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ForgetDevice.html
     */
    toForgetDevice() {
        return this.to('ForgetDevice');
    }
    /**
     * Grants permission to send a message to the end user with a confirmation code that is required to change the user's password
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ForgotPassword.html
     */
    toForgotPassword() {
        return this.to('ForgotPassword');
    }
    /**
     * Grants permission to generate headers for a user import .csv file
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_GetCSVHeader.html
     */
    toGetCSVHeader() {
        return this.to('GetCSVHeader');
    }
    /**
     * Grants permission to get the device
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_GetDevice.html
     */
    toGetDevice() {
        return this.to('GetDevice');
    }
    /**
     * Grants permission to describe a user pool group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_GetGroup.html
     */
    toGetGroup() {
        return this.to('GetGroup');
    }
    /**
     * Grants permission to correlate a user pool IdP identifier to the IdP Name
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_GetIdentityProviderByIdentifier.html
     */
    toGetIdentityProviderByIdentifier() {
        return this.to('GetIdentityProviderByIdentifier');
    }
    /**
     * Grants permission to get the detailed activity logging configuration for a user pool
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_GetLogDeliveryConfiguration.html
     */
    toGetLogDeliveryConfiguration() {
        return this.to('GetLogDeliveryConfiguration');
    }
    /**
     * Grants permission to look up signing certificates for user pools
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_GetSigningCertificate.html
     */
    toGetSigningCertificate() {
        return this.to('GetSigningCertificate');
    }
    /**
     * Grants permission to get UI customization information for the hosted UI of any app client
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_GetUICustomization.html
     */
    toGetUICustomization() {
        return this.to('GetUICustomization');
    }
    /**
     * Grants permission to get the user attributes and metadata for a user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_GetUser.html
     */
    toGetUser() {
        return this.to('GetUser');
    }
    /**
     * Grants permission to get the user attribute verification code for the specified attribute name
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_GetUserAttributeVerificationCode.html
     */
    toGetUserAttributeVerificationCode() {
        return this.to('GetUserAttributeVerificationCode');
    }
    /**
     * Grants permission to look up the MFA configuration of user pools
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_GetUserPoolMfaConfig.html
     */
    toGetUserPoolMfaConfig() {
        return this.to('GetUserPoolMfaConfig');
    }
    /**
     * Grants permission to get the AWS WAF web ACL that is associated with an Amazon Cognito user pool
     *
     * Access Level: Read
     */
    toGetWebACLForResource() {
        return this.to('GetWebACLForResource');
    }
    /**
     * Grants permission to sign out users from all devices
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_GlobalSignOut.html
     */
    toGlobalSignOut() {
        return this.to('GlobalSignOut');
    }
    /**
     * Grants permission to initiate the authentication flow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_InitiateAuth.html
     */
    toInitiateAuth() {
        return this.to('InitiateAuth');
    }
    /**
     * Grants permission to list the devices
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ListDevices.html
     */
    toListDevices() {
        return this.to('ListDevices');
    }
    /**
     * Grants permission to list all groups in user pools
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ListGroups.html
     */
    toListGroups() {
        return this.to('ListGroups');
    }
    /**
     * Grants permission to list all identity providers in user pools
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ListIdentityProviders.html
     */
    toListIdentityProviders() {
        return this.to('ListIdentityProviders');
    }
    /**
     * Grants permission to list all resource servers in user pools
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ListResourceServers.html
     */
    toListResourceServers() {
        return this.to('ListResourceServers');
    }
    /**
     * Grants permission to list the user pools that are associated with an AWS WAF web ACL
     *
     * Access Level: List
     */
    toListResourcesForWebACL() {
        return this.to('ListResourcesForWebACL');
    }
    /**
     * Grants permission to list the tags that are assigned to an Amazon Cognito user pool
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list all user import jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ListUserImportJobs.html
     */
    toListUserImportJobs() {
        return this.to('ListUserImportJobs');
    }
    /**
     * Grants permission to list all app clients in user pools
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ListUserPoolClients.html
     */
    toListUserPoolClients() {
        return this.to('ListUserPoolClients');
    }
    /**
     * Grants permission to list all user pools
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ListUserPools.html
     */
    toListUserPools() {
        return this.to('ListUserPools');
    }
    /**
     * Grants permission to list all user pool users
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ListUsers.html
     */
    toListUsers() {
        return this.to('ListUsers');
    }
    /**
     * Grants permission to list the users in any group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ListUsersInGroup.html
     */
    toListUsersInGroup() {
        return this.to('ListUsersInGroup');
    }
    /**
     * Grants permission to resend the confirmation (for confirmation of registration) to a specific user in the user pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ResendConfirmationCode.html
     */
    toResendConfirmationCode() {
        return this.to('ResendConfirmationCode');
    }
    /**
     * Grants permission to respond to the authentication challenge
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_RespondToAuthChallenge.html
     */
    toRespondToAuthChallenge() {
        return this.to('RespondToAuthChallenge');
    }
    /**
     * Grants permission to revoke all of the access tokens generated by the specified refresh token
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_RevokeToken.html
     */
    toRevokeToken() {
        return this.to('RevokeToken');
    }
    /**
     * Grants permission to set up or modify the detailed activity logging configuration of a user pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_SetLogDeliveryConfiguration.html
     */
    toSetLogDeliveryConfiguration() {
        return this.to('SetLogDeliveryConfiguration');
    }
    /**
     * Grants permission to set risk configuration for user pools and app clients
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_SetRiskConfiguration.html
     */
    toSetRiskConfiguration() {
        return this.to('SetRiskConfiguration');
    }
    /**
     * Grants permission to customize the hosted UI for any app client
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_SetUICustomization.html
     */
    toSetUICustomization() {
        return this.to('SetUICustomization');
    }
    /**
     * Grants permission to set MFA preference for the user in the userpool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_SetUserMFAPreference.html
     */
    toSetUserMFAPreference() {
        return this.to('SetUserMFAPreference');
    }
    /**
     * Grants permission to set user pool MFA configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_SetUserPoolMfaConfig.html
     */
    toSetUserPoolMfaConfig() {
        return this.to('SetUserPoolMfaConfig');
    }
    /**
     * Grants permission to set the user settings like multi-factor authentication (MFA)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_SetUserSettings.html
     */
    toSetUserSettings() {
        return this.to('SetUserSettings');
    }
    /**
     * Grants permission to register the user in the specified user pool and creates a user name, password, and user attributes
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_SignUp.html
     */
    toSignUp() {
        return this.to('SignUp');
    }
    /**
     * Grants permission to start any user import job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_StartUserImportJob.html
     */
    toStartUserImportJob() {
        return this.to('StartUserImportJob');
    }
    /**
     * Grants permission to stop any user import job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_StopUserImportJob.html
     */
    toStopUserImportJob() {
        return this.to('StopUserImportJob');
    }
    /**
     * Grants permission to tag a user pool
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag a user pool
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update the feedback for the user authentication event
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateAuthEventFeedback.html
     */
    toUpdateAuthEventFeedback() {
        return this.to('UpdateAuthEventFeedback');
    }
    /**
     * Grants permission to update the device status
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateDeviceStatus.html
     */
    toUpdateDeviceStatus() {
        return this.to('UpdateDeviceStatus');
    }
    /**
     * Grants permission to update the configuration of any group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateGroup.html
     */
    toUpdateGroup() {
        return this.to('UpdateGroup');
    }
    /**
     * Grants permission to update the configuration of any user pool IdP
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateIdentityProvider.html
     */
    toUpdateIdentityProvider() {
        return this.to('UpdateIdentityProvider');
    }
    /**
     * Grants permission to update the branding settings of a managed login
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateManagedLoginBranding.html
     */
    toUpdateManagedLoginBranding() {
        return this.to('UpdateManagedLoginBranding');
    }
    /**
     * Grants permission to update the configuration of any OAuth 2.0 resource server
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateResourceServer.html
     */
    toUpdateResourceServer() {
        return this.to('UpdateResourceServer');
    }
    /**
     * Grants permission to allow a user to update a specific attribute (one at a time)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateUserAttributes.html
     */
    toUpdateUserAttributes() {
        return this.to('UpdateUserAttributes');
    }
    /**
     * Grants permission to updates the configuration of user pools
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateUserPool.html
     */
    toUpdateUserPool() {
        return this.to('UpdateUserPool');
    }
    /**
     * Grants permission to update any user pool client
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateUserPoolClient.html
     */
    toUpdateUserPoolClient() {
        return this.to('UpdateUserPoolClient');
    }
    /**
     * Grants permission to replace the certificate for any custom domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateUserPoolDomain.html
     */
    toUpdateUserPoolDomain() {
        return this.to('UpdateUserPoolDomain');
    }
    /**
     * Grants permission to register a user's entered TOTP code and mark the user's software token MFA status as verified if successful
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_VerifySoftwareToken.html
     */
    toVerifySoftwareToken() {
        return this.to('VerifySoftwareToken');
    }
    /**
     * Grants permission to verify a user attribute using a one time verification code
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_VerifyUserAttribute.html
     */
    toVerifyUserAttribute() {
        return this.to('VerifyUserAttribute');
    }
    /**
     * Adds a resource of type userpool to the statement
     *
     * https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools.html
     *
     * @param userPoolId - Identifier for the userPoolId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onUserpool(userPoolId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cognito-idp:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:userpool/${userPoolId}`);
    }
    /**
     * Adds a resource of type webacl to the statement
     *
     * https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-waf.html
     *
     * @param scope - Identifier for the scope.
     * @param name - Identifier for the name.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onWebacl(scope, name, id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:wafv2:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:${scope}/webacl/${name}/${id}`);
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateUserPool()
     * - .toTagResource()
     * - .toUpdateUserPool()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toCreateUserPool()
     *
     * Applies to resource types:
     * - userpool
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a key that is present in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateUserPool()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateUserPool()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [cognito-idp](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncognitouserpools.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'cognito-idp';
        this.accessLevelList = {
            Write: [
                'AddCustomAttributes',
                'AdminAddUserToGroup',
                'AdminConfirmSignUp',
                'AdminCreateUser',
                'AdminDeleteUser',
                'AdminDeleteUserAttributes',
                'AdminDisableProviderForUser',
                'AdminDisableUser',
                'AdminEnableUser',
                'AdminForgetDevice',
                'AdminInitiateAuth',
                'AdminLinkProviderForUser',
                'AdminRemoveUserFromGroup',
                'AdminResetUserPassword',
                'AdminRespondToAuthChallenge',
                'AdminSetUserMFAPreference',
                'AdminSetUserPassword',
                'AdminSetUserSettings',
                'AdminUpdateAuthEventFeedback',
                'AdminUpdateDeviceStatus',
                'AdminUpdateUserAttributes',
                'AdminUserGlobalSignOut',
                'AssociateSoftwareToken',
                'AssociateWebACL',
                'ChangePassword',
                'ConfirmDevice',
                'ConfirmForgotPassword',
                'ConfirmSignUp',
                'CreateGroup',
                'CreateIdentityProvider',
                'CreateManagedLoginBranding',
                'CreateResourceServer',
                'CreateUserImportJob',
                'CreateUserPool',
                'CreateUserPoolClient',
                'CreateUserPoolDomain',
                'DeleteGroup',
                'DeleteIdentityProvider',
                'DeleteManagedLoginBranding',
                'DeleteResourceServer',
                'DeleteUser',
                'DeleteUserAttributes',
                'DeleteUserPool',
                'DeleteUserPoolClient',
                'DeleteUserPoolDomain',
                'DisassociateWebACL',
                'ForgetDevice',
                'ForgotPassword',
                'GlobalSignOut',
                'InitiateAuth',
                'ResendConfirmationCode',
                'RespondToAuthChallenge',
                'RevokeToken',
                'SetLogDeliveryConfiguration',
                'SetRiskConfiguration',
                'SetUICustomization',
                'SetUserMFAPreference',
                'SetUserPoolMfaConfig',
                'SetUserSettings',
                'SignUp',
                'StartUserImportJob',
                'StopUserImportJob',
                'UpdateAuthEventFeedback',
                'UpdateDeviceStatus',
                'UpdateGroup',
                'UpdateIdentityProvider',
                'UpdateManagedLoginBranding',
                'UpdateResourceServer',
                'UpdateUserAttributes',
                'UpdateUserPool',
                'UpdateUserPoolClient',
                'UpdateUserPoolDomain',
                'VerifySoftwareToken',
                'VerifyUserAttribute'
            ],
            Read: [
                'AdminGetDevice',
                'AdminGetUser',
                'AdminListUserAuthEvents',
                'DescribeIdentityProvider',
                'DescribeManagedLoginBranding',
                'DescribeManagedLoginBrandingByClient',
                'DescribeResourceServer',
                'DescribeRiskConfiguration',
                'DescribeUserImportJob',
                'DescribeUserPool',
                'DescribeUserPoolClient',
                'DescribeUserPoolDomain',
                'GetCSVHeader',
                'GetDevice',
                'GetGroup',
                'GetIdentityProviderByIdentifier',
                'GetLogDeliveryConfiguration',
                'GetSigningCertificate',
                'GetUICustomization',
                'GetUser',
                'GetUserAttributeVerificationCode',
                'GetUserPoolMfaConfig',
                'GetWebACLForResource'
            ],
            List: [
                'AdminListDevices',
                'AdminListGroupsForUser',
                'ListDevices',
                'ListGroups',
                'ListIdentityProviders',
                'ListResourceServers',
                'ListResourcesForWebACL',
                'ListTagsForResource',
                'ListUserImportJobs',
                'ListUserPoolClients',
                'ListUserPools',
                'ListUsers',
                'ListUsersInGroup'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.CognitoIdp = CognitoIdp;
//# sourceMappingURL=data:application/json;base64,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