"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ComputeOptimizer = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [compute-optimizer](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscomputeoptimizer.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class ComputeOptimizer extends shared_1.PolicyStatement {
    /**
     * Grants permission to delete recommendation preferences
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifResourceType()
     *
     * Dependent actions:
     * - autoscaling:DescribeAutoScalingGroups
     * - ec2:DescribeInstances
     * - rds:DescribeDBClusters
     * - rds:DescribeDBInstances
     *
     * https://docs.aws.amazon.com/compute-optimizer/latest/APIReference/API_DeleteRecommendationPreferences.html
     */
    toDeleteRecommendationPreferences() {
        return this.to('DeleteRecommendationPreferences');
    }
    /**
     * Grants permission to view the status of recommendation export jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/compute-optimizer/latest/APIReference/API_DescribeRecommendationExportJobs.html
     */
    toDescribeRecommendationExportJobs() {
        return this.to('DescribeRecommendationExportJobs');
    }
    /**
     * Grants permission to export AutoScaling group recommendations to S3 for the provided accounts
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - autoscaling:DescribeAutoScalingGroups
     * - compute-optimizer:GetAutoScalingGroupRecommendations
     *
     * https://docs.aws.amazon.com/compute-optimizer/latest/APIReference/API_ExportAutoScalingGroupRecommendations.html
     */
    toExportAutoScalingGroupRecommendations() {
        return this.to('ExportAutoScalingGroupRecommendations');
    }
    /**
     * Grants permission to export EBS volume recommendations to S3 for the provided accounts
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - compute-optimizer:GetEBSVolumeRecommendations
     * - ec2:DescribeVolumes
     *
     * https://docs.aws.amazon.com/compute-optimizer/latest/APIReference/API_ExportEBSVolumeRecommendations.html
     */
    toExportEBSVolumeRecommendations() {
        return this.to('ExportEBSVolumeRecommendations');
    }
    /**
     * Grants permission to export EC2 instance recommendations to S3 for the provided accounts
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - compute-optimizer:GetEC2InstanceRecommendations
     * - ec2:DescribeInstances
     *
     * https://docs.aws.amazon.com/compute-optimizer/latest/APIReference/API_ExportEC2InstanceRecommendations.html
     */
    toExportEC2InstanceRecommendations() {
        return this.to('ExportEC2InstanceRecommendations');
    }
    /**
     * Grants permission to export ECS service recommendations to S3 for the provided accounts
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - compute-optimizer:GetECSServiceRecommendations
     * - ecs:ListClusters
     * - ecs:ListServices
     *
     * https://docs.aws.amazon.com/compute-optimizer/latest/APIReference/API_ExportECSServiceRecommendations.html
     */
    toExportECSServiceRecommendations() {
        return this.to('ExportECSServiceRecommendations');
    }
    /**
     * Grants permission to export idle recommendations to S3 for the provided accounts
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - compute-optimizer:GetIdleRecommendations
     *
     * https://docs.aws.amazon.com/compute-optimizer/latest/APIReference/API_ExportIdleRecommendations.html
     */
    toExportIdleRecommendations() {
        return this.to('ExportIdleRecommendations');
    }
    /**
     * Grants permission to export Lambda function recommendations to S3 for the provided accounts
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - compute-optimizer:GetLambdaFunctionRecommendations
     * - lambda:ListFunctions
     * - lambda:ListProvisionedConcurrencyConfigs
     *
     * https://docs.aws.amazon.com/compute-optimizer/latest/APIReference/API_ExportLambdaFunctionRecommendations.html
     */
    toExportLambdaFunctionRecommendations() {
        return this.to('ExportLambdaFunctionRecommendations');
    }
    /**
     * Grants permission to export license recommendations to S3 for the provided account(s)
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - compute-optimizer:GetLicenseRecommendations
     * - ec2:DescribeInstances
     *
     * https://docs.aws.amazon.com/compute-optimizer/latest/APIReference/API_ExportLicenseRecommendations.html
     */
    toExportLicenseRecommendations() {
        return this.to('ExportLicenseRecommendations');
    }
    /**
     * Grants permission to export rds recommendations to S3 for the provided accounts
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - compute-optimizer:GetRDSDatabaseRecommendations
     * - rds:DescribeDBClusters
     * - rds:DescribeDBInstances
     *
     * https://docs.aws.amazon.com/compute-optimizer/latest/APIReference/API_ExportRDSDatabaseRecommendations.html
     */
    toExportRDSDatabaseRecommendations() {
        return this.to('ExportRDSDatabaseRecommendations');
    }
    /**
     * Grants permission to get recommendations for the provided AutoScaling groups
     *
     * Access Level: List
     *
     * Dependent actions:
     * - autoscaling:DescribeAutoScalingGroups
     *
     * https://docs.aws.amazon.com/compute-optimizer/latest/APIReference/API_GetAutoScalingGroupRecommendations.html
     */
    toGetAutoScalingGroupRecommendations() {
        return this.to('GetAutoScalingGroupRecommendations');
    }
    /**
     * Grants permission to get recommendations for the provided EBS volumes
     *
     * Access Level: List
     *
     * Dependent actions:
     * - ec2:DescribeVolumes
     *
     * https://docs.aws.amazon.com/compute-optimizer/latest/APIReference/API_GetEBSVolumeRecommendations.html
     */
    toGetEBSVolumeRecommendations() {
        return this.to('GetEBSVolumeRecommendations');
    }
    /**
     * Grants permission to get recommendations for the provided EC2 instances
     *
     * Access Level: List
     *
     * Dependent actions:
     * - ec2:DescribeInstances
     *
     * https://docs.aws.amazon.com/compute-optimizer/latest/APIReference/API_GetEC2InstanceRecommendations.html
     */
    toGetEC2InstanceRecommendations() {
        return this.to('GetEC2InstanceRecommendations');
    }
    /**
     * Grants permission to get the recommendation projected metrics of the specified instance
     *
     * Access Level: List
     *
     * Dependent actions:
     * - ec2:DescribeInstances
     *
     * https://docs.aws.amazon.com/compute-optimizer/latest/APIReference/API_GetEC2RecommendationProjectedMetrics.html
     */
    toGetEC2RecommendationProjectedMetrics() {
        return this.to('GetEC2RecommendationProjectedMetrics');
    }
    /**
     * Grants permission to get the recommendation projected metrics of the specified ECS service
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/compute-optimizer/latest/APIReference/API_GetECSServiceRecommendationProjectedMetrics.html
     */
    toGetECSServiceRecommendationProjectedMetrics() {
        return this.to('GetECSServiceRecommendationProjectedMetrics');
    }
    /**
     * Grants permission to get recommendations for the provided ECS services
     *
     * Access Level: List
     *
     * Dependent actions:
     * - ecs:ListClusters
     * - ecs:ListServices
     *
     * https://docs.aws.amazon.com/compute-optimizer/latest/APIReference/API_GetECSServiceRecommendations.html
     */
    toGetECSServiceRecommendations() {
        return this.to('GetECSServiceRecommendations');
    }
    /**
     * Grants permission to get recommendation preferences that are in effect
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifResourceType()
     *
     * Dependent actions:
     * - autoscaling:DescribeAutoScalingGroups
     * - autoscaling:DescribeAutoScalingInstances
     * - ec2:DescribeInstances
     * - rds:DescribeDBClusters
     * - rds:DescribeDBInstances
     *
     * https://docs.aws.amazon.com/compute-optimizer/latest/APIReference/API_GetEffectiveRecommendationPreferences.html
     */
    toGetEffectiveRecommendationPreferences() {
        return this.to('GetEffectiveRecommendationPreferences');
    }
    /**
     * Grants permission to get the enrollment status for the specified account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/compute-optimizer/latest/APIReference/API_GetEnrollmentStatus.html
     */
    toGetEnrollmentStatus() {
        return this.to('GetEnrollmentStatus');
    }
    /**
     * Grants permission to get the enrollment statuses for member accounts of the organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/compute-optimizer/latest/APIReference/API_GetEnrollmentStatusesForOrganization.html
     */
    toGetEnrollmentStatusesForOrganization() {
        return this.to('GetEnrollmentStatusesForOrganization');
    }
    /**
     * Grants permission to get idle recommendations for the specified account(s)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/compute-optimizer/latest/APIReference/API_GetIdleRecommendations.html
     */
    toGetIdleRecommendations() {
        return this.to('GetIdleRecommendations');
    }
    /**
     * Grants permission to get recommendations for the provided Lambda functions
     *
     * Access Level: List
     *
     * Dependent actions:
     * - lambda:ListFunctions
     * - lambda:ListProvisionedConcurrencyConfigs
     *
     * https://docs.aws.amazon.com/compute-optimizer/latest/APIReference/API_GetLambdaFunctionRecommendations.html
     */
    toGetLambdaFunctionRecommendations() {
        return this.to('GetLambdaFunctionRecommendations');
    }
    /**
     * Grants permission to get license recommendations for the specified account(s)
     *
     * Access Level: List
     *
     * Dependent actions:
     * - ec2:DescribeInstances
     *
     * https://docs.aws.amazon.com/compute-optimizer/latest/APIReference/API_GetLicenseRecommendations.html
     */
    toGetLicenseRecommendations() {
        return this.to('GetLicenseRecommendations');
    }
    /**
     * Grants permission to get the recommendation projected metrics of the specified instance
     *
     * Access Level: List
     *
     * Dependent actions:
     * - rds:DescribeDBClusters
     * - rds:DescribeDBInstances
     *
     * https://docs.aws.amazon.com/compute-optimizer/latest/APIReference/API_GetRDSDatabaseRecommendationProjectedMetrics.html
     */
    toGetRDSDatabaseRecommendationProjectedMetrics() {
        return this.to('GetRDSDatabaseRecommendationProjectedMetrics');
    }
    /**
     * Grants permission to get rds recommendations for the specified account(s)
     *
     * Access Level: List
     *
     * Dependent actions:
     * - rds:DescribeDBClusters
     * - rds:DescribeDBInstances
     *
     * https://docs.aws.amazon.com/compute-optimizer/latest/APIReference/API_GetRDSDatabaseRecommendations.html
     */
    toGetRDSDatabaseRecommendations() {
        return this.to('GetRDSDatabaseRecommendations');
    }
    /**
     * Grants permission to get recommendation preferences
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifResourceType()
     *
     * https://docs.aws.amazon.com/compute-optimizer/latest/APIReference/API_GetRecommendationPreferences.html
     */
    toGetRecommendationPreferences() {
        return this.to('GetRecommendationPreferences');
    }
    /**
     * Grants permission to get the recommendation summaries for the specified account(s)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/compute-optimizer/latest/APIReference/API_GetRecommendationSummaries.html
     */
    toGetRecommendationSummaries() {
        return this.to('GetRecommendationSummaries');
    }
    /**
     * Grants permission to put recommendation preferences
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifResourceType()
     *
     * Dependent actions:
     * - autoscaling:DescribeAutoScalingGroups
     * - autoscaling:DescribeAutoScalingInstances
     * - ec2:DescribeInstances
     * - rds:DescribeDBClusters
     * - rds:DescribeDBInstances
     *
     * https://docs.aws.amazon.com/compute-optimizer/latest/APIReference/API_PutRecommendationPreferences.html
     */
    toPutRecommendationPreferences() {
        return this.to('PutRecommendationPreferences');
    }
    /**
     * Grants permission to update the enrollment status
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/compute-optimizer/latest/APIReference/API_UpdateEnrollmentStatus.html
     */
    toUpdateEnrollmentStatus() {
        return this.to('UpdateEnrollmentStatus');
    }
    /**
     * Filters access by the resource type
     *
     * https://docs.aws.amazon.com/compute-optimizer/latest/ug/security-iam.html
     *
     * Applies to actions:
     * - .toDeleteRecommendationPreferences()
     * - .toGetEffectiveRecommendationPreferences()
     * - .toGetRecommendationPreferences()
     * - .toPutRecommendationPreferences()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceType(value, operator) {
        return this.if(`ResourceType`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [compute-optimizer](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscomputeoptimizer.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'compute-optimizer';
        this.accessLevelList = {
            Write: [
                'DeleteRecommendationPreferences',
                'ExportAutoScalingGroupRecommendations',
                'ExportEBSVolumeRecommendations',
                'ExportEC2InstanceRecommendations',
                'ExportECSServiceRecommendations',
                'ExportIdleRecommendations',
                'ExportLambdaFunctionRecommendations',
                'ExportLicenseRecommendations',
                'ExportRDSDatabaseRecommendations',
                'PutRecommendationPreferences',
                'UpdateEnrollmentStatus'
            ],
            List: [
                'DescribeRecommendationExportJobs',
                'GetAutoScalingGroupRecommendations',
                'GetEBSVolumeRecommendations',
                'GetEC2InstanceRecommendations',
                'GetEC2RecommendationProjectedMetrics',
                'GetECSServiceRecommendationProjectedMetrics',
                'GetECSServiceRecommendations',
                'GetEnrollmentStatus',
                'GetEnrollmentStatusesForOrganization',
                'GetIdleRecommendations',
                'GetLambdaFunctionRecommendations',
                'GetLicenseRecommendations',
                'GetRDSDatabaseRecommendationProjectedMetrics',
                'GetRDSDatabaseRecommendations',
                'GetRecommendationSummaries'
            ],
            Read: [
                'GetEffectiveRecommendationPreferences',
                'GetRecommendationPreferences'
            ]
        };
    }
}
exports.ComputeOptimizer = ComputeOptimizer;
//# sourceMappingURL=data:application/json;base64,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