"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Config = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [config](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsconfig.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Config extends shared_1.PolicyStatement {
    /**
     * Grants permission to add all specified resource types to the RecordingGroup of configuration recorder and includes those resource types when recording
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_AssociateResourceTypes.html
     */
    toAssociateResourceTypes() {
        return this.to('AssociateResourceTypes');
    }
    /**
     * Grants permission to return the current configuration items for resources that are present in your AWS Config aggregator
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_BatchGetAggregateResourceConfig.html
     */
    toBatchGetAggregateResourceConfig() {
        return this.to('BatchGetAggregateResourceConfig');
    }
    /**
     * Grants permission to return the current configuration for one or more requested resources
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_BatchGetResourceConfig.html
     */
    toBatchGetResourceConfig() {
        return this.to('BatchGetResourceConfig');
    }
    /**
     * Grants permission to delete the authorization granted to the specified configuration aggregator account in a specified region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteAggregationAuthorization.html
     */
    toDeleteAggregationAuthorization() {
        return this.to('DeleteAggregationAuthorization');
    }
    /**
     * Grants permission to delete the specified AWS Config rule and all of its evaluation results
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteConfigRule.html
     */
    toDeleteConfigRule() {
        return this.to('DeleteConfigRule');
    }
    /**
     * Grants permission to delete the specified configuration aggregator and the aggregated data associated with the aggregator
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteConfigurationAggregator.html
     */
    toDeleteConfigurationAggregator() {
        return this.to('DeleteConfigurationAggregator');
    }
    /**
     * Grants permission to delete the customer managed configuration recorder
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteConfigurationRecorder.html
     */
    toDeleteConfigurationRecorder() {
        return this.to('DeleteConfigurationRecorder');
    }
    /**
     * Grants permission to delete the specified conformance pack and all the AWS Config rules and all evaluation results within that conformance pack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteConformancePack.html
     */
    toDeleteConformancePack() {
        return this.to('DeleteConformancePack');
    }
    /**
     * Grants permission to delete the delivery channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteDeliveryChannel.html
     */
    toDeleteDeliveryChannel() {
        return this.to('DeleteDeliveryChannel');
    }
    /**
     * Grants permission to delete the evaluation results for the specified Config rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteEvaluationResults.html
     */
    toDeleteEvaluationResults() {
        return this.to('DeleteEvaluationResults');
    }
    /**
     * Grants permission to delete the specified organization config rule and all of its evaluation results from all member accounts in that organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteOrganizationConfigRule.html
     */
    toDeleteOrganizationConfigRule() {
        return this.to('DeleteOrganizationConfigRule');
    }
    /**
     * Grants permission to delete the specified organization conformance pack and all of its evaluation results from all member accounts in that organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteOrganizationConformancePack.html
     */
    toDeleteOrganizationConformancePack() {
        return this.to('DeleteOrganizationConformancePack');
    }
    /**
     * Grants permission to delete pending authorization requests for a specified aggregator account in a specified region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeletePendingAggregationRequest.html
     */
    toDeletePendingAggregationRequest() {
        return this.to('DeletePendingAggregationRequest');
    }
    /**
     * Grants permission to delete the remediation configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteRemediationConfiguration.html
     */
    toDeleteRemediationConfiguration() {
        return this.to('DeleteRemediationConfiguration');
    }
    /**
     * Grants permission to delete one or more remediation exceptions for specific resource keys for a specific AWS Config Rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteRemediationExceptions.html
     */
    toDeleteRemediationExceptions() {
        return this.to('DeleteRemediationExceptions');
    }
    /**
     * Grants permission to record the configuration state for a custom resource that has been deleted
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteResourceConfig.html
     */
    toDeleteResourceConfig() {
        return this.to('DeleteResourceConfig');
    }
    /**
     * Grants permission to delete the retention configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteRetentionConfiguration.html
     */
    toDeleteRetentionConfiguration() {
        return this.to('DeleteRetentionConfiguration');
    }
    /**
     * Grants permission to delete the service-linked configuration recorder
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifConfigurationRecorderServicePrincipal()
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteServiceLinkedConfigurationRecorder.html
     */
    toDeleteServiceLinkedConfigurationRecorder() {
        return this.to('DeleteServiceLinkedConfigurationRecorder');
    }
    /**
     * Grants permission to delete the stored query for an AWS account in an AWS Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteStoredQuery.html
     */
    toDeleteStoredQuery() {
        return this.to('DeleteStoredQuery');
    }
    /**
     * Grants permission to schedule delivery of a configuration snapshot to the Amazon S3 bucket in the specified delivery channel
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeliverConfigSnapshot.html
     */
    toDeliverConfigSnapshot() {
        return this.to('DeliverConfigSnapshot');
    }
    /**
     * Grants permission to return a list of compliant and noncompliant rules with the number of resources for compliant and noncompliant rules
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeAggregateComplianceByConfigRules.html
     */
    toDescribeAggregateComplianceByConfigRules() {
        return this.to('DescribeAggregateComplianceByConfigRules');
    }
    /**
     * Grants permission to return a list of compliant and noncompliant conformance packs along with count of compliant, non-compliant and total rules within each conformance pack
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeAggregateComplianceByConformancePacks.html
     */
    toDescribeAggregateComplianceByConformancePacks() {
        return this.to('DescribeAggregateComplianceByConformancePacks');
    }
    /**
     * Grants permission to return a list of authorizations granted to various aggregator accounts and regions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeAggregationAuthorizations.html
     */
    toDescribeAggregationAuthorizations() {
        return this.to('DescribeAggregationAuthorizations');
    }
    /**
     * Grants permission to indicate whether the specified AWS Config rules are compliant
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeComplianceByConfigRule.html
     */
    toDescribeComplianceByConfigRule() {
        return this.to('DescribeComplianceByConfigRule');
    }
    /**
     * Grants permission to indicate whether the specified AWS resources are compliant
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeComplianceByResource.html
     */
    toDescribeComplianceByResource() {
        return this.to('DescribeComplianceByResource');
    }
    /**
     * Grants permission to return status information for each of your AWS managed Config rules
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigRuleEvaluationStatus.html
     */
    toDescribeConfigRuleEvaluationStatus() {
        return this.to('DescribeConfigRuleEvaluationStatus');
    }
    /**
     * Grants permission to return details about your AWS Config rules
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigRules.html
     */
    toDescribeConfigRules() {
        return this.to('DescribeConfigRules');
    }
    /**
     * Grants permission to return status information for sources within an aggregator
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigurationAggregatorSourcesStatus.html
     */
    toDescribeConfigurationAggregatorSourcesStatus() {
        return this.to('DescribeConfigurationAggregatorSourcesStatus');
    }
    /**
     * Grants permission to return the details of one or more configuration aggregators
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigurationAggregators.html
     */
    toDescribeConfigurationAggregators() {
        return this.to('DescribeConfigurationAggregators');
    }
    /**
     * Grants permission to return the current status of the specified configuration recorder
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifConfigurationRecorderServicePrincipal()
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigurationRecorderStatus.html
     */
    toDescribeConfigurationRecorderStatus() {
        return this.to('DescribeConfigurationRecorderStatus');
    }
    /**
     * Grants permission to return the names of one or more specified configuration recorders
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifConfigurationRecorderServicePrincipal()
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigurationRecorders.html
     */
    toDescribeConfigurationRecorders() {
        return this.to('DescribeConfigurationRecorders');
    }
    /**
     * Grants permission to return compliance information for each rule in that conformance pack
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConformancePackCompliance.html
     */
    toDescribeConformancePackCompliance() {
        return this.to('DescribeConformancePackCompliance');
    }
    /**
     * Grants permission to provide one or more conformance packs deployment status
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConformancePackStatus.html
     */
    toDescribeConformancePackStatus() {
        return this.to('DescribeConformancePackStatus');
    }
    /**
     * Grants permission to return a list of one or more conformance packs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConformancePacks.html
     */
    toDescribeConformancePacks() {
        return this.to('DescribeConformancePacks');
    }
    /**
     * Grants permission to return the current status of the specified delivery channel
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeDeliveryChannelStatus.html
     */
    toDescribeDeliveryChannelStatus() {
        return this.to('DescribeDeliveryChannelStatus');
    }
    /**
     * Grants permission to return details about the specified delivery channel
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeDeliveryChannels.html
     */
    toDescribeDeliveryChannels() {
        return this.to('DescribeDeliveryChannels');
    }
    /**
     * Grants permission to provide organization config rule deployment status for an organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeOrganizationConfigRuleStatuses.html
     */
    toDescribeOrganizationConfigRuleStatuses() {
        return this.to('DescribeOrganizationConfigRuleStatuses');
    }
    /**
     * Grants permission to return a list of organization config rules
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeOrganizationConfigRules.html
     */
    toDescribeOrganizationConfigRules() {
        return this.to('DescribeOrganizationConfigRules');
    }
    /**
     * Grants permission to provide organization conformance pack deployment status for an organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeOrganizationConformancePackStatuses.html
     */
    toDescribeOrganizationConformancePackStatuses() {
        return this.to('DescribeOrganizationConformancePackStatuses');
    }
    /**
     * Grants permission to return a list of organization conformance packs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeOrganizationConformancePacks.html
     */
    toDescribeOrganizationConformancePacks() {
        return this.to('DescribeOrganizationConformancePacks');
    }
    /**
     * Grants permission to return a list of all pending aggregation requests
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribePendingAggregationRequests.html
     */
    toDescribePendingAggregationRequests() {
        return this.to('DescribePendingAggregationRequests');
    }
    /**
     * Grants permission to return the details of one or more remediation configurations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeRemediationConfigurations.html
     */
    toDescribeRemediationConfigurations() {
        return this.to('DescribeRemediationConfigurations');
    }
    /**
     * Grants permission to return the details of one or more remediation exceptions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeRemediationExceptions.html
     */
    toDescribeRemediationExceptions() {
        return this.to('DescribeRemediationExceptions');
    }
    /**
     * Grants permission to provide a detailed view of a Remediation Execution for a set of resources including state, timestamps and any error messages for steps that have failed
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeRemediationExecutionStatus.html
     */
    toDescribeRemediationExecutionStatus() {
        return this.to('DescribeRemediationExecutionStatus');
    }
    /**
     * Grants permission to return the details of one or more retention configurations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeRetentionConfigurations.html
     */
    toDescribeRetentionConfigurations() {
        return this.to('DescribeRetentionConfigurations');
    }
    /**
     * Grants permission to remove all specified resource types from the RecordingGroup of configuration recorder and excludes these resource types when recording
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DisassociateResourceTypes.html
     */
    toDisassociateResourceTypes() {
        return this.to('DisassociateResourceTypes');
    }
    /**
     * Grants permission to return the evaluation results for the specified AWS Config rule for a specific resource in a rule
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetAggregateComplianceDetailsByConfigRule.html
     */
    toGetAggregateComplianceDetailsByConfigRule() {
        return this.to('GetAggregateComplianceDetailsByConfigRule');
    }
    /**
     * Grants permission to return the number of compliant and noncompliant rules for one or more accounts and regions in an aggregator
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetAggregateConfigRuleComplianceSummary.html
     */
    toGetAggregateConfigRuleComplianceSummary() {
        return this.to('GetAggregateConfigRuleComplianceSummary');
    }
    /**
     * Grants permission to return the number of compliant and noncompliant conformance packs for one or more accounts and regions in an aggregator
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetAggregateConformancePackComplianceSummary.html
     */
    toGetAggregateConformancePackComplianceSummary() {
        return this.to('GetAggregateConformancePackComplianceSummary');
    }
    /**
     * Grants permission to return the resource counts across accounts and regions that are present in your AWS Config aggregator
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetAggregateDiscoveredResourceCounts.html
     */
    toGetAggregateDiscoveredResourceCounts() {
        return this.to('GetAggregateDiscoveredResourceCounts');
    }
    /**
     * Grants permission to return configuration item that is aggregated for your specific resource in a specific source account and region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetAggregateResourceConfig.html
     */
    toGetAggregateResourceConfig() {
        return this.to('GetAggregateResourceConfig');
    }
    /**
     * Grants permission to return the evaluation results for the specified AWS Config rule
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetComplianceDetailsByConfigRule.html
     */
    toGetComplianceDetailsByConfigRule() {
        return this.to('GetComplianceDetailsByConfigRule');
    }
    /**
     * Grants permission to return the evaluation results for the specified AWS resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetComplianceDetailsByResource.html
     */
    toGetComplianceDetailsByResource() {
        return this.to('GetComplianceDetailsByResource');
    }
    /**
     * Grants permission to return the number of AWS Config rules that are compliant and noncompliant, up to a maximum of 25 for each
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetComplianceSummaryByConfigRule.html
     */
    toGetComplianceSummaryByConfigRule() {
        return this.to('GetComplianceSummaryByConfigRule');
    }
    /**
     * Grants permission to return the number of resources that are compliant and the number that are noncompliant
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetComplianceSummaryByResourceType.html
     */
    toGetComplianceSummaryByResourceType() {
        return this.to('GetComplianceSummaryByResourceType');
    }
    /**
     * Grants permission to return compliance details of a conformance pack for all AWS resources that are monitered by conformance pack
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetConformancePackComplianceDetails.html
     */
    toGetConformancePackComplianceDetails() {
        return this.to('GetConformancePackComplianceDetails');
    }
    /**
     * Grants permission to provide compliance summary for one or more conformance packs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetConformancePackComplianceSummary.html
     */
    toGetConformancePackComplianceSummary() {
        return this.to('GetConformancePackComplianceSummary');
    }
    /**
     * Grants permission to return the policy definition containing the logic for your AWS Config Custom Policy rule
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetCustomRulePolicy.html
     */
    toGetCustomRulePolicy() {
        return this.to('GetCustomRulePolicy');
    }
    /**
     * Grants permission to return the resource types, the number of each resource type, and the total number of resources that AWS Config is recording in this region for your AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetDiscoveredResourceCounts.html
     */
    toGetDiscoveredResourceCounts() {
        return this.to('GetDiscoveredResourceCounts');
    }
    /**
     * Grants permission to return detailed status for each member account within an organization for a given organization config rule
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetOrganizationConfigRuleDetailedStatus.html
     */
    toGetOrganizationConfigRuleDetailedStatus() {
        return this.to('GetOrganizationConfigRuleDetailedStatus');
    }
    /**
     * Grants permission to return detailed status for each member account within an organization for a given organization conformance pack
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetOrganizationConformancePackDetailedStatus.html
     */
    toGetOrganizationConformancePackDetailedStatus() {
        return this.to('GetOrganizationConformancePackDetailedStatus');
    }
    /**
     * Grants permission to return the policy definition containing the logic for your organization AWS Config Custom Policy rule
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetOrganizationCustomRulePolicy.html
     */
    toGetOrganizationCustomRulePolicy() {
        return this.to('GetOrganizationCustomRulePolicy');
    }
    /**
     * Grants permission to return a list of configuration items for the specified resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetResourceConfigHistory.html
     */
    toGetResourceConfigHistory() {
        return this.to('GetResourceConfigHistory');
    }
    /**
     * Grants permission to return the summary of resource evaluations for a specific resource evaluation ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetResourceEvaluationSummary.html
     */
    toGetResourceEvaluationSummary() {
        return this.to('GetResourceEvaluationSummary');
    }
    /**
     * Grants permission to return the details of a specific stored query
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetStoredQuery.html
     */
    toGetStoredQuery() {
        return this.to('GetStoredQuery');
    }
    /**
     * Grants permission to accept a resource type and returns a list of resource identifiers that are aggregated for a specific resource type across accounts and regions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_ListAggregateDiscoveredResources.html
     */
    toListAggregateDiscoveredResources() {
        return this.to('ListAggregateDiscoveredResources');
    }
    /**
     * Grants permission to list the configuration recorder summaries for an AWS account in an AWS Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_ListConfigurationRecorders.html
     */
    toListConfigurationRecorders() {
        return this.to('ListConfigurationRecorders');
    }
    /**
     * Grants permission to return the percentage of compliant rule-resource combinations in a conformance pack compared to the number of total possible rule-resource combinations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_ListConformancePackComplianceScores.html
     */
    toListConformancePackComplianceScores() {
        return this.to('ListConformancePackComplianceScores');
    }
    /**
     * Grants permission to accept a resource type and returns a list of resource identifiers for the resources of that type
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_ListDiscoveredResources.html
     */
    toListDiscoveredResources() {
        return this.to('ListDiscoveredResources');
    }
    /**
     * Grants permission to list the resource evaluation summaries for an AWS account in an AWS Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_ListResourceEvaluations.html
     */
    toListResourceEvaluations() {
        return this.to('ListResourceEvaluations');
    }
    /**
     * Grants permission to list the stored queries for an AWS account in an AWS Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_ListStoredQueries.html
     */
    toListStoredQueries() {
        return this.to('ListStoredQueries');
    }
    /**
     * Grants permission to list the tags for AWS Config resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to authorize the aggregator account and region to collect data from the source account and region
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutAggregationAuthorization.html
     */
    toPutAggregationAuthorization() {
        return this.to('PutAggregationAuthorization');
    }
    /**
     * Grants permission to add or update an AWS Config rule for evaluating whether your AWS resources comply with your desired configurations
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutConfigRule.html
     */
    toPutConfigRule() {
        return this.to('PutConfigRule');
    }
    /**
     * Grants permission to create and update the configuration aggregator with the selected source accounts and regions
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     * - organizations:EnableAWSServiceAccess
     * - organizations:ListDelegatedAdministrators
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutConfigurationAggregator.html
     */
    toPutConfigurationAggregator() {
        return this.to('PutConfigurationAggregator');
    }
    /**
     * Grants permission to create or update a customer managed configuration recorder to record the selected resource configurations
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutConfigurationRecorder.html
     */
    toPutConfigurationRecorder() {
        return this.to('PutConfigurationRecorder');
    }
    /**
     * Grants permission to create or update a conformance pack
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - iam:PassRole
     * - s3:GetObject
     * - s3:ListBucket
     * - ssm:GetDocument
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutConformancePack.html
     */
    toPutConformancePack() {
        return this.to('PutConformancePack');
    }
    /**
     * Grants permission to create a delivery channel object to deliver configuration information to an Amazon S3 bucket and Amazon SNS topic
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutDeliveryChannel.html
     */
    toPutDeliveryChannel() {
        return this.to('PutDeliveryChannel');
    }
    /**
     * Grants permission to be used by an AWS Lambda function to deliver evaluation results to AWS Config
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutEvaluations.html
     */
    toPutEvaluations() {
        return this.to('PutEvaluations');
    }
    /**
     * Grants permission to deliver evaluation result to AWS Config
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutExternalEvaluation.html
     */
    toPutExternalEvaluation() {
        return this.to('PutExternalEvaluation');
    }
    /**
     * Grants permission to add or update organization config rule for your entire organization evaluating whether your AWS resources comply with your desired configurations
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - iam:PassRole
     * - organizations:EnableAWSServiceAccess
     * - organizations:ListDelegatedAdministrators
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutOrganizationConfigRule.html
     */
    toPutOrganizationConfigRule() {
        return this.to('PutOrganizationConfigRule');
    }
    /**
     * Grants permission to add or update organization conformance pack for your entire organization evaluating whether your AWS resources comply with your desired configurations
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - iam:PassRole
     * - organizations:EnableAWSServiceAccess
     * - organizations:ListDelegatedAdministrators
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutOrganizationConformancePack.html
     */
    toPutOrganizationConformancePack() {
        return this.to('PutOrganizationConformancePack');
    }
    /**
     * Grants permission to add or update the remediation configuration with a specific AWS Config rule with the selected target or action
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutRemediationConfigurations.html
     */
    toPutRemediationConfigurations() {
        return this.to('PutRemediationConfigurations');
    }
    /**
     * Grants permission to add or update remediation exceptions for specific resources for a specific AWS Config rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutRemediationExceptions.html
     */
    toPutRemediationExceptions() {
        return this.to('PutRemediationExceptions');
    }
    /**
     * Grants permission to record the configuration state for the resource provided in the request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutResourceConfig.html
     */
    toPutResourceConfig() {
        return this.to('PutResourceConfig');
    }
    /**
     * Grants permission to create and update the retention configuration with details about retention period (number of days) that AWS Config stores your historical information
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutRetentionConfiguration.html
     */
    toPutRetentionConfiguration() {
        return this.to('PutRetentionConfiguration');
    }
    /**
     * Grants permission to create a new service-linked configuration recorder to record the resource configurations in scope for the linked service
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifConfigurationRecorderServicePrincipal()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutServiceLinkedConfigurationRecorder.html
     */
    toPutServiceLinkedConfigurationRecorder() {
        return this.to('PutServiceLinkedConfigurationRecorder');
    }
    /**
     * Grants permission to save a new query or updates an existing saved query
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutStoredQuery.html
     */
    toPutStoredQuery() {
        return this.to('PutStoredQuery');
    }
    /**
     * Grants permission to accept a structured query language (SQL) SELECT command and an aggregator to query configuration state of AWS resources across multiple accounts and regions, performs the corresponding search, and returns resource configurations matching the properties
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_SelectAggregateResourceConfig.html
     */
    toSelectAggregateResourceConfig() {
        return this.to('SelectAggregateResourceConfig');
    }
    /**
     * Grants permission to accept a structured query language (SQL) SELECT command, performs the corresponding search, and returns resource configurations matching the properties
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_SelectResourceConfig.html
     */
    toSelectResourceConfig() {
        return this.to('SelectResourceConfig');
    }
    /**
     * Grants permission to evaluate your resources against the specified Config rules
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_StartConfigRulesEvaluation.html
     */
    toStartConfigRulesEvaluation() {
        return this.to('StartConfigRulesEvaluation');
    }
    /**
     * Grants permission to the customer managed configuration recorder to start recording configurations of the AWS resources you have selected to record in your AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_StartConfigurationRecorder.html
     */
    toStartConfigurationRecorder() {
        return this.to('StartConfigurationRecorder');
    }
    /**
     * Grants permission to run an on-demand remediation for the specified AWS Config rules against the last known remediation configuration
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_StartRemediationExecution.html
     */
    toStartRemediationExecution() {
        return this.to('StartRemediationExecution');
    }
    /**
     * Grants permission to evaluate your resource details against the AWS Config rules in your account
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - cloudformation:DescribeType
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_StartResourceEvaluation.html
     */
    toStartResourceEvaluation() {
        return this.to('StartResourceEvaluation');
    }
    /**
     * Grants permission to the customer managed configuration recorder to stop recording configurations of the AWS resources you have selected to record in your AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_StopConfigurationRecorder.html
     */
    toStopConfigurationRecorder() {
        return this.to('StopConfigurationRecorder');
    }
    /**
     * Grants permission to associate the specified tags to a resource with the specified resourceArn
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to delete specified tags from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Adds a resource of type AggregationAuthorization to the statement
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_AggregationAuthorization.html
     *
     * @param aggregatorAccount - Identifier for the aggregatorAccount.
     * @param aggregatorRegion - Identifier for the aggregatorRegion.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAggregationAuthorization(aggregatorAccount, aggregatorRegion, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:config:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:aggregation-authorization/${aggregatorAccount}/${aggregatorRegion}`);
    }
    /**
     * Adds a resource of type ConfigurationAggregator to the statement
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_ConfigurationAggregator.html
     *
     * @param aggregatorId - Identifier for the aggregatorId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConfigurationAggregator(aggregatorId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:config:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:config-aggregator/${aggregatorId}`);
    }
    /**
     * Adds a resource of type ConfigRule to the statement
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_ConfigRule.html
     *
     * @param configRuleId - Identifier for the configRuleId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConfigRule(configRuleId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:config:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:config-rule/${configRuleId}`);
    }
    /**
     * Adds a resource of type ConformancePack to the statement
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_ConformancePackDetail.html
     *
     * @param conformancePackName - Identifier for the conformancePackName.
     * @param conformancePackId - Identifier for the conformancePackId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConformancePack(conformancePackName, conformancePackId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:config:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:conformance-pack/${conformancePackName}/${conformancePackId}`);
    }
    /**
     * Adds a resource of type OrganizationConfigRule to the statement
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_OrganizationConfigRule.html
     *
     * @param organizationConfigRuleId - Identifier for the organizationConfigRuleId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onOrganizationConfigRule(organizationConfigRuleId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:config:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:organization-config-rule/${organizationConfigRuleId}`);
    }
    /**
     * Adds a resource of type OrganizationConformancePack to the statement
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_OrganizationConformancePack.html
     *
     * @param organizationConformancePackId - Identifier for the organizationConformancePackId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onOrganizationConformancePack(organizationConformancePackId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:config:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:organization-conformance-pack/${organizationConformancePackId}`);
    }
    /**
     * Adds a resource of type RemediationConfiguration to the statement
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_RemediationConfiguration.html
     *
     * @param remediationConfigurationId - Identifier for the remediationConfigurationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onRemediationConfiguration(remediationConfigurationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:config:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:remediation-configuration/${remediationConfigurationId}`);
    }
    /**
     * Adds a resource of type StoredQuery to the statement
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_StoredQuery.html
     *
     * @param storedQueryName - Identifier for the storedQueryName.
     * @param storedQueryId - Identifier for the storedQueryId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStoredQuery(storedQueryName, storedQueryId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:config:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:stored-query/${storedQueryName}/${storedQueryId}`);
    }
    /**
     * Adds a resource of type ConfigurationRecorder to the statement
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_ConfigurationRecorder.html
     *
     * @param recorderName - Identifier for the recorderName.
     * @param recorderId - Identifier for the recorderId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConfigurationRecorder(recorderName, recorderId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:config:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:configuration-recorder/${recorderName}/${recorderId}`);
    }
    /**
     * Filters access by the allowed set of values for each of the tags
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toPutAggregationAuthorization()
     * - .toPutConfigRule()
     * - .toPutConfigurationAggregator()
     * - .toPutConfigurationRecorder()
     * - .toPutServiceLinkedConfigurationRecorder()
     * - .toPutStoredQuery()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag-value associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - AggregationAuthorization
     * - ConfigurationAggregator
     * - ConfigRule
     * - ConformancePack
     * - OrganizationConfigRule
     * - OrganizationConformancePack
     * - StoredQuery
     * - ConfigurationRecorder
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of mandatory tags in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toPutAggregationAuthorization()
     * - .toPutConfigRule()
     * - .toPutConfigurationAggregator()
     * - .toPutConfigurationRecorder()
     * - .toPutServiceLinkedConfigurationRecorder()
     * - .toPutStoredQuery()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by service principal of the configuration recorder
     *
     * https://docs.aws.amazon.com/config/latest/developerguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * Applies to actions:
     * - .toDeleteServiceLinkedConfigurationRecorder()
     * - .toDescribeConfigurationRecorderStatus()
     * - .toDescribeConfigurationRecorders()
     * - .toPutServiceLinkedConfigurationRecorder()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifConfigurationRecorderServicePrincipal(value, operator) {
        return this.if(`ConfigurationRecorderServicePrincipal`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [config](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsconfig.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'config';
        this.accessLevelList = {
            Write: [
                'AssociateResourceTypes',
                'DeleteAggregationAuthorization',
                'DeleteConfigRule',
                'DeleteConfigurationAggregator',
                'DeleteConfigurationRecorder',
                'DeleteConformancePack',
                'DeleteDeliveryChannel',
                'DeleteEvaluationResults',
                'DeleteOrganizationConfigRule',
                'DeleteOrganizationConformancePack',
                'DeletePendingAggregationRequest',
                'DeleteRemediationConfiguration',
                'DeleteRemediationExceptions',
                'DeleteResourceConfig',
                'DeleteRetentionConfiguration',
                'DeleteServiceLinkedConfigurationRecorder',
                'DeleteStoredQuery',
                'DisassociateResourceTypes',
                'PutAggregationAuthorization',
                'PutConfigRule',
                'PutConfigurationAggregator',
                'PutConfigurationRecorder',
                'PutConformancePack',
                'PutDeliveryChannel',
                'PutEvaluations',
                'PutExternalEvaluation',
                'PutOrganizationConfigRule',
                'PutOrganizationConformancePack',
                'PutRemediationConfigurations',
                'PutRemediationExceptions',
                'PutResourceConfig',
                'PutRetentionConfiguration',
                'PutServiceLinkedConfigurationRecorder',
                'PutStoredQuery',
                'StartConfigRulesEvaluation',
                'StartConfigurationRecorder',
                'StartRemediationExecution',
                'StartResourceEvaluation',
                'StopConfigurationRecorder'
            ],
            Read: [
                'BatchGetAggregateResourceConfig',
                'BatchGetResourceConfig',
                'DeliverConfigSnapshot',
                'DescribeAggregateComplianceByConfigRules',
                'DescribeAggregateComplianceByConformancePacks',
                'DescribeComplianceByConfigRule',
                'DescribeComplianceByResource',
                'DescribeConfigRuleEvaluationStatus',
                'DescribeConfigurationAggregatorSourcesStatus',
                'DescribeConfigurationRecorderStatus',
                'DescribeConfigurationRecorders',
                'DescribeConformancePackCompliance',
                'DescribeConformancePackStatus',
                'DescribeDeliveryChannelStatus',
                'DescribeOrganizationConfigRuleStatuses',
                'DescribeOrganizationConformancePackStatuses',
                'DescribeRemediationExecutionStatus',
                'GetAggregateComplianceDetailsByConfigRule',
                'GetAggregateConfigRuleComplianceSummary',
                'GetAggregateConformancePackComplianceSummary',
                'GetAggregateDiscoveredResourceCounts',
                'GetAggregateResourceConfig',
                'GetComplianceDetailsByConfigRule',
                'GetComplianceDetailsByResource',
                'GetComplianceSummaryByConfigRule',
                'GetComplianceSummaryByResourceType',
                'GetConformancePackComplianceDetails',
                'GetConformancePackComplianceSummary',
                'GetCustomRulePolicy',
                'GetDiscoveredResourceCounts',
                'GetOrganizationConfigRuleDetailedStatus',
                'GetOrganizationConformancePackDetailedStatus',
                'GetOrganizationCustomRulePolicy',
                'GetResourceConfigHistory',
                'GetResourceEvaluationSummary',
                'GetStoredQuery',
                'ListTagsForResource',
                'SelectAggregateResourceConfig',
                'SelectResourceConfig'
            ],
            List: [
                'DescribeAggregationAuthorizations',
                'DescribeConfigRules',
                'DescribeConfigurationAggregators',
                'DescribeConformancePacks',
                'DescribeDeliveryChannels',
                'DescribeOrganizationConfigRules',
                'DescribeOrganizationConformancePacks',
                'DescribePendingAggregationRequests',
                'DescribeRemediationConfigurations',
                'DescribeRemediationExceptions',
                'DescribeRetentionConfigurations',
                'ListAggregateDiscoveredResources',
                'ListConfigurationRecorders',
                'ListConformancePackComplianceScores',
                'ListDiscoveredResources',
                'ListResourceEvaluations',
                'ListStoredQueries'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Config = Config;
//# sourceMappingURL=data:application/json;base64,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