"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Profile = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [profile](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonconnectcustomerprofiles.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Profile extends shared_1.PolicyStatement {
    /**
     * Grants permission to add a profile key
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_AddProfileKey.html
     */
    toAddProfileKey() {
        return this.to('AddProfileKey');
    }
    /**
     * Grants permission to retrieve a calculated attribute for the specific profiles in the domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_BatchGetCalculatedAttributeForProfile.html
     */
    toBatchGetCalculatedAttributeForProfile() {
        return this.to('BatchGetCalculatedAttributeForProfile');
    }
    /**
     * Grants permission to get profiles in the domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_BatchGetProfile.html
     */
    toBatchGetProfile() {
        return this.to('BatchGetProfile');
    }
    /**
     * Grants permission to create a calculated attribute definition in the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_CreateCalculatedAttributeDefinition.html
     */
    toCreateCalculatedAttributeDefinition() {
        return this.to('CreateCalculatedAttributeDefinition');
    }
    /**
     * Grants permission to create a Domain
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_CreateDomain.html
     */
    toCreateDomain() {
        return this.to('CreateDomain');
    }
    /**
     * Grants permission to put an event stream in a domain
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PutRolePolicy
     * - kinesis:DescribeStreamSummary
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_CreateEventStream.html
     */
    toCreateEventStream() {
        return this.to('CreateEventStream');
    }
    /**
     * Grants permission to create an event trigger in the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_CreateEventTrigger.html
     */
    toCreateEventTrigger() {
        return this.to('CreateEventTrigger');
    }
    /**
     * Grants permission to create an integration workflow in a domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_CreateIntegrationWorkflow.html
     */
    toCreateIntegrationWorkflow() {
        return this.to('CreateIntegrationWorkflow');
    }
    /**
     * Grants permission to create a profile in the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_CreateProfile.html
     */
    toCreateProfile() {
        return this.to('CreateProfile');
    }
    /**
     * Grants permission to create a segment definition in the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_CreateSegmentDefinition.html
     */
    toCreateSegmentDefinition() {
        return this.to('CreateSegmentDefinition');
    }
    /**
     * Grants permission to create a segment estimate in the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_CreateSegmentEstimate.html
     */
    toCreateSegmentEstimate() {
        return this.to('CreateSegmentEstimate');
    }
    /**
     * Grants permission to create a segment snapshot in the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_CreateSegmentSnapshot.html
     */
    toCreateSegmentSnapshot() {
        return this.to('CreateSegmentSnapshot');
    }
    /**
     * Grants permission to create a snapshot in the domain
     *
     * Access Level: Write
     */
    toCreateSnapshot() {
        return this.to('CreateSnapshot');
    }
    /**
     * Grants permission to delete a calculated attribute definition in the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_DeleteCalculatedAttributeDefinition.html
     */
    toDeleteCalculatedAttributeDefinition() {
        return this.to('DeleteCalculatedAttributeDefinition');
    }
    /**
     * Grants permission to delete a Domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_DeleteDomain.html
     */
    toDeleteDomain() {
        return this.to('DeleteDomain');
    }
    /**
     * Grants permission to delete an event stream in a domain
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:DeleteRolePolicy
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_DeleteEventStream.html
     */
    toDeleteEventStream() {
        return this.to('DeleteEventStream');
    }
    /**
     * Grants permission to delete an event trigger in the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_DeleteEventTrigger.html
     */
    toDeleteEventTrigger() {
        return this.to('DeleteEventTrigger');
    }
    /**
     * Grants permission to delete a integration in a domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_DeleteIntegration.html
     */
    toDeleteIntegration() {
        return this.to('DeleteIntegration');
    }
    /**
     * Grants permission to delete a profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_DeleteProfile.html
     */
    toDeleteProfile() {
        return this.to('DeleteProfile');
    }
    /**
     * Grants permission to delete a profile key
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_DeleteProfileKey.html
     */
    toDeleteProfileKey() {
        return this.to('DeleteProfileKey');
    }
    /**
     * Grants permission to delete a profile object
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_DeleteProfileObject.html
     */
    toDeleteProfileObject() {
        return this.to('DeleteProfileObject');
    }
    /**
     * Grants permission to delete a specific profile object type in the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_DeleteProfileObjectType.html
     */
    toDeleteProfileObjectType() {
        return this.to('DeleteProfileObjectType');
    }
    /**
     * Grants permission to delete a segment definition in the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_DeleteSegmentDefinition.html
     */
    toDeleteSegmentDefinition() {
        return this.to('DeleteSegmentDefinition');
    }
    /**
     * Grants permission to delete a workflow in a domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_DeleteWorkflow.html
     */
    toDeleteWorkflow() {
        return this.to('DeleteWorkflow');
    }
    /**
     * Grants permission to auto detect object type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_DetectProfileObjectType.html
     */
    toDetectProfileObjectType() {
        return this.to('DetectProfileObjectType');
    }
    /**
     * Grants permission to get a preview of auto merging in a domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetAutoMergingPreview.html
     */
    toGetAutoMergingPreview() {
        return this.to('GetAutoMergingPreview');
    }
    /**
     * Grants permission to get a calculated attribute definition in the domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetCalculatedAttributeDefinition.html
     */
    toGetCalculatedAttributeDefinition() {
        return this.to('GetCalculatedAttributeDefinition');
    }
    /**
     * Grants permission to retrieve a calculated attribute for a specific profile in the domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetCalculatedAttributeForProfile.html
     */
    toGetCalculatedAttributeForProfile() {
        return this.to('GetCalculatedAttributeForProfile');
    }
    /**
     * Grants permission to get a specific domain in an account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetDomain.html
     */
    toGetDomain() {
        return this.to('GetDomain');
    }
    /**
     * Grants permission to get a specific event stream in a domain
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kinesis:DescribeStreamSummary
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetEventStream.html
     */
    toGetEventStream() {
        return this.to('GetEventStream');
    }
    /**
     * Grants permission to get an event trigger in the domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetEventTrigger.html
     */
    toGetEventTrigger() {
        return this.to('GetEventTrigger');
    }
    /**
     * Grants permission to get an identity resolution job in a domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetIdentityResolutionJob.html
     */
    toGetIdentityResolutionJob() {
        return this.to('GetIdentityResolutionJob');
    }
    /**
     * Grants permission to get a specific integrations in a domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetIntegration.html
     */
    toGetIntegration() {
        return this.to('GetIntegration');
    }
    /**
     * Grants permission to get profile matches in a domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetMatches.html
     */
    toGetMatches() {
        return this.to('GetMatches');
    }
    /**
     * Grants permission to get a specific profile object type in the domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetProfileObjectType.html
     */
    toGetProfileObjectType() {
        return this.to('GetProfileObjectType');
    }
    /**
     * Grants permission to get a specific object type template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetProfileObjectTypeTemplate.html
     */
    toGetProfileObjectTypeTemplate() {
        return this.to('GetProfileObjectTypeTemplate');
    }
    /**
     * Grants permission to get a segment definition in the domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetSegmentDefinition.html
     */
    toGetSegmentDefinition() {
        return this.to('GetSegmentDefinition');
    }
    /**
     * Grants permission to get a segment estimate in the domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetSegmentEstimate.html
     */
    toGetSegmentEstimate() {
        return this.to('GetSegmentEstimate');
    }
    /**
     * Grants permission to determine if the given profiles are part of a segment in the domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetSegmentMembership.html
     */
    toGetSegmentMembership() {
        return this.to('GetSegmentMembership');
    }
    /**
     * Grants permission to get a segment snapshot in the domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetSegmentSnapshot.html
     */
    toGetSegmentSnapshot() {
        return this.to('GetSegmentSnapshot');
    }
    /**
     * Grants permission to get all the similar profiles in the domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetSimilarProfiles.html
     */
    toGetSimilarProfiles() {
        return this.to('GetSimilarProfiles');
    }
    /**
     * Grants permission to get a snapshot in the domain
     *
     * Access Level: Read
     */
    toGetSnapshot() {
        return this.to('GetSnapshot');
    }
    /**
     * Grants permission to get workflow details in a domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetWorkflow.html
     */
    toGetWorkflow() {
        return this.to('GetWorkflow');
    }
    /**
     * Grants permission to get workflow step details in a domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetWorkflowSteps.html
     */
    toGetWorkflowSteps() {
        return this.to('GetWorkflowSteps');
    }
    /**
     * Grants permission to list all the integrations in the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListAccountIntegrations.html
     */
    toListAccountIntegrations() {
        return this.to('ListAccountIntegrations');
    }
    /**
     * Grants permission to list all the calculated attribute definitions in the domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListCalculatedAttributeDefinitions.html
     */
    toListCalculatedAttributeDefinitions() {
        return this.to('ListCalculatedAttributeDefinitions');
    }
    /**
     * Grants permission to list all calculated attributes for a specific profile in the domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListCalculatedAttributesForProfile.html
     */
    toListCalculatedAttributesForProfile() {
        return this.to('ListCalculatedAttributesForProfile');
    }
    /**
     * Grants permission to list all the domains in an account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListDomains.html
     */
    toListDomains() {
        return this.to('ListDomains');
    }
    /**
     * Grants permission to list all the event streams in a specific domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListEventStreams.html
     */
    toListEventStreams() {
        return this.to('ListEventStreams');
    }
    /**
     * Grants permission to list all the event triggers in the domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListEventTriggers.html
     */
    toListEventTriggers() {
        return this.to('ListEventTriggers');
    }
    /**
     * Grants permission to list identity resolution jobs in a domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListIdentityResolutionJobs.html
     */
    toListIdentityResolutionJobs() {
        return this.to('ListIdentityResolutionJobs');
    }
    /**
     * Grants permission to list all the integrations in a specific domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListIntegrations.html
     */
    toListIntegrations() {
        return this.to('ListIntegrations');
    }
    /**
     * Grants permission to list all the attributes of a specific object type in the domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListObjectTypeAttributes.html
     */
    toListObjectTypeAttributes() {
        return this.to('ListObjectTypeAttributes');
    }
    /**
     * Grants permission to list all the values of a profile attribute in the domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListProfileAttributeValues.html
     */
    toListProfileAttributeValues() {
        return this.to('ListProfileAttributeValues');
    }
    /**
     * Grants permission to list all the profile object type templates in the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListProfileObjectTypeTemplates.html
     */
    toListProfileObjectTypeTemplates() {
        return this.to('ListProfileObjectTypeTemplates');
    }
    /**
     * Grants permission to list all the profile object types in the domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListProfileObjectTypes.html
     */
    toListProfileObjectTypes() {
        return this.to('ListProfileObjectTypes');
    }
    /**
     * Grants permission to list all the profile objects for a profile
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListProfileObjects.html
     */
    toListProfileObjects() {
        return this.to('ListProfileObjects');
    }
    /**
     * Grants permission to list all the rule-based matching result in the domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListRuleBasedMatches.html
     */
    toListRuleBasedMatches() {
        return this.to('ListRuleBasedMatches');
    }
    /**
     * Grants permission to list all the segment definitions in the domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListSegmentDefinitions.html
     */
    toListSegmentDefinitions() {
        return this.to('ListSegmentDefinitions');
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list all the workflows in a specific domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListWorkflows.html
     */
    toListWorkflows() {
        return this.to('ListWorkflows');
    }
    /**
     * Grants permission to merge profiles in a domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_MergeProfiles.html
     */
    toMergeProfiles() {
        return this.to('MergeProfiles');
    }
    /**
     * Grants permission to put a integration in a domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_PutIntegration.html
     */
    toPutIntegration() {
        return this.to('PutIntegration');
    }
    /**
     * Grants permission to put an object for a profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_PutProfileObject.html
     */
    toPutProfileObject() {
        return this.to('PutProfileObject');
    }
    /**
     * Grants permission to put a specific profile object type in the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_PutProfileObjectType.html
     */
    toPutProfileObjectType() {
        return this.to('PutProfileObjectType');
    }
    /**
     * Grants permission to search for profiles in a domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_SearchProfiles.html
     */
    toSearchProfiles() {
        return this.to('SearchProfiles');
    }
    /**
     * Grants permission to adds tags to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a calculated attribute definition in the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_UpdateCalculatedAttributeDefinition.html
     */
    toUpdateCalculatedAttributeDefinition() {
        return this.to('UpdateCalculatedAttributeDefinition');
    }
    /**
     * Grants permission to update a Domain
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_UpdateDomain.html
     */
    toUpdateDomain() {
        return this.to('UpdateDomain');
    }
    /**
     * Grants permission to update an event trigger in the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_UpdateEventTrigger.html
     */
    toUpdateEventTrigger() {
        return this.to('UpdateEventTrigger');
    }
    /**
     * Grants permission to update a profile in the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_UpdateProfile.html
     */
    toUpdateProfile() {
        return this.to('UpdateProfile');
    }
    /**
     * Adds a resource of type domains to the statement
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/
     *
     * @param domainName - Identifier for the domainName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDomains(domainName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:profile:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:domains/${domainName}`);
    }
    /**
     * Adds a resource of type object-types to the statement
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/
     *
     * @param domainName - Identifier for the domainName.
     * @param objectTypeName - Identifier for the objectTypeName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onObjectTypes(domainName, objectTypeName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:profile:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:domains/${domainName}/object-types/${objectTypeName}`);
    }
    /**
     * Adds a resource of type integrations to the statement
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/
     *
     * @param domainName - Identifier for the domainName.
     * @param uri - Identifier for the uri.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIntegrations(domainName, uri, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:profile:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:domains/${domainName}/integrations/${uri}`);
    }
    /**
     * Adds a resource of type event-streams to the statement
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/
     *
     * @param domainName - Identifier for the domainName.
     * @param eventStreamName - Identifier for the eventStreamName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEventStreams(domainName, eventStreamName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:profile:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:domains/${domainName}/event-streams/${eventStreamName}`);
    }
    /**
     * Adds a resource of type calculated-attributes to the statement
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/
     *
     * @param domainName - Identifier for the domainName.
     * @param calculatedAttributeName - Identifier for the calculatedAttributeName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCalculatedAttributes(domainName, calculatedAttributeName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:profile:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:domains/${domainName}/calculated-attributes/${calculatedAttributeName}`);
    }
    /**
     * Adds a resource of type segment-definitions to the statement
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/
     *
     * @param domainName - Identifier for the domainName.
     * @param segmentDefinitionName - Identifier for the segmentDefinitionName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSegmentDefinitions(domainName, segmentDefinitionName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:profile:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:domains/${domainName}/segment-definitions/${segmentDefinitionName}`);
    }
    /**
     * Adds a resource of type event-triggers to the statement
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/
     *
     * @param domainName - Identifier for the domainName.
     * @param eventTriggerName - Identifier for the eventTriggerName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEventTriggers(domainName, eventTriggerName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:profile:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:domains/${domainName}/event-triggers/${eventTriggerName}`);
    }
    /**
     * Filters access by a key that is present in the request the user makes to the customer profile service
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toCreateCalculatedAttributeDefinition()
     * - .toCreateDomain()
     * - .toCreateEventStream()
     * - .toCreateEventTrigger()
     * - .toCreateIntegrationWorkflow()
     * - .toCreateSegmentDefinition()
     * - .toPutIntegration()
     * - .toPutProfileObjectType()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a tag key and value pair
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to resource types:
     * - domains
     * - object-types
     * - integrations
     * - event-streams
     * - calculated-attributes
     * - segment-definitions
     * - event-triggers
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the list of all the tag key names present in the request the user makes to the customer profile service
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toCreateCalculatedAttributeDefinition()
     * - .toCreateDomain()
     * - .toCreateEventStream()
     * - .toCreateEventTrigger()
     * - .toCreateIntegrationWorkflow()
     * - .toCreateSegmentDefinition()
     * - .toPutIntegration()
     * - .toPutProfileObjectType()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [profile](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonconnectcustomerprofiles.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'profile';
        this.accessLevelList = {
            Write: [
                'AddProfileKey',
                'CreateCalculatedAttributeDefinition',
                'CreateDomain',
                'CreateEventStream',
                'CreateEventTrigger',
                'CreateIntegrationWorkflow',
                'CreateProfile',
                'CreateSegmentDefinition',
                'CreateSegmentEstimate',
                'CreateSegmentSnapshot',
                'CreateSnapshot',
                'DeleteCalculatedAttributeDefinition',
                'DeleteDomain',
                'DeleteEventStream',
                'DeleteEventTrigger',
                'DeleteIntegration',
                'DeleteProfile',
                'DeleteProfileKey',
                'DeleteProfileObject',
                'DeleteProfileObjectType',
                'DeleteSegmentDefinition',
                'DeleteWorkflow',
                'MergeProfiles',
                'PutIntegration',
                'PutProfileObject',
                'PutProfileObjectType',
                'UpdateCalculatedAttributeDefinition',
                'UpdateDomain',
                'UpdateEventTrigger',
                'UpdateProfile'
            ],
            Read: [
                'BatchGetCalculatedAttributeForProfile',
                'BatchGetProfile',
                'DetectProfileObjectType',
                'GetAutoMergingPreview',
                'GetCalculatedAttributeDefinition',
                'GetCalculatedAttributeForProfile',
                'GetDomain',
                'GetEventStream',
                'GetEventTrigger',
                'GetIdentityResolutionJob',
                'GetIntegration',
                'GetProfileObjectType',
                'GetProfileObjectTypeTemplate',
                'GetSegmentDefinition',
                'GetSegmentEstimate',
                'GetSegmentMembership',
                'GetSegmentSnapshot',
                'GetSnapshot',
                'GetWorkflow',
                'GetWorkflowSteps',
                'ListTagsForResource',
                'SearchProfiles'
            ],
            List: [
                'GetMatches',
                'GetSimilarProfiles',
                'ListAccountIntegrations',
                'ListCalculatedAttributeDefinitions',
                'ListCalculatedAttributesForProfile',
                'ListDomains',
                'ListEventStreams',
                'ListEventTriggers',
                'ListIdentityResolutionJobs',
                'ListIntegrations',
                'ListObjectTypeAttributes',
                'ListProfileAttributeValues',
                'ListProfileObjectTypeTemplates',
                'ListProfileObjectTypes',
                'ListProfileObjects',
                'ListRuleBasedMatches',
                'ListSegmentDefinitions',
                'ListWorkflows'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Profile = Profile;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29ubmVjdGN1c3RvbWVycHJvZmlsZXMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJjb25uZWN0Y3VzdG9tZXJwcm9maWxlcy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFDQSx5Q0FBeUQ7QUFHekQ7Ozs7R0FJRztBQUNILE1BQWEsT0FBUSxTQUFRLHdCQUFlO0lBRzFDOzs7Ozs7T0FNRztJQUNJLGVBQWU7UUFDcEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1Q0FBdUM7UUFDNUMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVDQUF1QyxDQUFDLENBQUM7SUFDMUQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUJBQWlCLENBQUMsQ0FBQztJQUNwQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUNBQXFDO1FBQzFDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQ0FBcUMsQ0FBQyxDQUFDO0lBQ3hELENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxjQUFjO1FBQ25CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxjQUFjLENBQUMsQ0FBQztJQUNqQyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO0lBQ3ZDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwyQkFBMkI7UUFDaEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDJCQUEyQixDQUFDLENBQUM7SUFDOUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGVBQWU7UUFDcEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx5QkFBeUI7UUFDOUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7SUFDNUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHVCQUF1QjtRQUM1QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsdUJBQXVCLENBQUMsQ0FBQztJQUMxQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksdUJBQXVCO1FBQzVCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO0lBQzFDLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksZ0JBQWdCO1FBQ3JCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO0lBQ25DLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQ0FBcUM7UUFDMUMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFDQUFxQyxDQUFDLENBQUM7SUFDeEQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGNBQWM7UUFDbkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGNBQWMsQ0FBQyxDQUFDO0lBQ2pDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxtQkFBbUI7UUFDeEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxlQUFlO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHlCQUF5QjtRQUM5QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMseUJBQXlCLENBQUMsQ0FBQztJQUM1QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUJBQXlCO1FBQzlCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO0lBQzVDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHlCQUF5QjtRQUM5QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMseUJBQXlCLENBQUMsQ0FBQztJQUM1QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksdUJBQXVCO1FBQzVCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO0lBQzFDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQ0FBa0M7UUFDdkMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtDQUFrQyxDQUFDLENBQUM7SUFDckQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtDQUFrQztRQUN2QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0NBQWtDLENBQUMsQ0FBQztJQUNyRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksV0FBVztRQUNoQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsV0FBVyxDQUFDLENBQUM7SUFDOUIsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLGdCQUFnQjtRQUNyQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztJQUNuQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUJBQWlCO1FBQ3RCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO0lBQ3BDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwwQkFBMEI7UUFDL0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztJQUNuQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksWUFBWTtRQUNqQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsWUFBWSxDQUFDLENBQUM7SUFDL0IsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksOEJBQThCO1FBQ25DLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO0lBQ2pELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxzQkFBc0I7UUFDM0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUM7SUFDekMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0JBQXNCO1FBQzNCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG9CQUFvQixDQUFDLENBQUM7SUFDdkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLGFBQWE7UUFDbEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ2hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxhQUFhO1FBQ2xCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUNoQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx5QkFBeUI7UUFDOUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7SUFDNUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9DQUFvQztRQUN6QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0NBQW9DLENBQUMsQ0FBQztJQUN2RCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0NBQW9DO1FBQ3pDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQ0FBb0MsQ0FBQyxDQUFDO0lBQ3ZELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxhQUFhO1FBQ2xCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUNoQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDRCQUE0QjtRQUNqQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNEJBQTRCLENBQUMsQ0FBQztJQUMvQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwwQkFBMEI7UUFDL0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDRCQUE0QjtRQUNqQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNEJBQTRCLENBQUMsQ0FBQztJQUMvQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0NBQWdDO1FBQ3JDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO0lBQ25ELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx3QkFBd0I7UUFDN0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixDQUFDLENBQUM7SUFDM0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0JBQXNCO1FBQzNCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx3QkFBd0I7UUFDN0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixDQUFDLENBQUM7SUFDM0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGVBQWU7UUFDcEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0JBQWtCLENBQUMsQ0FBQztJQUNyQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0JBQXNCO1FBQzNCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxhQUFhO1FBQ2xCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUNoQyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksZUFBZTtRQUNwQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFDQUFxQztRQUMxQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUNBQXFDLENBQUMsQ0FBQztJQUN4RCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksY0FBYztRQUNuQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsY0FBYyxDQUFDLENBQUM7SUFDakMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQXFGRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxTQUFTLENBQUMsVUFBa0IsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUN4RixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixZQUFhLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxZQUFhLFVBQVcsRUFBRSxDQUFDLENBQUM7SUFDdEssQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxhQUFhLENBQUMsVUFBa0IsRUFBRSxjQUFzQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3BILE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLFlBQWEsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLFlBQWEsVUFBVyxpQkFBa0IsY0FBZSxFQUFFLENBQUMsQ0FBQztJQUN2TSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLGNBQWMsQ0FBQyxVQUFrQixFQUFFLEdBQVcsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUMxRyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixZQUFhLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxZQUFhLFVBQVcsaUJBQWtCLEdBQUksRUFBRSxDQUFDLENBQUM7SUFDNUwsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxjQUFjLENBQUMsVUFBa0IsRUFBRSxlQUF1QixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3RILE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLFlBQWEsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLFlBQWEsVUFBVyxrQkFBbUIsZUFBZ0IsRUFBRSxDQUFDLENBQUM7SUFDek0sQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxzQkFBc0IsQ0FBQyxVQUFrQixFQUFFLHVCQUErQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3RJLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLFlBQWEsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLFlBQWEsVUFBVywwQkFBMkIsdUJBQXdCLEVBQUUsQ0FBQyxDQUFDO0lBQ3pOLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksb0JBQW9CLENBQUMsVUFBa0IsRUFBRSxxQkFBNkIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNsSSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixZQUFhLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxZQUFhLFVBQVcsd0JBQXlCLHFCQUFzQixFQUFFLENBQUMsQ0FBQztJQUNyTixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLGVBQWUsQ0FBQyxVQUFrQixFQUFFLGdCQUF3QixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3hILE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLFlBQWEsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLFlBQWEsVUFBVyxtQkFBb0IsZ0JBQWlCLEVBQUUsQ0FBQyxDQUFDO0lBQzNNLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQW1CRztJQUNJLGVBQWUsQ0FBQyxNQUFjLEVBQUUsS0FBd0IsRUFBRSxRQUE0QjtRQUMzRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0JBQW1CLE1BQU8sRUFBRSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDaEYsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7OztPQWlCRztJQUNJLGdCQUFnQixDQUFDLE1BQWMsRUFBRSxLQUF3QixFQUFFLFFBQTRCO1FBQzVGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBb0IsTUFBTyxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNqRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FtQkc7SUFDSSxZQUFZLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUN4RSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsYUFBYSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDakUsQ0FBQztJQUVEOzs7T0FHRztJQUNILFlBQVksS0FBZ0M7UUFDMUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBeGtDUixrQkFBYSxHQUFHLFNBQVMsQ0FBQztRQTZ5QnZCLG9CQUFlLEdBQW9CO1lBQzNDLEtBQUssRUFBRTtnQkFDTCxlQUFlO2dCQUNmLHFDQUFxQztnQkFDckMsY0FBYztnQkFDZCxtQkFBbUI7Z0JBQ25CLG9CQUFvQjtnQkFDcEIsMkJBQTJCO2dCQUMzQixlQUFlO2dCQUNmLHlCQUF5QjtnQkFDekIsdUJBQXVCO2dCQUN2Qix1QkFBdUI7Z0JBQ3ZCLGdCQUFnQjtnQkFDaEIscUNBQXFDO2dCQUNyQyxjQUFjO2dCQUNkLG1CQUFtQjtnQkFDbkIsb0JBQW9CO2dCQUNwQixtQkFBbUI7Z0JBQ25CLGVBQWU7Z0JBQ2Ysa0JBQWtCO2dCQUNsQixxQkFBcUI7Z0JBQ3JCLHlCQUF5QjtnQkFDekIseUJBQXlCO2dCQUN6QixnQkFBZ0I7Z0JBQ2hCLGVBQWU7Z0JBQ2YsZ0JBQWdCO2dCQUNoQixrQkFBa0I7Z0JBQ2xCLHNCQUFzQjtnQkFDdEIscUNBQXFDO2dCQUNyQyxjQUFjO2dCQUNkLG9CQUFvQjtnQkFDcEIsZUFBZTthQUNoQjtZQUNELElBQUksRUFBRTtnQkFDSix1Q0FBdUM7Z0JBQ3ZDLGlCQUFpQjtnQkFDakIseUJBQXlCO2dCQUN6Qix1QkFBdUI7Z0JBQ3ZCLGtDQUFrQztnQkFDbEMsa0NBQWtDO2dCQUNsQyxXQUFXO2dCQUNYLGdCQUFnQjtnQkFDaEIsaUJBQWlCO2dCQUNqQiwwQkFBMEI7Z0JBQzFCLGdCQUFnQjtnQkFDaEIsc0JBQXNCO2dCQUN0Qiw4QkFBOEI7Z0JBQzlCLHNCQUFzQjtnQkFDdEIsb0JBQW9CO2dCQUNwQixzQkFBc0I7Z0JBQ3RCLG9CQUFvQjtnQkFDcEIsYUFBYTtnQkFDYixhQUFhO2dCQUNiLGtCQUFrQjtnQkFDbEIscUJBQXFCO2dCQUNyQixnQkFBZ0I7YUFDakI7WUFDRCxJQUFJLEVBQUU7Z0JBQ0osWUFBWTtnQkFDWixvQkFBb0I7Z0JBQ3BCLHlCQUF5QjtnQkFDekIsb0NBQW9DO2dCQUNwQyxvQ0FBb0M7Z0JBQ3BDLGFBQWE7Z0JBQ2Isa0JBQWtCO2dCQUNsQixtQkFBbUI7Z0JBQ25CLDRCQUE0QjtnQkFDNUIsa0JBQWtCO2dCQUNsQiwwQkFBMEI7Z0JBQzFCLDRCQUE0QjtnQkFDNUIsZ0NBQWdDO2dCQUNoQyx3QkFBd0I7Z0JBQ3hCLG9CQUFvQjtnQkFDcEIsc0JBQXNCO2dCQUN0Qix3QkFBd0I7Z0JBQ3hCLGVBQWU7YUFDaEI7WUFDRCxPQUFPLEVBQUU7Z0JBQ1AsYUFBYTtnQkFDYixlQUFlO2FBQ2hCO1NBQ0YsQ0FBQztJQTJNRixDQUFDO0NBQ0Y7QUEza0NELDBCQTJrQ0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBBY2Nlc3NMZXZlbExpc3QgfSBmcm9tICcuLi8uLi9zaGFyZWQvYWNjZXNzLWxldmVsJztcbmltcG9ydCB7IFBvbGljeVN0YXRlbWVudCwgT3BlcmF0b3IgfSBmcm9tICcuLi8uLi9zaGFyZWQnO1xuaW1wb3J0IHsgYXdzX2lhbSBhcyBpYW0gfSBmcm9tIFwiYXdzLWNkay1saWJcIjtcblxuLyoqXG4gKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW3Byb2ZpbGVdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXJ2aWNlLWF1dGhvcml6YXRpb24vbGF0ZXN0L3JlZmVyZW5jZS9saXN0X2FtYXpvbmNvbm5lY3RjdXN0b21lcnByb2ZpbGVzLmh0bWwpLlxuICpcbiAqIEBwYXJhbSBzaWQgW1NJRF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19zaWQuaHRtbCkgb2YgdGhlIHN0YXRlbWVudFxuICovXG5leHBvcnQgY2xhc3MgUHJvZmlsZSBleHRlbmRzIFBvbGljeVN0YXRlbWVudCB7XG4gIHB1YmxpYyBzZXJ2aWNlUHJlZml4ID0gJ3Byb2ZpbGUnO1xuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBhZGQgYSBwcm9maWxlIGtleVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jdXN0b21lcnByb2ZpbGVzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0FkZFByb2ZpbGVLZXkuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQWRkUHJvZmlsZUtleSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQWRkUHJvZmlsZUtleScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGEgY2FsY3VsYXRlZCBhdHRyaWJ1dGUgZm9yIHRoZSBzcGVjaWZpYyBwcm9maWxlcyBpbiB0aGUgZG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY3VzdG9tZXJwcm9maWxlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9CYXRjaEdldENhbGN1bGF0ZWRBdHRyaWJ1dGVGb3JQcm9maWxlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0JhdGNoR2V0Q2FsY3VsYXRlZEF0dHJpYnV0ZUZvclByb2ZpbGUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0JhdGNoR2V0Q2FsY3VsYXRlZEF0dHJpYnV0ZUZvclByb2ZpbGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgcHJvZmlsZXMgaW4gdGhlIGRvbWFpblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2N1c3RvbWVycHJvZmlsZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQmF0Y2hHZXRQcm9maWxlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0JhdGNoR2V0UHJvZmlsZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQmF0Y2hHZXRQcm9maWxlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgY2FsY3VsYXRlZCBhdHRyaWJ1dGUgZGVmaW5pdGlvbiBpbiB0aGUgZG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2N1c3RvbWVycHJvZmlsZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlQ2FsY3VsYXRlZEF0dHJpYnV0ZURlZmluaXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlQ2FsY3VsYXRlZEF0dHJpYnV0ZURlZmluaXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZUNhbGN1bGF0ZWRBdHRyaWJ1dGVEZWZpbml0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgRG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gaWFtOkNyZWF0ZVNlcnZpY2VMaW5rZWRSb2xlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jdXN0b21lcnByb2ZpbGVzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZURvbWFpbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVEb21haW4oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZURvbWFpbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHB1dCBhbiBldmVudCBzdHJlYW0gaW4gYSBkb21haW5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBpYW06UHV0Um9sZVBvbGljeVxuICAgKiAtIGtpbmVzaXM6RGVzY3JpYmVTdHJlYW1TdW1tYXJ5XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jdXN0b21lcnByb2ZpbGVzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZUV2ZW50U3RyZWFtLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUV2ZW50U3RyZWFtKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVFdmVudFN0cmVhbScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhbiBldmVudCB0cmlnZ2VyIGluIHRoZSBkb21haW5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY3VzdG9tZXJwcm9maWxlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVFdmVudFRyaWdnZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlRXZlbnRUcmlnZ2VyKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVFdmVudFRyaWdnZXInKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYW4gaW50ZWdyYXRpb24gd29ya2Zsb3cgaW4gYSBkb21haW5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY3VzdG9tZXJwcm9maWxlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVJbnRlZ3JhdGlvbldvcmtmbG93Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUludGVncmF0aW9uV29ya2Zsb3coKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZUludGVncmF0aW9uV29ya2Zsb3cnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBwcm9maWxlIGluIHRoZSBkb21haW5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY3VzdG9tZXJwcm9maWxlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVQcm9maWxlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVByb2ZpbGUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZVByb2ZpbGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBzZWdtZW50IGRlZmluaXRpb24gaW4gdGhlIGRvbWFpblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jdXN0b21lcnByb2ZpbGVzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZVNlZ21lbnREZWZpbml0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVNlZ21lbnREZWZpbml0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVTZWdtZW50RGVmaW5pdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIHNlZ21lbnQgZXN0aW1hdGUgaW4gdGhlIGRvbWFpblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jdXN0b21lcnByb2ZpbGVzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZVNlZ21lbnRFc3RpbWF0ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVTZWdtZW50RXN0aW1hdGUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZVNlZ21lbnRFc3RpbWF0ZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIHNlZ21lbnQgc25hcHNob3QgaW4gdGhlIGRvbWFpblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jdXN0b21lcnByb2ZpbGVzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZVNlZ21lbnRTbmFwc2hvdC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVTZWdtZW50U25hcHNob3QoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZVNlZ21lbnRTbmFwc2hvdCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIHNuYXBzaG90IGluIHRoZSBkb21haW5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlU25hcHNob3QoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZVNuYXBzaG90Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGEgY2FsY3VsYXRlZCBhdHRyaWJ1dGUgZGVmaW5pdGlvbiBpbiB0aGUgZG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2N1c3RvbWVycHJvZmlsZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlQ2FsY3VsYXRlZEF0dHJpYnV0ZURlZmluaXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlQ2FsY3VsYXRlZEF0dHJpYnV0ZURlZmluaXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZUNhbGN1bGF0ZWRBdHRyaWJ1dGVEZWZpbml0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGEgRG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2N1c3RvbWVycHJvZmlsZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlRG9tYWluLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZURvbWFpbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlRG9tYWluJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGFuIGV2ZW50IHN0cmVhbSBpbiBhIGRvbWFpblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIGlhbTpEZWxldGVSb2xlUG9saWN5XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jdXN0b21lcnByb2ZpbGVzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUV2ZW50U3RyZWFtLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUV2ZW50U3RyZWFtKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVFdmVudFN0cmVhbScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhbiBldmVudCB0cmlnZ2VyIGluIHRoZSBkb21haW5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY3VzdG9tZXJwcm9maWxlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVFdmVudFRyaWdnZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlRXZlbnRUcmlnZ2VyKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVFdmVudFRyaWdnZXInKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBpbnRlZ3JhdGlvbiBpbiBhIGRvbWFpblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jdXN0b21lcnByb2ZpbGVzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUludGVncmF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUludGVncmF0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVJbnRlZ3JhdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhIHByb2ZpbGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY3VzdG9tZXJwcm9maWxlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVQcm9maWxlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVByb2ZpbGUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZVByb2ZpbGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBwcm9maWxlIGtleVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jdXN0b21lcnByb2ZpbGVzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZVByb2ZpbGVLZXkuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlUHJvZmlsZUtleSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlUHJvZmlsZUtleScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhIHByb2ZpbGUgb2JqZWN0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2N1c3RvbWVycHJvZmlsZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlUHJvZmlsZU9iamVjdC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVQcm9maWxlT2JqZWN0KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVQcm9maWxlT2JqZWN0Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGEgc3BlY2lmaWMgcHJvZmlsZSBvYmplY3QgdHlwZSBpbiB0aGUgZG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2N1c3RvbWVycHJvZmlsZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlUHJvZmlsZU9iamVjdFR5cGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlUHJvZmlsZU9iamVjdFR5cGUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZVByb2ZpbGVPYmplY3RUeXBlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGEgc2VnbWVudCBkZWZpbml0aW9uIGluIHRoZSBkb21haW5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY3VzdG9tZXJwcm9maWxlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVTZWdtZW50RGVmaW5pdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVTZWdtZW50RGVmaW5pdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlU2VnbWVudERlZmluaXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSB3b3JrZmxvdyBpbiBhIGRvbWFpblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jdXN0b21lcnByb2ZpbGVzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZVdvcmtmbG93Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVdvcmtmbG93KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVXb3JrZmxvdycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGF1dG8gZGV0ZWN0IG9iamVjdCB0eXBlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY3VzdG9tZXJwcm9maWxlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXRlY3RQcm9maWxlT2JqZWN0VHlwZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXRlY3RQcm9maWxlT2JqZWN0VHlwZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGV0ZWN0UHJvZmlsZU9iamVjdFR5cGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgYSBwcmV2aWV3IG9mIGF1dG8gbWVyZ2luZyBpbiBhIGRvbWFpblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2N1c3RvbWVycHJvZmlsZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0QXV0b01lcmdpbmdQcmV2aWV3Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEF1dG9NZXJnaW5nUHJldmlldygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0QXV0b01lcmdpbmdQcmV2aWV3Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IGEgY2FsY3VsYXRlZCBhdHRyaWJ1dGUgZGVmaW5pdGlvbiBpbiB0aGUgZG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY3VzdG9tZXJwcm9maWxlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9HZXRDYWxjdWxhdGVkQXR0cmlidXRlRGVmaW5pdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRDYWxjdWxhdGVkQXR0cmlidXRlRGVmaW5pdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0Q2FsY3VsYXRlZEF0dHJpYnV0ZURlZmluaXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBhIGNhbGN1bGF0ZWQgYXR0cmlidXRlIGZvciBhIHNwZWNpZmljIHByb2ZpbGUgaW4gdGhlIGRvbWFpblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2N1c3RvbWVycHJvZmlsZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0Q2FsY3VsYXRlZEF0dHJpYnV0ZUZvclByb2ZpbGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0Q2FsY3VsYXRlZEF0dHJpYnV0ZUZvclByb2ZpbGUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldENhbGN1bGF0ZWRBdHRyaWJ1dGVGb3JQcm9maWxlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IGEgc3BlY2lmaWMgZG9tYWluIGluIGFuIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jdXN0b21lcnByb2ZpbGVzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldERvbWFpbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXREb21haW4oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldERvbWFpbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBhIHNwZWNpZmljIGV2ZW50IHN0cmVhbSBpbiBhIGRvbWFpblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0ga2luZXNpczpEZXNjcmliZVN0cmVhbVN1bW1hcnlcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2N1c3RvbWVycHJvZmlsZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0RXZlbnRTdHJlYW0uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0RXZlbnRTdHJlYW0oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldEV2ZW50U3RyZWFtJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IGFuIGV2ZW50IHRyaWdnZXIgaW4gdGhlIGRvbWFpblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2N1c3RvbWVycHJvZmlsZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0RXZlbnRUcmlnZ2VyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEV2ZW50VHJpZ2dlcigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0RXZlbnRUcmlnZ2VyJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IGFuIGlkZW50aXR5IHJlc29sdXRpb24gam9iIGluIGEgZG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY3VzdG9tZXJwcm9maWxlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9HZXRJZGVudGl0eVJlc29sdXRpb25Kb2IuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0SWRlbnRpdHlSZXNvbHV0aW9uSm9iKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRJZGVudGl0eVJlc29sdXRpb25Kb2InKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgYSBzcGVjaWZpYyBpbnRlZ3JhdGlvbnMgaW4gYSBkb21haW5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jdXN0b21lcnByb2ZpbGVzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldEludGVncmF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEludGVncmF0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRJbnRlZ3JhdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBwcm9maWxlIG1hdGNoZXMgaW4gYSBkb21haW5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jdXN0b21lcnByb2ZpbGVzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldE1hdGNoZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0TWF0Y2hlcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0TWF0Y2hlcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBhIHNwZWNpZmljIHByb2ZpbGUgb2JqZWN0IHR5cGUgaW4gdGhlIGRvbWFpblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2N1c3RvbWVycHJvZmlsZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0UHJvZmlsZU9iamVjdFR5cGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0UHJvZmlsZU9iamVjdFR5cGUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldFByb2ZpbGVPYmplY3RUeXBlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IGEgc3BlY2lmaWMgb2JqZWN0IHR5cGUgdGVtcGxhdGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jdXN0b21lcnByb2ZpbGVzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldFByb2ZpbGVPYmplY3RUeXBlVGVtcGxhdGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0UHJvZmlsZU9iamVjdFR5cGVUZW1wbGF0ZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0UHJvZmlsZU9iamVjdFR5cGVUZW1wbGF0ZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBhIHNlZ21lbnQgZGVmaW5pdGlvbiBpbiB0aGUgZG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY3VzdG9tZXJwcm9maWxlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9HZXRTZWdtZW50RGVmaW5pdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRTZWdtZW50RGVmaW5pdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0U2VnbWVudERlZmluaXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgYSBzZWdtZW50IGVzdGltYXRlIGluIHRoZSBkb21haW5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jdXN0b21lcnByb2ZpbGVzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldFNlZ21lbnRFc3RpbWF0ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRTZWdtZW50RXN0aW1hdGUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldFNlZ21lbnRFc3RpbWF0ZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRldGVybWluZSBpZiB0aGUgZ2l2ZW4gcHJvZmlsZXMgYXJlIHBhcnQgb2YgYSBzZWdtZW50IGluIHRoZSBkb21haW5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jdXN0b21lcnByb2ZpbGVzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldFNlZ21lbnRNZW1iZXJzaGlwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFNlZ21lbnRNZW1iZXJzaGlwKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRTZWdtZW50TWVtYmVyc2hpcCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBhIHNlZ21lbnQgc25hcHNob3QgaW4gdGhlIGRvbWFpblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2N1c3RvbWVycHJvZmlsZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0U2VnbWVudFNuYXBzaG90Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFNlZ21lbnRTbmFwc2hvdCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0U2VnbWVudFNuYXBzaG90Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IGFsbCB0aGUgc2ltaWxhciBwcm9maWxlcyBpbiB0aGUgZG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY3VzdG9tZXJwcm9maWxlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9HZXRTaW1pbGFyUHJvZmlsZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0U2ltaWxhclByb2ZpbGVzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRTaW1pbGFyUHJvZmlsZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgYSBzbmFwc2hvdCBpbiB0aGUgZG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKi9cbiAgcHVibGljIHRvR2V0U25hcHNob3QoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldFNuYXBzaG90Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IHdvcmtmbG93IGRldGFpbHMgaW4gYSBkb21haW5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jdXN0b21lcnByb2ZpbGVzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldFdvcmtmbG93Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFdvcmtmbG93KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRXb3JrZmxvdycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCB3b3JrZmxvdyBzdGVwIGRldGFpbHMgaW4gYSBkb21haW5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jdXN0b21lcnByb2ZpbGVzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldFdvcmtmbG93U3RlcHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0V29ya2Zsb3dTdGVwcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0V29ya2Zsb3dTdGVwcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgYWxsIHRoZSBpbnRlZ3JhdGlvbnMgaW4gdGhlIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jdXN0b21lcnByb2ZpbGVzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RBY2NvdW50SW50ZWdyYXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RBY2NvdW50SW50ZWdyYXRpb25zKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0QWNjb3VudEludGVncmF0aW9ucycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgYWxsIHRoZSBjYWxjdWxhdGVkIGF0dHJpYnV0ZSBkZWZpbml0aW9ucyBpbiB0aGUgZG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY3VzdG9tZXJwcm9maWxlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0Q2FsY3VsYXRlZEF0dHJpYnV0ZURlZmluaXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RDYWxjdWxhdGVkQXR0cmlidXRlRGVmaW5pdGlvbnMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RDYWxjdWxhdGVkQXR0cmlidXRlRGVmaW5pdGlvbnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IGFsbCBjYWxjdWxhdGVkIGF0dHJpYnV0ZXMgZm9yIGEgc3BlY2lmaWMgcHJvZmlsZSBpbiB0aGUgZG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY3VzdG9tZXJwcm9maWxlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0Q2FsY3VsYXRlZEF0dHJpYnV0ZXNGb3JQcm9maWxlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RDYWxjdWxhdGVkQXR0cmlidXRlc0ZvclByb2ZpbGUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RDYWxjdWxhdGVkQXR0cmlidXRlc0ZvclByb2ZpbGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IGFsbCB0aGUgZG9tYWlucyBpbiBhbiBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY3VzdG9tZXJwcm9maWxlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0RG9tYWlucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0RG9tYWlucygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdERvbWFpbnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IGFsbCB0aGUgZXZlbnQgc3RyZWFtcyBpbiBhIHNwZWNpZmljIGRvbWFpblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2N1c3RvbWVycHJvZmlsZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdEV2ZW50U3RyZWFtcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0RXZlbnRTdHJlYW1zKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0RXZlbnRTdHJlYW1zJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBhbGwgdGhlIGV2ZW50IHRyaWdnZXJzIGluIHRoZSBkb21haW5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jdXN0b21lcnByb2ZpbGVzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RFdmVudFRyaWdnZXJzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RFdmVudFRyaWdnZXJzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0RXZlbnRUcmlnZ2VycycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgaWRlbnRpdHkgcmVzb2x1dGlvbiBqb2JzIGluIGEgZG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY3VzdG9tZXJwcm9maWxlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0SWRlbnRpdHlSZXNvbHV0aW9uSm9icy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0SWRlbnRpdHlSZXNvbHV0aW9uSm9icygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdElkZW50aXR5UmVzb2x1dGlvbkpvYnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IGFsbCB0aGUgaW50ZWdyYXRpb25zIGluIGEgc3BlY2lmaWMgZG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY3VzdG9tZXJwcm9maWxlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0SW50ZWdyYXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RJbnRlZ3JhdGlvbnMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RJbnRlZ3JhdGlvbnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IGFsbCB0aGUgYXR0cmlidXRlcyBvZiBhIHNwZWNpZmljIG9iamVjdCB0eXBlIGluIHRoZSBkb21haW5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jdXN0b21lcnByb2ZpbGVzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RPYmplY3RUeXBlQXR0cmlidXRlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0T2JqZWN0VHlwZUF0dHJpYnV0ZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RPYmplY3RUeXBlQXR0cmlidXRlcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgYWxsIHRoZSB2YWx1ZXMgb2YgYSBwcm9maWxlIGF0dHJpYnV0ZSBpbiB0aGUgZG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY3VzdG9tZXJwcm9maWxlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0UHJvZmlsZUF0dHJpYnV0ZVZhbHVlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0UHJvZmlsZUF0dHJpYnV0ZVZhbHVlcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdFByb2ZpbGVBdHRyaWJ1dGVWYWx1ZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IGFsbCB0aGUgcHJvZmlsZSBvYmplY3QgdHlwZSB0ZW1wbGF0ZXMgaW4gdGhlIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jdXN0b21lcnByb2ZpbGVzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RQcm9maWxlT2JqZWN0VHlwZVRlbXBsYXRlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0UHJvZmlsZU9iamVjdFR5cGVUZW1wbGF0ZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RQcm9maWxlT2JqZWN0VHlwZVRlbXBsYXRlcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgYWxsIHRoZSBwcm9maWxlIG9iamVjdCB0eXBlcyBpbiB0aGUgZG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY3VzdG9tZXJwcm9maWxlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0UHJvZmlsZU9iamVjdFR5cGVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RQcm9maWxlT2JqZWN0VHlwZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RQcm9maWxlT2JqZWN0VHlwZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IGFsbCB0aGUgcHJvZmlsZSBvYmplY3RzIGZvciBhIHByb2ZpbGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jdXN0b21lcnByb2ZpbGVzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RQcm9maWxlT2JqZWN0cy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0UHJvZmlsZU9iamVjdHMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RQcm9maWxlT2JqZWN0cycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgYWxsIHRoZSBydWxlLWJhc2VkIG1hdGNoaW5nIHJlc3VsdCBpbiB0aGUgZG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY3VzdG9tZXJwcm9maWxlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0UnVsZUJhc2VkTWF0Y2hlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0UnVsZUJhc2VkTWF0Y2hlcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdFJ1bGVCYXNlZE1hdGNoZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IGFsbCB0aGUgc2VnbWVudCBkZWZpbml0aW9ucyBpbiB0aGUgZG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY3VzdG9tZXJwcm9maWxlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0U2VnbWVudERlZmluaXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RTZWdtZW50RGVmaW5pdGlvbnMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RTZWdtZW50RGVmaW5pdGlvbnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHRhZ3MgZm9yIGEgcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jdXN0b21lcnByb2ZpbGVzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RUYWdzRm9yUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFRhZ3NGb3JSZXNvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdFRhZ3NGb3JSZXNvdXJjZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgYWxsIHRoZSB3b3JrZmxvd3MgaW4gYSBzcGVjaWZpYyBkb21haW5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jdXN0b21lcnByb2ZpbGVzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RXb3JrZmxvd3MuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFdvcmtmbG93cygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdFdvcmtmbG93cycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIG1lcmdlIHByb2ZpbGVzIGluIGEgZG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2N1c3RvbWVycHJvZmlsZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTWVyZ2VQcm9maWxlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9NZXJnZVByb2ZpbGVzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdNZXJnZVByb2ZpbGVzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcHV0IGEgaW50ZWdyYXRpb24gaW4gYSBkb21haW5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY3VzdG9tZXJwcm9maWxlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9QdXRJbnRlZ3JhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QdXRJbnRlZ3JhdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnUHV0SW50ZWdyYXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBwdXQgYW4gb2JqZWN0IGZvciBhIHByb2ZpbGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY3VzdG9tZXJwcm9maWxlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9QdXRQcm9maWxlT2JqZWN0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1B1dFByb2ZpbGVPYmplY3QoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1B1dFByb2ZpbGVPYmplY3QnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBwdXQgYSBzcGVjaWZpYyBwcm9maWxlIG9iamVjdCB0eXBlIGluIHRoZSBkb21haW5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY3VzdG9tZXJwcm9maWxlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9QdXRQcm9maWxlT2JqZWN0VHlwZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QdXRQcm9maWxlT2JqZWN0VHlwZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnUHV0UHJvZmlsZU9iamVjdFR5cGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBzZWFyY2ggZm9yIHByb2ZpbGVzIGluIGEgZG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY3VzdG9tZXJwcm9maWxlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TZWFyY2hQcm9maWxlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TZWFyY2hQcm9maWxlcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnU2VhcmNoUHJvZmlsZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBhZGRzIHRhZ3MgdG8gYSByZXNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFRhZ2dpbmdcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2N1c3RvbWVycHJvZmlsZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVGFnUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVGFnUmVzb3VyY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1RhZ1Jlc291cmNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmVtb3ZlIHRhZ3MgZnJvbSBhIHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogVGFnZ2luZ1xuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY3VzdG9tZXJwcm9maWxlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VbnRhZ1Jlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VudGFnUmVzb3VyY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VudGFnUmVzb3VyY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYSBjYWxjdWxhdGVkIGF0dHJpYnV0ZSBkZWZpbml0aW9uIGluIHRoZSBkb21haW5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY3VzdG9tZXJwcm9maWxlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVDYWxjdWxhdGVkQXR0cmlidXRlRGVmaW5pdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVDYWxjdWxhdGVkQXR0cmlidXRlRGVmaW5pdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlQ2FsY3VsYXRlZEF0dHJpYnV0ZURlZmluaXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYSBEb21haW5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBpYW06Q3JlYXRlU2VydmljZUxpbmtlZFJvbGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2N1c3RvbWVycHJvZmlsZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlRG9tYWluLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZURvbWFpbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlRG9tYWluJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIGFuIGV2ZW50IHRyaWdnZXIgaW4gdGhlIGRvbWFpblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jdXN0b21lcnByb2ZpbGVzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZUV2ZW50VHJpZ2dlci5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVFdmVudFRyaWdnZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZUV2ZW50VHJpZ2dlcicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhIHByb2ZpbGUgaW4gdGhlIGRvbWFpblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jdXN0b21lcnByb2ZpbGVzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZVByb2ZpbGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlUHJvZmlsZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlUHJvZmlsZScpO1xuICB9XG5cbiAgcHJvdGVjdGVkIGFjY2Vzc0xldmVsTGlzdDogQWNjZXNzTGV2ZWxMaXN0ID0ge1xuICAgIFdyaXRlOiBbXG4gICAgICAnQWRkUHJvZmlsZUtleScsXG4gICAgICAnQ3JlYXRlQ2FsY3VsYXRlZEF0dHJpYnV0ZURlZmluaXRpb24nLFxuICAgICAgJ0NyZWF0ZURvbWFpbicsXG4gICAgICAnQ3JlYXRlRXZlbnRTdHJlYW0nLFxuICAgICAgJ0NyZWF0ZUV2ZW50VHJpZ2dlcicsXG4gICAgICAnQ3JlYXRlSW50ZWdyYXRpb25Xb3JrZmxvdycsXG4gICAgICAnQ3JlYXRlUHJvZmlsZScsXG4gICAgICAnQ3JlYXRlU2VnbWVudERlZmluaXRpb24nLFxuICAgICAgJ0NyZWF0ZVNlZ21lbnRFc3RpbWF0ZScsXG4gICAgICAnQ3JlYXRlU2VnbWVudFNuYXBzaG90JyxcbiAgICAgICdDcmVhdGVTbmFwc2hvdCcsXG4gICAgICAnRGVsZXRlQ2FsY3VsYXRlZEF0dHJpYnV0ZURlZmluaXRpb24nLFxuICAgICAgJ0RlbGV0ZURvbWFpbicsXG4gICAgICAnRGVsZXRlRXZlbnRTdHJlYW0nLFxuICAgICAgJ0RlbGV0ZUV2ZW50VHJpZ2dlcicsXG4gICAgICAnRGVsZXRlSW50ZWdyYXRpb24nLFxuICAgICAgJ0RlbGV0ZVByb2ZpbGUnLFxuICAgICAgJ0RlbGV0ZVByb2ZpbGVLZXknLFxuICAgICAgJ0RlbGV0ZVByb2ZpbGVPYmplY3QnLFxuICAgICAgJ0RlbGV0ZVByb2ZpbGVPYmplY3RUeXBlJyxcbiAgICAgICdEZWxldGVTZWdtZW50RGVmaW5pdGlvbicsXG4gICAgICAnRGVsZXRlV29ya2Zsb3cnLFxuICAgICAgJ01lcmdlUHJvZmlsZXMnLFxuICAgICAgJ1B1dEludGVncmF0aW9uJyxcbiAgICAgICdQdXRQcm9maWxlT2JqZWN0JyxcbiAgICAgICdQdXRQcm9maWxlT2JqZWN0VHlwZScsXG4gICAgICAnVXBkYXRlQ2FsY3VsYXRlZEF0dHJpYnV0ZURlZmluaXRpb24nLFxuICAgICAgJ1VwZGF0ZURvbWFpbicsXG4gICAgICAnVXBkYXRlRXZlbnRUcmlnZ2VyJyxcbiAgICAgICdVcGRhdGVQcm9maWxlJ1xuICAgIF0sXG4gICAgUmVhZDogW1xuICAgICAgJ0JhdGNoR2V0Q2FsY3VsYXRlZEF0dHJpYnV0ZUZvclByb2ZpbGUnLFxuICAgICAgJ0JhdGNoR2V0UHJvZmlsZScsXG4gICAgICAnRGV0ZWN0UHJvZmlsZU9iamVjdFR5cGUnLFxuICAgICAgJ0dldEF1dG9NZXJnaW5nUHJldmlldycsXG4gICAgICAnR2V0Q2FsY3VsYXRlZEF0dHJpYnV0ZURlZmluaXRpb24nLFxuICAgICAgJ0dldENhbGN1bGF0ZWRBdHRyaWJ1dGVGb3JQcm9maWxlJyxcbiAgICAgICdHZXREb21haW4nLFxuICAgICAgJ0dldEV2ZW50U3RyZWFtJyxcbiAgICAgICdHZXRFdmVudFRyaWdnZXInLFxuICAgICAgJ0dldElkZW50aXR5UmVzb2x1dGlvbkpvYicsXG4gICAgICAnR2V0SW50ZWdyYXRpb24nLFxuICAgICAgJ0dldFByb2ZpbGVPYmplY3RUeXBlJyxcbiAgICAgICdHZXRQcm9maWxlT2JqZWN0VHlwZVRlbXBsYXRlJyxcbiAgICAgICdHZXRTZWdtZW50RGVmaW5pdGlvbicsXG4gICAgICAnR2V0U2VnbWVudEVzdGltYXRlJyxcbiAgICAgICdHZXRTZWdtZW50TWVtYmVyc2hpcCcsXG4gICAgICAnR2V0U2VnbWVudFNuYXBzaG90JyxcbiAgICAgICdHZXRTbmFwc2hvdCcsXG4gICAgICAnR2V0V29ya2Zsb3cnLFxuICAgICAgJ0dldFdvcmtmbG93U3RlcHMnLFxuICAgICAgJ0xpc3RUYWdzRm9yUmVzb3VyY2UnLFxuICAgICAgJ1NlYXJjaFByb2ZpbGVzJ1xuICAgIF0sXG4gICAgTGlzdDogW1xuICAgICAgJ0dldE1hdGNoZXMnLFxuICAgICAgJ0dldFNpbWlsYXJQcm9maWxlcycsXG4gICAgICAnTGlzdEFjY291bnRJbnRlZ3JhdGlvbnMnLFxuICAgICAgJ0xpc3RDYWxjdWxhdGVkQXR0cmlidXRlRGVmaW5pdGlvbnMnLFxuICAgICAgJ0xpc3RDYWxjdWxhdGVkQXR0cmlidXRlc0ZvclByb2ZpbGUnLFxuICAgICAgJ0xpc3REb21haW5zJyxcbiAgICAgICdMaXN0RXZlbnRTdHJlYW1zJyxcbiAgICAgICdMaXN0RXZlbnRUcmlnZ2VycycsXG4gICAgICAnTGlzdElkZW50aXR5UmVzb2x1dGlvbkpvYnMnLFxuICAgICAgJ0xpc3RJbnRlZ3JhdGlvbnMnLFxuICAgICAgJ0xpc3RPYmplY3RUeXBlQXR0cmlidXRlcycsXG4gICAgICAnTGlzdFByb2ZpbGVBdHRyaWJ1dGVWYWx1ZXMnLFxuICAgICAgJ0xpc3RQcm9maWxlT2JqZWN0VHlwZVRlbXBsYXRlcycsXG4gICAgICAnTGlzdFByb2ZpbGVPYmplY3RUeXBlcycsXG4gICAgICAnTGlzdFByb2ZpbGVPYmplY3RzJyxcbiAgICAgICdMaXN0UnVsZUJhc2VkTWF0Y2hlcycsXG4gICAgICAnTGlzdFNlZ21lbnREZWZpbml0aW9ucycsXG4gICAgICAnTGlzdFdvcmtmbG93cydcbiAgICBdLFxuICAgIFRhZ2dpbmc6IFtcbiAgICAgICdUYWdSZXNvdXJjZScsXG4gICAgICAnVW50YWdSZXNvdXJjZSdcbiAgICBdXG4gIH07XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGRvbWFpbnMgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY3VzdG9tZXJwcm9maWxlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL1xuICAgKlxuICAgKiBAcGFyYW0gZG9tYWluTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBkb21haW5OYW1lLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25Eb21haW5zKGRvbWFpbk5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTpwcm9maWxlOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpkb21haW5zLyR7IGRvbWFpbk5hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIG9iamVjdC10eXBlcyB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jdXN0b21lcnByb2ZpbGVzL2xhdGVzdC9BUElSZWZlcmVuY2UvXG4gICAqXG4gICAqIEBwYXJhbSBkb21haW5OYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGRvbWFpbk5hbWUuXG4gICAqIEBwYXJhbSBvYmplY3RUeXBlTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBvYmplY3RUeXBlTmFtZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uT2JqZWN0VHlwZXMoZG9tYWluTmFtZTogc3RyaW5nLCBvYmplY3RUeXBlTmFtZTogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OnByb2ZpbGU6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OmRvbWFpbnMvJHsgZG9tYWluTmFtZSB9L29iamVjdC10eXBlcy8keyBvYmplY3RUeXBlTmFtZSB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgaW50ZWdyYXRpb25zIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2N1c3RvbWVycHJvZmlsZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9cbiAgICpcbiAgICogQHBhcmFtIGRvbWFpbk5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgZG9tYWluTmFtZS5cbiAgICogQHBhcmFtIHVyaSAtIElkZW50aWZpZXIgZm9yIHRoZSB1cmkuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkludGVncmF0aW9ucyhkb21haW5OYW1lOiBzdHJpbmcsIHVyaTogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OnByb2ZpbGU6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OmRvbWFpbnMvJHsgZG9tYWluTmFtZSB9L2ludGVncmF0aW9ucy8keyB1cmkgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGV2ZW50LXN0cmVhbXMgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY3VzdG9tZXJwcm9maWxlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL1xuICAgKlxuICAgKiBAcGFyYW0gZG9tYWluTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBkb21haW5OYW1lLlxuICAgKiBAcGFyYW0gZXZlbnRTdHJlYW1OYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGV2ZW50U3RyZWFtTmFtZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uRXZlbnRTdHJlYW1zKGRvbWFpbk5hbWU6IHN0cmluZywgZXZlbnRTdHJlYW1OYW1lOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06cHJvZmlsZTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06ZG9tYWlucy8keyBkb21haW5OYW1lIH0vZXZlbnQtc3RyZWFtcy8keyBldmVudFN0cmVhbU5hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGNhbGN1bGF0ZWQtYXR0cmlidXRlcyB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jdXN0b21lcnByb2ZpbGVzL2xhdGVzdC9BUElSZWZlcmVuY2UvXG4gICAqXG4gICAqIEBwYXJhbSBkb21haW5OYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGRvbWFpbk5hbWUuXG4gICAqIEBwYXJhbSBjYWxjdWxhdGVkQXR0cmlidXRlTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBjYWxjdWxhdGVkQXR0cmlidXRlTmFtZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uQ2FsY3VsYXRlZEF0dHJpYnV0ZXMoZG9tYWluTmFtZTogc3RyaW5nLCBjYWxjdWxhdGVkQXR0cmlidXRlTmFtZTogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OnByb2ZpbGU6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OmRvbWFpbnMvJHsgZG9tYWluTmFtZSB9L2NhbGN1bGF0ZWQtYXR0cmlidXRlcy8keyBjYWxjdWxhdGVkQXR0cmlidXRlTmFtZSB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgc2VnbWVudC1kZWZpbml0aW9ucyB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jdXN0b21lcnByb2ZpbGVzL2xhdGVzdC9BUElSZWZlcmVuY2UvXG4gICAqXG4gICAqIEBwYXJhbSBkb21haW5OYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGRvbWFpbk5hbWUuXG4gICAqIEBwYXJhbSBzZWdtZW50RGVmaW5pdGlvbk5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgc2VnbWVudERlZmluaXRpb25OYW1lLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25TZWdtZW50RGVmaW5pdGlvbnMoZG9tYWluTmFtZTogc3RyaW5nLCBzZWdtZW50RGVmaW5pdGlvbk5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTpwcm9maWxlOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpkb21haW5zLyR7IGRvbWFpbk5hbWUgfS9zZWdtZW50LWRlZmluaXRpb25zLyR7IHNlZ21lbnREZWZpbml0aW9uTmFtZSB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgZXZlbnQtdHJpZ2dlcnMgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY3VzdG9tZXJwcm9maWxlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL1xuICAgKlxuICAgKiBAcGFyYW0gZG9tYWluTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBkb21haW5OYW1lLlxuICAgKiBAcGFyYW0gZXZlbnRUcmlnZ2VyTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBldmVudFRyaWdnZXJOYW1lLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25FdmVudFRyaWdnZXJzKGRvbWFpbk5hbWU6IHN0cmluZywgZXZlbnRUcmlnZ2VyTmFtZTogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OnByb2ZpbGU6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OmRvbWFpbnMvJHsgZG9tYWluTmFtZSB9L2V2ZW50LXRyaWdnZXJzLyR7IGV2ZW50VHJpZ2dlck5hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IGEga2V5IHRoYXQgaXMgcHJlc2VudCBpbiB0aGUgcmVxdWVzdCB0aGUgdXNlciBtYWtlcyB0byB0aGUgY3VzdG9tZXIgcHJvZmlsZSBzZXJ2aWNlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfaWFtLXBlcm1pc3Npb25zLmh0bWwjaWFtLWNvbnRleHRrZXlzXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9DcmVhdGVDYWxjdWxhdGVkQXR0cmlidXRlRGVmaW5pdGlvbigpXG4gICAqIC0gLnRvQ3JlYXRlRG9tYWluKClcbiAgICogLSAudG9DcmVhdGVFdmVudFN0cmVhbSgpXG4gICAqIC0gLnRvQ3JlYXRlRXZlbnRUcmlnZ2VyKClcbiAgICogLSAudG9DcmVhdGVJbnRlZ3JhdGlvbldvcmtmbG93KClcbiAgICogLSAudG9DcmVhdGVTZWdtZW50RGVmaW5pdGlvbigpXG4gICAqIC0gLnRvUHV0SW50ZWdyYXRpb24oKVxuICAgKiAtIC50b1B1dFByb2ZpbGVPYmplY3RUeXBlKClcbiAgICogLSAudG9UYWdSZXNvdXJjZSgpXG4gICAqXG4gICAqIEBwYXJhbSB0YWdLZXkgVGhlIHRhZyBrZXkgdG8gY2hlY2tcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NSZXF1ZXN0VGFnKHRhZ0tleTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYXdzOlJlcXVlc3RUYWcvJHsgdGFnS2V5IH1gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSBhIHRhZyBrZXkgYW5kIHZhbHVlIHBhaXJcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9pYW0tcGVybWlzc2lvbnMuaHRtbCNpYW0tY29udGV4dGtleXNcbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBkb21haW5zXG4gICAqIC0gb2JqZWN0LXR5cGVzXG4gICAqIC0gaW50ZWdyYXRpb25zXG4gICAqIC0gZXZlbnQtc3RyZWFtc1xuICAgKiAtIGNhbGN1bGF0ZWQtYXR0cmlidXRlc1xuICAgKiAtIHNlZ21lbnQtZGVmaW5pdGlvbnNcbiAgICogLSBldmVudC10cmlnZ2Vyc1xuICAgKlxuICAgKiBAcGFyYW0gdGFnS2V5IFRoZSB0YWcga2V5IHRvIGNoZWNrXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzUmVzb3VyY2VUYWcodGFnS2V5OiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6UmVzb3VyY2VUYWcvJHsgdGFnS2V5IH1gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgbGlzdCBvZiBhbGwgdGhlIHRhZyBrZXkgbmFtZXMgcHJlc2VudCBpbiB0aGUgcmVxdWVzdCB0aGUgdXNlciBtYWtlcyB0byB0aGUgY3VzdG9tZXIgcHJvZmlsZSBzZXJ2aWNlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfaWFtLXBlcm1pc3Npb25zLmh0bWwjaWFtLWNvbnRleHRrZXlzXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9DcmVhdGVDYWxjdWxhdGVkQXR0cmlidXRlRGVmaW5pdGlvbigpXG4gICAqIC0gLnRvQ3JlYXRlRG9tYWluKClcbiAgICogLSAudG9DcmVhdGVFdmVudFN0cmVhbSgpXG4gICAqIC0gLnRvQ3JlYXRlRXZlbnRUcmlnZ2VyKClcbiAgICogLSAudG9DcmVhdGVJbnRlZ3JhdGlvbldvcmtmbG93KClcbiAgICogLSAudG9DcmVhdGVTZWdtZW50RGVmaW5pdGlvbigpXG4gICAqIC0gLnRvUHV0SW50ZWdyYXRpb24oKVxuICAgKiAtIC50b1B1dFByb2ZpbGVPYmplY3RUeXBlKClcbiAgICogLSAudG9UYWdSZXNvdXJjZSgpXG4gICAqIC0gLnRvVW50YWdSZXNvdXJjZSgpXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzVGFnS2V5cyh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYXdzOlRhZ0tleXNgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW3Byb2ZpbGVdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXJ2aWNlLWF1dGhvcml6YXRpb24vbGF0ZXN0L3JlZmVyZW5jZS9saXN0X2FtYXpvbmNvbm5lY3RjdXN0b21lcnByb2ZpbGVzLmh0bWwpLlxuICAgKlxuICAgKi9cbiAgY29uc3RydWN0b3IocHJvcHM/OiBpYW0uUG9saWN5U3RhdGVtZW50UHJvcHMpIHtcbiAgICBzdXBlcihwcm9wcyk7XG4gIH1cbn1cbiJdfQ==