"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Controltower = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [controltower](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscontroltower.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Controltower extends shared_1.PolicyStatement {
    /**
     * Grants permission to create a landing zone
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - controltower:TagResource
     *
     * https://docs.aws.amazon.com/controltower/latest/APIReference/API_CreateLandingZone.html
     */
    toCreateLandingZone() {
        return this.to('CreateLandingZone');
    }
    /**
     * Grants permission to create an account managed by AWS Control Tower
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/account-factory.html
     */
    toCreateManagedAccount() {
        return this.to('CreateManagedAccount');
    }
    /**
     * Grants permission to delete AWS Control Tower landing zone
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/controltower/latest/APIReference/API_DeleteLandingZone.html
     */
    toDeleteLandingZone() {
        return this.to('DeleteLandingZone');
    }
    /**
     * Grants permission to deregister an account created through the account factory from AWS Control Tower
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/account-factory.html
     */
    toDeregisterManagedAccount() {
        return this.to('DeregisterManagedAccount');
    }
    /**
     * Grants permission to deregister an organizational unit from AWS Control Tower management
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/organizations.html
     */
    toDeregisterOrganizationalUnit() {
        return this.to('DeregisterOrganizationalUnit');
    }
    /**
     * Grants permission to describe the current account factory configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/account-factory.html
     */
    toDescribeAccountFactoryConfig() {
        return this.to('DescribeAccountFactoryConfig');
    }
    /**
     * Grants permission to describe resources managed by core accounts in AWS Control Tower
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/how-control-tower-works.html#what-shared
     */
    toDescribeCoreService() {
        return this.to('DescribeCoreService');
    }
    /**
     * Grants permission to describe a guardrail
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/controls.html
     */
    toDescribeGuardrail() {
        return this.to('DescribeGuardrail');
    }
    /**
     * Grants permission to describe a guardrail for a organizational unit
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/controls.html
     */
    toDescribeGuardrailForTarget() {
        return this.to('DescribeGuardrailForTarget');
    }
    /**
     * Grants permission to describe the current Landing Zone configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/step-two.html
     */
    toDescribeLandingZoneConfiguration() {
        return this.to('DescribeLandingZoneConfiguration');
    }
    /**
     * Grants permission to describe an account created through account factory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/account-factory.html
     */
    toDescribeManagedAccount() {
        return this.to('DescribeManagedAccount');
    }
    /**
     * Grants permission to describe an AWS Organizations organizational unit managed by AWS Control Tower
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/organizations.html
     */
    toDescribeManagedOrganizationalUnit() {
        return this.to('DescribeManagedOrganizationalUnit');
    }
    /**
     * Grants permission to describe a Register Organizational Unit Operation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/about-extending-governance.html
     */
    toDescribeRegisterOrganizationalUnitOperation() {
        return this.to('DescribeRegisterOrganizationalUnitOperation');
    }
    /**
     * Grants permission to describe the current AWS Control Tower IAM Identity Center configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/sso.html
     */
    toDescribeSingleSignOn() {
        return this.to('DescribeSingleSignOn');
    }
    /**
     * Grants permission to disable a Baseline on a target
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/controltower/latest/APIReference/API_DisableBaseline.html
     */
    toDisableBaseline() {
        return this.to('DisableBaseline');
    }
    /**
     * Grants permission to remove a control from an organizational unit
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/controltower/latest/APIReference/API_DisableControl.html
     */
    toDisableControl() {
        return this.to('DisableControl');
    }
    /**
     * Grants permission to disable a guardrail from an organizational unit
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/enable-controls-on-ou.html
     */
    toDisableGuardrail() {
        return this.to('DisableGuardrail');
    }
    /**
     * Grants permission to enable a Baseline on a target
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - controltower:TagResource
     *
     * https://docs.aws.amazon.com/controltower/latest/APIReference/API_EnableBaseline.html
     */
    toEnableBaseline() {
        return this.to('EnableBaseline');
    }
    /**
     * Grants permission to activate a control for an organizational unit
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - controltower:TagResource
     *
     * https://docs.aws.amazon.com/controltower/latest/APIReference/API_EnableControl.html
     */
    toEnableControl() {
        return this.to('EnableControl');
    }
    /**
     * Grants permission to enable a guardrail to an organizational unit
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/enable-controls-on-ou.html
     */
    toEnableGuardrail() {
        return this.to('EnableGuardrail');
    }
    /**
     * Grants permission to describe an account email and validate that it exists
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/accounts.html
     */
    toGetAccountInfo() {
        return this.to('GetAccountInfo');
    }
    /**
     * Grants permission to list available updates for the current AWS Control Tower deployment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/configuration-updates.html
     */
    toGetAvailableUpdates() {
        return this.to('GetAvailableUpdates');
    }
    /**
     * Grants permission to get Baseline details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/controltower/latest/APIReference/API_GetBaseline.html
     */
    toGetBaseline() {
        return this.to('GetBaseline');
    }
    /**
     * Grants permission to get the current status of a particular Baseline operation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/controltower/latest/APIReference/API_GetBaselineOperation.html
     */
    toGetBaselineOperation() {
        return this.to('GetBaselineOperation');
    }
    /**
     * Grants permission to get the current status of a particular EnabledControl or DisableControl operation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/controltower/latest/APIReference/API_GetControlOperation.html
     */
    toGetControlOperation() {
        return this.to('GetControlOperation');
    }
    /**
     * Grants permission to get an enabled Baseline
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/controltower/latest/APIReference/API_GetEnabledBaseline.html
     */
    toGetEnabledBaseline() {
        return this.to('GetEnabledBaseline');
    }
    /**
     * Grants permission to get an enabled control from an organizational unit
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/controltower/latest/APIReference/API_GetEnabledControl.html
     */
    toGetEnabledControl() {
        return this.to('GetEnabledControl');
    }
    /**
     * Grants permission to get the current compliance status of a guardrail
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/controls.html
     */
    toGetGuardrailComplianceStatus() {
        return this.to('GetGuardrailComplianceStatus');
    }
    /**
     * Grants permission to get the home region of the AWS Control Tower setup
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/how-control-tower-works.html#region-how
     */
    toGetHomeRegion() {
        return this.to('GetHomeRegion');
    }
    /**
     * Grants permission to get the current status of the landing zone setup
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/controltower/latest/APIReference/API_GetLandingZone.html
     */
    toGetLandingZone() {
        return this.to('GetLandingZone');
    }
    /**
     * Grants permission to get the current landing zone drift status
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/drift.html
     */
    toGetLandingZoneDriftStatus() {
        return this.to('GetLandingZoneDriftStatus');
    }
    /**
     * Grants permission to get the current status of a particular landing zone operation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/controltower/latest/APIReference/API_GetLandingZoneOperation.html
     */
    toGetLandingZoneOperation() {
        return this.to('GetLandingZoneOperation');
    }
    /**
     * Grants permission to get the current status of the landing zone setup
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/getting-started-with-control-tower.html#step-two
     */
    toGetLandingZoneStatus() {
        return this.to('GetLandingZoneStatus');
    }
    /**
     * Grants permission to list Baselines
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/controltower/latest/APIReference/API_ListBaselines.html
     */
    toListBaselines() {
        return this.to('ListBaselines');
    }
    /**
     * Grants permission to list all control operations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/controltower/latest/APIReference/API_ListControlOperations.html
     */
    toListControlOperations() {
        return this.to('ListControlOperations');
    }
    /**
     * Grants permission to list the current directory groups available through IAM Identity Center
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/sso.html
     */
    toListDirectoryGroups() {
        return this.to('ListDirectoryGroups');
    }
    /**
     * Grants permission to list occurrences of drift in AWS Control Tower
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/drift.html
     */
    toListDriftDetails() {
        return this.to('ListDriftDetails');
    }
    /**
     * Grants permission to list enabled Baselines
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/controltower/latest/APIReference/API_ListEnabledBaselines.html
     */
    toListEnabledBaselines() {
        return this.to('ListEnabledBaselines');
    }
    /**
     * Grants permission to list all enabled controls in a specified organizational unit
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/controltower/latest/APIReference/API_ListEnabledControls.html
     */
    toListEnabledControls() {
        return this.to('ListEnabledControls');
    }
    /**
     * Grants permission to list currently enabled guardrails
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/controls.html
     */
    toListEnabledGuardrails() {
        return this.to('ListEnabledGuardrails');
    }
    /**
     * Grants permission to list Precheck details for an Organizational Unit
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/about-extending-governance.html
     */
    toListExtendGovernancePrecheckDetails() {
        return this.to('ListExtendGovernancePrecheckDetails');
    }
    /**
     * Grants permission to list the compliance of external AWS Config rules
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/review-compliance.html
     */
    toListExternalConfigRuleCompliance() {
        return this.to('ListExternalConfigRuleCompliance');
    }
    /**
     * Grants permission to list existing guardrail violations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/controls.html
     */
    toListGuardrailViolations() {
        return this.to('ListGuardrailViolations');
    }
    /**
     * Grants permission to list all available guardrails
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/controls.html
     */
    toListGuardrails() {
        return this.to('ListGuardrails');
    }
    /**
     * Grants permission to list guardrails and their current state for a organizational unit
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/controls.html
     */
    toListGuardrailsForTarget() {
        return this.to('ListGuardrailsForTarget');
    }
    /**
     * Grants permission to list all landing zone operations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/controltower/latest/APIReference/API_ListLandingZoneOperations.html
     */
    toListLandingZoneOperations() {
        return this.to('ListLandingZoneOperations');
    }
    /**
     * Grants permission to list all landing zones
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/controltower/latest/APIReference/API_ListLandingZones.html
     */
    toListLandingZones() {
        return this.to('ListLandingZones');
    }
    /**
     * Grants permission to list accounts managed through AWS Control Tower
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/account-factory.html
     */
    toListManagedAccounts() {
        return this.to('ListManagedAccounts');
    }
    /**
     * Grants permission to list managed accounts with a specified guardrail applied
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/account-factory.html
     */
    toListManagedAccountsForGuardrail() {
        return this.to('ListManagedAccountsForGuardrail');
    }
    /**
     * Grants permission to list managed accounts under an organizational unit
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/account-factory.html
     */
    toListManagedAccountsForParent() {
        return this.to('ListManagedAccountsForParent');
    }
    /**
     * Grants permission to list organizational units managed by AWS Control Tower
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/organizations.html
     */
    toListManagedOrganizationalUnits() {
        return this.to('ListManagedOrganizationalUnits');
    }
    /**
     * Grants permission to list managed organizational units that have a specified guardrail applied
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/organizations.html
     */
    toListManagedOrganizationalUnitsForGuardrail() {
        return this.to('ListManagedOrganizationalUnitsForGuardrail');
    }
    /**
     * Grants permission to list the tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/controltower/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to set up an organizational unit to be managed by AWS Control Tower
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/organizations.html
     */
    toManageOrganizationalUnit() {
        return this.to('ManageOrganizationalUnit');
    }
    /**
     * Grants permission to perform validations in an account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/getting-started-prereqs.html
     */
    toPerformPreLaunchChecks() {
        return this.to('PerformPreLaunchChecks');
    }
    /**
     * Grants permission to reset an enabled Baseline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/controltower/latest/APIReference/API_ResetEnabledBaseline.html
     */
    toResetEnabledBaseline() {
        return this.to('ResetEnabledBaseline');
    }
    /**
     * Grants permission to reset an enabled control for an organizational unit
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/controltower/latest/APIReference/API_ResetEnabledControl.html
     */
    toResetEnabledControl() {
        return this.to('ResetEnabledControl');
    }
    /**
     * Grants permission to reset a landing zone
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/controltower/latest/APIReference/API_ResetLandingZone.html
     */
    toResetLandingZone() {
        return this.to('ResetLandingZone');
    }
    /**
     * Grants permission to set up or update AWS Control Tower landing zone
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/getting-started-with-control-tower.html#step-two
     */
    toSetupLandingZone() {
        return this.to('SetupLandingZone');
    }
    /**
     * Grants permission to add tags to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/controltower/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/controltower/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update the account factory configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/controltower/latest/userguide/account-factory.html
     */
    toUpdateAccountFactoryConfig() {
        return this.to('UpdateAccountFactoryConfig');
    }
    /**
     * Grants permission to update an enabled Baseline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/controltower/latest/APIReference/API_UpdateEnabledBaseline.html
     */
    toUpdateEnabledBaseline() {
        return this.to('UpdateEnabledBaseline');
    }
    /**
     * Grants permission to update an enabled control for an organizational unit
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/controltower/latest/APIReference/API_UpdateEnabledControl.html
     */
    toUpdateEnabledControl() {
        return this.to('UpdateEnabledControl');
    }
    /**
     * Grants permission to update a landing zone
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/controltower/latest/APIReference/API_UpdateLandingZone.html
     */
    toUpdateLandingZone() {
        return this.to('UpdateLandingZone');
    }
    /**
     * Adds a resource of type EnabledControl to the statement
     *
     * https://docs.aws.amazon.com/controltower/latest/APIReference/API_EnableControl.html
     *
     * @param enabledControlId - Identifier for the enabledControlId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEnabledControl(enabledControlId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:controltower:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:enabledcontrol/${enabledControlId}`);
    }
    /**
     * Adds a resource of type Baseline to the statement
     *
     * https://docs.aws.amazon.com/controltower/latest/APIReference/API_GetBaseline.html
     *
     * @param baselineId - Identifier for the baselineId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onBaseline(baselineId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:controltower:${region ?? this.defaultRegion}::baseline/${baselineId}`);
    }
    /**
     * Adds a resource of type EnabledBaseline to the statement
     *
     * https://docs.aws.amazon.com/controltower/latest/APIReference/API_EnableBaseline.html
     *
     * @param enabledBaselineId - Identifier for the enabledBaselineId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEnabledBaseline(enabledBaselineId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:controltower:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:enabledbaseline/${enabledBaselineId}`);
    }
    /**
     * Adds a resource of type LandingZone to the statement
     *
     * https://docs.aws.amazon.com/controltower/latest/APIReference/API_CreateLandingZone.html
     *
     * @param landingZoneId - Identifier for the landingZoneId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onLandingZone(landingZoneId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:controltower:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:landingzone/${landingZoneId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateLandingZone()
     * - .toEnableBaseline()
     * - .toEnableControl()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - EnabledControl
     * - EnabledBaseline
     * - LandingZone
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateLandingZone()
     * - .toEnableBaseline()
     * - .toEnableControl()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [controltower](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscontroltower.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'controltower';
        this.accessLevelList = {
            Write: [
                'CreateLandingZone',
                'CreateManagedAccount',
                'DeleteLandingZone',
                'DeregisterManagedAccount',
                'DeregisterOrganizationalUnit',
                'DisableBaseline',
                'DisableControl',
                'DisableGuardrail',
                'EnableBaseline',
                'EnableControl',
                'EnableGuardrail',
                'ManageOrganizationalUnit',
                'ResetEnabledBaseline',
                'ResetEnabledControl',
                'ResetLandingZone',
                'SetupLandingZone',
                'UpdateAccountFactoryConfig',
                'UpdateEnabledBaseline',
                'UpdateEnabledControl',
                'UpdateLandingZone'
            ],
            Read: [
                'DescribeAccountFactoryConfig',
                'DescribeCoreService',
                'DescribeGuardrail',
                'DescribeGuardrailForTarget',
                'DescribeLandingZoneConfiguration',
                'DescribeManagedAccount',
                'DescribeManagedOrganizationalUnit',
                'DescribeRegisterOrganizationalUnitOperation',
                'DescribeSingleSignOn',
                'GetAccountInfo',
                'GetAvailableUpdates',
                'GetBaseline',
                'GetBaselineOperation',
                'GetControlOperation',
                'GetEnabledBaseline',
                'GetEnabledControl',
                'GetGuardrailComplianceStatus',
                'GetHomeRegion',
                'GetLandingZone',
                'GetLandingZoneDriftStatus',
                'GetLandingZoneOperation',
                'GetLandingZoneStatus',
                'ListDriftDetails',
                'ListExternalConfigRuleCompliance',
                'ListTagsForResource',
                'PerformPreLaunchChecks'
            ],
            List: [
                'ListBaselines',
                'ListControlOperations',
                'ListDirectoryGroups',
                'ListEnabledBaselines',
                'ListEnabledControls',
                'ListEnabledGuardrails',
                'ListExtendGovernancePrecheckDetails',
                'ListGuardrailViolations',
                'ListGuardrails',
                'ListGuardrailsForTarget',
                'ListLandingZoneOperations',
                'ListLandingZones',
                'ListManagedAccounts',
                'ListManagedAccountsForGuardrail',
                'ListManagedAccountsForParent',
                'ListManagedOrganizationalUnits',
                'ListManagedOrganizationalUnitsForGuardrail'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Controltower = Controltower;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29udHJvbHRvd2VyLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiY29udHJvbHRvd2VyLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUNBLHlDQUF5RDtBQUd6RDs7OztHQUlHO0FBQ0gsTUFBYSxZQUFhLFNBQVEsd0JBQWU7SUFHL0M7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0JBQXNCO1FBQzNCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDBCQUEwQjtRQUMvQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMEJBQTBCLENBQUMsQ0FBQztJQUM3QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksOEJBQThCO1FBQ25DLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO0lBQ2pELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw4QkFBOEI7UUFDbkMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDhCQUE4QixDQUFDLENBQUM7SUFDakQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw0QkFBNEI7UUFDakMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDRCQUE0QixDQUFDLENBQUM7SUFDL0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtDQUFrQztRQUN2QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0NBQWtDLENBQUMsQ0FBQztJQUNyRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksd0JBQXdCO1FBQzdCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO0lBQzNDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQ0FBbUM7UUFDeEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1DQUFtQyxDQUFDLENBQUM7SUFDdEQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDZDQUE2QztRQUNsRCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNkNBQTZDLENBQUMsQ0FBQztJQUNoRSxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0JBQXNCO1FBQzNCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUI7UUFDdEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGlCQUFpQixDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztJQUNuQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksZ0JBQWdCO1FBQ3JCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO0lBQ25DLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksZUFBZTtRQUNwQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUJBQWlCLENBQUMsQ0FBQztJQUNwQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO0lBQ25DLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ2hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxzQkFBc0I7UUFDM0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUM7SUFDekMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO0lBQ3ZDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDhCQUE4QjtRQUNuQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsOEJBQThCLENBQUMsQ0FBQztJQUNqRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztJQUNuQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMkJBQTJCO1FBQ2hDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO0lBQzlDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx5QkFBeUI7UUFDOUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7SUFDNUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHVCQUF1QjtRQUM1QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsdUJBQXVCLENBQUMsQ0FBQztJQUMxQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFrQixDQUFDLENBQUM7SUFDckMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7SUFDMUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFDQUFxQztRQUMxQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUNBQXFDLENBQUMsQ0FBQztJQUN4RCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0NBQWtDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDO0lBQ3JELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx5QkFBeUI7UUFDOUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7SUFDNUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztJQUNuQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUJBQXlCO1FBQzlCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO0lBQzVDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwyQkFBMkI7UUFDaEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDJCQUEyQixDQUFDLENBQUM7SUFDOUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0JBQWtCLENBQUMsQ0FBQztJQUNyQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQ0FBaUM7UUFDdEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGlDQUFpQyxDQUFDLENBQUM7SUFDcEQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDhCQUE4QjtRQUNuQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsOEJBQThCLENBQUMsQ0FBQztJQUNqRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0NBQWdDO1FBQ3JDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO0lBQ25ELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw0Q0FBNEM7UUFDakQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDRDQUE0QyxDQUFDLENBQUM7SUFDL0QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMEJBQTBCO1FBQy9CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO0lBQzdDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx3QkFBd0I7UUFDN0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixDQUFDLENBQUM7SUFDM0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFrQixDQUFDLENBQUM7SUFDckMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0JBQWtCLENBQUMsQ0FBQztJQUNyQyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGFBQWE7UUFDbEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ2hDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxlQUFlO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksNEJBQTRCO1FBQ2pDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO0lBQy9DLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7SUFDMUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUE4RUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksZ0JBQWdCLENBQUMsZ0JBQXdCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDckcsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsaUJBQWtCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxtQkFBb0IsZ0JBQWlCLEVBQUUsQ0FBQyxDQUFDO0lBQ3hMLENBQUM7SUFFRDs7Ozs7Ozs7T0FRRztJQUNJLFVBQVUsQ0FBQyxVQUFrQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUN2RSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixpQkFBa0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLGNBQWUsVUFBVyxFQUFFLENBQUMsQ0FBQztJQUN6SSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksaUJBQWlCLENBQUMsaUJBQXlCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDdkcsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsaUJBQWtCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxvQkFBcUIsaUJBQWtCLEVBQUUsQ0FBQyxDQUFDO0lBQzFMLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxhQUFhLENBQUMsYUFBcUIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUMvRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixpQkFBa0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLGdCQUFpQixhQUFjLEVBQUUsQ0FBQyxDQUFDO0lBQ2xMLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7T0FjRztJQUNJLGVBQWUsQ0FBQyxNQUFjLEVBQUUsS0FBd0IsRUFBRSxRQUE0QjtRQUMzRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0JBQW1CLE1BQU8sRUFBRSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDaEYsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxnQkFBZ0IsQ0FBQyxNQUFjLEVBQUUsS0FBd0IsRUFBRSxRQUE0QjtRQUM1RixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW9CLE1BQU8sRUFBRSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDakYsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7OztPQWNHO0lBQ0ksWUFBWSxDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDeEUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2pFLENBQUM7SUFFRDs7O09BR0c7SUFDSCxZQUFZLEtBQWdDO1FBQzFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztRQWw3QlIsa0JBQWEsR0FBRyxjQUFjLENBQUM7UUF5dUI1QixvQkFBZSxHQUFvQjtZQUMzQyxLQUFLLEVBQUU7Z0JBQ0wsbUJBQW1CO2dCQUNuQixzQkFBc0I7Z0JBQ3RCLG1CQUFtQjtnQkFDbkIsMEJBQTBCO2dCQUMxQiw4QkFBOEI7Z0JBQzlCLGlCQUFpQjtnQkFDakIsZ0JBQWdCO2dCQUNoQixrQkFBa0I7Z0JBQ2xCLGdCQUFnQjtnQkFDaEIsZUFBZTtnQkFDZixpQkFBaUI7Z0JBQ2pCLDBCQUEwQjtnQkFDMUIsc0JBQXNCO2dCQUN0QixxQkFBcUI7Z0JBQ3JCLGtCQUFrQjtnQkFDbEIsa0JBQWtCO2dCQUNsQiw0QkFBNEI7Z0JBQzVCLHVCQUF1QjtnQkFDdkIsc0JBQXNCO2dCQUN0QixtQkFBbUI7YUFDcEI7WUFDRCxJQUFJLEVBQUU7Z0JBQ0osOEJBQThCO2dCQUM5QixxQkFBcUI7Z0JBQ3JCLG1CQUFtQjtnQkFDbkIsNEJBQTRCO2dCQUM1QixrQ0FBa0M7Z0JBQ2xDLHdCQUF3QjtnQkFDeEIsbUNBQW1DO2dCQUNuQyw2Q0FBNkM7Z0JBQzdDLHNCQUFzQjtnQkFDdEIsZ0JBQWdCO2dCQUNoQixxQkFBcUI7Z0JBQ3JCLGFBQWE7Z0JBQ2Isc0JBQXNCO2dCQUN0QixxQkFBcUI7Z0JBQ3JCLG9CQUFvQjtnQkFDcEIsbUJBQW1CO2dCQUNuQiw4QkFBOEI7Z0JBQzlCLGVBQWU7Z0JBQ2YsZ0JBQWdCO2dCQUNoQiwyQkFBMkI7Z0JBQzNCLHlCQUF5QjtnQkFDekIsc0JBQXNCO2dCQUN0QixrQkFBa0I7Z0JBQ2xCLGtDQUFrQztnQkFDbEMscUJBQXFCO2dCQUNyQix3QkFBd0I7YUFDekI7WUFDRCxJQUFJLEVBQUU7Z0JBQ0osZUFBZTtnQkFDZix1QkFBdUI7Z0JBQ3ZCLHFCQUFxQjtnQkFDckIsc0JBQXNCO2dCQUN0QixxQkFBcUI7Z0JBQ3JCLHVCQUF1QjtnQkFDdkIscUNBQXFDO2dCQUNyQyx5QkFBeUI7Z0JBQ3pCLGdCQUFnQjtnQkFDaEIseUJBQXlCO2dCQUN6QiwyQkFBMkI7Z0JBQzNCLGtCQUFrQjtnQkFDbEIscUJBQXFCO2dCQUNyQixpQ0FBaUM7Z0JBQ2pDLDhCQUE4QjtnQkFDOUIsZ0NBQWdDO2dCQUNoQyw0Q0FBNEM7YUFDN0M7WUFDRCxPQUFPLEVBQUU7Z0JBQ1AsYUFBYTtnQkFDYixlQUFlO2FBQ2hCO1NBQ0YsQ0FBQztJQWdJRixDQUFDO0NBQ0Y7QUFyN0JELG9DQXE3QkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBBY2Nlc3NMZXZlbExpc3QgfSBmcm9tICcuLi8uLi9zaGFyZWQvYWNjZXNzLWxldmVsJztcbmltcG9ydCB7IFBvbGljeVN0YXRlbWVudCwgT3BlcmF0b3IgfSBmcm9tICcuLi8uLi9zaGFyZWQnO1xuaW1wb3J0IHsgYXdzX2lhbSBhcyBpYW0gfSBmcm9tIFwiYXdzLWNkay1saWJcIjtcblxuLyoqXG4gKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW2NvbnRyb2x0b3dlcl0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYXdzY29udHJvbHRvd2VyLmh0bWwpLlxuICpcbiAqIEBwYXJhbSBzaWQgW1NJRF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19zaWQuaHRtbCkgb2YgdGhlIHN0YXRlbWVudFxuICovXG5leHBvcnQgY2xhc3MgQ29udHJvbHRvd2VyIGV4dGVuZHMgUG9saWN5U3RhdGVtZW50IHtcbiAgcHVibGljIHNlcnZpY2VQcmVmaXggPSAnY29udHJvbHRvd2VyJztcblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgbGFuZGluZyB6b25lXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gY29udHJvbHRvd2VyOlRhZ1Jlc291cmNlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb250cm9sdG93ZXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlTGFuZGluZ1pvbmUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlTGFuZGluZ1pvbmUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZUxhbmRpbmdab25lJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGFuIGFjY291bnQgbWFuYWdlZCBieSBBV1MgQ29udHJvbCBUb3dlclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb250cm9sdG93ZXIvbGF0ZXN0L3VzZXJndWlkZS9hY2NvdW50LWZhY3RvcnkuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlTWFuYWdlZEFjY291bnQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZU1hbmFnZWRBY2NvdW50Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIEFXUyBDb250cm9sIFRvd2VyIGxhbmRpbmcgem9uZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb250cm9sdG93ZXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlTGFuZGluZ1pvbmUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlTGFuZGluZ1pvbmUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZUxhbmRpbmdab25lJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVyZWdpc3RlciBhbiBhY2NvdW50IGNyZWF0ZWQgdGhyb3VnaCB0aGUgYWNjb3VudCBmYWN0b3J5IGZyb20gQVdTIENvbnRyb2wgVG93ZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29udHJvbHRvd2VyL2xhdGVzdC91c2VyZ3VpZGUvYWNjb3VudC1mYWN0b3J5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlcmVnaXN0ZXJNYW5hZ2VkQWNjb3VudCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVyZWdpc3Rlck1hbmFnZWRBY2NvdW50Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVyZWdpc3RlciBhbiBvcmdhbml6YXRpb25hbCB1bml0IGZyb20gQVdTIENvbnRyb2wgVG93ZXIgbWFuYWdlbWVudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb250cm9sdG93ZXIvbGF0ZXN0L3VzZXJndWlkZS9vcmdhbml6YXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlcmVnaXN0ZXJPcmdhbml6YXRpb25hbFVuaXQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlcmVnaXN0ZXJPcmdhbml6YXRpb25hbFVuaXQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZXNjcmliZSB0aGUgY3VycmVudCBhY2NvdW50IGZhY3RvcnkgY29uZmlndXJhdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbnRyb2x0b3dlci9sYXRlc3QvdXNlcmd1aWRlL2FjY291bnQtZmFjdG9yeS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZUFjY291bnRGYWN0b3J5Q29uZmlnKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZUFjY291bnRGYWN0b3J5Q29uZmlnJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVzY3JpYmUgcmVzb3VyY2VzIG1hbmFnZWQgYnkgY29yZSBhY2NvdW50cyBpbiBBV1MgQ29udHJvbCBUb3dlclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbnRyb2x0b3dlci9sYXRlc3QvdXNlcmd1aWRlL2hvdy1jb250cm9sLXRvd2VyLXdvcmtzLmh0bWwjd2hhdC1zaGFyZWRcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlQ29yZVNlcnZpY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlQ29yZVNlcnZpY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZXNjcmliZSBhIGd1YXJkcmFpbFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbnRyb2x0b3dlci9sYXRlc3QvdXNlcmd1aWRlL2NvbnRyb2xzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlR3VhcmRyYWlsKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZUd1YXJkcmFpbCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlc2NyaWJlIGEgZ3VhcmRyYWlsIGZvciBhIG9yZ2FuaXphdGlvbmFsIHVuaXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb250cm9sdG93ZXIvbGF0ZXN0L3VzZXJndWlkZS9jb250cm9scy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZUd1YXJkcmFpbEZvclRhcmdldCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVHdWFyZHJhaWxGb3JUYXJnZXQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZXNjcmliZSB0aGUgY3VycmVudCBMYW5kaW5nIFpvbmUgY29uZmlndXJhdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbnRyb2x0b3dlci9sYXRlc3QvdXNlcmd1aWRlL3N0ZXAtdHdvLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlTGFuZGluZ1pvbmVDb25maWd1cmF0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZUxhbmRpbmdab25lQ29uZmlndXJhdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlc2NyaWJlIGFuIGFjY291bnQgY3JlYXRlZCB0aHJvdWdoIGFjY291bnQgZmFjdG9yeVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbnRyb2x0b3dlci9sYXRlc3QvdXNlcmd1aWRlL2FjY291bnQtZmFjdG9yeS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZU1hbmFnZWRBY2NvdW50KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZU1hbmFnZWRBY2NvdW50Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVzY3JpYmUgYW4gQVdTIE9yZ2FuaXphdGlvbnMgb3JnYW5pemF0aW9uYWwgdW5pdCBtYW5hZ2VkIGJ5IEFXUyBDb250cm9sIFRvd2VyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29udHJvbHRvd2VyL2xhdGVzdC91c2VyZ3VpZGUvb3JnYW5pemF0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZU1hbmFnZWRPcmdhbml6YXRpb25hbFVuaXQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlTWFuYWdlZE9yZ2FuaXphdGlvbmFsVW5pdCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlc2NyaWJlIGEgUmVnaXN0ZXIgT3JnYW5pemF0aW9uYWwgVW5pdCBPcGVyYXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb250cm9sdG93ZXIvbGF0ZXN0L3VzZXJndWlkZS9hYm91dC1leHRlbmRpbmctZ292ZXJuYW5jZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZVJlZ2lzdGVyT3JnYW5pemF0aW9uYWxVbml0T3BlcmF0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZVJlZ2lzdGVyT3JnYW5pemF0aW9uYWxVbml0T3BlcmF0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVzY3JpYmUgdGhlIGN1cnJlbnQgQVdTIENvbnRyb2wgVG93ZXIgSUFNIElkZW50aXR5IENlbnRlciBjb25maWd1cmF0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29udHJvbHRvd2VyL2xhdGVzdC91c2VyZ3VpZGUvc3NvLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlU2luZ2xlU2lnbk9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZVNpbmdsZVNpZ25PbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRpc2FibGUgYSBCYXNlbGluZSBvbiBhIHRhcmdldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb250cm9sdG93ZXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGlzYWJsZUJhc2VsaW5lLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rpc2FibGVCYXNlbGluZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGlzYWJsZUJhc2VsaW5lJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmVtb3ZlIGEgY29udHJvbCBmcm9tIGFuIG9yZ2FuaXphdGlvbmFsIHVuaXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29udHJvbHRvd2VyL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rpc2FibGVDb250cm9sLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rpc2FibGVDb250cm9sKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEaXNhYmxlQ29udHJvbCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRpc2FibGUgYSBndWFyZHJhaWwgZnJvbSBhbiBvcmdhbml6YXRpb25hbCB1bml0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbnRyb2x0b3dlci9sYXRlc3QvdXNlcmd1aWRlL2VuYWJsZS1jb250cm9scy1vbi1vdS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EaXNhYmxlR3VhcmRyYWlsKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEaXNhYmxlR3VhcmRyYWlsJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZW5hYmxlIGEgQmFzZWxpbmUgb24gYSB0YXJnZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBjb250cm9sdG93ZXI6VGFnUmVzb3VyY2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbnRyb2x0b3dlci9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9FbmFibGVCYXNlbGluZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9FbmFibGVCYXNlbGluZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRW5hYmxlQmFzZWxpbmUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBhY3RpdmF0ZSBhIGNvbnRyb2wgZm9yIGFuIG9yZ2FuaXphdGlvbmFsIHVuaXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBjb250cm9sdG93ZXI6VGFnUmVzb3VyY2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbnRyb2x0b3dlci9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9FbmFibGVDb250cm9sLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0VuYWJsZUNvbnRyb2woKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0VuYWJsZUNvbnRyb2wnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBlbmFibGUgYSBndWFyZHJhaWwgdG8gYW4gb3JnYW5pemF0aW9uYWwgdW5pdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb250cm9sdG93ZXIvbGF0ZXN0L3VzZXJndWlkZS9lbmFibGUtY29udHJvbHMtb24tb3UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRW5hYmxlR3VhcmRyYWlsKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdFbmFibGVHdWFyZHJhaWwnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZXNjcmliZSBhbiBhY2NvdW50IGVtYWlsIGFuZCB2YWxpZGF0ZSB0aGF0IGl0IGV4aXN0c1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbnRyb2x0b3dlci9sYXRlc3QvdXNlcmd1aWRlL2FjY291bnRzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEFjY291bnRJbmZvKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRBY2NvdW50SW5mbycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgYXZhaWxhYmxlIHVwZGF0ZXMgZm9yIHRoZSBjdXJyZW50IEFXUyBDb250cm9sIFRvd2VyIGRlcGxveW1lbnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb250cm9sdG93ZXIvbGF0ZXN0L3VzZXJndWlkZS9jb25maWd1cmF0aW9uLXVwZGF0ZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0QXZhaWxhYmxlVXBkYXRlcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0QXZhaWxhYmxlVXBkYXRlcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBCYXNlbGluZSBkZXRhaWxzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29udHJvbHRvd2VyL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldEJhc2VsaW5lLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEJhc2VsaW5lKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRCYXNlbGluZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCB0aGUgY3VycmVudCBzdGF0dXMgb2YgYSBwYXJ0aWN1bGFyIEJhc2VsaW5lIG9wZXJhdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbnRyb2x0b3dlci9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9HZXRCYXNlbGluZU9wZXJhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRCYXNlbGluZU9wZXJhdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0QmFzZWxpbmVPcGVyYXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgdGhlIGN1cnJlbnQgc3RhdHVzIG9mIGEgcGFydGljdWxhciBFbmFibGVkQ29udHJvbCBvciBEaXNhYmxlQ29udHJvbCBvcGVyYXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb250cm9sdG93ZXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0Q29udHJvbE9wZXJhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRDb250cm9sT3BlcmF0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRDb250cm9sT3BlcmF0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IGFuIGVuYWJsZWQgQmFzZWxpbmVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb250cm9sdG93ZXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0RW5hYmxlZEJhc2VsaW5lLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEVuYWJsZWRCYXNlbGluZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0RW5hYmxlZEJhc2VsaW5lJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IGFuIGVuYWJsZWQgY29udHJvbCBmcm9tIGFuIG9yZ2FuaXphdGlvbmFsIHVuaXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb250cm9sdG93ZXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0RW5hYmxlZENvbnRyb2wuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0RW5hYmxlZENvbnRyb2woKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldEVuYWJsZWRDb250cm9sJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IHRoZSBjdXJyZW50IGNvbXBsaWFuY2Ugc3RhdHVzIG9mIGEgZ3VhcmRyYWlsXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29udHJvbHRvd2VyL2xhdGVzdC91c2VyZ3VpZGUvY29udHJvbHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0R3VhcmRyYWlsQ29tcGxpYW5jZVN0YXR1cygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0R3VhcmRyYWlsQ29tcGxpYW5jZVN0YXR1cycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCB0aGUgaG9tZSByZWdpb24gb2YgdGhlIEFXUyBDb250cm9sIFRvd2VyIHNldHVwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29udHJvbHRvd2VyL2xhdGVzdC91c2VyZ3VpZGUvaG93LWNvbnRyb2wtdG93ZXItd29ya3MuaHRtbCNyZWdpb24taG93XG4gICAqL1xuICBwdWJsaWMgdG9HZXRIb21lUmVnaW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRIb21lUmVnaW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IHRoZSBjdXJyZW50IHN0YXR1cyBvZiB0aGUgbGFuZGluZyB6b25lIHNldHVwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29udHJvbHRvd2VyL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldExhbmRpbmdab25lLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldExhbmRpbmdab25lKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRMYW5kaW5nWm9uZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCB0aGUgY3VycmVudCBsYW5kaW5nIHpvbmUgZHJpZnQgc3RhdHVzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29udHJvbHRvd2VyL2xhdGVzdC91c2VyZ3VpZGUvZHJpZnQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0TGFuZGluZ1pvbmVEcmlmdFN0YXR1cygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0TGFuZGluZ1pvbmVEcmlmdFN0YXR1cycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCB0aGUgY3VycmVudCBzdGF0dXMgb2YgYSBwYXJ0aWN1bGFyIGxhbmRpbmcgem9uZSBvcGVyYXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb250cm9sdG93ZXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0TGFuZGluZ1pvbmVPcGVyYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0TGFuZGluZ1pvbmVPcGVyYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldExhbmRpbmdab25lT3BlcmF0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IHRoZSBjdXJyZW50IHN0YXR1cyBvZiB0aGUgbGFuZGluZyB6b25lIHNldHVwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29udHJvbHRvd2VyL2xhdGVzdC91c2VyZ3VpZGUvZ2V0dGluZy1zdGFydGVkLXdpdGgtY29udHJvbC10b3dlci5odG1sI3N0ZXAtdHdvXG4gICAqL1xuICBwdWJsaWMgdG9HZXRMYW5kaW5nWm9uZVN0YXR1cygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0TGFuZGluZ1pvbmVTdGF0dXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IEJhc2VsaW5lc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbnRyb2x0b3dlci9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0QmFzZWxpbmVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RCYXNlbGluZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RCYXNlbGluZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IGFsbCBjb250cm9sIG9wZXJhdGlvbnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb250cm9sdG93ZXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdENvbnRyb2xPcGVyYXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RDb250cm9sT3BlcmF0aW9ucygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdENvbnRyb2xPcGVyYXRpb25zJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCB0aGUgY3VycmVudCBkaXJlY3RvcnkgZ3JvdXBzIGF2YWlsYWJsZSB0aHJvdWdoIElBTSBJZGVudGl0eSBDZW50ZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb250cm9sdG93ZXIvbGF0ZXN0L3VzZXJndWlkZS9zc28uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdERpcmVjdG9yeUdyb3VwcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdERpcmVjdG9yeUdyb3VwcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3Qgb2NjdXJyZW5jZXMgb2YgZHJpZnQgaW4gQVdTIENvbnRyb2wgVG93ZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb250cm9sdG93ZXIvbGF0ZXN0L3VzZXJndWlkZS9kcmlmdC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0RHJpZnREZXRhaWxzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0RHJpZnREZXRhaWxzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBlbmFibGVkIEJhc2VsaW5lc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbnRyb2x0b3dlci9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0RW5hYmxlZEJhc2VsaW5lcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0RW5hYmxlZEJhc2VsaW5lcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdEVuYWJsZWRCYXNlbGluZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IGFsbCBlbmFibGVkIGNvbnRyb2xzIGluIGEgc3BlY2lmaWVkIG9yZ2FuaXphdGlvbmFsIHVuaXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb250cm9sdG93ZXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdEVuYWJsZWRDb250cm9scy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0RW5hYmxlZENvbnRyb2xzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0RW5hYmxlZENvbnRyb2xzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBjdXJyZW50bHkgZW5hYmxlZCBndWFyZHJhaWxzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29udHJvbHRvd2VyL2xhdGVzdC91c2VyZ3VpZGUvY29udHJvbHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdEVuYWJsZWRHdWFyZHJhaWxzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0RW5hYmxlZEd1YXJkcmFpbHMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IFByZWNoZWNrIGRldGFpbHMgZm9yIGFuIE9yZ2FuaXphdGlvbmFsIFVuaXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb250cm9sdG93ZXIvbGF0ZXN0L3VzZXJndWlkZS9hYm91dC1leHRlbmRpbmctZ292ZXJuYW5jZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0RXh0ZW5kR292ZXJuYW5jZVByZWNoZWNrRGV0YWlscygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdEV4dGVuZEdvdmVybmFuY2VQcmVjaGVja0RldGFpbHMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHRoZSBjb21wbGlhbmNlIG9mIGV4dGVybmFsIEFXUyBDb25maWcgcnVsZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb250cm9sdG93ZXIvbGF0ZXN0L3VzZXJndWlkZS9yZXZpZXctY29tcGxpYW5jZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0RXh0ZXJuYWxDb25maWdSdWxlQ29tcGxpYW5jZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdEV4dGVybmFsQ29uZmlnUnVsZUNvbXBsaWFuY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IGV4aXN0aW5nIGd1YXJkcmFpbCB2aW9sYXRpb25zXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29udHJvbHRvd2VyL2xhdGVzdC91c2VyZ3VpZGUvY29udHJvbHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdEd1YXJkcmFpbFZpb2xhdGlvbnMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RHdWFyZHJhaWxWaW9sYXRpb25zJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBhbGwgYXZhaWxhYmxlIGd1YXJkcmFpbHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb250cm9sdG93ZXIvbGF0ZXN0L3VzZXJndWlkZS9jb250cm9scy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0R3VhcmRyYWlscygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdEd1YXJkcmFpbHMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IGd1YXJkcmFpbHMgYW5kIHRoZWlyIGN1cnJlbnQgc3RhdGUgZm9yIGEgb3JnYW5pemF0aW9uYWwgdW5pdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbnRyb2x0b3dlci9sYXRlc3QvdXNlcmd1aWRlL2NvbnRyb2xzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RHdWFyZHJhaWxzRm9yVGFyZ2V0KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0R3VhcmRyYWlsc0ZvclRhcmdldCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgYWxsIGxhbmRpbmcgem9uZSBvcGVyYXRpb25zXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29udHJvbHRvd2VyL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RMYW5kaW5nWm9uZU9wZXJhdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdExhbmRpbmdab25lT3BlcmF0aW9ucygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdExhbmRpbmdab25lT3BlcmF0aW9ucycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgYWxsIGxhbmRpbmcgem9uZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb250cm9sdG93ZXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdExhbmRpbmdab25lcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0TGFuZGluZ1pvbmVzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0TGFuZGluZ1pvbmVzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBhY2NvdW50cyBtYW5hZ2VkIHRocm91Z2ggQVdTIENvbnRyb2wgVG93ZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb250cm9sdG93ZXIvbGF0ZXN0L3VzZXJndWlkZS9hY2NvdW50LWZhY3RvcnkuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdE1hbmFnZWRBY2NvdW50cygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdE1hbmFnZWRBY2NvdW50cycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgbWFuYWdlZCBhY2NvdW50cyB3aXRoIGEgc3BlY2lmaWVkIGd1YXJkcmFpbCBhcHBsaWVkXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29udHJvbHRvd2VyL2xhdGVzdC91c2VyZ3VpZGUvYWNjb3VudC1mYWN0b3J5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RNYW5hZ2VkQWNjb3VudHNGb3JHdWFyZHJhaWwoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RNYW5hZ2VkQWNjb3VudHNGb3JHdWFyZHJhaWwnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IG1hbmFnZWQgYWNjb3VudHMgdW5kZXIgYW4gb3JnYW5pemF0aW9uYWwgdW5pdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbnRyb2x0b3dlci9sYXRlc3QvdXNlcmd1aWRlL2FjY291bnQtZmFjdG9yeS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0TWFuYWdlZEFjY291bnRzRm9yUGFyZW50KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0TWFuYWdlZEFjY291bnRzRm9yUGFyZW50Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBvcmdhbml6YXRpb25hbCB1bml0cyBtYW5hZ2VkIGJ5IEFXUyBDb250cm9sIFRvd2VyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29udHJvbHRvd2VyL2xhdGVzdC91c2VyZ3VpZGUvb3JnYW5pemF0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0TWFuYWdlZE9yZ2FuaXphdGlvbmFsVW5pdHMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RNYW5hZ2VkT3JnYW5pemF0aW9uYWxVbml0cycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgbWFuYWdlZCBvcmdhbml6YXRpb25hbCB1bml0cyB0aGF0IGhhdmUgYSBzcGVjaWZpZWQgZ3VhcmRyYWlsIGFwcGxpZWRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb250cm9sdG93ZXIvbGF0ZXN0L3VzZXJndWlkZS9vcmdhbml6YXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RNYW5hZ2VkT3JnYW5pemF0aW9uYWxVbml0c0Zvckd1YXJkcmFpbCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdE1hbmFnZWRPcmdhbml6YXRpb25hbFVuaXRzRm9yR3VhcmRyYWlsJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCB0aGUgdGFncyBmb3IgYSByZXNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbnRyb2x0b3dlci9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0VGFnc0ZvclJlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RUYWdzRm9yUmVzb3VyY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RUYWdzRm9yUmVzb3VyY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBzZXQgdXAgYW4gb3JnYW5pemF0aW9uYWwgdW5pdCB0byBiZSBtYW5hZ2VkIGJ5IEFXUyBDb250cm9sIFRvd2VyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbnRyb2x0b3dlci9sYXRlc3QvdXNlcmd1aWRlL29yZ2FuaXphdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTWFuYWdlT3JnYW5pemF0aW9uYWxVbml0KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdNYW5hZ2VPcmdhbml6YXRpb25hbFVuaXQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBwZXJmb3JtIHZhbGlkYXRpb25zIGluIGFuIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb250cm9sdG93ZXIvbGF0ZXN0L3VzZXJndWlkZS9nZXR0aW5nLXN0YXJ0ZWQtcHJlcmVxcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QZXJmb3JtUHJlTGF1bmNoQ2hlY2tzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdQZXJmb3JtUHJlTGF1bmNoQ2hlY2tzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmVzZXQgYW4gZW5hYmxlZCBCYXNlbGluZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb250cm9sdG93ZXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUmVzZXRFbmFibGVkQmFzZWxpbmUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUmVzZXRFbmFibGVkQmFzZWxpbmUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1Jlc2V0RW5hYmxlZEJhc2VsaW5lJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmVzZXQgYW4gZW5hYmxlZCBjb250cm9sIGZvciBhbiBvcmdhbml6YXRpb25hbCB1bml0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbnRyb2x0b3dlci9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9SZXNldEVuYWJsZWRDb250cm9sLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1Jlc2V0RW5hYmxlZENvbnRyb2woKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1Jlc2V0RW5hYmxlZENvbnRyb2wnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXNldCBhIGxhbmRpbmcgem9uZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb250cm9sdG93ZXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUmVzZXRMYW5kaW5nWm9uZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9SZXNldExhbmRpbmdab25lKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdSZXNldExhbmRpbmdab25lJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc2V0IHVwIG9yIHVwZGF0ZSBBV1MgQ29udHJvbCBUb3dlciBsYW5kaW5nIHpvbmVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29udHJvbHRvd2VyL2xhdGVzdC91c2VyZ3VpZGUvZ2V0dGluZy1zdGFydGVkLXdpdGgtY29udHJvbC10b3dlci5odG1sI3N0ZXAtdHdvXG4gICAqL1xuICBwdWJsaWMgdG9TZXR1cExhbmRpbmdab25lKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTZXR1cExhbmRpbmdab25lJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYWRkIHRhZ3MgdG8gYSByZXNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFRhZ2dpbmdcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbnRyb2x0b3dlci9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9UYWdSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9UYWdSZXNvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVGFnUmVzb3VyY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZW1vdmUgdGFncyBmcm9tIGEgcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb250cm9sdG93ZXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVW50YWdSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VbnRhZ1Jlc291cmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVbnRhZ1Jlc291cmNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIHRoZSBhY2NvdW50IGZhY3RvcnkgY29uZmlndXJhdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb250cm9sdG93ZXIvbGF0ZXN0L3VzZXJndWlkZS9hY2NvdW50LWZhY3RvcnkuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlQWNjb3VudEZhY3RvcnlDb25maWcoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZUFjY291bnRGYWN0b3J5Q29uZmlnJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIGFuIGVuYWJsZWQgQmFzZWxpbmVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29udHJvbHRvd2VyL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZUVuYWJsZWRCYXNlbGluZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVFbmFibGVkQmFzZWxpbmUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZUVuYWJsZWRCYXNlbGluZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhbiBlbmFibGVkIGNvbnRyb2wgZm9yIGFuIG9yZ2FuaXphdGlvbmFsIHVuaXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29udHJvbHRvd2VyL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZUVuYWJsZWRDb250cm9sLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUVuYWJsZWRDb250cm9sKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVFbmFibGVkQ29udHJvbCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhIGxhbmRpbmcgem9uZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb250cm9sdG93ZXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlTGFuZGluZ1pvbmUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlTGFuZGluZ1pvbmUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZUxhbmRpbmdab25lJyk7XG4gIH1cblxuICBwcm90ZWN0ZWQgYWNjZXNzTGV2ZWxMaXN0OiBBY2Nlc3NMZXZlbExpc3QgPSB7XG4gICAgV3JpdGU6IFtcbiAgICAgICdDcmVhdGVMYW5kaW5nWm9uZScsXG4gICAgICAnQ3JlYXRlTWFuYWdlZEFjY291bnQnLFxuICAgICAgJ0RlbGV0ZUxhbmRpbmdab25lJyxcbiAgICAgICdEZXJlZ2lzdGVyTWFuYWdlZEFjY291bnQnLFxuICAgICAgJ0RlcmVnaXN0ZXJPcmdhbml6YXRpb25hbFVuaXQnLFxuICAgICAgJ0Rpc2FibGVCYXNlbGluZScsXG4gICAgICAnRGlzYWJsZUNvbnRyb2wnLFxuICAgICAgJ0Rpc2FibGVHdWFyZHJhaWwnLFxuICAgICAgJ0VuYWJsZUJhc2VsaW5lJyxcbiAgICAgICdFbmFibGVDb250cm9sJyxcbiAgICAgICdFbmFibGVHdWFyZHJhaWwnLFxuICAgICAgJ01hbmFnZU9yZ2FuaXphdGlvbmFsVW5pdCcsXG4gICAgICAnUmVzZXRFbmFibGVkQmFzZWxpbmUnLFxuICAgICAgJ1Jlc2V0RW5hYmxlZENvbnRyb2wnLFxuICAgICAgJ1Jlc2V0TGFuZGluZ1pvbmUnLFxuICAgICAgJ1NldHVwTGFuZGluZ1pvbmUnLFxuICAgICAgJ1VwZGF0ZUFjY291bnRGYWN0b3J5Q29uZmlnJyxcbiAgICAgICdVcGRhdGVFbmFibGVkQmFzZWxpbmUnLFxuICAgICAgJ1VwZGF0ZUVuYWJsZWRDb250cm9sJyxcbiAgICAgICdVcGRhdGVMYW5kaW5nWm9uZSdcbiAgICBdLFxuICAgIFJlYWQ6IFtcbiAgICAgICdEZXNjcmliZUFjY291bnRGYWN0b3J5Q29uZmlnJyxcbiAgICAgICdEZXNjcmliZUNvcmVTZXJ2aWNlJyxcbiAgICAgICdEZXNjcmliZUd1YXJkcmFpbCcsXG4gICAgICAnRGVzY3JpYmVHdWFyZHJhaWxGb3JUYXJnZXQnLFxuICAgICAgJ0Rlc2NyaWJlTGFuZGluZ1pvbmVDb25maWd1cmF0aW9uJyxcbiAgICAgICdEZXNjcmliZU1hbmFnZWRBY2NvdW50JyxcbiAgICAgICdEZXNjcmliZU1hbmFnZWRPcmdhbml6YXRpb25hbFVuaXQnLFxuICAgICAgJ0Rlc2NyaWJlUmVnaXN0ZXJPcmdhbml6YXRpb25hbFVuaXRPcGVyYXRpb24nLFxuICAgICAgJ0Rlc2NyaWJlU2luZ2xlU2lnbk9uJyxcbiAgICAgICdHZXRBY2NvdW50SW5mbycsXG4gICAgICAnR2V0QXZhaWxhYmxlVXBkYXRlcycsXG4gICAgICAnR2V0QmFzZWxpbmUnLFxuICAgICAgJ0dldEJhc2VsaW5lT3BlcmF0aW9uJyxcbiAgICAgICdHZXRDb250cm9sT3BlcmF0aW9uJyxcbiAgICAgICdHZXRFbmFibGVkQmFzZWxpbmUnLFxuICAgICAgJ0dldEVuYWJsZWRDb250cm9sJyxcbiAgICAgICdHZXRHdWFyZHJhaWxDb21wbGlhbmNlU3RhdHVzJyxcbiAgICAgICdHZXRIb21lUmVnaW9uJyxcbiAgICAgICdHZXRMYW5kaW5nWm9uZScsXG4gICAgICAnR2V0TGFuZGluZ1pvbmVEcmlmdFN0YXR1cycsXG4gICAgICAnR2V0TGFuZGluZ1pvbmVPcGVyYXRpb24nLFxuICAgICAgJ0dldExhbmRpbmdab25lU3RhdHVzJyxcbiAgICAgICdMaXN0RHJpZnREZXRhaWxzJyxcbiAgICAgICdMaXN0RXh0ZXJuYWxDb25maWdSdWxlQ29tcGxpYW5jZScsXG4gICAgICAnTGlzdFRhZ3NGb3JSZXNvdXJjZScsXG4gICAgICAnUGVyZm9ybVByZUxhdW5jaENoZWNrcydcbiAgICBdLFxuICAgIExpc3Q6IFtcbiAgICAgICdMaXN0QmFzZWxpbmVzJyxcbiAgICAgICdMaXN0Q29udHJvbE9wZXJhdGlvbnMnLFxuICAgICAgJ0xpc3REaXJlY3RvcnlHcm91cHMnLFxuICAgICAgJ0xpc3RFbmFibGVkQmFzZWxpbmVzJyxcbiAgICAgICdMaXN0RW5hYmxlZENvbnRyb2xzJyxcbiAgICAgICdMaXN0RW5hYmxlZEd1YXJkcmFpbHMnLFxuICAgICAgJ0xpc3RFeHRlbmRHb3Zlcm5hbmNlUHJlY2hlY2tEZXRhaWxzJyxcbiAgICAgICdMaXN0R3VhcmRyYWlsVmlvbGF0aW9ucycsXG4gICAgICAnTGlzdEd1YXJkcmFpbHMnLFxuICAgICAgJ0xpc3RHdWFyZHJhaWxzRm9yVGFyZ2V0JyxcbiAgICAgICdMaXN0TGFuZGluZ1pvbmVPcGVyYXRpb25zJyxcbiAgICAgICdMaXN0TGFuZGluZ1pvbmVzJyxcbiAgICAgICdMaXN0TWFuYWdlZEFjY291bnRzJyxcbiAgICAgICdMaXN0TWFuYWdlZEFjY291bnRzRm9yR3VhcmRyYWlsJyxcbiAgICAgICdMaXN0TWFuYWdlZEFjY291bnRzRm9yUGFyZW50JyxcbiAgICAgICdMaXN0TWFuYWdlZE9yZ2FuaXphdGlvbmFsVW5pdHMnLFxuICAgICAgJ0xpc3RNYW5hZ2VkT3JnYW5pemF0aW9uYWxVbml0c0Zvckd1YXJkcmFpbCdcbiAgICBdLFxuICAgIFRhZ2dpbmc6IFtcbiAgICAgICdUYWdSZXNvdXJjZScsXG4gICAgICAnVW50YWdSZXNvdXJjZSdcbiAgICBdXG4gIH07XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIEVuYWJsZWRDb250cm9sIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbnRyb2x0b3dlci9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9FbmFibGVDb250cm9sLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGVuYWJsZWRDb250cm9sSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgZW5hYmxlZENvbnRyb2xJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uRW5hYmxlZENvbnRyb2woZW5hYmxlZENvbnRyb2xJZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmNvbnRyb2x0b3dlcjokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06ZW5hYmxlZGNvbnRyb2wvJHsgZW5hYmxlZENvbnRyb2xJZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgQmFzZWxpbmUgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29udHJvbHRvd2VyL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldEJhc2VsaW5lLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGJhc2VsaW5lSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgYmFzZWxpbmVJZC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKi9cbiAgcHVibGljIG9uQmFzZWxpbmUoYmFzZWxpbmVJZDogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmNvbnRyb2x0b3dlcjokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06OmJhc2VsaW5lLyR7IGJhc2VsaW5lSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIEVuYWJsZWRCYXNlbGluZSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb250cm9sdG93ZXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRW5hYmxlQmFzZWxpbmUuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gZW5hYmxlZEJhc2VsaW5lSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgZW5hYmxlZEJhc2VsaW5lSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkVuYWJsZWRCYXNlbGluZShlbmFibGVkQmFzZWxpbmVJZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmNvbnRyb2x0b3dlcjokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06ZW5hYmxlZGJhc2VsaW5lLyR7IGVuYWJsZWRCYXNlbGluZUlkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBMYW5kaW5nWm9uZSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb250cm9sdG93ZXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlTGFuZGluZ1pvbmUuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gbGFuZGluZ1pvbmVJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBsYW5kaW5nWm9uZUlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25MYW5kaW5nWm9uZShsYW5kaW5nWm9uZUlkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06Y29udHJvbHRvd2VyOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpsYW5kaW5nem9uZS8keyBsYW5kaW5nWm9uZUlkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgdGFncyB0aGF0IGFyZSBwYXNzZWQgaW4gdGhlIHJlcXVlc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXJlcXVlc3R0YWdcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0NyZWF0ZUxhbmRpbmdab25lKClcbiAgICogLSAudG9FbmFibGVCYXNlbGluZSgpXG4gICAqIC0gLnRvRW5hYmxlQ29udHJvbCgpXG4gICAqIC0gLnRvVGFnUmVzb3VyY2UoKVxuICAgKlxuICAgKiBAcGFyYW0gdGFnS2V5IFRoZSB0YWcga2V5IHRvIGNoZWNrXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzUmVxdWVzdFRhZyh0YWdLZXk6IHN0cmluZywgdmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpSZXF1ZXN0VGFnLyR7IHRhZ0tleSB9YCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgdGhlIHRhZ3MgYXNzb2NpYXRlZCB3aXRoIHRoZSByZXNvdXJjZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtcmVzb3VyY2V0YWdcbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBFbmFibGVkQ29udHJvbFxuICAgKiAtIEVuYWJsZWRCYXNlbGluZVxuICAgKiAtIExhbmRpbmdab25lXG4gICAqXG4gICAqIEBwYXJhbSB0YWdLZXkgVGhlIHRhZyBrZXkgdG8gY2hlY2tcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NSZXNvdXJjZVRhZyh0YWdLZXk6IHN0cmluZywgdmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpSZXNvdXJjZVRhZy8keyB0YWdLZXkgfWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSB0YWcga2V5cyB0aGF0IGFyZSBwYXNzZWQgaW4gdGhlIHJlcXVlc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXRhZ2tleXNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0NyZWF0ZUxhbmRpbmdab25lKClcbiAgICogLSAudG9FbmFibGVCYXNlbGluZSgpXG4gICAqIC0gLnRvRW5hYmxlQ29udHJvbCgpXG4gICAqIC0gLnRvVGFnUmVzb3VyY2UoKVxuICAgKiAtIC50b1VudGFnUmVzb3VyY2UoKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1RhZ0tleXModmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpUYWdLZXlzYCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFtjb250cm9sdG93ZXJdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXJ2aWNlLWF1dGhvcml6YXRpb24vbGF0ZXN0L3JlZmVyZW5jZS9saXN0X2F3c2NvbnRyb2x0b3dlci5odG1sKS5cbiAgICpcbiAgICovXG4gIGNvbnN0cnVjdG9yKHByb3BzPzogaWFtLlBvbGljeVN0YXRlbWVudFByb3BzKSB7XG4gICAgc3VwZXIocHJvcHMpO1xuICB9XG59XG4iXX0=