import { AccessLevelList } from '../../shared/access-level';
import { PolicyStatement, Operator } from '../../shared';
import { aws_iam as iam } from "aws-cdk-lib";
/**
 * Statement provider for service [dms](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatabasemigrationservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class Dms extends PolicyStatement {
    servicePrefix: string;
    /**
     * Grants permission to add metadata tags to DMS resources, including replication instances, endpoints, security groups, and migration tasks
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_AddTagsToResource.html
     */
    toAddTagsToResource(): this;
    /**
     * Grants permission to apply a pending maintenance action to a resource (for example, to a replication instance)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ApplyPendingMaintenanceAction.html
     */
    toApplyPendingMaintenanceAction(): this;
    /**
     * Grants permission to associate a extension pack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_StartExtensionPackAssociation.html
     */
    toAssociateExtensionPack(): this;
    /**
     * Grants permission to start the analysis of up to 20 source databases to recommend target engines for each source database
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_BatchStartRecommendations.html
     */
    toBatchStartRecommendations(): this;
    /**
     * Grants permission to cancel a single premigration assessment run
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_CancelReplicationTaskAssessmentRun.html
     */
    toCancelReplicationTaskAssessmentRun(): this;
    /**
     * Grants permission to create a database migration using the provided settings
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toCreateDataMigration(): this;
    /**
     * Grants permission to create an data provider using the provided settings
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateDataProvider.html
     */
    toCreateDataProvider(): this;
    /**
     * Grants permission to create an endpoint using the provided settings
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateEndpoint.html
     */
    toCreateEndpoint(): this;
    /**
     * Grants permission to create an AWS DMS event notification subscription
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateEventSubscription.html
     */
    toCreateEventSubscription(): this;
    /**
     * Grants permission to create a Fleet Advisor collector using the specified parameters
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateFleetAdvisorCollector.html
     */
    toCreateFleetAdvisorCollector(): this;
    /**
     * Grants permission to create an instance profile using the provided settings
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateInstanceProfile.html
     */
    toCreateInstanceProfile(): this;
    /**
     * Grants permission to create an migration project using the provided settings
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateMigrationProject.html
     */
    toCreateMigrationProject(): this;
    /**
     * Grants permission to create a replication config using the provided settings
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toCreateReplicationConfig(): this;
    /**
     * Grants permission to create a replication instance using the specified parameters
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateReplicationInstance.html
     */
    toCreateReplicationInstance(): this;
    /**
     * Grants permission to create a replication subnet group given a list of the subnet IDs in a VPC
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateReplicationSubnetGroup.html
     */
    toCreateReplicationSubnetGroup(): this;
    /**
     * Grants permission to create a replication task using the specified parameters
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateReplicationTask.html
     */
    toCreateReplicationTask(): this;
    /**
     * Grants permission to delete the specified certificate
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteCertificate.html
     */
    toDeleteCertificate(): this;
    /**
     * Grants permission to delete the specified connection between a replication instance and an endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteConnection.html
     */
    toDeleteConnection(): this;
    /**
     * Grants permission to delete the specified database migration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toDeleteDataMigration(): this;
    /**
     * Grants permission to delete the specified data provider
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteDataProvider.html
     */
    toDeleteDataProvider(): this;
    /**
     * Grants permission to delete the specified endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteEndpoint.html
     */
    toDeleteEndpoint(): this;
    /**
     * Grants permission to delete an AWS DMS event subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteEventSubscription.html
     */
    toDeleteEventSubscription(): this;
    /**
     * Grants permission to delete the specified Fleet Advisor collector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteFleetAdvisorCollector.html
     */
    toDeleteFleetAdvisorCollector(): this;
    /**
     * Grants permission to delete the specified Fleet Advisor databases
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteFleetAdvisorDatabases.html
     */
    toDeleteFleetAdvisorDatabases(): this;
    /**
     * Grants permission to delete the specified instance profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteInstanceProfile.html
     */
    toDeleteInstanceProfile(): this;
    /**
     * Grants permission to delete the specified migration project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteMigrationProject.html
     */
    toDeleteMigrationProject(): this;
    /**
     * Grants permission to delete the specified replication config
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toDeleteReplicationConfig(): this;
    /**
     * Grants permission to delete the specified replication instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteReplicationInstance.html
     */
    toDeleteReplicationInstance(): this;
    /**
     * Grants permission to deletes a subnet group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteReplicationSubnetGroup.html
     */
    toDeleteReplicationSubnetGroup(): this;
    /**
     * Grants permission to delete the specified replication task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteReplicationTask.html
     */
    toDeleteReplicationTask(): this;
    /**
     * Grants permission to delete the record of a single premigration assessment run
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteReplicationTaskAssessmentRun.html
     */
    toDeleteReplicationTaskAssessmentRun(): this;
    /**
     * Grants permission to list all of the AWS DMS attributes for a customer account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeAccountAttributes.html
     */
    toDescribeAccountAttributes(): this;
    /**
     * Grants permission to list individual assessments that you can specify for a new premigration assessment run
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeApplicableIndividualAssessments.html
     */
    toDescribeApplicableIndividualAssessments(): this;
    /**
     * Grants permission to provide a description of the certificate
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeCertificates.html
     */
    toDescribeCertificates(): this;
    /**
     * Grants permission to describe the status of the connections that have been made between the replication instance and an endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeConnections.html
     */
    toDescribeConnections(): this;
    /**
     * Grants permission to return information about DMS Schema Conversion project configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeConversionConfiguration.html
     */
    toDescribeConversionConfiguration(): this;
    /**
     * Grants permission to return information about database migrations for your account in the specified region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toDescribeDataMigrations(): this;
    /**
     * Grants permission to return the possible endpoint settings available when you create an endpoint for a specific database engine
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEndpointSettings.html
     */
    toDescribeEndpointSettings(): this;
    /**
     * Grants permission to return information about the type of endpoints available
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEndpointTypes.html
     */
    toDescribeEndpointTypes(): this;
    /**
     * Grants permission to return information about the endpoints for your account in the current region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEndpoints.html
     */
    toDescribeEndpoints(): this;
    /**
     * Grants permission to return information about the available versions for DMS replication instances
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEngineVersions.html
     */
    toDescribeEngineVersions(): this;
    /**
     * Grants permission to list categories for all event source types, or, if specified, for a specified source type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEventCategories.html
     */
    toDescribeEventCategories(): this;
    /**
     * Grants permission to list all the event subscriptions for a customer account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEventSubscriptions.html
     */
    toDescribeEventSubscriptions(): this;
    /**
     * Grants permission to list events for a given source identifier and source type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEvents.html
     */
    toDescribeEvents(): this;
    /**
     * Grants permission to return a paginated list of Fleet Advisor collectors in your account based on filter settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeFleetAdvisorCollectors.html
     */
    toDescribeFleetAdvisorCollectors(): this;
    /**
     * Grants permission to return a paginated list of Fleet Advisor databases in your account based on filter settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeFleetAdvisorDatabases.html
     */
    toDescribeFleetAdvisorDatabases(): this;
    /**
     * Grants permission to return a paginated list of descriptions of large-scale assessment (LSA) analyses produced by your Fleet Advisor collectors
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeFleetAdvisorLsaAnalysis.html
     */
    toDescribeFleetAdvisorLsaAnalysis(): this;
    /**
     * Grants permission to return a paginated list of descriptions of schemas discovered by your Fleet Advisor collectors based on filter settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeFleetAdvisorSchemaObjectSummary.html
     */
    toDescribeFleetAdvisorSchemaObjectSummary(): this;
    /**
     * Grants permission to return a paginated list of schemas discovered by your Fleet Advisor collectors based on filter settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeFleetAdvisorSchemas.html
     */
    toDescribeFleetAdvisorSchemas(): this;
    /**
     * Grants permission to return information about start metadata model import operations for a migration project
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeMetadataModelImports.html
     */
    toDescribeMetadataModelImports(): this;
    /**
     * Grants permission to return information about the replication instance types that can be created in the specified region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeOrderableReplicationInstances.html
     */
    toDescribeOrderableReplicationInstances(): this;
    /**
     * Grants permission to return information about pending maintenance actions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribePendingMaintenanceActions.html
     */
    toDescribePendingMaintenanceActions(): this;
    /**
     * Grants permission to return a paginated list of descriptions of limitations for recommendations of target AWS engines
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeFleetAdvisorLsaAnalysis.html
     */
    toDescribeRecommendationLimitations(): this;
    /**
     * Grants permission to return a paginated list of descriptions of target engine recommendations for your source databases
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeRecommendations.html
     */
    toDescribeRecommendations(): this;
    /**
     * Grants permission to returns the status of the RefreshSchemas operation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeRefreshSchemasStatus.html
     */
    toDescribeRefreshSchemasStatus(): this;
    /**
     * Grants permission to describe replication configs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toDescribeReplicationConfigs(): this;
    /**
     * Grants permission to return information about the task logs for the specified task
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationInstanceTaskLogs.html
     */
    toDescribeReplicationInstanceTaskLogs(): this;
    /**
     * Grants permission to return information about replication instances for your account in the current region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationInstances.html
     */
    toDescribeReplicationInstances(): this;
    /**
     * Grants permission to return information about the replication subnet groups
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationSubnetGroups.html
     */
    toDescribeReplicationSubnetGroups(): this;
    /**
     * Grants permission to describe replication table statistics
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toDescribeReplicationTableStatistics(): this;
    /**
     * Grants permission to return the latest task assessment results from Amazon S3
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationTaskAssessmentResults.html
     */
    toDescribeReplicationTaskAssessmentResults(): this;
    /**
     * Grants permission to return a paginated list of premigration assessment runs based on filter settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationTaskAssessmentRuns.html
     */
    toDescribeReplicationTaskAssessmentRuns(): this;
    /**
     * Grants permission to return a paginated list of individual assessments based on filter settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationTaskIndividualAssessments.html
     */
    toDescribeReplicationTaskIndividualAssessments(): this;
    /**
     * Grants permission to return information about replication tasks for your account in the current region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationTasks.html
     */
    toDescribeReplicationTasks(): this;
    /**
     * Grants permission to describe replications
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toDescribeReplications(): this;
    /**
     * Grants permission to return information about the schema for the specified endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeSchemas.html
     */
    toDescribeSchemas(): this;
    /**
     * Grants permission to return table statistics on the database migration task, including table name, rows inserted, rows updated, and rows deleted
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeTableStatistics.html
     */
    toDescribeTableStatistics(): this;
    /**
     * Grants permission to export the specified metadata model assessment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ExportMetadataModelAssessment.html
     */
    toExportMetadataModelAssessment(): this;
    /**
     * Grants permission to upload the specified certificate
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ImportCertificate.html
     */
    toImportCertificate(): this;
    /**
     * Grants permission to list the AWS DMS attributes for a data providers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeDataProviders.html
     */
    toListDataProviders(): this;
    /**
     * Grants permission to list the AWS DMS attributes for a extension packs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeExtensionPackAssociations.html
     */
    toListExtensionPacks(): this;
    /**
     * Grants permission to list the AWS DMS attributes for a instance profiles
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeInstanceProfiles.html
     */
    toListInstanceProfiles(): this;
    /**
     * Grants permission to list the AWS DMS attributes for a metadata model assessment action items. Note. Despite this action requires StartMetadataModelImport, the latter does not currently authorize the described Schema Conversion operation
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - dms:StartMetadataModelImport
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toListMetadataModelAssessmentActionItems(): this;
    /**
     * Grants permission to list the AWS DMS attributes for a metadata model assessments
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ListMetadataModelExports.html
     */
    toListMetadataModelAssessments(): this;
    /**
     * Grants permission to list the AWS DMS attributes for a metadata model conversions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeMetadataModelConversions.html
     */
    toListMetadataModelConversions(): this;
    /**
     * Grants permission to list the AWS DMS attributes for a metadata model exports
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toListMetadataModelExports(): this;
    /**
     * Grants permission to list the AWS DMS attributes for a migration projects. Note. Despite this action requires DescribeMigrationProjects and DescribeConversionConfiguration, both required actions do not currently authorize the described Schema Conversion operation
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - dms:DescribeConversionConfiguration
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toListMigrationProjects(): this;
    /**
     * Grants permission to list all tags for an AWS DMS resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource(): this;
    /**
     * Grants permission to modify the specified database migration
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toModifyDataMigration(): this;
    /**
     * Grants permission to modify the specified endpoint
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyEndpoint.html
     */
    toModifyEndpoint(): this;
    /**
     * Grants permission to modify an existing AWS DMS event notification subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyEventSubscription.html
     */
    toModifyEventSubscription(): this;
    /**
     * Grants permission to modify the name and description of the specified Fleet Advisor collector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toModifyFleetAdvisorCollector(): this;
    /**
     * Grants permission to modify the status of the specified Fleet Advisor collector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toModifyFleetAdvisorCollectorStatuses(): this;
    /**
     * Grants permission to modify the specified replication config
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toModifyReplicationConfig(): this;
    /**
     * Grants permission to modify the replication instance to apply new settings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyReplicationInstance.html
     */
    toModifyReplicationInstance(): this;
    /**
     * Grants permission to modify the settings for the specified replication subnet group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyReplicationSubnetGroup.html
     */
    toModifyReplicationSubnetGroup(): this;
    /**
     * Grants permission to modify the specified replication task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyReplicationTask.html
     */
    toModifyReplicationTask(): this;
    /**
     * Grants permission to move the specified replication task to a different replication instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_MoveReplicationTask.html
     */
    toMoveReplicationTask(): this;
    /**
     * Grants permission to reboot a replication instance. Rebooting results in a momentary outage, until the replication instance becomes available again
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_RebootReplicationInstance.html
     */
    toRebootReplicationInstance(): this;
    /**
     * Grants permission to populate the schema for the specified endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_RefreshSchemas.html
     */
    toRefreshSchemas(): this;
    /**
     * Grants permission to reload the target database table with the source for a replication
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toReloadReplicationTables(): this;
    /**
     * Grants permission to reload the target database table with the source data
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ReloadTables.html
     */
    toReloadTables(): this;
    /**
     * Grants permission to remove metadata tags from a DMS resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_RemoveTagsFromResource.html
     */
    toRemoveTagsFromResource(): this;
    /**
     * Grants permission to run a large-scale assessment (LSA) analysis on every Fleet Advisor collector in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_RunFleetAdvisorLsaAnalysis.html
     */
    toRunFleetAdvisorLsaAnalysis(): this;
    /**
     * Grants permission to start the database migration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toStartDataMigration(): this;
    /**
     * Grants permission to start a new assessment of metadata model
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_StartMetadataModelAssessment.html
     */
    toStartMetadataModelAssessment(): this;
    /**
     * Grants permission to start a new conversion of metadata model
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_StartMetadataModelConversion.html
     */
    toStartMetadataModelConversion(): this;
    /**
     * Grants permission to start a new export of metadata model as script
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_StartMetadataModelExportAsScript.html
     */
    toStartMetadataModelExportAsScripts(): this;
    /**
     * Grants permission to start a new export of metadata model to target
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_StartMetadataModelExportToTarget.html
     */
    toStartMetadataModelExportToTarget(): this;
    /**
     * Grants permission to start a new import of metadata model
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_StartMetadataModelImport.html
     */
    toStartMetadataModelImport(): this;
    /**
     * Grants permission to start the analysis of your source database to provide recommendations of target engines
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_StartRecommendations.html
     */
    toStartRecommendations(): this;
    /**
     * Grants permission to start a replication
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toStartReplication(): this;
    /**
     * Grants permission to start the replication task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_StartReplicationTask.html
     */
    toStartReplicationTask(): this;
    /**
     * Grants permission to start the replication task assessment for unsupported data types in the source database
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_StartReplicationTaskAssessment.html
     */
    toStartReplicationTaskAssessment(): this;
    /**
     * Grants permission to start a new premigration assessment run for one or more individual assessments of a migration task
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_StartReplicationTaskAssessmentRun.html
     */
    toStartReplicationTaskAssessmentRun(): this;
    /**
     * Grants permission to stop the database migration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toStopDataMigration(): this;
    /**
     * Grants permission to stop a replication
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toStopReplication(): this;
    /**
     * Grants permission to stop the replication task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_StopReplicationTask.html
     */
    toStopReplicationTask(): this;
    /**
     * Grants permission to test the connection between the replication instance and the endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_TestConnection.html
     */
    toTestConnection(): this;
    /**
     * Grants permission to update a conversion configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyConversionConfiguration.html
     */
    toUpdateConversionConfiguration(): this;
    /**
     * Grants permission to update the specified data provider
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyDataProvider.html
     */
    toUpdateDataProvider(): this;
    /**
     * Grants permission to update the specified instance profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyInstanceProfile.html
     */
    toUpdateInstanceProfile(): this;
    /**
     * Grants permission to update the specified migration project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyMigrationProject.html
     */
    toUpdateMigrationProject(): this;
    /**
     * Grants permission to migrate DMS subcriptions to Eventbridge
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_UpdateSubscriptionsToEventBridge.html
     */
    toUpdateSubscriptionsToEventBridge(): this;
    /**
     * Grants permission to upload files to your Amazon S3 bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toUploadFileMetadataList(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type Certificate to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_Certificate.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifCertTag()
     */
    onCertificate(resourceName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type DataProvider to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifDataProviderTag()
     */
    onDataProvider(resourceName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type DataMigration to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifDataMigrationTag()
     */
    onDataMigration(resourceName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type Endpoint to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_Endpoint.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifEndpointTag()
     */
    onEndpoint(resourceName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type EventSubscription to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_EventSubscription.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifEsTag()
     */
    onEventSubscription(resourceName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type InstanceProfile to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceProfileTag()
     */
    onInstanceProfile(resourceName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type MigrationProject to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifMigrationProjectTag()
     */
    onMigrationProject(resourceName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type ReplicationConfig to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifReplicationConfigTag()
     */
    onReplicationConfig(resourceName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type ReplicationInstance to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ReplicationInstance.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifRepTag()
     */
    onReplicationInstance(resourceName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type ReplicationSubnetGroup to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ReplicationSubnetGroup.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifSubgrpTag()
     */
    onReplicationSubnetGroup(resourceName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type ReplicationTask to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ReplicationTask.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifTaskTag()
     */
    onReplicationTask(resourceName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type ReplicationTaskAssessmentRun to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ReplicationTaskAssessmentRun.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAssessmentRunTag()
     */
    onReplicationTaskAssessmentRun(resourceName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type ReplicationTaskIndividualAssessment to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ReplicationTaskIndividualAssessment.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifIndividualAssessmentTag()
     */
    onReplicationTaskIndividualAssessment(resourceName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toAddTagsToResource()
     * - .toCreateDataMigration()
     * - .toCreateDataProvider()
     * - .toCreateEndpoint()
     * - .toCreateEventSubscription()
     * - .toCreateInstanceProfile()
     * - .toCreateMigrationProject()
     * - .toCreateReplicationConfig()
     * - .toCreateReplicationInstance()
     * - .toCreateReplicationSubnetGroup()
     * - .toCreateReplicationTask()
     * - .toImportCertificate()
     * - .toRemoveTagsFromResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the presence of tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toAddTagsToResource()
     * - .toCreateDataMigration()
     * - .toCreateDataProvider()
     * - .toCreateEndpoint()
     * - .toCreateEventSubscription()
     * - .toCreateInstanceProfile()
     * - .toCreateMigrationProject()
     * - .toCreateReplicationConfig()
     * - .toCreateReplicationInstance()
     * - .toCreateReplicationSubnetGroup()
     * - .toCreateReplicationTask()
     * - .toDescribeReplicationInstanceTaskLogs()
     * - .toImportCertificate()
     * - .toRemoveTagsFromResource()
     *
     * Applies to resource types:
     * - Certificate
     * - DataProvider
     * - DataMigration
     * - Endpoint
     * - EventSubscription
     * - InstanceProfile
     * - MigrationProject
     * - ReplicationConfig
     * - ReplicationInstance
     * - ReplicationSubnetGroup
     * - ReplicationTask
     * - ReplicationTaskAssessmentRun
     * - ReplicationTaskIndividualAssessment
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toAddTagsToResource()
     * - .toCreateDataMigration()
     * - .toCreateDataProvider()
     * - .toCreateEndpoint()
     * - .toCreateEventSubscription()
     * - .toCreateInstanceProfile()
     * - .toCreateMigrationProject()
     * - .toCreateReplicationConfig()
     * - .toCreateReplicationInstance()
     * - .toCreateReplicationSubnetGroup()
     * - .toCreateReplicationTask()
     * - .toDescribeReplicationInstanceTaskLogs()
     * - .toImportCertificate()
     * - .toRemoveTagsFromResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the presence of tag key-value pairs in the request for AssessmentRun
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatabasemigrationservice.html#awsdatabasemigrationservice-dms_assessment-run-tag___TagKey_
     *
     * Applies to resource types:
     * - ReplicationTaskAssessmentRun
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAssessmentRunTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the presence of tag key-value pairs in the request for Certificate
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatabasemigrationservice.html#awsdatabasemigrationservice--dms_cert-tag___TagKey_
     *
     * Applies to resource types:
     * - Certificate
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCertTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the presence of tag key-value pairs in the request for DataMigration
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatabasemigrationservice.html#awsdatabasemigrationservice--dms_data-migration-tag___TagKey_
     *
     * Applies to resource types:
     * - DataMigration
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDataMigrationTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the presence of tag key-value pairs in the request for DataProvider
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatabasemigrationservice.html#awsdatabasemigrationservice--dms_dp-tag___TagKey_
     *
     * Applies to resource types:
     * - DataProvider
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDataProviderTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the presence of tag key-value pairs in the request for Endpoint
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatabasemigrationservice.html#awsdatabasemigrationservice-dms_endpoint-tag___TagKey_
     *
     * Applies to resource types:
     * - Endpoint
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEndpointTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the presence of tag key-value pairs in the request for EventSubscription
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatabasemigrationservice.html#awsdatabasemigrationservice-dms_es-tag___TagKey_
     *
     * Applies to resource types:
     * - EventSubscription
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEsTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the presence of tag key-value pairs in the request for IndividualAssessment
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatabasemigrationservice.html#awsdatabasemigrationservice-dms_individual-assessment-tag___TagKey_
     *
     * Applies to resource types:
     * - ReplicationTaskIndividualAssessment
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifIndividualAssessmentTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the presence of tag key-value pairs in the request for InstanceProfile
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatabasemigrationservice.html#awsdatabasemigrationservice--dms_ip-tag___TagKey_
     *
     * Applies to resource types:
     * - InstanceProfile
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInstanceProfileTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the presence of tag key-value pairs in the request for MigrationProject
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatabasemigrationservice.html#awsdatabasemigrationservice--dms_mp-tag___TagKey_
     *
     * Applies to resource types:
     * - MigrationProject
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifMigrationProjectTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the presence of tag key-value pairs in the request for ReplicationInstance
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatabasemigrationservice.html#awsdatabasemigrationservice-dms_rep-tag___TagKey_
     *
     * Applies to resource types:
     * - ReplicationInstance
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRepTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the presence of tag key-value pairs in the request for ReplicationConfig
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatabasemigrationservice.html#awsdatabasemigrationservice-dms_replication-config-tag___TagKey_
     *
     * Applies to resource types:
     * - ReplicationConfig
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReplicationConfigTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the presence of tag key-value pairs in the given request
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatabasemigrationservice.html#awsdatabasemigrationservice-dms_req-tag___TagKey_
     *
     * Applies to actions:
     * - .toAddTagsToResource()
     * - .toCreateDataMigration()
     * - .toCreateDataProvider()
     * - .toCreateEndpoint()
     * - .toCreateEventSubscription()
     * - .toCreateInstanceProfile()
     * - .toCreateMigrationProject()
     * - .toCreateReplicationConfig()
     * - .toCreateReplicationInstance()
     * - .toCreateReplicationSubnetGroup()
     * - .toCreateReplicationTask()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReqTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the presence of tag key-value pairs in the request for ReplicationSubnetGroup
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatabasemigrationservice.html#awsdatabasemigrationservice-dms_subgrp-tag___TagKey_
     *
     * Applies to resource types:
     * - ReplicationSubnetGroup
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSubgrpTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the presence of tag key-value pairs in the request for ReplicationTask
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatabasemigrationservice.html#awsdatabasemigrationservice-dms_task-tag___TagKey_
     *
     * Applies to resource types:
     * - ReplicationTask
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTaskTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Statement provider for service [dms](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatabasemigrationservice.html).
     *
     */
    constructor(props?: iam.PolicyStatementProps);
}
