"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Detective = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [detective](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazondetective.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Detective extends shared_1.PolicyStatement {
    /**
     * Grants permission to accept an invitation to become a member of a behavior graph
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_AcceptInvitation.html
     */
    toAcceptInvitation() {
        return this.to('AcceptInvitation');
    }
    /**
     * Grants permission to retrieve the datasource package history for the specified member accounts in a behavior graph managed by this account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_BatchGetGraphMemberDatasources.html
     */
    toBatchGetGraphMemberDatasources() {
        return this.to('BatchGetGraphMemberDatasources');
    }
    /**
     * Grants permission to retrieve the datasource package history of the caller account for the specified graphs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_BatchGetMembershipDatasources.html
     */
    toBatchGetMembershipDatasources() {
        return this.to('BatchGetMembershipDatasources');
    }
    /**
     * Grants permission to create a behavior graph and begin to aggregate security information
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - detective:TagResource
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_CreateGraph.html
     */
    toCreateGraph() {
        return this.to('CreateGraph');
    }
    /**
     * Grants permission to request the membership of one or more accounts in a behavior graph managed by this account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_CreateMembers.html
     */
    toCreateMembers() {
        return this.to('CreateMembers');
    }
    /**
     * Grants permission to delete a behavior graph and stop aggregating security information
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_DeleteGraph.html
     */
    toDeleteGraph() {
        return this.to('DeleteGraph');
    }
    /**
     * Grants permission to remove member accounts from a behavior graph managed by this account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_DeleteMembers.html
     */
    toDeleteMembers() {
        return this.to('DeleteMembers');
    }
    /**
     * Grants permission to view the current configuration related to the Amazon Detective integration with AWS Organizations
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - organizations:DescribeOrganization
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_DescribeOrganizationConfiguration.html
     */
    toDescribeOrganizationConfiguration() {
        return this.to('DescribeOrganizationConfiguration');
    }
    /**
     * Grants permission to remove the Amazon Detective delegated administrator account for an organization
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - organizations:DescribeOrganization
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_DisableOrganizationAdminAccount.html
     */
    toDisableOrganizationAdminAccount() {
        return this.to('DisableOrganizationAdminAccount');
    }
    /**
     * Grants permission to remove the association of this account with a behavior graph
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_DisassociateMembership.html
     */
    toDisassociateMembership() {
        return this.to('DisassociateMembership');
    }
    /**
     * Grants permission to designate the Amazon Detective delegated administrator account for an organization
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - organizations:DescribeOrganization
     * - organizations:EnableAWSServiceAccess
     * - organizations:RegisterDelegatedAdministrator
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_EnableOrganizationAdminAccount.html
     */
    toEnableOrganizationAdminAccount() {
        return this.to('EnableOrganizationAdminAccount');
    }
    /**
     * Grants permission to retrieve a behavior graph's eligibility for a free trial period
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/detective/latest/adminguide/free-trial-overview.html
     */
    toGetFreeTrialEligibility() {
        return this.to('GetFreeTrialEligibility');
    }
    /**
     * Grants permission to retrieve the data ingestion state of a behavior graph
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/detective/latest/adminguide/detective-source-data-about.html
     */
    toGetGraphIngestState() {
        return this.to('GetGraphIngestState');
    }
    /**
     * Grants permission to get an investigation's status and metadata
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_GetInvestigation.html
     */
    toGetInvestigation() {
        return this.to('GetInvestigation');
    }
    /**
     * Grants permission to retrieve details on specified members of a behavior graph
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_GetMembers.html
     */
    toGetMembers() {
        return this.to('GetMembers');
    }
    /**
     * Grants permission to retrieve information about Amazon Detective's pricing
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/detective/latest/adminguide/usage-projected-cost-calculation.html
     */
    toGetPricingInformation() {
        return this.to('GetPricingInformation');
    }
    /**
     * Grants permission to list usage information of a behavior graph
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/detective/latest/adminguide/tracking-usage-logging.html
     */
    toGetUsageInformation() {
        return this.to('GetUsageInformation');
    }
    /**
     * Grants permission to invoke Detective's Assistant
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/detective/latest/userguide/finding-groups-summary.html
     */
    toInvokeAssistant() {
        return this.to('InvokeAssistant');
    }
    /**
     * Grants permission to list a graph's datasource package ingest states and timestamps for the most recent state changes in a behavior graph managed by this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_ListDatasourcePackages.html
     */
    toListDatasourcePackages() {
        return this.to('ListDatasourcePackages');
    }
    /**
     * Grants permission to list behavior graphs managed by this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_ListGraphs.html
     */
    toListGraphs() {
        return this.to('ListGraphs');
    }
    /**
     * Grants permission to retrieve high volume entities whose relationships cannot be stored by Detective
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/detective/latest/userguide/high-volume-entities.html
     */
    toListHighDegreeEntities() {
        return this.to('ListHighDegreeEntities');
    }
    /**
     * Grants permission to list the indicators of an investigation
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_ListIndicators.html
     */
    toListIndicators() {
        return this.to('ListIndicators');
    }
    /**
     * Grants permission to list the investigations of a behavior graph
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_ListInvestigations.html
     */
    toListInvestigations() {
        return this.to('ListInvestigations');
    }
    /**
     * Grants permission to retrieve details on the behavior graphs to which this account has been invited to join
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_ListInvitations.html
     */
    toListInvitations() {
        return this.to('ListInvitations');
    }
    /**
     * Grants permission to retrieve details on all members of a behavior graph
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_ListMembers.html
     */
    toListMembers() {
        return this.to('ListMembers');
    }
    /**
     * Grants permission to view the current Amazon Detective delegated administrator account for an organization
     *
     * Access Level: List
     *
     * Dependent actions:
     * - organizations:DescribeOrganization
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_ListOrganizationAdminAccounts.html
     */
    toListOrganizationAdminAccount() {
        return this.to('ListOrganizationAdminAccount');
    }
    /**
     * Grants permission to list the tag values that are assigned to a behavior graph
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to reject an invitation to become a member of a behavior graph
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_RejectInvitation.html
     */
    toRejectInvitation() {
        return this.to('RejectInvitation');
    }
    /**
     * Grants permission to search the data stored in a behavior graph
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/detective/latest/userguide/detective-search.html
     */
    toSearchGraph() {
        return this.to('SearchGraph');
    }
    /**
     * Grants permission to start investigations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_StartInvestigation.html
     */
    toStartInvestigation() {
        return this.to('StartInvestigation');
    }
    /**
     * Grants permission to start data ingest for a member account that has a status of ACCEPTED_BUT_DISABLED
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_StartMonitoringMember.html
     */
    toStartMonitoringMember() {
        return this.to('StartMonitoringMember');
    }
    /**
     * Grants permission to assign tag values to a behavior graph
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tag values from a behavior graph
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to enable or disable datasource package(s) in a behavior graph managed by this account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_UpdateDatasourcePackages.html
     */
    toUpdateDatasourcePackages() {
        return this.to('UpdateDatasourcePackages');
    }
    /**
     * Grants permission to update an investigation's state and metadata
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_UpdateInvestigationState.html
     */
    toUpdateInvestigationState() {
        return this.to('UpdateInvestigationState');
    }
    /**
     * Grants permission to update the current configuration related to the Amazon Detective integration with AWS Organizations
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - organizations:DescribeOrganization
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_UpdateOrganizationConfiguration.html
     */
    toUpdateOrganizationConfiguration() {
        return this.to('UpdateOrganizationConfiguration');
    }
    /**
     * Adds a resource of type Graph to the statement
     *
     * https://docs.aws.amazon.com/detective/latest/adminguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-resources
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGraph(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:detective:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:graph:${resourceId}`);
    }
    /**
     * Filters access by specifying the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateGraph()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by specifying the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toCreateGraph()
     * - .toListTagsForResource()
     * - .toTagResource()
     *
     * Applies to resource types:
     * - Graph
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by specifying the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateGraph()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [detective](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazondetective.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'detective';
        this.accessLevelList = {
            Write: [
                'AcceptInvitation',
                'CreateGraph',
                'CreateMembers',
                'DeleteGraph',
                'DeleteMembers',
                'DisableOrganizationAdminAccount',
                'DisassociateMembership',
                'EnableOrganizationAdminAccount',
                'RejectInvitation',
                'StartInvestigation',
                'StartMonitoringMember',
                'UpdateDatasourcePackages',
                'UpdateInvestigationState',
                'UpdateOrganizationConfiguration'
            ],
            Read: [
                'BatchGetGraphMemberDatasources',
                'BatchGetMembershipDatasources',
                'DescribeOrganizationConfiguration',
                'GetFreeTrialEligibility',
                'GetGraphIngestState',
                'GetInvestigation',
                'GetMembers',
                'GetPricingInformation',
                'GetUsageInformation',
                'InvokeAssistant',
                'SearchGraph'
            ],
            List: [
                'ListDatasourcePackages',
                'ListGraphs',
                'ListHighDegreeEntities',
                'ListIndicators',
                'ListInvestigations',
                'ListInvitations',
                'ListMembers',
                'ListOrganizationAdminAccount',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Detective = Detective;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZGV0ZWN0aXZlLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiZGV0ZWN0aXZlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUNBLHlDQUF5RDtBQUd6RDs7OztHQUlHO0FBQ0gsTUFBYSxTQUFVLFNBQVEsd0JBQWU7SUFHNUM7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQ0FBZ0M7UUFDckMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdDQUFnQyxDQUFDLENBQUM7SUFDbkQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLCtCQUErQjtRQUNwQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsK0JBQStCLENBQUMsQ0FBQztJQUNsRCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7O09BY0c7SUFDSSxhQUFhO1FBQ2xCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUNoQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ2hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxlQUFlO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksbUNBQW1DO1FBQ3hDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQ0FBbUMsQ0FBQyxDQUFDO0lBQ3RELENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxpQ0FBaUM7UUFDdEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGlDQUFpQyxDQUFDLENBQUM7SUFDcEQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdCQUF3QjtRQUM3QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksZ0NBQWdDO1FBQ3JDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO0lBQ25ELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx5QkFBeUI7UUFDOUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7SUFDNUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxZQUFZO1FBQ2pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxZQUFZLENBQUMsQ0FBQztJQUMvQixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksdUJBQXVCO1FBQzVCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO0lBQzFDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUJBQWlCLENBQUMsQ0FBQztJQUNwQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksd0JBQXdCO1FBQzdCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO0lBQzNDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxZQUFZO1FBQ2pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxZQUFZLENBQUMsQ0FBQztJQUMvQixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksd0JBQXdCO1FBQzdCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO0lBQzNDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUJBQWlCO1FBQ3RCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO0lBQ3BDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxhQUFhO1FBQ2xCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUNoQyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksOEJBQThCO1FBQ25DLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO0lBQ2pELENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0JBQWtCLENBQUMsQ0FBQztJQUNyQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsYUFBYSxDQUFDLENBQUM7SUFDaEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksdUJBQXVCO1FBQzVCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO0lBQzFDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLGFBQWE7UUFDbEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ2hDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxlQUFlO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMEJBQTBCO1FBQy9CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO0lBQzdDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwwQkFBMEI7UUFDL0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLGlDQUFpQztRQUN0QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUNBQWlDLENBQUMsQ0FBQztJQUNwRCxDQUFDO0lBaUREOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLE9BQU8sQ0FBQyxVQUFrQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3RGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGNBQWUsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLFVBQVcsVUFBVyxFQUFFLENBQUMsQ0FBQztJQUN0SyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksZUFBZSxDQUFDLE1BQWMsRUFBRSxLQUF3QixFQUFFLFFBQTRCO1FBQzNGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBbUIsTUFBTyxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNoRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7T0FnQkc7SUFDSSxnQkFBZ0IsQ0FBQyxNQUFjLEVBQUUsS0FBd0IsRUFBRSxRQUE0QjtRQUM1RixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW9CLE1BQU8sRUFBRSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDakYsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLFlBQVksQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ3hFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNqRSxDQUFDO0lBRUQ7OztPQUdHO0lBQ0gsWUFBWSxLQUFnQztRQUMxQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUM7UUEvaUJSLGtCQUFhLEdBQUcsV0FBVyxDQUFDO1FBbWJ6QixvQkFBZSxHQUFvQjtZQUMzQyxLQUFLLEVBQUU7Z0JBQ0wsa0JBQWtCO2dCQUNsQixhQUFhO2dCQUNiLGVBQWU7Z0JBQ2YsYUFBYTtnQkFDYixlQUFlO2dCQUNmLGlDQUFpQztnQkFDakMsd0JBQXdCO2dCQUN4QixnQ0FBZ0M7Z0JBQ2hDLGtCQUFrQjtnQkFDbEIsb0JBQW9CO2dCQUNwQix1QkFBdUI7Z0JBQ3ZCLDBCQUEwQjtnQkFDMUIsMEJBQTBCO2dCQUMxQixpQ0FBaUM7YUFDbEM7WUFDRCxJQUFJLEVBQUU7Z0JBQ0osZ0NBQWdDO2dCQUNoQywrQkFBK0I7Z0JBQy9CLG1DQUFtQztnQkFDbkMseUJBQXlCO2dCQUN6QixxQkFBcUI7Z0JBQ3JCLGtCQUFrQjtnQkFDbEIsWUFBWTtnQkFDWix1QkFBdUI7Z0JBQ3ZCLHFCQUFxQjtnQkFDckIsaUJBQWlCO2dCQUNqQixhQUFhO2FBQ2Q7WUFDRCxJQUFJLEVBQUU7Z0JBQ0osd0JBQXdCO2dCQUN4QixZQUFZO2dCQUNaLHdCQUF3QjtnQkFDeEIsZ0JBQWdCO2dCQUNoQixvQkFBb0I7Z0JBQ3BCLGlCQUFpQjtnQkFDakIsYUFBYTtnQkFDYiw4QkFBOEI7Z0JBQzlCLHFCQUFxQjthQUN0QjtZQUNELE9BQU8sRUFBRTtnQkFDUCxhQUFhO2dCQUNiLGVBQWU7YUFDaEI7U0FDRixDQUFDO0lBZ0ZGLENBQUM7Q0FDRjtBQWxqQkQsOEJBa2pCQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEFjY2Vzc0xldmVsTGlzdCB9IGZyb20gJy4uLy4uL3NoYXJlZC9hY2Nlc3MtbGV2ZWwnO1xuaW1wb3J0IHsgUG9saWN5U3RhdGVtZW50LCBPcGVyYXRvciB9IGZyb20gJy4uLy4uL3NoYXJlZCc7XG5pbXBvcnQgeyBhd3NfaWFtIGFzIGlhbSB9IGZyb20gXCJhd3MtY2RrLWxpYlwiO1xuXG4vKipcbiAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbZGV0ZWN0aXZlXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9hbWF6b25kZXRlY3RpdmUuaHRtbCkuXG4gKlxuICogQHBhcmFtIHNpZCBbU0lEXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX3NpZC5odG1sKSBvZiB0aGUgc3RhdGVtZW50XG4gKi9cbmV4cG9ydCBjbGFzcyBEZXRlY3RpdmUgZXh0ZW5kcyBQb2xpY3lTdGF0ZW1lbnQge1xuICBwdWJsaWMgc2VydmljZVByZWZpeCA9ICdkZXRlY3RpdmUnO1xuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBhY2NlcHQgYW4gaW52aXRhdGlvbiB0byBiZWNvbWUgYSBtZW1iZXIgb2YgYSBiZWhhdmlvciBncmFwaFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9kZXRlY3RpdmUvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQWNjZXB0SW52aXRhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9BY2NlcHRJbnZpdGF0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdBY2NlcHRJbnZpdGF0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgdGhlIGRhdGFzb3VyY2UgcGFja2FnZSBoaXN0b3J5IGZvciB0aGUgc3BlY2lmaWVkIG1lbWJlciBhY2NvdW50cyBpbiBhIGJlaGF2aW9yIGdyYXBoIG1hbmFnZWQgYnkgdGhpcyBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZGV0ZWN0aXZlL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0JhdGNoR2V0R3JhcGhNZW1iZXJEYXRhc291cmNlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9CYXRjaEdldEdyYXBoTWVtYmVyRGF0YXNvdXJjZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0JhdGNoR2V0R3JhcGhNZW1iZXJEYXRhc291cmNlcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIHRoZSBkYXRhc291cmNlIHBhY2thZ2UgaGlzdG9yeSBvZiB0aGUgY2FsbGVyIGFjY291bnQgZm9yIHRoZSBzcGVjaWZpZWQgZ3JhcGhzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZGV0ZWN0aXZlL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0JhdGNoR2V0TWVtYmVyc2hpcERhdGFzb3VyY2VzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0JhdGNoR2V0TWVtYmVyc2hpcERhdGFzb3VyY2VzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdCYXRjaEdldE1lbWJlcnNoaXBEYXRhc291cmNlcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIGJlaGF2aW9yIGdyYXBoIGFuZCBiZWdpbiB0byBhZ2dyZWdhdGUgc2VjdXJpdHkgaW5mb3JtYXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gZGV0ZWN0aXZlOlRhZ1Jlc291cmNlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9kZXRlY3RpdmUvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlR3JhcGguaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlR3JhcGgoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZUdyYXBoJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmVxdWVzdCB0aGUgbWVtYmVyc2hpcCBvZiBvbmUgb3IgbW9yZSBhY2NvdW50cyBpbiBhIGJlaGF2aW9yIGdyYXBoIG1hbmFnZWQgYnkgdGhpcyBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2RldGVjdGl2ZS9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVNZW1iZXJzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZU1lbWJlcnMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZU1lbWJlcnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBiZWhhdmlvciBncmFwaCBhbmQgc3RvcCBhZ2dyZWdhdGluZyBzZWN1cml0eSBpbmZvcm1hdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9kZXRlY3RpdmUvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlR3JhcGguaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlR3JhcGgoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZUdyYXBoJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmVtb3ZlIG1lbWJlciBhY2NvdW50cyBmcm9tIGEgYmVoYXZpb3IgZ3JhcGggbWFuYWdlZCBieSB0aGlzIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZGV0ZWN0aXZlL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZU1lbWJlcnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlTWVtYmVycygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlTWVtYmVycycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHZpZXcgdGhlIGN1cnJlbnQgY29uZmlndXJhdGlvbiByZWxhdGVkIHRvIHRoZSBBbWF6b24gRGV0ZWN0aXZlIGludGVncmF0aW9uIHdpdGggQVdTIE9yZ2FuaXphdGlvbnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIG9yZ2FuaXphdGlvbnM6RGVzY3JpYmVPcmdhbml6YXRpb25cbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2RldGVjdGl2ZS9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZU9yZ2FuaXphdGlvbkNvbmZpZ3VyYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVPcmdhbml6YXRpb25Db25maWd1cmF0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZU9yZ2FuaXphdGlvbkNvbmZpZ3VyYXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZW1vdmUgdGhlIEFtYXpvbiBEZXRlY3RpdmUgZGVsZWdhdGVkIGFkbWluaXN0cmF0b3IgYWNjb3VudCBmb3IgYW4gb3JnYW5pemF0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gb3JnYW5pemF0aW9uczpEZXNjcmliZU9yZ2FuaXphdGlvblxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZGV0ZWN0aXZlL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rpc2FibGVPcmdhbml6YXRpb25BZG1pbkFjY291bnQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGlzYWJsZU9yZ2FuaXphdGlvbkFkbWluQWNjb3VudCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGlzYWJsZU9yZ2FuaXphdGlvbkFkbWluQWNjb3VudCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJlbW92ZSB0aGUgYXNzb2NpYXRpb24gb2YgdGhpcyBhY2NvdW50IHdpdGggYSBiZWhhdmlvciBncmFwaFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9kZXRlY3RpdmUvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGlzYXNzb2NpYXRlTWVtYmVyc2hpcC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EaXNhc3NvY2lhdGVNZW1iZXJzaGlwKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEaXNhc3NvY2lhdGVNZW1iZXJzaGlwJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVzaWduYXRlIHRoZSBBbWF6b24gRGV0ZWN0aXZlIGRlbGVnYXRlZCBhZG1pbmlzdHJhdG9yIGFjY291bnQgZm9yIGFuIG9yZ2FuaXphdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIGlhbTpDcmVhdGVTZXJ2aWNlTGlua2VkUm9sZVxuICAgKiAtIG9yZ2FuaXphdGlvbnM6RGVzY3JpYmVPcmdhbml6YXRpb25cbiAgICogLSBvcmdhbml6YXRpb25zOkVuYWJsZUFXU1NlcnZpY2VBY2Nlc3NcbiAgICogLSBvcmdhbml6YXRpb25zOlJlZ2lzdGVyRGVsZWdhdGVkQWRtaW5pc3RyYXRvclxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZGV0ZWN0aXZlL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0VuYWJsZU9yZ2FuaXphdGlvbkFkbWluQWNjb3VudC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9FbmFibGVPcmdhbml6YXRpb25BZG1pbkFjY291bnQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0VuYWJsZU9yZ2FuaXphdGlvbkFkbWluQWNjb3VudCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGEgYmVoYXZpb3IgZ3JhcGgncyBlbGlnaWJpbGl0eSBmb3IgYSBmcmVlIHRyaWFsIHBlcmlvZFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2RldGVjdGl2ZS9sYXRlc3QvYWRtaW5ndWlkZS9mcmVlLXRyaWFsLW92ZXJ2aWV3Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEZyZWVUcmlhbEVsaWdpYmlsaXR5KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRGcmVlVHJpYWxFbGlnaWJpbGl0eScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIHRoZSBkYXRhIGluZ2VzdGlvbiBzdGF0ZSBvZiBhIGJlaGF2aW9yIGdyYXBoXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZGV0ZWN0aXZlL2xhdGVzdC9hZG1pbmd1aWRlL2RldGVjdGl2ZS1zb3VyY2UtZGF0YS1hYm91dC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRHcmFwaEluZ2VzdFN0YXRlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRHcmFwaEluZ2VzdFN0YXRlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IGFuIGludmVzdGlnYXRpb24ncyBzdGF0dXMgYW5kIG1ldGFkYXRhXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZGV0ZWN0aXZlL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldEludmVzdGlnYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0SW52ZXN0aWdhdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0SW52ZXN0aWdhdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGRldGFpbHMgb24gc3BlY2lmaWVkIG1lbWJlcnMgb2YgYSBiZWhhdmlvciBncmFwaFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2RldGVjdGl2ZS9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9HZXRNZW1iZXJzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldE1lbWJlcnMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldE1lbWJlcnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBpbmZvcm1hdGlvbiBhYm91dCBBbWF6b24gRGV0ZWN0aXZlJ3MgcHJpY2luZ1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2RldGVjdGl2ZS9sYXRlc3QvYWRtaW5ndWlkZS91c2FnZS1wcm9qZWN0ZWQtY29zdC1jYWxjdWxhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRQcmljaW5nSW5mb3JtYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldFByaWNpbmdJbmZvcm1hdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgdXNhZ2UgaW5mb3JtYXRpb24gb2YgYSBiZWhhdmlvciBncmFwaFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2RldGVjdGl2ZS9sYXRlc3QvYWRtaW5ndWlkZS90cmFja2luZy11c2FnZS1sb2dnaW5nLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFVzYWdlSW5mb3JtYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldFVzYWdlSW5mb3JtYXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBpbnZva2UgRGV0ZWN0aXZlJ3MgQXNzaXN0YW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZGV0ZWN0aXZlL2xhdGVzdC91c2VyZ3VpZGUvZmluZGluZy1ncm91cHMtc3VtbWFyeS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9JbnZva2VBc3Npc3RhbnQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0ludm9rZUFzc2lzdGFudCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgYSBncmFwaCdzIGRhdGFzb3VyY2UgcGFja2FnZSBpbmdlc3Qgc3RhdGVzIGFuZCB0aW1lc3RhbXBzIGZvciB0aGUgbW9zdCByZWNlbnQgc3RhdGUgY2hhbmdlcyBpbiBhIGJlaGF2aW9yIGdyYXBoIG1hbmFnZWQgYnkgdGhpcyBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZGV0ZWN0aXZlL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3REYXRhc291cmNlUGFja2FnZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdERhdGFzb3VyY2VQYWNrYWdlcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdERhdGFzb3VyY2VQYWNrYWdlcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgYmVoYXZpb3IgZ3JhcGhzIG1hbmFnZWQgYnkgdGhpcyBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZGV0ZWN0aXZlL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RHcmFwaHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdEdyYXBocygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdEdyYXBocycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGhpZ2ggdm9sdW1lIGVudGl0aWVzIHdob3NlIHJlbGF0aW9uc2hpcHMgY2Fubm90IGJlIHN0b3JlZCBieSBEZXRlY3RpdmVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9kZXRlY3RpdmUvbGF0ZXN0L3VzZXJndWlkZS9oaWdoLXZvbHVtZS1lbnRpdGllcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0SGlnaERlZ3JlZUVudGl0aWVzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0SGlnaERlZ3JlZUVudGl0aWVzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCB0aGUgaW5kaWNhdG9ycyBvZiBhbiBpbnZlc3RpZ2F0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZGV0ZWN0aXZlL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RJbmRpY2F0b3JzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RJbmRpY2F0b3JzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0SW5kaWNhdG9ycycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgdGhlIGludmVzdGlnYXRpb25zIG9mIGEgYmVoYXZpb3IgZ3JhcGhcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9kZXRlY3RpdmUvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdEludmVzdGlnYXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RJbnZlc3RpZ2F0aW9ucygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdEludmVzdGlnYXRpb25zJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgZGV0YWlscyBvbiB0aGUgYmVoYXZpb3IgZ3JhcGhzIHRvIHdoaWNoIHRoaXMgYWNjb3VudCBoYXMgYmVlbiBpbnZpdGVkIHRvIGpvaW5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9kZXRlY3RpdmUvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdEludml0YXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RJbnZpdGF0aW9ucygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdEludml0YXRpb25zJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgZGV0YWlscyBvbiBhbGwgbWVtYmVycyBvZiBhIGJlaGF2aW9yIGdyYXBoXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZGV0ZWN0aXZlL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RNZW1iZXJzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RNZW1iZXJzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0TWVtYmVycycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHZpZXcgdGhlIGN1cnJlbnQgQW1hem9uIERldGVjdGl2ZSBkZWxlZ2F0ZWQgYWRtaW5pc3RyYXRvciBhY2NvdW50IGZvciBhbiBvcmdhbml6YXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIG9yZ2FuaXphdGlvbnM6RGVzY3JpYmVPcmdhbml6YXRpb25cbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2RldGVjdGl2ZS9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0T3JnYW5pemF0aW9uQWRtaW5BY2NvdW50cy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0T3JnYW5pemF0aW9uQWRtaW5BY2NvdW50KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0T3JnYW5pemF0aW9uQWRtaW5BY2NvdW50Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCB0aGUgdGFnIHZhbHVlcyB0aGF0IGFyZSBhc3NpZ25lZCB0byBhIGJlaGF2aW9yIGdyYXBoXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2RldGVjdGl2ZS9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0VGFnc0ZvclJlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RUYWdzRm9yUmVzb3VyY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RUYWdzRm9yUmVzb3VyY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZWplY3QgYW4gaW52aXRhdGlvbiB0byBiZWNvbWUgYSBtZW1iZXIgb2YgYSBiZWhhdmlvciBncmFwaFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9kZXRlY3RpdmUvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUmVqZWN0SW52aXRhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9SZWplY3RJbnZpdGF0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdSZWplY3RJbnZpdGF0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc2VhcmNoIHRoZSBkYXRhIHN0b3JlZCBpbiBhIGJlaGF2aW9yIGdyYXBoXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZGV0ZWN0aXZlL2xhdGVzdC91c2VyZ3VpZGUvZGV0ZWN0aXZlLXNlYXJjaC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TZWFyY2hHcmFwaCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnU2VhcmNoR3JhcGgnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBzdGFydCBpbnZlc3RpZ2F0aW9uc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9kZXRlY3RpdmUvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU3RhcnRJbnZlc3RpZ2F0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N0YXJ0SW52ZXN0aWdhdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnU3RhcnRJbnZlc3RpZ2F0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc3RhcnQgZGF0YSBpbmdlc3QgZm9yIGEgbWVtYmVyIGFjY291bnQgdGhhdCBoYXMgYSBzdGF0dXMgb2YgQUNDRVBURURfQlVUX0RJU0FCTEVEXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2RldGVjdGl2ZS9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TdGFydE1vbml0b3JpbmdNZW1iZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU3RhcnRNb25pdG9yaW5nTWVtYmVyKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTdGFydE1vbml0b3JpbmdNZW1iZXInKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBhc3NpZ24gdGFnIHZhbHVlcyB0byBhIGJlaGF2aW9yIGdyYXBoXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogVGFnZ2luZ1xuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2RldGVjdGl2ZS9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9UYWdSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9UYWdSZXNvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVGFnUmVzb3VyY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZW1vdmUgdGFnIHZhbHVlcyBmcm9tIGEgYmVoYXZpb3IgZ3JhcGhcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9kZXRlY3RpdmUvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVW50YWdSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VbnRhZ1Jlc291cmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVbnRhZ1Jlc291cmNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZW5hYmxlIG9yIGRpc2FibGUgZGF0YXNvdXJjZSBwYWNrYWdlKHMpIGluIGEgYmVoYXZpb3IgZ3JhcGggbWFuYWdlZCBieSB0aGlzIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZGV0ZWN0aXZlL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZURhdGFzb3VyY2VQYWNrYWdlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVEYXRhc291cmNlUGFja2FnZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZURhdGFzb3VyY2VQYWNrYWdlcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhbiBpbnZlc3RpZ2F0aW9uJ3Mgc3RhdGUgYW5kIG1ldGFkYXRhXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2RldGVjdGl2ZS9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVJbnZlc3RpZ2F0aW9uU3RhdGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlSW52ZXN0aWdhdGlvblN0YXRlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVJbnZlc3RpZ2F0aW9uU3RhdGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgdGhlIGN1cnJlbnQgY29uZmlndXJhdGlvbiByZWxhdGVkIHRvIHRoZSBBbWF6b24gRGV0ZWN0aXZlIGludGVncmF0aW9uIHdpdGggQVdTIE9yZ2FuaXphdGlvbnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBvcmdhbml6YXRpb25zOkRlc2NyaWJlT3JnYW5pemF0aW9uXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9kZXRlY3RpdmUvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlT3JnYW5pemF0aW9uQ29uZmlndXJhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVPcmdhbml6YXRpb25Db25maWd1cmF0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVPcmdhbml6YXRpb25Db25maWd1cmF0aW9uJyk7XG4gIH1cblxuICBwcm90ZWN0ZWQgYWNjZXNzTGV2ZWxMaXN0OiBBY2Nlc3NMZXZlbExpc3QgPSB7XG4gICAgV3JpdGU6IFtcbiAgICAgICdBY2NlcHRJbnZpdGF0aW9uJyxcbiAgICAgICdDcmVhdGVHcmFwaCcsXG4gICAgICAnQ3JlYXRlTWVtYmVycycsXG4gICAgICAnRGVsZXRlR3JhcGgnLFxuICAgICAgJ0RlbGV0ZU1lbWJlcnMnLFxuICAgICAgJ0Rpc2FibGVPcmdhbml6YXRpb25BZG1pbkFjY291bnQnLFxuICAgICAgJ0Rpc2Fzc29jaWF0ZU1lbWJlcnNoaXAnLFxuICAgICAgJ0VuYWJsZU9yZ2FuaXphdGlvbkFkbWluQWNjb3VudCcsXG4gICAgICAnUmVqZWN0SW52aXRhdGlvbicsXG4gICAgICAnU3RhcnRJbnZlc3RpZ2F0aW9uJyxcbiAgICAgICdTdGFydE1vbml0b3JpbmdNZW1iZXInLFxuICAgICAgJ1VwZGF0ZURhdGFzb3VyY2VQYWNrYWdlcycsXG4gICAgICAnVXBkYXRlSW52ZXN0aWdhdGlvblN0YXRlJyxcbiAgICAgICdVcGRhdGVPcmdhbml6YXRpb25Db25maWd1cmF0aW9uJ1xuICAgIF0sXG4gICAgUmVhZDogW1xuICAgICAgJ0JhdGNoR2V0R3JhcGhNZW1iZXJEYXRhc291cmNlcycsXG4gICAgICAnQmF0Y2hHZXRNZW1iZXJzaGlwRGF0YXNvdXJjZXMnLFxuICAgICAgJ0Rlc2NyaWJlT3JnYW5pemF0aW9uQ29uZmlndXJhdGlvbicsXG4gICAgICAnR2V0RnJlZVRyaWFsRWxpZ2liaWxpdHknLFxuICAgICAgJ0dldEdyYXBoSW5nZXN0U3RhdGUnLFxuICAgICAgJ0dldEludmVzdGlnYXRpb24nLFxuICAgICAgJ0dldE1lbWJlcnMnLFxuICAgICAgJ0dldFByaWNpbmdJbmZvcm1hdGlvbicsXG4gICAgICAnR2V0VXNhZ2VJbmZvcm1hdGlvbicsXG4gICAgICAnSW52b2tlQXNzaXN0YW50JyxcbiAgICAgICdTZWFyY2hHcmFwaCdcbiAgICBdLFxuICAgIExpc3Q6IFtcbiAgICAgICdMaXN0RGF0YXNvdXJjZVBhY2thZ2VzJyxcbiAgICAgICdMaXN0R3JhcGhzJyxcbiAgICAgICdMaXN0SGlnaERlZ3JlZUVudGl0aWVzJyxcbiAgICAgICdMaXN0SW5kaWNhdG9ycycsXG4gICAgICAnTGlzdEludmVzdGlnYXRpb25zJyxcbiAgICAgICdMaXN0SW52aXRhdGlvbnMnLFxuICAgICAgJ0xpc3RNZW1iZXJzJyxcbiAgICAgICdMaXN0T3JnYW5pemF0aW9uQWRtaW5BY2NvdW50JyxcbiAgICAgICdMaXN0VGFnc0ZvclJlc291cmNlJ1xuICAgIF0sXG4gICAgVGFnZ2luZzogW1xuICAgICAgJ1RhZ1Jlc291cmNlJyxcbiAgICAgICdVbnRhZ1Jlc291cmNlJ1xuICAgIF1cbiAgfTtcblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgR3JhcGggdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZGV0ZWN0aXZlL2xhdGVzdC9hZG1pbmd1aWRlL3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLmh0bWwjc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0taWQtYmFzZWQtcG9saWNpZXMtcmVzb3VyY2VzXG4gICAqXG4gICAqIEBwYXJhbSByZXNvdXJjZUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc291cmNlSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkdyYXBoKHJlc291cmNlSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTpkZXRlY3RpdmU6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OmdyYXBoOiR7IHJlc291cmNlSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHNwZWNpZnlpbmcgdGhlIHRhZ3MgdGhhdCBhcmUgcGFzc2VkIGluIHRoZSByZXF1ZXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy1yZXF1ZXN0dGFnXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9DcmVhdGVHcmFwaCgpXG4gICAqIC0gLnRvVGFnUmVzb3VyY2UoKVxuICAgKlxuICAgKiBAcGFyYW0gdGFnS2V5IFRoZSB0YWcga2V5IHRvIGNoZWNrXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzUmVxdWVzdFRhZyh0YWdLZXk6IHN0cmluZywgdmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpSZXF1ZXN0VGFnLyR7IHRhZ0tleSB9YCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgc3BlY2lmeWluZyB0aGUgdGFncyBhc3NvY2lhdGVkIHdpdGggdGhlIHJlc291cmNlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy1yZXNvdXJjZXRhZ1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQ3JlYXRlR3JhcGgoKVxuICAgKiAtIC50b0xpc3RUYWdzRm9yUmVzb3VyY2UoKVxuICAgKiAtIC50b1RhZ1Jlc291cmNlKClcbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBHcmFwaFxuICAgKlxuICAgKiBAcGFyYW0gdGFnS2V5IFRoZSB0YWcga2V5IHRvIGNoZWNrXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzUmVzb3VyY2VUYWcodGFnS2V5OiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6UmVzb3VyY2VUYWcvJHsgdGFnS2V5IH1gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSBzcGVjaWZ5aW5nIHRoZSB0YWcga2V5cyB0aGF0IGFyZSBwYXNzZWQgaW4gdGhlIHJlcXVlc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXRhZ2tleXNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0NyZWF0ZUdyYXBoKClcbiAgICogLSAudG9UYWdSZXNvdXJjZSgpXG4gICAqIC0gLnRvVW50YWdSZXNvdXJjZSgpXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzVGFnS2V5cyh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYXdzOlRhZ0tleXNgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW2RldGVjdGl2ZV0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYW1hem9uZGV0ZWN0aXZlLmh0bWwpLlxuICAgKlxuICAgKi9cbiAgY29uc3RydWN0b3IocHJvcHM/OiBpYW0uUG9saWN5U3RhdGVtZW50UHJvcHMpIHtcbiAgICBzdXBlcihwcm9wcyk7XG4gIH1cbn1cbiJdfQ==