"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Directconnect = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [directconnect](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdirectconnect.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Directconnect extends shared_1.PolicyStatement {
    /**
     * Grants permission to accept a proposal request to attach a virtual private gateway to a Direct Connect gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AcceptDirectConnectGatewayAssociationProposal.html
     */
    toAcceptDirectConnectGatewayAssociationProposal() {
        return this.to('AcceptDirectConnectGatewayAssociationProposal');
    }
    /**
     * Grants permission to create a hosted connection on an interconnect
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocateConnectionOnInterconnect.html
     */
    toAllocateConnectionOnInterconnect() {
        return this.to('AllocateConnectionOnInterconnect');
    }
    /**
     * Grants permission to create a new hosted connection between a AWS Direct Connect partner's network and a specific AWS Direct Connect location
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocateHostedConnection.html
     */
    toAllocateHostedConnection() {
        return this.to('AllocateHostedConnection');
    }
    /**
     * Grants permission to provision a private virtual interface to be owned by a different customer
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocatePrivateVirtualInterface.html
     */
    toAllocatePrivateVirtualInterface() {
        return this.to('AllocatePrivateVirtualInterface');
    }
    /**
     * Grants permission to provision a public virtual interface to be owned by a different customer
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocatePublicVirtualInterface.html
     */
    toAllocatePublicVirtualInterface() {
        return this.to('AllocatePublicVirtualInterface');
    }
    /**
     * Grants permission to provision a transit virtual interface to be owned by a different customer
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocateTransitVirtualInterface.html
     */
    toAllocateTransitVirtualInterface() {
        return this.to('AllocateTransitVirtualInterface');
    }
    /**
     * Grants permission to associate a connection with a LAG
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AssociateConnectionWithLag.html
     */
    toAssociateConnectionWithLag() {
        return this.to('AssociateConnectionWithLag');
    }
    /**
     * Grants permission to associate a hosted connection and its virtual interfaces with a link aggregation group (LAG) or interconnect
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AssociateHostedConnection.html
     */
    toAssociateHostedConnection() {
        return this.to('AssociateHostedConnection');
    }
    /**
     * Grants permission to associate a MAC Security (MACsec) Connection Key Name (CKN)/ Connectivity Association Key (CAK) pair with an AWS Direct Connect dedicated connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AssociateMacSecKey.html
     */
    toAssociateMacSecKey() {
        return this.to('AssociateMacSecKey');
    }
    /**
     * Grants permission to associate a virtual interface with a specified link aggregation group (LAG) or connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AssociateVirtualInterface.html
     */
    toAssociateVirtualInterface() {
        return this.to('AssociateVirtualInterface');
    }
    /**
     * Grants permission to confirm the creation of a hosted connection on an interconnect
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ConfirmConnection.html
     */
    toConfirmConnection() {
        return this.to('ConfirmConnection');
    }
    /**
     * Grants permission to confirm the the terms of agreement when creating the connection or link aggregation group (LAG)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ConfirmCustomerAgreement.html
     */
    toConfirmCustomerAgreement() {
        return this.to('ConfirmCustomerAgreement');
    }
    /**
     * Grants permission to accept ownership of a private virtual interface created by another customer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ConfirmPrivateVirtualInterface.html
     */
    toConfirmPrivateVirtualInterface() {
        return this.to('ConfirmPrivateVirtualInterface');
    }
    /**
     * Grants permission to accept ownership of a public virtual interface created by another customer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ConfirmPublicVirtualInterface.html
     */
    toConfirmPublicVirtualInterface() {
        return this.to('ConfirmPublicVirtualInterface');
    }
    /**
     * Grants permission to accept ownership of a transit virtual interface created by another customer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ConfirmTransitVirtualInterface.html
     */
    toConfirmTransitVirtualInterface() {
        return this.to('ConfirmTransitVirtualInterface');
    }
    /**
     * Grants permission to create a BGP peer on the specified virtual interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateBGPPeer.html
     */
    toCreateBGPPeer() {
        return this.to('CreateBGPPeer');
    }
    /**
     * Grants permission to create a new connection between the customer network and a specific AWS Direct Connect location
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateConnection.html
     */
    toCreateConnection() {
        return this.to('CreateConnection');
    }
    /**
     * Grants permission to create a Direct Connect gateway, which is an intermediate object that enables you to connect a set of virtual interfaces and virtual private gateways
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateDirectConnectGateway.html
     */
    toCreateDirectConnectGateway() {
        return this.to('CreateDirectConnectGateway');
    }
    /**
     * Grants permission to create an association between a Direct Connect gateway and a virtual private gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateDirectConnectGatewayAssociation.html
     */
    toCreateDirectConnectGatewayAssociation() {
        return this.to('CreateDirectConnectGatewayAssociation');
    }
    /**
     * Grants permission to create a proposal to associate the specified virtual private gateway with the specified Direct Connect gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateDirectConnectGatewayAssociationProposal.html
     */
    toCreateDirectConnectGatewayAssociationProposal() {
        return this.to('CreateDirectConnectGatewayAssociationProposal');
    }
    /**
     * Grants permission to create a new interconnect between a AWS Direct Connect partner's network and a specific AWS Direct Connect location
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateInterconnect.html
     */
    toCreateInterconnect() {
        return this.to('CreateInterconnect');
    }
    /**
     * Grants permission to create a link aggregation group (LAG) with the specified number of bundled physical connections between the customer network and a specific AWS Direct Connect location
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateLag.html
     */
    toCreateLag() {
        return this.to('CreateLag');
    }
    /**
     * Grants permission to create a new private virtual interface
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreatePrivateVirtualInterface.html
     */
    toCreatePrivateVirtualInterface() {
        return this.to('CreatePrivateVirtualInterface');
    }
    /**
     * Grants permission to create a new public virtual interface
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreatePublicVirtualInterface.html
     */
    toCreatePublicVirtualInterface() {
        return this.to('CreatePublicVirtualInterface');
    }
    /**
     * Grants permission to create a new transit virtual interface
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateTransitVirtualInterface.html
     */
    toCreateTransitVirtualInterface() {
        return this.to('CreateTransitVirtualInterface');
    }
    /**
     * Grants permission to delete the specified BGP peer on the specified virtual interface with the specified customer address and ASN
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteBGPPeer.html
     */
    toDeleteBGPPeer() {
        return this.to('DeleteBGPPeer');
    }
    /**
     * Grants permission to delete the connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteConnection.html
     */
    toDeleteConnection() {
        return this.to('DeleteConnection');
    }
    /**
     * Grants permission to delete the specified Direct Connect gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteDirectConnectGateway.html
     */
    toDeleteDirectConnectGateway() {
        return this.to('DeleteDirectConnectGateway');
    }
    /**
     * Grants permission to delete the association between the specified Direct Connect gateway and virtual private gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteDirectConnectGatewayAssociation.html
     */
    toDeleteDirectConnectGatewayAssociation() {
        return this.to('DeleteDirectConnectGatewayAssociation');
    }
    /**
     * Grants permission to delete the association proposal request between the specified Direct Connect gateway and virtual private gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteDirectConnectGatewayAssociationProposal.html
     */
    toDeleteDirectConnectGatewayAssociationProposal() {
        return this.to('DeleteDirectConnectGatewayAssociationProposal');
    }
    /**
     * Grants permission to delete the specified interconnect
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteInterconnect.html
     */
    toDeleteInterconnect() {
        return this.to('DeleteInterconnect');
    }
    /**
     * Grants permission to delete the specified link aggregation group (LAG)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteLag.html
     */
    toDeleteLag() {
        return this.to('DeleteLag');
    }
    /**
     * Grants permission to delete a virtual interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteVirtualInterface.html
     */
    toDeleteVirtualInterface() {
        return this.to('DeleteVirtualInterface');
    }
    /**
     * Grants permission to describe the LOA-CFA for a Connection
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeConnectionLoa.html
     */
    toDescribeConnectionLoa() {
        return this.to('DescribeConnectionLoa');
    }
    /**
     * Grants permission to describe all connections in this region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeConnections.html
     */
    toDescribeConnections() {
        return this.to('DescribeConnections');
    }
    /**
     * Grants permission to describe a list of connections that have been provisioned on the given interconnect
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeConnectionsOnInterconnect.html
     */
    toDescribeConnectionsOnInterconnect() {
        return this.to('DescribeConnectionsOnInterconnect');
    }
    /**
     * Grants permission to view a list of customer agreements, along with their signed status and whether the customer is an NNIPartner, NNIPartnerV2, or a nonPartner
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeCustomerMetadata.html
     */
    toDescribeCustomerMetadata() {
        return this.to('DescribeCustomerMetadata');
    }
    /**
     * Grants permission to describe one or more association proposals for connection between a virtual private gateway and a Direct Connect gateway
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeDirectConnectGatewayAssociationProposals.html
     */
    toDescribeDirectConnectGatewayAssociationProposals() {
        return this.to('DescribeDirectConnectGatewayAssociationProposals');
    }
    /**
     * Grants permission to describe the associations between your Direct Connect gateways and virtual private gateways
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeDirectConnectGatewayAssociations.html
     */
    toDescribeDirectConnectGatewayAssociations() {
        return this.to('DescribeDirectConnectGatewayAssociations');
    }
    /**
     * Grants permission to describe the attachments between your Direct Connect gateways and virtual interfaces
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeDirectConnectGatewayAttachments.html
     */
    toDescribeDirectConnectGatewayAttachments() {
        return this.to('DescribeDirectConnectGatewayAttachments');
    }
    /**
     * Grants permission to describe all your Direct Connect gateways or only the specified Direct Connect gateway
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeDirectConnectGateways.html
     */
    toDescribeDirectConnectGateways() {
        return this.to('DescribeDirectConnectGateways');
    }
    /**
     * Grants permission to describe the hosted connections that have been provisioned on the specified interconnect or link aggregation group (LAG)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeHostedConnections.html
     */
    toDescribeHostedConnections() {
        return this.to('DescribeHostedConnections');
    }
    /**
     * Grants permission to describe the LOA-CFA for an Interconnect
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeInterconnectLoa.html
     */
    toDescribeInterconnectLoa() {
        return this.to('DescribeInterconnectLoa');
    }
    /**
     * Grants permission to describe a list of interconnects owned by the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeInterconnects.html
     */
    toDescribeInterconnects() {
        return this.to('DescribeInterconnects');
    }
    /**
     * Grants permission to describe all your link aggregation groups (LAG) or the specified LAG
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeLags.html
     */
    toDescribeLags() {
        return this.to('DescribeLags');
    }
    /**
     * Grants permission to describe the LOA-CFA for a connection, interconnect, or link aggregation group (LAG)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeLoa.html
     */
    toDescribeLoa() {
        return this.to('DescribeLoa');
    }
    /**
     * Grants permission to describe the list of AWS Direct Connect locations in the current AWS region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeLocations.html
     */
    toDescribeLocations() {
        return this.to('DescribeLocations');
    }
    /**
     * Grants permission to describe Details about the router for a virtual interface
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeRouterConfiguration.html
     */
    toDescribeRouterConfiguration() {
        return this.to('DescribeRouterConfiguration');
    }
    /**
     * Grants permission to describe the tags associated with the specified AWS Direct Connect resources
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeTags.html
     */
    toDescribeTags() {
        return this.to('DescribeTags');
    }
    /**
     * Grants permission to describe a list of virtual private gateways owned by the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeVirtualGateways.html
     */
    toDescribeVirtualGateways() {
        return this.to('DescribeVirtualGateways');
    }
    /**
     * Grants permission to describe all virtual interfaces for an AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeVirtualInterfaces.html
     */
    toDescribeVirtualInterfaces() {
        return this.to('DescribeVirtualInterfaces');
    }
    /**
     * Grants permission to disassociate a connection from a link aggregation group (LAG)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DisassociateConnectionFromLag.html
     */
    toDisassociateConnectionFromLag() {
        return this.to('DisassociateConnectionFromLag');
    }
    /**
     * Grants permission to remove the association between a MAC Security (MACsec) security key and an AWS Direct Connect dedicated connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DisassociateMacSecKey.html
     */
    toDisassociateMacSecKey() {
        return this.to('DisassociateMacSecKey');
    }
    /**
     * Grants permission to list the virtual interface failover test history
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ListVirtualInterfaceTestHistory.html
     */
    toListVirtualInterfaceTestHistory() {
        return this.to('ListVirtualInterfaceTestHistory');
    }
    /**
     * Grants permission to start the virtual interface failover test that verifies your configuration meets your resiliency requirements by placing the BGP peering session in the DOWN state. You can then send traffic to verify that there are no outages
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_StartBgpFailoverTest.html
     */
    toStartBgpFailoverTest() {
        return this.to('StartBgpFailoverTest');
    }
    /**
     * Grants permission to stop the virtual interface failover test
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_StopBgpFailoverTest.html
     */
    toStopBgpFailoverTest() {
        return this.to('StopBgpFailoverTest');
    }
    /**
     * Grants permission to add the specified tags to the specified AWS Direct Connect resource. Each resource can have a maximum of 50 tags
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove one or more tags from the specified AWS Direct Connect resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update the AWS Direct Connect dedicated connection configuration. You can update the following parameters for a connection: The connection name or The connection's MAC Security (MACsec) encryption mode
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_UpdateConnection.html
     */
    toUpdateConnection() {
        return this.to('UpdateConnection');
    }
    /**
     * Grants permission to update the name of a Direct Connect gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_UpdateDirectConnectGateway.html
     */
    toUpdateDirectConnectGateway() {
        return this.to('UpdateDirectConnectGateway');
    }
    /**
     * Grants permission to update the specified attributes of the Direct Connect gateway association
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_UpdateDirectConnectGatewayAssociation.html
     */
    toUpdateDirectConnectGatewayAssociation() {
        return this.to('UpdateDirectConnectGatewayAssociation');
    }
    /**
     * Grants permission to update the attributes of the specified link aggregation group (LAG)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_UpdateLag.html
     */
    toUpdateLag() {
        return this.to('UpdateLag');
    }
    /**
     * Grants permission to update the specified attributes of the specified virtual private interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_UpdateVirtualInterfaceAttributes.html
     */
    toUpdateVirtualInterfaceAttributes() {
        return this.to('UpdateVirtualInterfaceAttributes');
    }
    /**
     * Adds a resource of type dxcon to the statement
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_Connection.html
     *
     * @param connectionId - Identifier for the connectionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDxcon(connectionId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:directconnect:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:dxcon/${connectionId}`);
    }
    /**
     * Adds a resource of type dxlag to the statement
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_Lag.html
     *
     * @param lagId - Identifier for the lagId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDxlag(lagId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:directconnect:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:dxlag/${lagId}`);
    }
    /**
     * Adds a resource of type dxvif to the statement
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_VirtualInterface.html
     *
     * @param virtualInterfaceId - Identifier for the virtualInterfaceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDxvif(virtualInterfaceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:directconnect:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:dxvif/${virtualInterfaceId}`);
    }
    /**
     * Adds a resource of type dx-gateway to the statement
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DirectConnectGateway.html
     *
     * @param directConnectGatewayId - Identifier for the directConnectGatewayId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDxGateway(directConnectGatewayId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:directconnect::${account ?? this.defaultAccount}:dx-gateway/${directConnectGatewayId}`);
    }
    /**
     * Filters access by actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toAllocateHostedConnection()
     * - .toAllocatePrivateVirtualInterface()
     * - .toAllocatePublicVirtualInterface()
     * - .toAllocateTransitVirtualInterface()
     * - .toCreateConnection()
     * - .toCreateDirectConnectGateway()
     * - .toCreateInterconnect()
     * - .toCreateLag()
     * - .toCreatePrivateVirtualInterface()
     * - .toCreatePublicVirtualInterface()
     * - .toCreateTransitVirtualInterface()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - dxcon
     * - dxlag
     * - dxvif
     * - dx-gateway
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by actions based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toAllocateHostedConnection()
     * - .toAllocatePrivateVirtualInterface()
     * - .toAllocatePublicVirtualInterface()
     * - .toAllocateTransitVirtualInterface()
     * - .toCreateConnection()
     * - .toCreateDirectConnectGateway()
     * - .toCreateInterconnect()
     * - .toCreateLag()
     * - .toCreatePrivateVirtualInterface()
     * - .toCreatePublicVirtualInterface()
     * - .toCreateTransitVirtualInterface()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [directconnect](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdirectconnect.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'directconnect';
        this.accessLevelList = {
            Write: [
                'AcceptDirectConnectGatewayAssociationProposal',
                'AllocateConnectionOnInterconnect',
                'AllocateHostedConnection',
                'AllocatePrivateVirtualInterface',
                'AllocatePublicVirtualInterface',
                'AllocateTransitVirtualInterface',
                'AssociateConnectionWithLag',
                'AssociateHostedConnection',
                'AssociateMacSecKey',
                'AssociateVirtualInterface',
                'ConfirmConnection',
                'ConfirmCustomerAgreement',
                'ConfirmPrivateVirtualInterface',
                'ConfirmPublicVirtualInterface',
                'ConfirmTransitVirtualInterface',
                'CreateBGPPeer',
                'CreateConnection',
                'CreateDirectConnectGateway',
                'CreateDirectConnectGatewayAssociation',
                'CreateDirectConnectGatewayAssociationProposal',
                'CreateInterconnect',
                'CreateLag',
                'CreatePrivateVirtualInterface',
                'CreatePublicVirtualInterface',
                'CreateTransitVirtualInterface',
                'DeleteBGPPeer',
                'DeleteConnection',
                'DeleteDirectConnectGateway',
                'DeleteDirectConnectGatewayAssociation',
                'DeleteDirectConnectGatewayAssociationProposal',
                'DeleteInterconnect',
                'DeleteLag',
                'DeleteVirtualInterface',
                'DisassociateConnectionFromLag',
                'DisassociateMacSecKey',
                'StartBgpFailoverTest',
                'StopBgpFailoverTest',
                'UpdateConnection',
                'UpdateDirectConnectGateway',
                'UpdateDirectConnectGatewayAssociation',
                'UpdateLag',
                'UpdateVirtualInterfaceAttributes'
            ],
            Read: [
                'DescribeConnectionLoa',
                'DescribeConnections',
                'DescribeConnectionsOnInterconnect',
                'DescribeCustomerMetadata',
                'DescribeDirectConnectGatewayAssociationProposals',
                'DescribeDirectConnectGatewayAssociations',
                'DescribeDirectConnectGatewayAttachments',
                'DescribeDirectConnectGateways',
                'DescribeHostedConnections',
                'DescribeInterconnectLoa',
                'DescribeInterconnects',
                'DescribeLags',
                'DescribeLoa',
                'DescribeLocations',
                'DescribeRouterConfiguration',
                'DescribeTags',
                'DescribeVirtualGateways',
                'DescribeVirtualInterfaces'
            ],
            List: [
                'ListVirtualInterfaceTestHistory'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Directconnect = Directconnect;
//# sourceMappingURL=data:application/json;base64,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