"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Dynamodb = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [dynamodb](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazondynamodb.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Dynamodb extends shared_1.PolicyStatement {
    /**
     * Grants permission to return the attributes of one or more items from one or more tables
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     * - .ifSelect()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_BatchGetItem.html
     */
    toBatchGetItem() {
        return this.to('BatchGetItem');
    }
    /**
     * Grants permission to put or delete multiple items in one or more tables
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_BatchWriteItem.html
     */
    toBatchWriteItem() {
        return this.to('BatchWriteItem');
    }
    /**
     * Grants permission to the ConditionCheckItem operation checks the existence of a set of attributes for the item with the given primary key
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     * - .ifReturnValues()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ConditionCheck.html
     */
    toConditionCheckItem() {
        return this.to('ConditionCheckItem');
    }
    /**
     * Grants permission to create a backup for an existing table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_CreateBackup.html
     */
    toCreateBackup() {
        return this.to('CreateBackup');
    }
    /**
     * Grants permission to create a global table from an existing table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_CreateGlobalTable.html
     */
    toCreateGlobalTable() {
        return this.to('CreateGlobalTable');
    }
    /**
     * Grants permission to the CreateTable operation adds a new table to your account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_CreateTable.html
     */
    toCreateTable() {
        return this.to('CreateTable');
    }
    /**
     * Grants permission to add a new replica table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/V2gt_IAM.html
     */
    toCreateTableReplica() {
        return this.to('CreateTableReplica');
    }
    /**
     * Grants permission to delete an existing backup of a table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteBackup.html
     */
    toDeleteBackup() {
        return this.to('DeleteBackup');
    }
    /**
     * Grants permission to deletes a single item in a table by primary key
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifEnclosingOperation()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     * - .ifReturnValues()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteItem.html
     */
    toDeleteItem() {
        return this.to('DeleteItem');
    }
    /**
     * Grants permission to delete the resource-based policy attached to the resource
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to the DeleteTable operation which deletes a table and all of its items
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteTable.html
     */
    toDeleteTable() {
        return this.to('DeleteTable');
    }
    /**
     * Grants permission to delete a replica table and all of its items
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/V2gt_IAM.html
     */
    toDeleteTableReplica() {
        return this.to('DeleteTableReplica');
    }
    /**
     * Grants permission to describe an existing backup of a table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeBackup.html
     */
    toDescribeBackup() {
        return this.to('DescribeBackup');
    }
    /**
     * Grants permission to check the status of the backup restore settings on the specified table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeContinuousBackups.html
     */
    toDescribeContinuousBackups() {
        return this.to('DescribeContinuousBackups');
    }
    /**
     * Grants permission to describe the contributor insights status and related details for a given table or global secondary index
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeContributorInsights.html
     */
    toDescribeContributorInsights() {
        return this.to('DescribeContributorInsights');
    }
    /**
     * Grants permission to return the regional endpoint information
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeEndpoints.html
     */
    toDescribeEndpoints() {
        return this.to('DescribeEndpoints');
    }
    /**
     * Grants permission to describe an existing Export of a table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeExport.html
     */
    toDescribeExport() {
        return this.to('DescribeExport');
    }
    /**
     * Grants permission to return information about the specified global table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeGlobalTable.html
     */
    toDescribeGlobalTable() {
        return this.to('DescribeGlobalTable');
    }
    /**
     * Grants permission to return settings information about the specified global table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeGlobalTableSettings.html
     */
    toDescribeGlobalTableSettings() {
        return this.to('DescribeGlobalTableSettings');
    }
    /**
     * Grants permission to describe an existing import
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeImport.html
     */
    toDescribeImport() {
        return this.to('DescribeImport');
    }
    /**
     * Grants permission to grant permission to describe the status of Kinesis streaming and related details for a given table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeKinesisStreamingDestination.html
     */
    toDescribeKinesisStreamingDestination() {
        return this.to('DescribeKinesisStreamingDestination');
    }
    /**
     * Grants permission to return the current provisioned-capacity limits for your AWS account in a region, both for the region as a whole and for any one DynamoDB table that you create there
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeLimits.html
     */
    toDescribeLimits() {
        return this.to('DescribeLimits');
    }
    /**
     * Grants permission to describe one or more of the Reserved Capacity purchased
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/iam-policy-prevent-purchase-reserved-capacity.html
     */
    toDescribeReservedCapacity() {
        return this.to('DescribeReservedCapacity');
    }
    /**
     * Grants permission to describe Reserved Capacity offerings that are available for purchase
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/iam-policy-prevent-purchase-reserved-capacity.html
     */
    toDescribeReservedCapacityOfferings() {
        return this.to('DescribeReservedCapacityOfferings');
    }
    /**
     * Grants permission to return information about a stream, including the current status of the stream, its Amazon Resource Name (ARN), the composition of its shards, and its corresponding DynamoDB table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_streams_DescribeStream.html
     */
    toDescribeStream() {
        return this.to('DescribeStream');
    }
    /**
     * Grants permission to return information about the table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeTable.html
     */
    toDescribeTable() {
        return this.to('DescribeTable');
    }
    /**
     * Grants permission to describe the auto scaling settings across all replicas of the global table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeTableReplicaAutoScaling.html
     */
    toDescribeTableReplicaAutoScaling() {
        return this.to('DescribeTableReplicaAutoScaling');
    }
    /**
     * Grants permission to give a description of the Time to Live (TTL) status on the specified table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeTimeToLive.html
     */
    toDescribeTimeToLive() {
        return this.to('DescribeTimeToLive');
    }
    /**
     * Grants permission to grant permission to stop replication from the DynamoDB table to the Kinesis data stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DisableKinesisStreamingDestination.html
     */
    toDisableKinesisStreamingDestination() {
        return this.to('DisableKinesisStreamingDestination');
    }
    /**
     * Grants permission to grant permission to start table data replication to the specified Kinesis data stream at a timestamp chosen during the enable workflow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_EnableKinesisStreamingDestination.html
     */
    toEnableKinesisStreamingDestination() {
        return this.to('EnableKinesisStreamingDestination');
    }
    /**
     * Grants permission to initiate an Export of a DynamoDB table to S3
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ExportTableToPointInTime.html
     */
    toExportTableToPointInTime() {
        return this.to('ExportTableToPointInTime');
    }
    /**
     * Grants permission to view the status of Attribute Based Access Control for the account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/abac-enable-ddb.html
     */
    toGetAbacStatus() {
        return this.to('GetAbacStatus');
    }
    /**
     * Grants permission to the GetItem operation that returns a set of attributes for the item with the given primary key
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifEnclosingOperation()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     * - .ifSelect()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetItem.html
     */
    toGetItem() {
        return this.to('GetItem');
    }
    /**
     * Grants permission to retrieve the stream records from a given shard
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_streams_GetRecords.html
     */
    toGetRecords() {
        return this.to('GetRecords');
    }
    /**
     * Grants permission to view a resource-based policy for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetResourcePolicy.html
     */
    toGetResourcePolicy() {
        return this.to('GetResourcePolicy');
    }
    /**
     * Grants permission to return a shard iterator
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_streams_GetShardIterator.html
     */
    toGetShardIterator() {
        return this.to('GetShardIterator');
    }
    /**
     * Grants permission to initiate an import from S3 to a DynamoDB table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ImportTable.html
     */
    toImportTable() {
        return this.to('ImportTable');
    }
    /**
     * Grants permission to list backups associated with the account and endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListBackups.html
     */
    toListBackups() {
        return this.to('ListBackups');
    }
    /**
     * Grants permission to list the ContributorInsightsSummary for all tables and global secondary indexes associated with the current account and endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListContributorInsights.html
     */
    toListContributorInsights() {
        return this.to('ListContributorInsights');
    }
    /**
     * Grants permission to list exports associated with the account and endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListExports.html
     */
    toListExports() {
        return this.to('ListExports');
    }
    /**
     * Grants permission to list all global tables that have a replica in the specified region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListGlobalTables.html
     */
    toListGlobalTables() {
        return this.to('ListGlobalTables');
    }
    /**
     * Grants permission to list imports associated with the account and endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListImports.html
     */
    toListImports() {
        return this.to('ListImports');
    }
    /**
     * Grants permission to return an array of stream ARNs associated with the current account and endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_streams_ListStreams.html
     */
    toListStreams() {
        return this.to('ListStreams');
    }
    /**
     * Grants permission to return an array of table names associated with the current account and endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListTables.html
     */
    toListTables() {
        return this.to('ListTables');
    }
    /**
     * Grants permission to list all tags on an Amazon DynamoDB resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListTagsOfResource.html
     */
    toListTagsOfResource() {
        return this.to('ListTagsOfResource');
    }
    /**
     * Grants permission to delete a single item in a table by primary key
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifEnclosingOperation()
     * - .ifLeadingKeys()
     * - .ifReturnValues()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ExecuteStatement.html
     */
    toPartiQLDelete() {
        return this.to('PartiQLDelete');
    }
    /**
     * Grants permission to create a new item, if an item with same primary key does not exist in the table
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifEnclosingOperation()
     * - .ifLeadingKeys()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ExecuteStatement.html
     */
    toPartiQLInsert() {
        return this.to('PartiQLInsert');
    }
    /**
     * Grants permission to read a set of attributes for items from a table or index
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifEnclosingOperation()
     * - .ifFullTableScan()
     * - .ifLeadingKeys()
     * - .ifSelect()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ExecuteStatement.html
     */
    toPartiQLSelect() {
        return this.to('PartiQLSelect');
    }
    /**
     * Grants permission to edit an existing item's attributes
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifEnclosingOperation()
     * - .ifLeadingKeys()
     * - .ifReturnValues()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ExecuteStatement.html
     */
    toPartiQLUpdate() {
        return this.to('PartiQLUpdate');
    }
    /**
     * Grants permission to purchases reserved capacity for use with your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/iam-policy-prevent-purchase-reserved-capacity.html
     */
    toPurchaseReservedCapacityOfferings() {
        return this.to('PurchaseReservedCapacityOfferings');
    }
    /**
     * Grants permission to create a new item, or replace an old item with a new item
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifEnclosingOperation()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     * - .ifReturnValues()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_PutItem.html
     */
    toPutItem() {
        return this.to('PutItem');
    }
    /**
     * Grants permission to attach a resource-based policy to the resource
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_PutResourcePolicy.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to use the primary key of a table or a secondary index to directly access items from that table or index
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     * - .ifReturnValues()
     * - .ifSelect()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_Query.html
     */
    toQuery() {
        return this.to('Query');
    }
    /**
     * Grants permission to create a new table from recovery point on AWS Backup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/UsageNotesAWS.html
     */
    toRestoreTableFromAwsBackup() {
        return this.to('RestoreTableFromAwsBackup');
    }
    /**
     * Grants permission to create a new table from an existing backup
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - dynamodb:BatchWriteItem
     * - dynamodb:DeleteItem
     * - dynamodb:GetItem
     * - dynamodb:PutItem
     * - dynamodb:Query
     * - dynamodb:Scan
     * - dynamodb:UpdateItem
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_RestoreTableFromBackup.html
     */
    toRestoreTableFromBackup() {
        return this.to('RestoreTableFromBackup');
    }
    /**
     * Grants permission to restore a table to a point in time
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - dynamodb:BatchWriteItem
     * - dynamodb:DeleteItem
     * - dynamodb:GetItem
     * - dynamodb:PutItem
     * - dynamodb:Query
     * - dynamodb:Scan
     * - dynamodb:UpdateItem
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_RestoreTableToPointInTime.html
     */
    toRestoreTableToPointInTime() {
        return this.to('RestoreTableToPointInTime');
    }
    /**
     * Grants permission to return one or more items and item attributes by accessing every item in a table or a secondary index
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifReturnConsumedCapacity()
     * - .ifReturnValues()
     * - .ifSelect()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_Scan.html
     */
    toScan() {
        return this.to('Scan');
    }
    /**
     * Grants permission to create a backup on AWS Backup with advanced features enabled
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/UsageNotesAWS.html
     */
    toStartAwsBackupJob() {
        return this.to('StartAwsBackupJob');
    }
    /**
     * Grants permission to associate a set of tags with an Amazon DynamoDB resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove the association of tags from an Amazon DynamoDB resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update the status of Attribute Based Access Control for the account
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/abac-enable-ddb.html
     */
    toUpdateAbacStatus() {
        return this.to('UpdateAbacStatus');
    }
    /**
     * Grants permission to enable or disable continuous backups
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateContinuousBackups.html
     */
    toUpdateContinuousBackups() {
        return this.to('UpdateContinuousBackups');
    }
    /**
     * Grants permission to update the status for contributor insights for a specific table or global secondary index
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateContributorInsights.html
     */
    toUpdateContributorInsights() {
        return this.to('UpdateContributorInsights');
    }
    /**
     * Grants permission to add or remove replicas in the specified global table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateGlobalTable.html
     */
    toUpdateGlobalTable() {
        return this.to('UpdateGlobalTable');
    }
    /**
     * Grants permission to update settings of the specified global table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateGlobalTableSettings.html
     */
    toUpdateGlobalTableSettings() {
        return this.to('UpdateGlobalTableSettings');
    }
    /**
     * Grants permission to update version of the specified global table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/V2globaltables_upgrade.html
     */
    toUpdateGlobalTableVersion() {
        return this.to('UpdateGlobalTableVersion');
    }
    /**
     * Grants permission to edit an existing item's attributes, or adds a new item to the table if it does not already exist
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifEnclosingOperation()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     * - .ifReturnValues()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateItem.html
     */
    toUpdateItem() {
        return this.to('UpdateItem');
    }
    /**
     * Grants permission to update data replication configurations for the specified Kinesis data stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateKinesisStreamingDestination.html
     */
    toUpdateKinesisStreamingDestination() {
        return this.to('UpdateKinesisStreamingDestination');
    }
    /**
     * Grants permission to modify the provisioned throughput settings, global secondary indexes, or DynamoDB Streams settings for a given table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateTable.html
     */
    toUpdateTable() {
        return this.to('UpdateTable');
    }
    /**
     * Grants permission to update auto scaling settings on your replica table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateTableReplicaAutoScaling.html
     */
    toUpdateTableReplicaAutoScaling() {
        return this.to('UpdateTableReplicaAutoScaling');
    }
    /**
     * Grants permission to enable or disable TTL for the specified table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateTimeToLive.html
     */
    toUpdateTimeToLive() {
        return this.to('UpdateTimeToLive');
    }
    /**
     * Adds a resource of type index to the statement
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.CoreComponents.html#HowItWorks.CoreComponents.PrimaryKey
     *
     * @param tableName - Identifier for the tableName.
     * @param indexName - Identifier for the indexName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIndex(tableName, indexName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dynamodb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:table/${tableName}/index/${indexName}`);
    }
    /**
     * Adds a resource of type stream to the statement
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.CoreComponents.html#HowItWorks.CoreComponents.Streams
     *
     * @param tableName - Identifier for the tableName.
     * @param streamLabel - Identifier for the streamLabel.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onStream(tableName, streamLabel, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dynamodb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:table/${tableName}/stream/${streamLabel}`);
    }
    /**
     * Adds a resource of type table to the statement
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.CoreComponents.html#HowItWorks.CoreComponents.TablesItemsAttributes
     *
     * @param tableName - Identifier for the tableName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTable(tableName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dynamodb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:table/${tableName}`);
    }
    /**
     * Adds a resource of type backup to the statement
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/backuprestore_HowItWorks.html
     *
     * @param tableName - Identifier for the tableName.
     * @param backupName - Identifier for the backupName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onBackup(tableName, backupName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dynamodb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:table/${tableName}/backup/${backupName}`);
    }
    /**
     * Adds a resource of type export to the statement
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/S3DataExport.HowItWorks.html
     *
     * @param tableName - Identifier for the tableName.
     * @param exportName - Identifier for the exportName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onExport(tableName, exportName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dynamodb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:table/${tableName}/export/${exportName}`);
    }
    /**
     * Adds a resource of type global-table to the statement
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/globaltables_HowItWorks.html
     *
     * @param globalTableName - Identifier for the globalTableName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onGlobalTable(globalTableName, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dynamodb::${account ?? this.defaultAccount}:global-table/${globalTableName}`);
    }
    /**
     * Adds a resource of type import to the statement
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/S3DataImport.HowItWorks.html
     *
     * @param tableName - Identifier for the tableName.
     * @param importName - Identifier for the importName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onImport(tableName, importName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dynamodb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:table/${tableName}/import/${importName}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateTable()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - index
     * - table
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateTable()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by attribute (field or column) names of the table
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/specifying-conditions.html#FGAC_DDB.ConditionKeys
     *
     * Applies to actions:
     * - .toBatchGetItem()
     * - .toBatchWriteItem()
     * - .toConditionCheckItem()
     * - .toDeleteItem()
     * - .toGetItem()
     * - .toPartiQLDelete()
     * - .toPartiQLInsert()
     * - .toPartiQLSelect()
     * - .toPartiQLUpdate()
     * - .toPutItem()
     * - .toQuery()
     * - .toScan()
     * - .toUpdateItem()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAttributes(value, operator) {
        return this.if(`Attributes`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by blocking Transactions APIs calls and allow the non-Transaction APIs calls and vice-versa
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/specifying-conditions.html#FGAC_DDB.ConditionKeys
     *
     * Applies to actions:
     * - .toDeleteItem()
     * - .toGetItem()
     * - .toPartiQLDelete()
     * - .toPartiQLInsert()
     * - .toPartiQLSelect()
     * - .toPartiQLUpdate()
     * - .toPutItem()
     * - .toUpdateItem()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnclosingOperation(value, operator) {
        return this.if(`EnclosingOperation`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by blocking full table scan
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/ql-iam.html
     *
     * Applies to actions:
     * - .toPartiQLSelect()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifFullTableScan(value) {
        return this.if(`FullTableScan`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the partition key of the table
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/specifying-conditions.html#FGAC_DDB.ConditionKeys
     *
     * Applies to actions:
     * - .toBatchGetItem()
     * - .toBatchWriteItem()
     * - .toConditionCheckItem()
     * - .toDeleteItem()
     * - .toGetItem()
     * - .toPartiQLDelete()
     * - .toPartiQLInsert()
     * - .toPartiQLSelect()
     * - .toPartiQLUpdate()
     * - .toPutItem()
     * - .toQuery()
     * - .toUpdateItem()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLeadingKeys(value, operator) {
        return this.if(`LeadingKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the ReturnConsumedCapacity parameter of a request. Contains either "TOTAL" or "NONE"
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/specifying-conditions.html#FGAC_DDB.ConditionKeys
     *
     * Applies to actions:
     * - .toBatchGetItem()
     * - .toBatchWriteItem()
     * - .toConditionCheckItem()
     * - .toDeleteItem()
     * - .toGetItem()
     * - .toPutItem()
     * - .toQuery()
     * - .toScan()
     * - .toUpdateItem()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReturnConsumedCapacity(value, operator) {
        return this.if(`ReturnConsumedCapacity`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the ReturnValues parameter of request. Contains one of the following: "ALL_OLD", "UPDATED_OLD","ALL_NEW","UPDATED_NEW", or "NONE"
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/specifying-conditions.html#FGAC_DDB.ConditionKeys
     *
     * Applies to actions:
     * - .toConditionCheckItem()
     * - .toDeleteItem()
     * - .toPartiQLDelete()
     * - .toPartiQLUpdate()
     * - .toPutItem()
     * - .toQuery()
     * - .toScan()
     * - .toUpdateItem()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReturnValues(value, operator) {
        return this.if(`ReturnValues`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the Select parameter of a Query or Scan request
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/specifying-conditions.html#FGAC_DDB.ConditionKeys
     *
     * Applies to actions:
     * - .toBatchGetItem()
     * - .toGetItem()
     * - .toPartiQLSelect()
     * - .toQuery()
     * - .toScan()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSelect(value, operator) {
        return this.if(`Select`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [dynamodb](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazondynamodb.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'dynamodb';
        this.accessLevelList = {
            Read: [
                'BatchGetItem',
                'ConditionCheckItem',
                'DescribeBackup',
                'DescribeContinuousBackups',
                'DescribeContributorInsights',
                'DescribeEndpoints',
                'DescribeExport',
                'DescribeGlobalTable',
                'DescribeGlobalTableSettings',
                'DescribeImport',
                'DescribeKinesisStreamingDestination',
                'DescribeLimits',
                'DescribeReservedCapacity',
                'DescribeReservedCapacityOfferings',
                'DescribeStream',
                'DescribeTable',
                'DescribeTableReplicaAutoScaling',
                'DescribeTimeToLive',
                'GetAbacStatus',
                'GetItem',
                'GetRecords',
                'GetResourcePolicy',
                'GetShardIterator',
                'ListStreams',
                'ListTagsOfResource',
                'PartiQLSelect',
                'Query',
                'Scan'
            ],
            Write: [
                'BatchWriteItem',
                'CreateBackup',
                'CreateGlobalTable',
                'CreateTable',
                'CreateTableReplica',
                'DeleteBackup',
                'DeleteItem',
                'DeleteTable',
                'DeleteTableReplica',
                'DisableKinesisStreamingDestination',
                'EnableKinesisStreamingDestination',
                'ExportTableToPointInTime',
                'ImportTable',
                'PartiQLDelete',
                'PartiQLInsert',
                'PartiQLUpdate',
                'PurchaseReservedCapacityOfferings',
                'PutItem',
                'RestoreTableFromAwsBackup',
                'RestoreTableFromBackup',
                'RestoreTableToPointInTime',
                'StartAwsBackupJob',
                'UpdateContinuousBackups',
                'UpdateContributorInsights',
                'UpdateGlobalTable',
                'UpdateGlobalTableSettings',
                'UpdateGlobalTableVersion',
                'UpdateItem',
                'UpdateKinesisStreamingDestination',
                'UpdateTable',
                'UpdateTableReplicaAutoScaling',
                'UpdateTimeToLive'
            ],
            'Permissions management': [
                'DeleteResourcePolicy',
                'PutResourcePolicy',
                'UpdateAbacStatus'
            ],
            List: [
                'ListBackups',
                'ListContributorInsights',
                'ListExports',
                'ListGlobalTables',
                'ListImports',
                'ListTables'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Dynamodb = Dynamodb;
//# sourceMappingURL=data:application/json;base64,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