"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Dax = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [dax](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazondynamodbacceleratordax.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Dax extends shared_1.PolicyStatement {
    /**
     * Grants permission to return the attributes of one or more items from one or more tables
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_BatchGetItem.html
     */
    toBatchGetItem() {
        return this.to('BatchGetItem');
    }
    /**
     * Grants permission to put or delete multiple items in one or more tables
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_BatchWriteItem.html
     */
    toBatchWriteItem() {
        return this.to('BatchWriteItem');
    }
    /**
     * Grants permission to the ConditionCheckItem operation that checks the existence of a set of attributes for the item with the given primary key
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ConditionCheckItem.html
     */
    toConditionCheckItem() {
        return this.to('ConditionCheckItem');
    }
    /**
     * Grants permission to create a DAX cluster
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - dax:CreateParameterGroup
     * - dax:CreateSubnetGroup
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - iam:GetRole
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_CreateCluster.html
     */
    toCreateCluster() {
        return this.to('CreateCluster');
    }
    /**
     * Grants permission to create a parameter group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_CreateParameterGroup.html
     */
    toCreateParameterGroup() {
        return this.to('CreateParameterGroup');
    }
    /**
     * Grants permission to create a subnet group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_CreateSubnetGroup.html
     */
    toCreateSubnetGroup() {
        return this.to('CreateSubnetGroup');
    }
    /**
     * Grants permission to remove one or more nodes from a DAX cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DecreaseReplicationFactor.html
     */
    toDecreaseReplicationFactor() {
        return this.to('DecreaseReplicationFactor');
    }
    /**
     * Grants permission to delete a previously provisioned DAX cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DeleteCluster.html
     */
    toDeleteCluster() {
        return this.to('DeleteCluster');
    }
    /**
     * Grants permission to delete a single item in a table by primary key
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifEnclosingOperation()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteItem.html
     */
    toDeleteItem() {
        return this.to('DeleteItem');
    }
    /**
     * Grants permission to delete the specified parameter group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DeleteParameterGroup.html
     */
    toDeleteParameterGroup() {
        return this.to('DeleteParameterGroup');
    }
    /**
     * Grants permission to delete a subnet group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DeleteSubnetGroup.html
     */
    toDeleteSubnetGroup() {
        return this.to('DeleteSubnetGroup');
    }
    /**
     * Grants permission to return information about all provisioned DAX clusters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DescribeClusters.html
     */
    toDescribeClusters() {
        return this.to('DescribeClusters');
    }
    /**
     * Grants permission to return the default system parameter information for DAX
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DescribeDefaultParameters.html
     */
    toDescribeDefaultParameters() {
        return this.to('DescribeDefaultParameters');
    }
    /**
     * Grants permission to return events related to DAX clusters and parameter groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DescribeEvents.html
     */
    toDescribeEvents() {
        return this.to('DescribeEvents');
    }
    /**
     * Grants permission to return a list of parameter group descriptions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DescribeParameterGroups.html
     */
    toDescribeParameterGroups() {
        return this.to('DescribeParameterGroups');
    }
    /**
     * Grants permission to return the detailed parameter list for a particular parameter group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DescribeParameters.html
     */
    toDescribeParameters() {
        return this.to('DescribeParameters');
    }
    /**
     * Grants permission to return a list of subnet group descriptions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_DescribeSubnetGroups.html
     */
    toDescribeSubnetGroups() {
        return this.to('DescribeSubnetGroups');
    }
    /**
     * Grants permission to the GetItem operation that returns a set of attributes for the item with the given primary key
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifEnclosingOperation()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetItem.html
     */
    toGetItem() {
        return this.to('GetItem');
    }
    /**
     * Grants permission to add one or more nodes to a DAX cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_IncreaseReplicationFactor.html
     */
    toIncreaseReplicationFactor() {
        return this.to('IncreaseReplicationFactor');
    }
    /**
     * Grants permission to return a list all of the tags for a DAX cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_ListTags.html
     */
    toListTags() {
        return this.to('ListTags');
    }
    /**
     * Grants permission to create a new item, or replace an old item with a new item
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifEnclosingOperation()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_PutItem.html
     */
    toPutItem() {
        return this.to('PutItem');
    }
    /**
     * Grants permission to use the primary key of a table or a secondary index to directly access items from that table or index
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_Query.html
     */
    toQuery() {
        return this.to('Query');
    }
    /**
     * Grants permission to reboot a single node of a DAX cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_RebootNode.html
     */
    toRebootNode() {
        return this.to('RebootNode');
    }
    /**
     * Grants permission to return one or more items and item attributes by accessing every item in a table or a secondary index
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_Scan.html
     */
    toScan() {
        return this.to('Scan');
    }
    /**
     * Grants permission to associate a set of tags with a DAX resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove the association of tags from a DAX resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to modify the settings for a DAX cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_UpdateCluster.html
     */
    toUpdateCluster() {
        return this.to('UpdateCluster');
    }
    /**
     * Grants permission to edit an existing item's attributes, or adds a new item to the table if it does not already exist
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifEnclosingOperation()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateItem.html
     */
    toUpdateItem() {
        return this.to('UpdateItem');
    }
    /**
     * Grants permission to modify the parameters of a parameter group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_UpdateParameterGroup.html
     */
    toUpdateParameterGroup() {
        return this.to('UpdateParameterGroup');
    }
    /**
     * Grants permission to modify an existing subnet group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_dax_UpdateSubnetGroup.html
     */
    toUpdateSubnetGroup() {
        return this.to('UpdateSubnetGroup');
    }
    /**
     * Adds a resource of type application to the statement
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/DAX.access-control.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onApplication(clusterName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dax:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:cache/${clusterName}`);
    }
    /**
     * Used to block Transactions APIs calls and allow the non-Transaction APIs calls and vice-versa
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/DAX.access-control.htmlspecifying-conditions.html#FGAC_DDB.ConditionKeys
     *
     * Applies to actions:
     * - .toDeleteItem()
     * - .toGetItem()
     * - .toPutItem()
     * - .toUpdateItem()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnclosingOperation(value, operator) {
        return this.if(`EnclosingOperation`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [dax](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazondynamodbacceleratordax.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'dax';
        this.accessLevelList = {
            Read: [
                'BatchGetItem',
                'ConditionCheckItem',
                'DescribeParameters',
                'GetItem',
                'ListTags',
                'Query',
                'Scan'
            ],
            Write: [
                'BatchWriteItem',
                'CreateCluster',
                'CreateParameterGroup',
                'CreateSubnetGroup',
                'DecreaseReplicationFactor',
                'DeleteCluster',
                'DeleteItem',
                'DeleteParameterGroup',
                'DeleteSubnetGroup',
                'IncreaseReplicationFactor',
                'PutItem',
                'RebootNode',
                'UpdateCluster',
                'UpdateItem',
                'UpdateParameterGroup',
                'UpdateSubnetGroup'
            ],
            List: [
                'DescribeClusters',
                'DescribeDefaultParameters',
                'DescribeEvents',
                'DescribeParameterGroups',
                'DescribeSubnetGroups'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Dax = Dax;
//# sourceMappingURL=data:application/json;base64,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