"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Autoscaling = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [autoscaling](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonec2autoscaling.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Autoscaling extends shared_1.PolicyStatement {
    /**
     * Grants permission to attach one or more EC2 instances to the specified Auto Scaling group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_AttachInstances.html
     */
    toAttachInstances() {
        return this.to('AttachInstances');
    }
    /**
     * Grants permission to attach one or more target groups to the specified Auto Scaling group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTargetGroupARNs()
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_AttachLoadBalancerTargetGroups.html
     */
    toAttachLoadBalancerTargetGroups() {
        return this.to('AttachLoadBalancerTargetGroups');
    }
    /**
     * Grants permission to attach one or more load balancers to the specified Auto Scaling group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifLoadBalancerNames()
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_AttachLoadBalancers.html
     */
    toAttachLoadBalancers() {
        return this.to('AttachLoadBalancers');
    }
    /**
     * Grants permission to attach one or more traffic sources to an Auto Scaling group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTrafficSourceIdentifiers()
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_AttachTrafficSources.html
     */
    toAttachTrafficSources() {
        return this.to('AttachTrafficSources');
    }
    /**
     * Grants permission to delete the specified scheduled actions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_BatchDeleteScheduledAction.html
     */
    toBatchDeleteScheduledAction() {
        return this.to('BatchDeleteScheduledAction');
    }
    /**
     * Grants permission to create or update multiple scheduled scaling actions for an Auto Scaling group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_BatchPutScheduledUpdateGroupAction.html
     */
    toBatchPutScheduledUpdateGroupAction() {
        return this.to('BatchPutScheduledUpdateGroupAction');
    }
    /**
     * Grants permission to cancel an instance refresh operation in progress
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_CancelInstanceRefresh.html
     */
    toCancelInstanceRefresh() {
        return this.to('CancelInstanceRefresh');
    }
    /**
     * Grants permission to complete the lifecycle action for the specified token or instance with the specified result
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_CompleteLifecycleAction.html
     */
    toCompleteLifecycleAction() {
        return this.to('CompleteLifecycleAction');
    }
    /**
     * Grants permission to create an Auto Scaling group with the specified name and attributes
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCapacityReservationIds()
     * - .ifCapacityReservationResourceGroupArns()
     * - .ifInstanceTypes()
     * - .ifLaunchConfigurationName()
     * - .ifLaunchTemplateVersionSpecified()
     * - .ifLoadBalancerNames()
     * - .ifMaxSize()
     * - .ifMinSize()
     * - .ifTargetGroupARNs()
     * - .ifTrafficSourceIdentifiers()
     * - .ifVPCZoneIdentifiers()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_CreateAutoScalingGroup.html
     */
    toCreateAutoScalingGroup() {
        return this.to('CreateAutoScalingGroup');
    }
    /**
     * Grants permission to create a launch configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifImageId()
     * - .ifInstanceType()
     * - .ifSpotPrice()
     * - .ifMetadataHttpTokens()
     * - .ifMetadataHttpPutResponseHopLimit()
     * - .ifMetadataHttpEndpoint()
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_CreateLaunchConfiguration.html
     */
    toCreateLaunchConfiguration() {
        return this.to('CreateLaunchConfiguration');
    }
    /**
     * Grants permission to create or update tags for the specified Auto Scaling group
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_CreateOrUpdateTags.html
     */
    toCreateOrUpdateTags() {
        return this.to('CreateOrUpdateTags');
    }
    /**
     * Grants permission to delete the specified Auto Scaling group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteAutoScalingGroup.html
     */
    toDeleteAutoScalingGroup() {
        return this.to('DeleteAutoScalingGroup');
    }
    /**
     * Grants permission to delete the specified launch configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteLaunchConfiguration.html
     */
    toDeleteLaunchConfiguration() {
        return this.to('DeleteLaunchConfiguration');
    }
    /**
     * Grants permission to deletes the specified lifecycle hook
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteLifecycleHook.html
     */
    toDeleteLifecycleHook() {
        return this.to('DeleteLifecycleHook');
    }
    /**
     * Grants permission to delete the specified notification
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteNotificationConfiguration.html
     */
    toDeleteNotificationConfiguration() {
        return this.to('DeleteNotificationConfiguration');
    }
    /**
     * Grants permission to delete the specified Auto Scaling policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeletePolicy.html
     */
    toDeletePolicy() {
        return this.to('DeletePolicy');
    }
    /**
     * Grants permission to delete the specified scheduled action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteScheduledAction.html
     */
    toDeleteScheduledAction() {
        return this.to('DeleteScheduledAction');
    }
    /**
     * Grants permission to delete the specified tags
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteTags.html
     */
    toDeleteTags() {
        return this.to('DeleteTags');
    }
    /**
     * Grants permission to delete the warm pool associated with the Auto Scaling group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteWarmPool.html
     */
    toDeleteWarmPool() {
        return this.to('DeleteWarmPool');
    }
    /**
     * Grants permission to describe the current Auto Scaling resource limits for your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeAccountLimits.html
     */
    toDescribeAccountLimits() {
        return this.to('DescribeAccountLimits');
    }
    /**
     * Grants permission to describe the policy adjustment types for use with PutScalingPolicy
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeAdjustmentTypes.html
     */
    toDescribeAdjustmentTypes() {
        return this.to('DescribeAdjustmentTypes');
    }
    /**
     * Grants permission to describe one or more Auto Scaling groups. If a list of names is not provided, the call describes all Auto Scaling groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeAutoScalingGroups.html
     */
    toDescribeAutoScalingGroups() {
        return this.to('DescribeAutoScalingGroups');
    }
    /**
     * Grants permission to describe one or more Auto Scaling instances. If a list is not provided, the call describes all instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeAutoScalingInstances.html
     */
    toDescribeAutoScalingInstances() {
        return this.to('DescribeAutoScalingInstances');
    }
    /**
     * Grants permission to describe the notification types that are supported by Auto Scaling
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeAutoScalingNotificationTypes.html
     */
    toDescribeAutoScalingNotificationTypes() {
        return this.to('DescribeAutoScalingNotificationTypes');
    }
    /**
     * Grants permission to describe one or more instance refreshes for an Auto Scaling group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeInstanceRefreshes.html
     */
    toDescribeInstanceRefreshes() {
        return this.to('DescribeInstanceRefreshes');
    }
    /**
     * Grants permission to describe one or more launch configurations. If you omit the list of names, then the call describes all launch configurations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeLaunchConfigurations.html
     */
    toDescribeLaunchConfigurations() {
        return this.to('DescribeLaunchConfigurations');
    }
    /**
     * Grants permission to describe the available types of lifecycle hooks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeLifecycleHookTypes.html
     */
    toDescribeLifecycleHookTypes() {
        return this.to('DescribeLifecycleHookTypes');
    }
    /**
     * Grants permission to describe the lifecycle hooks for the specified Auto Scaling group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeLifecycleHooks.html
     */
    toDescribeLifecycleHooks() {
        return this.to('DescribeLifecycleHooks');
    }
    /**
     * Grants permission to describe the target groups for the specified Auto Scaling group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeLoadBalancerTargetGroups.html
     */
    toDescribeLoadBalancerTargetGroups() {
        return this.to('DescribeLoadBalancerTargetGroups');
    }
    /**
     * Grants permission to describe the load balancers for the specified Auto Scaling group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeLoadBalancers.html
     */
    toDescribeLoadBalancers() {
        return this.to('DescribeLoadBalancers');
    }
    /**
     * Grants permission to describe the available CloudWatch metrics for Auto Scaling
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeMetricCollectionTypes.html
     */
    toDescribeMetricCollectionTypes() {
        return this.to('DescribeMetricCollectionTypes');
    }
    /**
     * Grants permission to describe the notification actions associated with the specified Auto Scaling group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeNotificationConfigurations.html
     */
    toDescribeNotificationConfigurations() {
        return this.to('DescribeNotificationConfigurations');
    }
    /**
     * Grants permission to describe the policies for the specified Auto Scaling group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribePolicies.html
     */
    toDescribePolicies() {
        return this.to('DescribePolicies');
    }
    /**
     * Grants permission to describe one or more scaling activities for the specified Auto Scaling group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeScalingActivities.html
     */
    toDescribeScalingActivities() {
        return this.to('DescribeScalingActivities');
    }
    /**
     * Grants permission to describe the scaling process types for use with ResumeProcesses and SuspendProcesses
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeScalingProcessTypes.html
     */
    toDescribeScalingProcessTypes() {
        return this.to('DescribeScalingProcessTypes');
    }
    /**
     * Grants permission to describe the actions scheduled for your Auto Scaling group that haven't run
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeScheduledActions.html
     */
    toDescribeScheduledActions() {
        return this.to('DescribeScheduledActions');
    }
    /**
     * Grants permission to describe the specified tags
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeTags.html
     */
    toDescribeTags() {
        return this.to('DescribeTags');
    }
    /**
     * Grants permission to describe the termination policies supported by Auto Scaling
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeTerminationPolicyTypes.html
     */
    toDescribeTerminationPolicyTypes() {
        return this.to('DescribeTerminationPolicyTypes');
    }
    /**
     * Grants permission to describe the target groups for the specified Auto Scaling group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeTrafficSources.html
     */
    toDescribeTrafficSources() {
        return this.to('DescribeTrafficSources');
    }
    /**
     * Grants permission to describe the warm pool associated with the Auto Scaling group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeWarmPool.html
     */
    toDescribeWarmPool() {
        return this.to('DescribeWarmPool');
    }
    /**
     * Grants permission to remove one or more instances from the specified Auto Scaling group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DetachInstances.html
     */
    toDetachInstances() {
        return this.to('DetachInstances');
    }
    /**
     * Grants permission to detach one or more target groups from the specified Auto Scaling group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTargetGroupARNs()
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DetachLoadBalancerTargetGroups.html
     */
    toDetachLoadBalancerTargetGroups() {
        return this.to('DetachLoadBalancerTargetGroups');
    }
    /**
     * Grants permission to remove one or more load balancers from the specified Auto Scaling group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifLoadBalancerNames()
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DetachLoadBalancers.html
     */
    toDetachLoadBalancers() {
        return this.to('DetachLoadBalancers');
    }
    /**
     * Grants permission to detach one or more traffic sources from an Auto Scaling group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTrafficSourceIdentifiers()
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DetachTrafficSources.html
     */
    toDetachTrafficSources() {
        return this.to('DetachTrafficSources');
    }
    /**
     * Grants permission to disable monitoring of the specified metrics for the specified Auto Scaling group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DisableMetricsCollection.html
     */
    toDisableMetricsCollection() {
        return this.to('DisableMetricsCollection');
    }
    /**
     * Grants permission to enable monitoring of the specified metrics for the specified Auto Scaling group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_EnableMetricsCollection.html
     */
    toEnableMetricsCollection() {
        return this.to('EnableMetricsCollection');
    }
    /**
     * Grants permission to move the specified instances into Standby mode
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_EnterStandby.html
     */
    toEnterStandby() {
        return this.to('EnterStandby');
    }
    /**
     * Grants permission to execute the specified policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_ExecutePolicy.html
     */
    toExecutePolicy() {
        return this.to('ExecutePolicy');
    }
    /**
     * Grants permission to move the specified instances out of Standby mode
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_ExitStandby.html
     */
    toExitStandby() {
        return this.to('ExitStandby');
    }
    /**
     * Grants permission to retrieve the forecast data for a predictive scaling policy
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_GetPredictiveScalingForecast.html
     */
    toGetPredictiveScalingForecast() {
        return this.to('GetPredictiveScalingForecast');
    }
    /**
     * Grants permission to create or update a lifecycle hook for the specified Auto Scaling Group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_PutLifecycleHook.html
     */
    toPutLifecycleHook() {
        return this.to('PutLifecycleHook');
    }
    /**
     * Grants permission to configure an Auto Scaling group to send notifications when specified events take place
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_PutNotificationConfiguration.html
     */
    toPutNotificationConfiguration() {
        return this.to('PutNotificationConfiguration');
    }
    /**
     * Grants permission to create or update a policy for an Auto Scaling group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_PutScalingPolicy.html
     */
    toPutScalingPolicy() {
        return this.to('PutScalingPolicy');
    }
    /**
     * Grants permission to create or update a scheduled scaling action for an Auto Scaling group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifMaxSize()
     * - .ifMinSize()
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_PutScheduledUpdateGroupAction.html
     */
    toPutScheduledUpdateGroupAction() {
        return this.to('PutScheduledUpdateGroupAction');
    }
    /**
     * Grants permission to create or update the warm pool associated with the specified Auto Scaling group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_PutWarmPool.html
     */
    toPutWarmPool() {
        return this.to('PutWarmPool');
    }
    /**
     * Grants permission to record a heartbeat for the lifecycle action associated with the specified token or instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_RecordLifecycleActionHeartbeat.html
     */
    toRecordLifecycleActionHeartbeat() {
        return this.to('RecordLifecycleActionHeartbeat');
    }
    /**
     * Grants permission to resume the specified suspended Auto Scaling processes, or all suspended process, for the specified Auto Scaling group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_ResumeProcesses.html
     */
    toResumeProcesses() {
        return this.to('ResumeProcesses');
    }
    /**
     * Grants permission to rollback an instance refresh operation in progress
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_RollbackInstanceRefresh.html
     */
    toRollbackInstanceRefresh() {
        return this.to('RollbackInstanceRefresh');
    }
    /**
     * Grants permission to set the size of the specified Auto Scaling group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_SetDesiredCapacity.html
     */
    toSetDesiredCapacity() {
        return this.to('SetDesiredCapacity');
    }
    /**
     * Grants permission to set the health status of the specified instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_SetInstanceHealth.html
     */
    toSetInstanceHealth() {
        return this.to('SetInstanceHealth');
    }
    /**
     * Grants permission to update the instance protection settings of the specified instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_SetInstanceProtection.html
     */
    toSetInstanceProtection() {
        return this.to('SetInstanceProtection');
    }
    /**
     * Grants permission to start a new instance refresh operation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_StartInstanceRefresh.html
     */
    toStartInstanceRefresh() {
        return this.to('StartInstanceRefresh');
    }
    /**
     * Grants permission to suspend the specified Auto Scaling processes, or all processes, for the specified Auto Scaling group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_SuspendProcesses.html
     */
    toSuspendProcesses() {
        return this.to('SuspendProcesses');
    }
    /**
     * Grants permission to terminate the specified instance and optionally adjust the desired group size
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_TerminateInstanceInAutoScalingGroup.html
     */
    toTerminateInstanceInAutoScalingGroup() {
        return this.to('TerminateInstanceInAutoScalingGroup');
    }
    /**
     * Grants permission to update the configuration for the specified Auto Scaling group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCapacityReservationIds()
     * - .ifCapacityReservationResourceGroupArns()
     * - .ifInstanceTypes()
     * - .ifLaunchConfigurationName()
     * - .ifLaunchTemplateVersionSpecified()
     * - .ifMaxSize()
     * - .ifMinSize()
     * - .ifVPCZoneIdentifiers()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_UpdateAutoScalingGroup.html
     */
    toUpdateAutoScalingGroup() {
        return this.to('UpdateAutoScalingGroup');
    }
    /**
     * Adds a resource of type autoScalingGroup to the statement
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-resources
     *
     * @param groupId - Identifier for the groupId.
     * @param groupFriendlyName - Identifier for the groupFriendlyName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifResourceTag()
     * - .ifAwsResourceTag()
     */
    onAutoScalingGroup(groupId, groupFriendlyName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:autoscaling:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:autoScalingGroup:${groupId}:autoScalingGroupName/${groupFriendlyName}`);
    }
    /**
     * Adds a resource of type launchConfiguration to the statement
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-resources
     *
     * @param id - Identifier for the id.
     * @param launchConfigurationName - Identifier for the launchConfigurationName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onLaunchConfiguration(id, launchConfigurationName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:autoscaling:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:launchConfiguration:${id}:launchConfigurationName/${launchConfigurationName}`);
    }
    /**
     * Filters access based on the Capacity Reservation IDs
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateAutoScalingGroup()
     * - .toUpdateAutoScalingGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCapacityReservationIds(value, operator) {
        return this.if(`CapacityReservationIds`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the ARN of a Capacity Reservation resource group
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateAutoScalingGroup()
     * - .toUpdateAutoScalingGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCapacityReservationResourceGroupArns(value, operator) {
        return this.if(`CapacityReservationResourceGroupArns`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the AMI ID for the launch configuration
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateLaunchConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifImageId(value, operator) {
        return this.if(`ImageId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the instance type for the launch configuration
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateLaunchConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInstanceType(value, operator) {
        return this.if(`InstanceType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the instance types present as overrides to a launch template for a mixed instances policy. Use it to qualify which instance types can be explicitly defined in the policy
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateAutoScalingGroup()
     * - .toUpdateAutoScalingGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInstanceTypes(value, operator) {
        return this.if(`InstanceTypes`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the name of a launch configuration
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateAutoScalingGroup()
     * - .toUpdateAutoScalingGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLaunchConfigurationName(value, operator) {
        return this.if(`LaunchConfigurationName`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on whether users can specify any version of a launch template or only the Latest or Default version
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateAutoScalingGroup()
     * - .toUpdateAutoScalingGroup()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifLaunchTemplateVersionSpecified(value) {
        return this.if(`LaunchTemplateVersionSpecified`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access based on the name of the load balancer
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toAttachLoadBalancers()
     * - .toCreateAutoScalingGroup()
     * - .toDetachLoadBalancers()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLoadBalancerNames(value, operator) {
        return this.if(`LoadBalancerNames`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the maximum scaling size in the request
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateAutoScalingGroup()
     * - .toPutScheduledUpdateGroupAction()
     * - .toUpdateAutoScalingGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifMaxSize(value, operator) {
        return this.if(`MaxSize`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access based on whether the HTTP endpoint is enabled for the instance metadata service
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateLaunchConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifMetadataHttpEndpoint(value, operator) {
        return this.if(`MetadataHttpEndpoint`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the allowed number of hops when calling the instance metadata service
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateLaunchConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifMetadataHttpPutResponseHopLimit(value, operator) {
        return this.if(`MetadataHttpPutResponseHopLimit`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access based on whether tokens are required when calling the instance metadata service (optional or required)
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateLaunchConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifMetadataHttpTokens(value, operator) {
        return this.if(`MetadataHttpTokens`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the minimum scaling size in the request
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateAutoScalingGroup()
     * - .toPutScheduledUpdateGroupAction()
     * - .toUpdateAutoScalingGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifMinSize(value, operator) {
        return this.if(`MinSize`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access based on the tags associated with the resource
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toAttachInstances()
     * - .toAttachLoadBalancerTargetGroups()
     * - .toAttachLoadBalancers()
     * - .toAttachTrafficSources()
     * - .toBatchDeleteScheduledAction()
     * - .toBatchPutScheduledUpdateGroupAction()
     * - .toCancelInstanceRefresh()
     * - .toCompleteLifecycleAction()
     * - .toCreateAutoScalingGroup()
     * - .toCreateOrUpdateTags()
     * - .toDeleteAutoScalingGroup()
     * - .toDeleteLifecycleHook()
     * - .toDeleteNotificationConfiguration()
     * - .toDeletePolicy()
     * - .toDeleteScheduledAction()
     * - .toDeleteTags()
     * - .toDeleteWarmPool()
     * - .toDetachInstances()
     * - .toDetachLoadBalancerTargetGroups()
     * - .toDetachLoadBalancers()
     * - .toDetachTrafficSources()
     * - .toDisableMetricsCollection()
     * - .toEnableMetricsCollection()
     * - .toEnterStandby()
     * - .toExecutePolicy()
     * - .toExitStandby()
     * - .toPutLifecycleHook()
     * - .toPutNotificationConfiguration()
     * - .toPutScalingPolicy()
     * - .toPutScheduledUpdateGroupAction()
     * - .toPutWarmPool()
     * - .toRecordLifecycleActionHeartbeat()
     * - .toResumeProcesses()
     * - .toRollbackInstanceRefresh()
     * - .toSetDesiredCapacity()
     * - .toSetInstanceHealth()
     * - .toSetInstanceProtection()
     * - .toStartInstanceRefresh()
     * - .toSuspendProcesses()
     * - .toTerminateInstanceInAutoScalingGroup()
     * - .toUpdateAutoScalingGroup()
     *
     * Applies to resource types:
     * - autoScalingGroup
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTag(tagKey, value, operator) {
        return this.if(`ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the price for Spot Instances for the launch configuration
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateLaunchConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifSpotPrice(value, operator) {
        return this.if(`SpotPrice`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access based on the ARN of a target group
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toAttachLoadBalancerTargetGroups()
     * - .toCreateAutoScalingGroup()
     * - .toDetachLoadBalancerTargetGroups()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifTargetGroupARNs(value, operator) {
        return this.if(`TargetGroupARNs`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access based on the identifiers of the traffic sources
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toAttachTrafficSources()
     * - .toCreateAutoScalingGroup()
     * - .toDetachTrafficSources()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTrafficSourceIdentifiers(value, operator) {
        return this.if(`TrafficSourceIdentifiers`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the identifier of a VPC zone
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateAutoScalingGroup()
     * - .toUpdateAutoScalingGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVPCZoneIdentifiers(value, operator) {
        return this.if(`VPCZoneIdentifiers`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateAutoScalingGroup()
     * - .toCreateOrUpdateTags()
     * - .toDeleteTags()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the tags associated with the resource
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toAttachInstances()
     * - .toAttachLoadBalancerTargetGroups()
     * - .toAttachLoadBalancers()
     * - .toAttachTrafficSources()
     * - .toBatchDeleteScheduledAction()
     * - .toBatchPutScheduledUpdateGroupAction()
     * - .toCancelInstanceRefresh()
     * - .toCompleteLifecycleAction()
     * - .toCreateAutoScalingGroup()
     * - .toCreateOrUpdateTags()
     * - .toDeleteAutoScalingGroup()
     * - .toDeleteLifecycleHook()
     * - .toDeleteNotificationConfiguration()
     * - .toDeletePolicy()
     * - .toDeleteScheduledAction()
     * - .toDeleteTags()
     * - .toDeleteWarmPool()
     * - .toDetachInstances()
     * - .toDetachLoadBalancerTargetGroups()
     * - .toDetachLoadBalancers()
     * - .toDetachTrafficSources()
     * - .toDisableMetricsCollection()
     * - .toEnableMetricsCollection()
     * - .toEnterStandby()
     * - .toExecutePolicy()
     * - .toExitStandby()
     * - .toPutLifecycleHook()
     * - .toPutNotificationConfiguration()
     * - .toPutScalingPolicy()
     * - .toPutScheduledUpdateGroupAction()
     * - .toPutWarmPool()
     * - .toRecordLifecycleActionHeartbeat()
     * - .toResumeProcesses()
     * - .toRollbackInstanceRefresh()
     * - .toSetDesiredCapacity()
     * - .toSetInstanceHealth()
     * - .toSetInstanceProtection()
     * - .toStartInstanceRefresh()
     * - .toSuspendProcesses()
     * - .toTerminateInstanceInAutoScalingGroup()
     * - .toUpdateAutoScalingGroup()
     *
     * Applies to resource types:
     * - autoScalingGroup
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateAutoScalingGroup()
     * - .toCreateOrUpdateTags()
     * - .toDeleteTags()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [autoscaling](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonec2autoscaling.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'autoscaling';
        this.accessLevelList = {
            Write: [
                'AttachInstances',
                'AttachLoadBalancerTargetGroups',
                'AttachLoadBalancers',
                'AttachTrafficSources',
                'BatchDeleteScheduledAction',
                'BatchPutScheduledUpdateGroupAction',
                'CancelInstanceRefresh',
                'CompleteLifecycleAction',
                'CreateAutoScalingGroup',
                'CreateLaunchConfiguration',
                'DeleteAutoScalingGroup',
                'DeleteLaunchConfiguration',
                'DeleteLifecycleHook',
                'DeleteNotificationConfiguration',
                'DeletePolicy',
                'DeleteScheduledAction',
                'DeleteWarmPool',
                'DetachInstances',
                'DetachLoadBalancerTargetGroups',
                'DetachLoadBalancers',
                'DetachTrafficSources',
                'DisableMetricsCollection',
                'EnableMetricsCollection',
                'EnterStandby',
                'ExecutePolicy',
                'ExitStandby',
                'PutLifecycleHook',
                'PutNotificationConfiguration',
                'PutScalingPolicy',
                'PutScheduledUpdateGroupAction',
                'PutWarmPool',
                'RecordLifecycleActionHeartbeat',
                'ResumeProcesses',
                'RollbackInstanceRefresh',
                'SetDesiredCapacity',
                'SetInstanceHealth',
                'SetInstanceProtection',
                'StartInstanceRefresh',
                'SuspendProcesses',
                'TerminateInstanceInAutoScalingGroup',
                'UpdateAutoScalingGroup'
            ],
            Tagging: [
                'CreateOrUpdateTags',
                'DeleteTags'
            ],
            List: [
                'DescribeAccountLimits',
                'DescribeAdjustmentTypes',
                'DescribeAutoScalingGroups',
                'DescribeAutoScalingInstances',
                'DescribeAutoScalingNotificationTypes',
                'DescribeInstanceRefreshes',
                'DescribeLaunchConfigurations',
                'DescribeLifecycleHookTypes',
                'DescribeLifecycleHooks',
                'DescribeLoadBalancerTargetGroups',
                'DescribeLoadBalancers',
                'DescribeMetricCollectionTypes',
                'DescribeNotificationConfigurations',
                'DescribePolicies',
                'DescribeScalingActivities',
                'DescribeScalingProcessTypes',
                'DescribeScheduledActions',
                'DescribeTerminationPolicyTypes',
                'DescribeTrafficSources',
                'DescribeWarmPool',
                'GetPredictiveScalingForecast'
            ],
            Read: [
                'DescribeTags'
            ]
        };
    }
}
exports.Autoscaling = Autoscaling;
//# sourceMappingURL=data:application/json;base64,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