"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Imagebuilder = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [imagebuilder](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonec2imagebuilder.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Imagebuilder extends shared_1.PolicyStatement {
    /**
     * Grants permission to cancel an image creation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_CancelImageCreation.html
     */
    toCancelImageCreation() {
        return this.to('CancelImageCreation');
    }
    /**
     * Grants permission to cancel a lifecycle execution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_CancelLifecycleExecution.html
     */
    toCancelLifecycleExecution() {
        return this.to('CancelLifecycleExecution');
    }
    /**
     * Grants permission to create a new component
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - imagebuilder:TagResource
     * - kms:Encrypt
     * - kms:GenerateDataKey
     * - kms:GenerateDataKeyWithoutPlaintext
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_CreateComponent.html
     */
    toCreateComponent() {
        return this.to('CreateComponent');
    }
    /**
     * Grants permission to create a new Container Recipe
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ecr:DescribeImages
     * - ecr:DescribeRepositories
     * - iam:CreateServiceLinkedRole
     * - imagebuilder:GetComponent
     * - imagebuilder:GetImage
     * - imagebuilder:TagResource
     * - kms:Encrypt
     * - kms:GenerateDataKey
     * - kms:GenerateDataKeyWithoutPlaintext
     * - ssm:GetParameter
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_CreateContainerRecipe.html
     */
    toCreateContainerRecipe() {
        return this.to('CreateContainerRecipe');
    }
    /**
     * Grants permission to create a new distribution configuration
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - imagebuilder:TagResource
     * - ssm:GetParameter
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_CreateDistributionConfiguration.html
     */
    toCreateDistributionConfiguration() {
        return this.to('CreateDistributionConfiguration');
    }
    /**
     * Grants permission to create a new image
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - iam:PassRole
     * - imagebuilder:GetContainerRecipe
     * - imagebuilder:GetDistributionConfiguration
     * - imagebuilder:GetImageRecipe
     * - imagebuilder:GetInfrastructureConfiguration
     * - imagebuilder:GetWorkflow
     * - imagebuilder:TagResource
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_CreateImage.html
     */
    toCreateImage() {
        return this.to('CreateImage');
    }
    /**
     * Grants permission to create a new image pipeline
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - iam:PassRole
     * - imagebuilder:GetContainerRecipe
     * - imagebuilder:GetDistributionConfiguration
     * - imagebuilder:GetImageRecipe
     * - imagebuilder:GetInfrastructureConfiguration
     * - imagebuilder:GetWorkflow
     * - imagebuilder:TagResource
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_CreateImagePipeline.html
     */
    toCreateImagePipeline() {
        return this.to('CreateImagePipeline');
    }
    /**
     * Grants permission to create a new Image Recipe
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DescribeImages
     * - iam:CreateServiceLinkedRole
     * - imagebuilder:GetComponent
     * - imagebuilder:GetImage
     * - imagebuilder:TagResource
     * - ssm:GetParameter
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_CreateImageRecipe.html
     */
    toCreateImageRecipe() {
        return this.to('CreateImageRecipe');
    }
    /**
     * Grants permission to create a new infrastructure configuration
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - iam:PassRole
     * - imagebuilder:TagResource
     * - sns:Publish
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_CreateInfrastructureConfiguration.html
     */
    toCreateInfrastructureConfiguration() {
        return this.to('CreateInfrastructureConfiguration');
    }
    /**
     * Grants permission to create a new lifecycle policy
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     * - imagebuilder:TagResource
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_CreateLifecyclePolicy.html
     */
    toCreateLifecyclePolicy() {
        return this.to('CreateLifecyclePolicy');
    }
    /**
     * Grants permission to create a new workflow
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - imagebuilder:TagResource
     * - kms:Encrypt
     * - kms:GenerateDataKey
     * - kms:GenerateDataKeyWithoutPlaintext
     * - s3:GetObject
     * - s3:ListBucket
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_CreateWorkflow.html
     */
    toCreateWorkflow() {
        return this.to('CreateWorkflow');
    }
    /**
     * Grants permission to delete a component
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_DeleteComponent.html
     */
    toDeleteComponent() {
        return this.to('DeleteComponent');
    }
    /**
     * Grants permission to delete a container recipe
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_DeleteContainerRecipe.html
     */
    toDeleteContainerRecipe() {
        return this.to('DeleteContainerRecipe');
    }
    /**
     * Grants permission to delete a distribution configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_DeleteDistributionConfiguration.html
     */
    toDeleteDistributionConfiguration() {
        return this.to('DeleteDistributionConfiguration');
    }
    /**
     * Grants permission to delete an image
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_DeleteImage.html
     */
    toDeleteImage() {
        return this.to('DeleteImage');
    }
    /**
     * Grants permission to delete an image pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_DeleteImagePipeline.html
     */
    toDeleteImagePipeline() {
        return this.to('DeleteImagePipeline');
    }
    /**
     * Grants permission to delete an image recipe
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_DeleteImageRecipe.html
     */
    toDeleteImageRecipe() {
        return this.to('DeleteImageRecipe');
    }
    /**
     * Grants permission to delete an infrastructure configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_DeleteInfrastructureConfiguration.html
     */
    toDeleteInfrastructureConfiguration() {
        return this.to('DeleteInfrastructureConfiguration');
    }
    /**
     * Grants permission to delete a lifecycle policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_DeleteLifecyclePolicy.html
     */
    toDeleteLifecyclePolicy() {
        return this.to('DeleteLifecyclePolicy');
    }
    /**
     * Grants permission to delete a workflow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_DeleteWorkflow.html
     */
    toDeleteWorkflow() {
        return this.to('DeleteWorkflow');
    }
    /**
     * Grants permission to view details about a component
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_GetComponent.html
     */
    toGetComponent() {
        return this.to('GetComponent');
    }
    /**
     * Grants permission to view the resource policy associated with a component
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_GetComponentPolicy.html
     */
    toGetComponentPolicy() {
        return this.to('GetComponentPolicy');
    }
    /**
     * Grants permission to view details about a container recipe
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_GetContainerRecipe.html
     */
    toGetContainerRecipe() {
        return this.to('GetContainerRecipe');
    }
    /**
     * Grants permission to view the resource policy associated with a container recipe
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_GetContainerRecipePolicy.html
     */
    toGetContainerRecipePolicy() {
        return this.to('GetContainerRecipePolicy');
    }
    /**
     * Grants permission to view details about a distribution configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_GetDistributionConfiguration.html
     */
    toGetDistributionConfiguration() {
        return this.to('GetDistributionConfiguration');
    }
    /**
     * Grants permission to view details about an image
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_GetImage.html
     */
    toGetImage() {
        return this.to('GetImage');
    }
    /**
     * Grants permission to view details about an image pipeline
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_GetImagePipeline.html
     */
    toGetImagePipeline() {
        return this.to('GetImagePipeline');
    }
    /**
     * Grants permission to view the resource policy associated with an image
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_GetImagePolicy.html
     */
    toGetImagePolicy() {
        return this.to('GetImagePolicy');
    }
    /**
     * Grants permission to view details about an image recipe
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_GetImageRecipe.html
     */
    toGetImageRecipe() {
        return this.to('GetImageRecipe');
    }
    /**
     * Grants permission to view the resource policy associated with an image recipe
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_GetImageRecipePolicy.html
     */
    toGetImageRecipePolicy() {
        return this.to('GetImageRecipePolicy');
    }
    /**
     * Grants permission to view details about an infrastructure configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_GetInfrastructureConfiguration.html
     */
    toGetInfrastructureConfiguration() {
        return this.to('GetInfrastructureConfiguration');
    }
    /**
     * Grants permission to view details about a lifecycle execution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_GetLifecycleExecution.html
     */
    toGetLifecycleExecution() {
        return this.to('GetLifecycleExecution');
    }
    /**
     * Grants permission to view details about a lifecycle policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_GetLifecyclePolicy.html
     */
    toGetLifecyclePolicy() {
        return this.to('GetLifecyclePolicy');
    }
    /**
     * Grants permission to retrieve Marketplace provided resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_GetMarketplaceResource.html
     */
    toGetMarketplaceResource() {
        return this.to('GetMarketplaceResource');
    }
    /**
     * Grants permission to view details about a workflow
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_GetWorkflow.html
     */
    toGetWorkflow() {
        return this.to('GetWorkflow');
    }
    /**
     * Grants permission to view details about a workflow execution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_GetWorkflowExecution.html
     */
    toGetWorkflowExecution() {
        return this.to('GetWorkflowExecution');
    }
    /**
     * Grants permission to view details about a workflow step execution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_GetWorkflowStepExecution.html
     */
    toGetWorkflowStepExecution() {
        return this.to('GetWorkflowStepExecution');
    }
    /**
     * Grants permission to import a new component
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - imagebuilder:TagResource
     * - kms:Encrypt
     * - kms:GenerateDataKey
     * - kms:GenerateDataKeyWithoutPlaintext
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ImportComponent.html
     */
    toImportComponent() {
        return this.to('ImportComponent');
    }
    /**
     * Grants permission to import a disk image
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - iam:PassRole
     * - imagebuilder:GetInfrastructureConfiguration
     * - imagebuilder:GetWorkflow
     * - imagebuilder:TagResource
     * - s3:GetObject
     * - s3:ListBucket
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ImportDiskImage.html
     */
    toImportDiskImage() {
        return this.to('ImportDiskImage');
    }
    /**
     * Grants permission to import an image
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DescribeImages
     * - ec2:DescribeImportImageTasks
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ImportVmImage.html
     */
    toImportVmImage() {
        return this.to('ImportVmImage');
    }
    /**
     * Grants permission to list the component build versions in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListComponentBuildVersions.html
     */
    toListComponentBuildVersions() {
        return this.to('ListComponentBuildVersions');
    }
    /**
     * Grants permission to list the component versions owned by or shared with your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListComponents.html
     */
    toListComponents() {
        return this.to('ListComponents');
    }
    /**
     * Grants permission to list the container recipes owned by or shared with your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListContainerRecipes.html
     */
    toListContainerRecipes() {
        return this.to('ListContainerRecipes');
    }
    /**
     * Grants permission to list the distribution configurations in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListDistributionConfigurations.html
     */
    toListDistributionConfigurations() {
        return this.to('ListDistributionConfigurations');
    }
    /**
     * Grants permission to list the image build versions in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListImageBuildVersions.html
     */
    toListImageBuildVersions() {
        return this.to('ListImageBuildVersions');
    }
    /**
     * Grants permission to return a list of packages installed on the specified image
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListImagePackages.html
     */
    toListImagePackages() {
        return this.to('ListImagePackages');
    }
    /**
     * Grants permission to return a list of images created by the specified pipeline
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListImagePipelineImages.html
     */
    toListImagePipelineImages() {
        return this.to('ListImagePipelineImages');
    }
    /**
     * Grants permission to list the image pipelines in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListImagePipelines.html
     */
    toListImagePipelines() {
        return this.to('ListImagePipelines');
    }
    /**
     * Grants permission to list the image recipes owned by or shared with your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListImageRecipes.html
     */
    toListImageRecipes() {
        return this.to('ListImageRecipes');
    }
    /**
     * Grants permission to list aggregations on the image scan findings in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListImageScanFindingAggregations.html
     */
    toListImageScanFindingAggregations() {
        return this.to('ListImageScanFindingAggregations');
    }
    /**
     * Grants permission to list the image scan findings for the images in your account
     *
     * Access Level: List
     *
     * Dependent actions:
     * - inspector2:ListFindings
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListImageScanFindings.html
     */
    toListImageScanFindings() {
        return this.to('ListImageScanFindings');
    }
    /**
     * Grants permission to list the image versions owned by or shared with your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListImages.html
     */
    toListImages() {
        return this.to('ListImages');
    }
    /**
     * Grants permission to list the infrastructure configurations in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListInfrastructureConfigurations.html
     */
    toListInfrastructureConfigurations() {
        return this.to('ListInfrastructureConfigurations');
    }
    /**
     * Grants permission to list resources for the specified lifecycle execution
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListLifecycleExecutionResources.html
     */
    toListLifecycleExecutionResources() {
        return this.to('ListLifecycleExecutionResources');
    }
    /**
     * Grants permission to list lifecycle executions for the specified resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListLifecycleExecutions.html
     */
    toListLifecycleExecutions() {
        return this.to('ListLifecycleExecutions');
    }
    /**
     * Grants permission to list the lifecycle policies in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListLifecyclePolicies.html
     */
    toListLifecyclePolicies() {
        return this.to('ListLifecyclePolicies');
    }
    /**
     * Grants permission to list tags for an Image Builder resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list waiting workflow steps for the caller account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListWaitingWorkflowSteps.html
     */
    toListWaitingWorkflowSteps() {
        return this.to('ListWaitingWorkflowSteps');
    }
    /**
     * Grants permission to list the workflow build versions in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListWorkflowBuildVersions.html
     */
    toListWorkflowBuildVersions() {
        return this.to('ListWorkflowBuildVersions');
    }
    /**
     * Grants permission to list workflow executions for the specified image
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListWorkflowExecutions.html
     */
    toListWorkflowExecutions() {
        return this.to('ListWorkflowExecutions');
    }
    /**
     * Grants permission to list workflow step executions for the specified workflow
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListWorkflowStepExecutions.html
     */
    toListWorkflowStepExecutions() {
        return this.to('ListWorkflowStepExecutions');
    }
    /**
     * Grants permission to list the workflow versions owned by or shared with your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ListWorkflows.html
     */
    toListWorkflows() {
        return this.to('ListWorkflows');
    }
    /**
     * Grants permission to set the resource policy associated with a component
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_PutComponentPolicy.html
     */
    toPutComponentPolicy() {
        return this.to('PutComponentPolicy');
    }
    /**
     * Grants permission to set the resource policy associated with a container recipe
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_PutContainerRecipePolicy.html
     */
    toPutContainerRecipePolicy() {
        return this.to('PutContainerRecipePolicy');
    }
    /**
     * Grants permission to set the resource policy associated with an image
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_PutImagePolicy.html
     */
    toPutImagePolicy() {
        return this.to('PutImagePolicy');
    }
    /**
     * Grants permission to set the resource policy associated with an image recipe
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_PutImageRecipePolicy.html
     */
    toPutImageRecipePolicy() {
        return this.to('PutImageRecipePolicy');
    }
    /**
     * Grants permission to send an action to a workflow step
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_SendWorkflowStepAction.html
     */
    toSendWorkflowStepAction() {
        return this.to('SendWorkflowStepAction');
    }
    /**
     * Grants permission to create a new image from a pipeline
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - imagebuilder:GetImagePipeline
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_StartImagePipelineExecution.html
     */
    toStartImagePipelineExecution() {
        return this.to('StartImagePipelineExecution');
    }
    /**
     * Grants permission to start a state update for the specified resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_StartResourceStateUpdate.html
     */
    toStartResourceStateUpdate() {
        return this.to('StartResourceStateUpdate');
    }
    /**
     * Grants permission to tag an Image Builder resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag an Image Builder resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an existing distribution configuration
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ssm:GetParameter
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_UpdateDistributionConfiguration.html
     */
    toUpdateDistributionConfiguration() {
        return this.to('UpdateDistributionConfiguration');
    }
    /**
     * Grants permission to update an existing image pipeline
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - iam:PassRole
     * - imagebuilder:GetContainerRecipe
     * - imagebuilder:GetDistributionConfiguration
     * - imagebuilder:GetImageRecipe
     * - imagebuilder:GetInfrastructureConfiguration
     * - imagebuilder:GetWorkflow
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_UpdateImagePipeline.html
     */
    toUpdateImagePipeline() {
        return this.to('UpdateImagePipeline');
    }
    /**
     * Grants permission to update an existing infrastructure configuration
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     * - sns:Publish
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_UpdateInfrastructureConfiguration.html
     */
    toUpdateInfrastructureConfiguration() {
        return this.to('UpdateInfrastructureConfiguration');
    }
    /**
     * Grants permission to update an existing lifecycle policy
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_UpdateLifecyclePolicy.html
     */
    toUpdateLifecyclePolicy() {
        return this.to('UpdateLifecyclePolicy');
    }
    /**
     * Adds a resource of type component to the statement
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_Component.html
     *
     * @param componentName - Identifier for the componentName.
     * @param componentVersion - Identifier for the componentVersion.
     * @param componentBuildVersion - Identifier for the componentBuildVersion.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onComponent(componentName, componentVersion, componentBuildVersion, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:imagebuilder:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:component/${componentName}/${componentVersion}/${componentBuildVersion}`);
    }
    /**
     * Adds a resource of type componentVersion to the statement
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ComponentVersion.html
     *
     * @param componentName - Identifier for the componentName.
     * @param componentVersion - Identifier for the componentVersion.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onComponentVersion(componentName, componentVersion, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:imagebuilder:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:component/${componentName}/${componentVersion}`);
    }
    /**
     * Adds a resource of type distributionConfiguration to the statement
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_DistributionConfiguration.html
     *
     * @param distributionConfigurationName - Identifier for the distributionConfigurationName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDistributionConfiguration(distributionConfigurationName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:imagebuilder:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:distribution-configuration/${distributionConfigurationName}`);
    }
    /**
     * Adds a resource of type image to the statement
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_Image.html
     *
     * @param imageName - Identifier for the imageName.
     * @param imageVersion - Identifier for the imageVersion.
     * @param imageBuildVersion - Identifier for the imageBuildVersion.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onImage(imageName, imageVersion, imageBuildVersion, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:imagebuilder:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:image/${imageName}/${imageVersion}/${imageBuildVersion}`);
    }
    /**
     * Adds a resource of type imageVersion to the statement
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ImageVersion.html
     *
     * @param imageName - Identifier for the imageName.
     * @param imageVersion - Identifier for the imageVersion.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onImageVersion(imageName, imageVersion, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:imagebuilder:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:image/${imageName}/${imageVersion}`);
    }
    /**
     * Adds a resource of type imageRecipe to the statement
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ImageRecipe.html
     *
     * @param imageRecipeName - Identifier for the imageRecipeName.
     * @param imageRecipeVersion - Identifier for the imageRecipeVersion.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onImageRecipe(imageRecipeName, imageRecipeVersion, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:imagebuilder:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:image-recipe/${imageRecipeName}/${imageRecipeVersion}`);
    }
    /**
     * Adds a resource of type containerRecipe to the statement
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ContainerRecipe.html
     *
     * @param containerRecipeName - Identifier for the containerRecipeName.
     * @param containerRecipeVersion - Identifier for the containerRecipeVersion.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onContainerRecipe(containerRecipeName, containerRecipeVersion, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:imagebuilder:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:container-recipe/${containerRecipeName}/${containerRecipeVersion}`);
    }
    /**
     * Adds a resource of type imagePipeline to the statement
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_ImagePipeline.html
     *
     * @param imagePipelineName - Identifier for the imagePipelineName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onImagePipeline(imagePipelineName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:imagebuilder:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:image-pipeline/${imagePipelineName}`);
    }
    /**
     * Adds a resource of type infrastructureConfiguration to the statement
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_InfrastructureConfiguration.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onInfrastructureConfiguration(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:imagebuilder:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:infrastructure-configuration/${resourceId}`);
    }
    /**
     * Adds a resource of type kmsKey to the statement
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#master_keys
     *
     * @param keyId - Identifier for the keyId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onKmsKey(keyId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:kms:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:key/${keyId}`);
    }
    /**
     * Adds a resource of type lifecycleExecution to the statement
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_LifecycleExecution.html
     *
     * @param lifecycleExecutionId - Identifier for the lifecycleExecutionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onLifecycleExecution(lifecycleExecutionId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:imagebuilder:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:lifecycle-execution/${lifecycleExecutionId}`);
    }
    /**
     * Adds a resource of type lifecyclePolicy to the statement
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_LifecyclePolicy.html
     *
     * @param lifecyclePolicyName - Identifier for the lifecyclePolicyName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onLifecyclePolicy(lifecyclePolicyName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:imagebuilder:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:lifecycle-policy/${lifecyclePolicyName}`);
    }
    /**
     * Adds a resource of type workflow to the statement
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_Workflow.html
     *
     * @param workflowType - Identifier for the workflowType.
     * @param workflowName - Identifier for the workflowName.
     * @param workflowVersion - Identifier for the workflowVersion.
     * @param workflowBuildVersion - Identifier for the workflowBuildVersion.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorkflow(workflowType, workflowName, workflowVersion, workflowBuildVersion, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:imagebuilder:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workflow/${workflowType}/${workflowName}/${workflowVersion}/${workflowBuildVersion}`);
    }
    /**
     * Adds a resource of type workflowVersion to the statement
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_WorkflowVersion.html
     *
     * @param workflowType - Identifier for the workflowType.
     * @param workflowName - Identifier for the workflowName.
     * @param workflowVersion - Identifier for the workflowVersion.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorkflowVersion(workflowType, workflowName, workflowVersion, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:imagebuilder:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workflow/${workflowType}/${workflowName}/${workflowVersion}`);
    }
    /**
     * Adds a resource of type workflowExecution to the statement
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_WorkflowExecutionMetadata.html
     *
     * @param workflowExecutionId - Identifier for the workflowExecutionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onWorkflowExecution(workflowExecutionId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:imagebuilder:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workflow-execution/${workflowExecutionId}`);
    }
    /**
     * Adds a resource of type workflowStepExecution to the statement
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/APIReference/API_WorkflowStepMetadata.html
     *
     * @param workflowStepExecutionId - Identifier for the workflowStepExecutionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onWorkflowStepExecution(workflowStepExecutionId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:imagebuilder:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workflow-step-execution/${workflowStepExecutionId}`);
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateComponent()
     * - .toCreateContainerRecipe()
     * - .toCreateDistributionConfiguration()
     * - .toCreateImage()
     * - .toCreateImagePipeline()
     * - .toCreateImageRecipe()
     * - .toCreateInfrastructureConfiguration()
     * - .toCreateLifecyclePolicy()
     * - .toCreateWorkflow()
     * - .toImportComponent()
     * - .toImportDiskImage()
     * - .toImportVmImage()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toGetImage()
     * - .toListImagePackages()
     * - .toListTagsForResource()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateInfrastructureConfiguration()
     *
     * Applies to resource types:
     * - component
     * - componentVersion
     * - distributionConfiguration
     * - image
     * - imageVersion
     * - imageRecipe
     * - containerRecipe
     * - imagePipeline
     * - infrastructureConfiguration
     * - lifecyclePolicy
     * - workflow
     * - workflowVersion
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateComponent()
     * - .toCreateContainerRecipe()
     * - .toCreateDistributionConfiguration()
     * - .toCreateImage()
     * - .toCreateImagePipeline()
     * - .toCreateImageRecipe()
     * - .toCreateInfrastructureConfiguration()
     * - .toCreateLifecyclePolicy()
     * - .toCreateWorkflow()
     * - .toImportComponent()
     * - .toImportDiskImage()
     * - .toImportVmImage()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag key-value pairs attached to the resource created by Image Builder
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/userguide/security_iam_service-with-iam.html#image-builder-security-createdresourcetag
     *
     * Applies to actions:
     * - .toCreateInfrastructureConfiguration()
     * - .toUpdateInfrastructureConfiguration()
     *
     * @param key The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCreatedResourceTag(key, value, operator) {
        return this.if(`CreatedResourceTag/${key}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/userguide/security_iam_service-with-iam.html#image-builder-security-createdresourcetagkeys
     *
     * Applies to actions:
     * - .toCreateInfrastructureConfiguration()
     * - .toUpdateInfrastructureConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCreatedResourceTagKeys(value, operator) {
        return this.if(`CreatedResourceTagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the EC2 Instance Metadata HTTP Token Requirement specified in the request
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/userguide/security_iam_service-with-iam.html#image-builder-security-ec2metadatahttptokens
     *
     * Applies to actions:
     * - .toCreateInfrastructureConfiguration()
     * - .toUpdateInfrastructureConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEc2MetadataHttpTokens(value, operator) {
        return this.if(`Ec2MetadataHttpTokens`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the Lifecycle Policy Resource Type specified in the request
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/userguide/security_iam_service-with-iam.html#image-builder-security-ec2metadatahttptokens
     *
     * Applies to actions:
     * - .toCreateLifecyclePolicy()
     * - .toUpdateLifecyclePolicy()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLifecyclePolicyResourceType(value, operator) {
        return this.if(`LifecyclePolicyResourceType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the SNS Topic Arn in the request to which terminal state notifications will be published
     *
     * https://docs.aws.amazon.com/imagebuilder/latest/userguide/security_iam_service-with-iam.html#image-builder-security-statustopicarn
     *
     * Applies to actions:
     * - .toCreateInfrastructureConfiguration()
     * - .toUpdateInfrastructureConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifStatusTopicArn(value, operator) {
        return this.if(`StatusTopicArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Statement provider for service [imagebuilder](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonec2imagebuilder.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'imagebuilder';
        this.accessLevelList = {
            Write: [
                'CancelImageCreation',
                'CancelLifecycleExecution',
                'CreateComponent',
                'CreateContainerRecipe',
                'CreateDistributionConfiguration',
                'CreateImage',
                'CreateImagePipeline',
                'CreateImageRecipe',
                'CreateInfrastructureConfiguration',
                'CreateLifecyclePolicy',
                'CreateWorkflow',
                'DeleteComponent',
                'DeleteContainerRecipe',
                'DeleteDistributionConfiguration',
                'DeleteImage',
                'DeleteImagePipeline',
                'DeleteImageRecipe',
                'DeleteInfrastructureConfiguration',
                'DeleteLifecyclePolicy',
                'DeleteWorkflow',
                'ImportComponent',
                'ImportDiskImage',
                'ImportVmImage',
                'SendWorkflowStepAction',
                'StartImagePipelineExecution',
                'StartResourceStateUpdate',
                'UpdateDistributionConfiguration',
                'UpdateImagePipeline',
                'UpdateInfrastructureConfiguration',
                'UpdateLifecyclePolicy'
            ],
            Read: [
                'GetComponent',
                'GetComponentPolicy',
                'GetContainerRecipe',
                'GetContainerRecipePolicy',
                'GetDistributionConfiguration',
                'GetImage',
                'GetImagePipeline',
                'GetImagePolicy',
                'GetImageRecipe',
                'GetImageRecipePolicy',
                'GetInfrastructureConfiguration',
                'GetLifecycleExecution',
                'GetLifecyclePolicy',
                'GetMarketplaceResource',
                'GetWorkflow',
                'GetWorkflowExecution',
                'GetWorkflowStepExecution',
                'ListTagsForResource'
            ],
            List: [
                'ListComponentBuildVersions',
                'ListComponents',
                'ListContainerRecipes',
                'ListDistributionConfigurations',
                'ListImageBuildVersions',
                'ListImagePackages',
                'ListImagePipelineImages',
                'ListImagePipelines',
                'ListImageRecipes',
                'ListImageScanFindingAggregations',
                'ListImageScanFindings',
                'ListImages',
                'ListInfrastructureConfigurations',
                'ListLifecycleExecutionResources',
                'ListLifecycleExecutions',
                'ListLifecyclePolicies',
                'ListWaitingWorkflowSteps',
                'ListWorkflowBuildVersions',
                'ListWorkflowExecutions',
                'ListWorkflowStepExecutions',
                'ListWorkflows'
            ],
            'Permissions management': [
                'PutComponentPolicy',
                'PutContainerRecipePolicy',
                'PutImagePolicy',
                'PutImageRecipePolicy'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Imagebuilder = Imagebuilder;
//# sourceMappingURL=data:application/json;base64,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