"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Elasticmapreduce = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [elasticmapreduce](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonelasticmapreduce.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Elasticmapreduce extends shared_1.PolicyStatement {
    /**
     * Grants permission to add an instance fleet to a running cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_AddInstanceFleet.html
     */
    toAddInstanceFleet() {
        return this.to('AddInstanceFleet');
    }
    /**
     * Grants permission to add instance groups to a running cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_AddInstanceGroups.html
     */
    toAddInstanceGroups() {
        return this.to('AddInstanceGroups');
    }
    /**
     * Grants permission to add new steps to a running cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifExecutionRoleArn()
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_AddJobFlowSteps.html
     */
    toAddJobFlowSteps() {
        return this.to('AddJobFlowSteps');
    }
    /**
     * Grants permission to add tags to an Amazon EMR resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifRequestTag()
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_AddTags.html
     */
    toAddTags() {
        return this.to('AddTags');
    }
    /**
     * Grants permission to attach an EMR notebook to a compute engine
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-working-with.html
     */
    toAttachEditor() {
        return this.to('AttachEditor');
    }
    /**
     * Grants permission to cancel a pending step or steps in a running cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_CancelSteps.html
     */
    toCancelSteps() {
        return this.to('CancelSteps');
    }
    /**
     * Grants permission to create an EMR notebook
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifRequestTag()
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-create.html
     */
    toCreateEditor() {
        return this.to('CreateEditor');
    }
    /**
     * Grants permission to create a persistent application history server
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-studio-debug.html
     */
    toCreatePersistentAppUI() {
        return this.to('CreatePersistentAppUI');
    }
    /**
     * Grants permission to create an EMR notebook repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks.html#emr-managed-notebooks-editor
     */
    toCreateRepository() {
        return this.to('CreateRepository');
    }
    /**
     * Grants permission to create a security configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_CreateSecurityConfiguration.html
     */
    toCreateSecurityConfiguration() {
        return this.to('CreateSecurityConfiguration');
    }
    /**
     * Grants permission to create an EMR Studio
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifRequestTag()
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-studio.html
     */
    toCreateStudio() {
        return this.to('CreateStudio');
    }
    /**
     * Grants permission to launch an EMR Studio using IAM authentication mode
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-studio.html
     */
    toCreateStudioPresignedUrl() {
        return this.to('CreateStudioPresignedUrl');
    }
    /**
     * Grants permission to create an EMR Studio session mapping
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-studio.html
     */
    toCreateStudioSessionMapping() {
        return this.to('CreateStudioSessionMapping');
    }
    /**
     * Grants permission to delete an EMR notebook
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks.html#emr-managed-notebooks-deleting
     */
    toDeleteEditor() {
        return this.to('DeleteEditor');
    }
    /**
     * Grants permission to delete an EMR notebook repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks.html#emr-managed-notebooks-editor
     */
    toDeleteRepository() {
        return this.to('DeleteRepository');
    }
    /**
     * Grants permission to delete a security configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_DeleteSecurityConfiguration.html
     */
    toDeleteSecurityConfiguration() {
        return this.to('DeleteSecurityConfiguration');
    }
    /**
     * Grants permission to delete an EMR Studio
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-studio.html
     */
    toDeleteStudio() {
        return this.to('DeleteStudio');
    }
    /**
     * Grants permission to delete an EMR Studio session mapping
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-studio.html
     */
    toDeleteStudioSessionMapping() {
        return this.to('DeleteStudioSessionMapping');
    }
    /**
     * Grants permission to block an identity from opening a collaborative workspace
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-working-with.html
     */
    toDeleteWorkspaceAccess() {
        return this.to('DeleteWorkspaceAccess');
    }
    /**
     * Grants permission to get details about a cluster, including status, hardware and software configuration, VPC settings, and so on
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_DescribeCluster.html
     */
    toDescribeCluster() {
        return this.to('DescribeCluster');
    }
    /**
     * Grants permission to view information about a notebook, including status, user, role, tags, location, and more
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-working-with.html
     */
    toDescribeEditor() {
        return this.to('DescribeEditor');
    }
    /**
     * Grants permission to describe details of clusters (job flows). This API is deprecated and will eventually be removed. We recommend you use ListClusters, DescribeCluster, ListSteps, ListInstanceGroups and ListBootstrapActions instead
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_DescribeJobFlows.html
     */
    toDescribeJobFlows() {
        return this.to('DescribeJobFlows');
    }
    /**
     * Grants permission to view information about a notebook execution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-headless.html
     */
    toDescribeNotebookExecution() {
        return this.to('DescribeNotebookExecution');
    }
    /**
     * Grants permission to describe a persistent application history server
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-studio-debug.html
     */
    toDescribePersistentAppUI() {
        return this.to('DescribePersistentAppUI');
    }
    /**
     * Grants permission to view information about an EMR release, such as which applications are supported
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_DescribeReleaseLabel.html
     */
    toDescribeReleaseLabel() {
        return this.to('DescribeReleaseLabel');
    }
    /**
     * Grants permission to describe an EMR notebook repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks.html#emr-managed-notebooks-editor
     */
    toDescribeRepository() {
        return this.to('DescribeRepository');
    }
    /**
     * Grants permission to get details of a security configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_DescribeSecurityConfiguration.html
     */
    toDescribeSecurityConfiguration() {
        return this.to('DescribeSecurityConfiguration');
    }
    /**
     * Grants permission to get details about a cluster step
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_DescribeStep.html
     */
    toDescribeStep() {
        return this.to('DescribeStep');
    }
    /**
     * Grants permission to view information about an EMR Studio
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-studio.html
     */
    toDescribeStudio() {
        return this.to('DescribeStudio');
    }
    /**
     * Grants permission to detach an EMR notebook from a compute engine
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-working-with.html
     */
    toDetachEditor() {
        return this.to('DetachEditor');
    }
    /**
     * Grants permission to retrieve the auto-termination policy associated with a cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_GetAutoTerminationPolicy.html
     */
    toGetAutoTerminationPolicy() {
        return this.to('GetAutoTerminationPolicy');
    }
    /**
     * Grants permission to retrieve the EMR block public access configuration for the AWS account in the Region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_GetBlockPublicAccessConfiguration.html
     */
    toGetBlockPublicAccessConfiguration() {
        return this.to('GetBlockPublicAccessConfiguration');
    }
    /**
     * Grants permission to retrieve HTTP basic credentials associated with a given execution IAM Role for a fine-grained access control enabled EMR Cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifExecutionRoleArn()
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-steps-runtime-roles.html
     */
    toGetClusterSessionCredentials() {
        return this.to('GetClusterSessionCredentials');
    }
    /**
     * Grants permission to retrieve the managed scaling policy associated with a cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_GetManagedScalingPolicy.html
     */
    toGetManagedScalingPolicy() {
        return this.to('GetManagedScalingPolicy');
    }
    /**
     * Grants permission to get a presigned URL for an application history server running on the cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-studio-debug.html
     */
    toGetOnClusterAppUIPresignedURL() {
        return this.to('GetOnClusterAppUIPresignedURL');
    }
    /**
     * Grants permission to get a presigned URL for a persistent application history server
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifExecutionRoleArn()
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-studio-debug.html
     */
    toGetPersistentAppUIPresignedURL() {
        return this.to('GetPersistentAppUIPresignedURL');
    }
    /**
     * Grants permission to view information about an EMR Studio session mapping
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-studio.html
     */
    toGetStudioSessionMapping() {
        return this.to('GetStudioSessionMapping');
    }
    /**
     * Grants permission to link an EMR notebook repository to EMR notebooks
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks.html#emr-managed-notebooks-editor
     */
    toLinkRepository() {
        return this.to('LinkRepository');
    }
    /**
     * Grants permission to get details about the bootstrap actions associated with a cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_ListBootstrapActions.html
     */
    toListBootstrapActions() {
        return this.to('ListBootstrapActions');
    }
    /**
     * Grants permission to get the status of accessible clusters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_ListClusters.html
     */
    toListClusters() {
        return this.to('ListClusters');
    }
    /**
     * Grants permission to list summary information for accessible EMR notebooks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-working-with.html
     */
    toListEditors() {
        return this.to('ListEditors');
    }
    /**
     * Grants permission to get details of instance fleets in a cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_ListInstanceFleets.html
     */
    toListInstanceFleets() {
        return this.to('ListInstanceFleets');
    }
    /**
     * Grants permission to get details of instance groups in a cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_ListInstanceGroups.html
     */
    toListInstanceGroups() {
        return this.to('ListInstanceGroups');
    }
    /**
     * Grants permission to get details about the Amazon EC2 instances in a cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_ListInstances.html
     */
    toListInstances() {
        return this.to('ListInstances');
    }
    /**
     * Grants permission to list summary information for notebook executions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-headless.html
     */
    toListNotebookExecutions() {
        return this.to('ListNotebookExecutions');
    }
    /**
     * Grants permission to list and filter the available EMR releases in the current region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_ListReleaseLabels.html
     */
    toListReleaseLabels() {
        return this.to('ListReleaseLabels');
    }
    /**
     * Grants permission to list existing EMR notebook repositories
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks.html#emr-managed-notebooks-editor
     */
    toListRepositories() {
        return this.to('ListRepositories');
    }
    /**
     * Grants permission to list available security configurations in this account by name, along with creation dates and times
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_ListSecurityConfigurations.html
     */
    toListSecurityConfigurations() {
        return this.to('ListSecurityConfigurations');
    }
    /**
     * Grants permission to list steps associated with a cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_ListSteps.html
     */
    toListSteps() {
        return this.to('ListSteps');
    }
    /**
     * Grants permission to list summary information about EMR Studio session mappings
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-studio.html
     */
    toListStudioSessionMappings() {
        return this.to('ListStudioSessionMappings');
    }
    /**
     * Grants permission to list summary information about EMR Studios
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-studio.html
     */
    toListStudios() {
        return this.to('ListStudios');
    }
    /**
     * Grants permission to list the Amazon EC2 instance types that an Amazon EMR release supports
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_ListSupportedInstanceTypes.html
     */
    toListSupportedInstanceTypes() {
        return this.to('ListSupportedInstanceTypes');
    }
    /**
     * Grants permission to list identities that are granted access to a workspace
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-working-with.html
     */
    toListWorkspaceAccessIdentities() {
        return this.to('ListWorkspaceAccessIdentities');
    }
    /**
     * Grants permission to change cluster settings such as number of steps that can be executed concurrently for a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_ModifyCluster.html
     */
    toModifyCluster() {
        return this.to('ModifyCluster');
    }
    /**
     * Grants permission to change the target On-Demand and target Spot capacities for a instance fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_ModifyInstanceFleet.html
     */
    toModifyInstanceFleet() {
        return this.to('ModifyInstanceFleet');
    }
    /**
     * Grants permission to change the number and configuration of EC2 instances for an instance group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_ModifyInstanceGroups.html
     */
    toModifyInstanceGroups() {
        return this.to('ModifyInstanceGroups');
    }
    /**
     * Grants permission to launch the Jupyter notebook editor for an EMR notebook from within the console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks.html#emr-managed-notebooks-editor
     */
    toOpenEditorInConsole() {
        return this.to('OpenEditorInConsole');
    }
    /**
     * Grants permission to create or update an automatic scaling policy for a core instance group or task instance group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_PutAutoScalingPolicy.html
     */
    toPutAutoScalingPolicy() {
        return this.to('PutAutoScalingPolicy');
    }
    /**
     * Grants permission to create or update the auto-termination policy associated with a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_PutAutoTerminationPolicy.html
     */
    toPutAutoTerminationPolicy() {
        return this.to('PutAutoTerminationPolicy');
    }
    /**
     * Grants permission to create or update the EMR block public access configuration for the AWS account in the Region
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_PutBlockPublicAccessConfiguration.html
     */
    toPutBlockPublicAccessConfiguration() {
        return this.to('PutBlockPublicAccessConfiguration');
    }
    /**
     * Grants permission to create or update the managed scaling policy associated with a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_PutManagedScalingPolicy.html
     */
    toPutManagedScalingPolicy() {
        return this.to('PutManagedScalingPolicy');
    }
    /**
     * Grants permission to allow an identity to open a collaborative workspace
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-working-with.html
     */
    toPutWorkspaceAccess() {
        return this.to('PutWorkspaceAccess');
    }
    /**
     * Grants permission to remove an automatic scaling policy from an instance group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_RemoveAutoScalingPolicy.html
     */
    toRemoveAutoScalingPolicy() {
        return this.to('RemoveAutoScalingPolicy');
    }
    /**
     * Grants permission to remove the auto-termination policy associated with a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_RemoveAutoTerminationPolicy.html
     */
    toRemoveAutoTerminationPolicy() {
        return this.to('RemoveAutoTerminationPolicy');
    }
    /**
     * Grants permission to remove the managed scaling policy associated with a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_RemoveManagedScalingPolicy.html
     */
    toRemoveManagedScalingPolicy() {
        return this.to('RemoveManagedScalingPolicy');
    }
    /**
     * Grants permission to remove tags from an Amazon EMR resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_RemoveTags.html
     */
    toRemoveTags() {
        return this.to('RemoveTags');
    }
    /**
     * Grants permission to create and launch a cluster (job flow)
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifRequestTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_RunJobFlow.html
     */
    toRunJobFlow() {
        return this.to('RunJobFlow');
    }
    /**
     * Grants permission to add and remove auto terminate after step execution for a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_SetKeepJobFlowAliveWhenNoSteps.html
     */
    toSetKeepJobFlowAliveWhenNoSteps() {
        return this.to('SetKeepJobFlowAliveWhenNoSteps');
    }
    /**
     * Grants permission to add and remove termination protection for a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_SetTerminationProtection.html
     */
    toSetTerminationProtection() {
        return this.to('SetTerminationProtection');
    }
    /**
     * Grants permission to enable or disable unhealthy node replacement for a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_SetUnhealthyNodeReplacement.html
     */
    toSetUnhealthyNodeReplacement() {
        return this.to('SetUnhealthyNodeReplacement');
    }
    /**
     * Grants permission to set whether all AWS Identity and Access Management (IAM) users in the AWS account can view a cluster. This API is deprecated and your cluster may be visible to all users in your account. To restrict cluster access using an IAM policy, see AWS Identity and Access Management for Amazon EMR (https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-plan-access-iam.html)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_SetVisibleToAllUsers.html
     */
    toSetVisibleToAllUsers() {
        return this.to('SetVisibleToAllUsers');
    }
    /**
     * Grants permission to start an EMR notebook
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-working-with.html
     */
    toStartEditor() {
        return this.to('StartEditor');
    }
    /**
     * Grants permission to start an EMR notebook execution
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifRequestTag()
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-headless.html
     */
    toStartNotebookExecution() {
        return this.to('StartNotebookExecution');
    }
    /**
     * Grants permission to shut down an EMR notebook
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks.html
     */
    toStopEditor() {
        return this.to('StopEditor');
    }
    /**
     * Grants permission to stop notebook execution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-headless.html
     */
    toStopNotebookExecution() {
        return this.to('StopNotebookExecution');
    }
    /**
     * Grants permission to terminate a cluster (job flow)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_TerminateJobFlows.html
     */
    toTerminateJobFlows() {
        return this.to('TerminateJobFlows');
    }
    /**
     * Grants permission to unlink an EMR notebook repository from EMR notebooks
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks.html#emr-managed-notebooks-editor
     */
    toUnlinkRepository() {
        return this.to('UnlinkRepository');
    }
    /**
     * Grants permission to update an EMR notebook
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-working-with.html
     */
    toUpdateEditor() {
        return this.to('UpdateEditor');
    }
    /**
     * Grants permission to update an EMR notebook repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks.html#emr-managed-notebooks-editor
     */
    toUpdateRepository() {
        return this.to('UpdateRepository');
    }
    /**
     * Grants permission to update information about an EMR Studio
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-studio.html
     */
    toUpdateStudio() {
        return this.to('UpdateStudio');
    }
    /**
     * Grants permission to update an EMR Studio session mapping
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-studio.html
     */
    toUpdateStudioSessionMapping() {
        return this.to('UpdateStudioSessionMapping');
    }
    /**
     * Grants permission to use the EMR console to view events from all clusters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonelasticmapreduce.html
     */
    toViewEventsFromAllClustersInConsole() {
        return this.to('ViewEventsFromAllClustersInConsole');
    }
    /**
     * Adds a resource of type cluster to the statement
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-overview.html
     *
     * @param clusterId - Identifier for the clusterId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onCluster(clusterId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elasticmapreduce:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:cluster/${clusterId}`);
    }
    /**
     * Adds a resource of type editor to the statement
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks.html
     *
     * @param editorId - Identifier for the editorId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onEditor(editorId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elasticmapreduce:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:editor/${editorId}`);
    }
    /**
     * Adds a resource of type notebook-execution to the statement
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-headless.html
     *
     * @param notebookExecutionId - Identifier for the notebookExecutionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onNotebookExecution(notebookExecutionId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elasticmapreduce:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:notebook-execution/${notebookExecutionId}`);
    }
    /**
     * Adds a resource of type studio to the statement
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-studio.html
     *
     * @param studioId - Identifier for the studioId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onStudio(studioId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elasticmapreduce:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:studio/${studioId}`);
    }
    /**
     * Filters access by whether the tag and value pair is provided with the action
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-plan-access-iam.html#emr-fine-grained-cluster-access
     *
     * Applies to actions:
     * - .toAddTags()
     * - .toCreateEditor()
     * - .toCreateStudio()
     * - .toRunJobFlow()
     * - .toStartNotebookExecution()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag and value pair associated with an Amazon EMR resource
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-plan-access-iam.html#emr-fine-grained-cluster-access
     *
     * Applies to resource types:
     * - cluster
     * - editor
     * - notebook-execution
     * - studio
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by whether the tag keys are provided with the action regardless of tag value
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-plan-access-iam.html#emr-fine-grained-cluster-access
     *
     * Applies to actions:
     * - .toAddTags()
     * - .toCreateEditor()
     * - .toCreateStudio()
     * - .toRemoveTags()
     * - .toRunJobFlow()
     * - .toStartNotebookExecution()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by whether the execution role ARN is provided with the action
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-plan-access-iam.html#emr-security
     *
     * Applies to actions:
     * - .toAddJobFlowSteps()
     * - .toGetClusterSessionCredentials()
     * - .toGetPersistentAppUIPresignedURL()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifExecutionRoleArn(value, operator) {
        return this.if(`ExecutionRoleArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by whether the tag and value pair is provided with the action
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-plan-access-iam.html#emr-fine-grained-cluster-access
     *
     * Applies to actions:
     * - .toAddTags()
     * - .toCreateEditor()
     * - .toCreateStudio()
     * - .toRunJobFlow()
     * - .toStartNotebookExecution()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestTag(tagKey, value, operator) {
        return this.if(`RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag and value pair associated with an Amazon EMR resource
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-plan-access-iam.html#emr-fine-grained-cluster-access
     *
     * Applies to resource types:
     * - cluster
     * - editor
     * - notebook-execution
     * - studio
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTag(tagKey, value, operator) {
        return this.if(`ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [elasticmapreduce](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonelasticmapreduce.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'elasticmapreduce';
        this.accessLevelList = {
            Write: [
                'AddInstanceFleet',
                'AddInstanceGroups',
                'AddJobFlowSteps',
                'AttachEditor',
                'CancelSteps',
                'CreateEditor',
                'CreatePersistentAppUI',
                'CreateRepository',
                'CreateSecurityConfiguration',
                'CreateStudio',
                'CreateStudioPresignedUrl',
                'CreateStudioSessionMapping',
                'DeleteEditor',
                'DeleteRepository',
                'DeleteSecurityConfiguration',
                'DeleteStudio',
                'DeleteStudioSessionMapping',
                'DetachEditor',
                'GetClusterSessionCredentials',
                'GetOnClusterAppUIPresignedURL',
                'GetPersistentAppUIPresignedURL',
                'LinkRepository',
                'ModifyCluster',
                'ModifyInstanceFleet',
                'ModifyInstanceGroups',
                'OpenEditorInConsole',
                'PutAutoScalingPolicy',
                'PutAutoTerminationPolicy',
                'PutManagedScalingPolicy',
                'RemoveAutoScalingPolicy',
                'RemoveAutoTerminationPolicy',
                'RemoveManagedScalingPolicy',
                'RunJobFlow',
                'SetKeepJobFlowAliveWhenNoSteps',
                'SetTerminationProtection',
                'SetUnhealthyNodeReplacement',
                'SetVisibleToAllUsers',
                'StartEditor',
                'StartNotebookExecution',
                'StopEditor',
                'StopNotebookExecution',
                'TerminateJobFlows',
                'UnlinkRepository',
                'UpdateEditor',
                'UpdateRepository',
                'UpdateStudio',
                'UpdateStudioSessionMapping'
            ],
            Tagging: [
                'AddTags',
                'RemoveTags'
            ],
            'Permissions management': [
                'DeleteWorkspaceAccess',
                'PutBlockPublicAccessConfiguration',
                'PutWorkspaceAccess'
            ],
            Read: [
                'DescribeCluster',
                'DescribeEditor',
                'DescribeJobFlows',
                'DescribeNotebookExecution',
                'DescribePersistentAppUI',
                'DescribeReleaseLabel',
                'DescribeRepository',
                'DescribeSecurityConfiguration',
                'DescribeStep',
                'DescribeStudio',
                'GetAutoTerminationPolicy',
                'GetBlockPublicAccessConfiguration',
                'GetManagedScalingPolicy',
                'GetStudioSessionMapping',
                'ListBootstrapActions',
                'ListInstanceFleets',
                'ListInstanceGroups',
                'ListInstances',
                'ListSteps'
            ],
            List: [
                'ListClusters',
                'ListEditors',
                'ListNotebookExecutions',
                'ListReleaseLabels',
                'ListRepositories',
                'ListSecurityConfigurations',
                'ListStudioSessionMappings',
                'ListStudios',
                'ListSupportedInstanceTypes',
                'ListWorkspaceAccessIdentities',
                'ViewEventsFromAllClustersInConsole'
            ]
        };
    }
}
exports.Elasticmapreduce = Elasticmapreduce;
//# sourceMappingURL=data:application/json;base64,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