"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SocialMessaging = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [social-messaging](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsendusermessagingsocial.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class SocialMessaging extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate a WhatsApp Business Account with your AWS account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_AssociateWhatsAppBusinessAccount.html
     */
    toAssociateWhatsAppBusinessAccount() {
        return this.to('AssociateWhatsAppBusinessAccount');
    }
    /**
     * Grants permission to delete a media object from WhatsApp
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_DeleteWhatsAppMessageMedia.html
     */
    toDeleteWhatsAppMessageMedia() {
        return this.to('DeleteWhatsAppMessageMedia');
    }
    /**
     * Grants permission to disassociate a WhatsApp Business Account from your AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_DisassociateWhatsAppBusinessAccount.html
     */
    toDisassociateWhatsAppBusinessAccount() {
        return this.to('DisassociateWhatsAppBusinessAccount');
    }
    /**
     * Grants permission to view the details of a WhatsApp Business Account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_GetLinkedWhatsAppBusinessAccount.html
     */
    toGetLinkedWhatsAppBusinessAccount() {
        return this.to('GetLinkedWhatsAppBusinessAccount');
    }
    /**
     * Grants permission to view the details of a phone number
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_GetLinkedWhatsAppBusinessAccountPhoneNumber.html
     */
    toGetLinkedWhatsAppBusinessAccountPhoneNumber() {
        return this.to('GetLinkedWhatsAppBusinessAccountPhoneNumber');
    }
    /**
     * Grants permission to get a media object from WhatsApp
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_GetWhatsAppMessageMedia.html
     */
    toGetWhatsAppMessageMedia() {
        return this.to('GetWhatsAppMessageMedia');
    }
    /**
     * Grants permission to view all of your WhatsApp Business Accounts
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_ListLinkedWhatsAppBusinessAccounts.html
     */
    toListLinkedWhatsAppBusinessAccounts() {
        return this.to('ListLinkedWhatsAppBusinessAccounts');
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to upload a media object to WhatsApp
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_PostWhatsAppMessageMedia.html
     */
    toPostWhatsAppMessageMedia() {
        return this.to('PostWhatsAppMessageMedia');
    }
    /**
     * Grants permission to update a WhatsApp Business Accounts event destination
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_PutWhatsAppBusinessAccountEventDestinations.html
     */
    toPutWhatsAppBusinessAccountEventDestinations() {
        return this.to('PutWhatsAppBusinessAccountEventDestinations');
    }
    /**
     * Grants permission to send a message through WhatsApp
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_SendWhatsAppMessage.html
     */
    toSendWhatsAppMessage() {
        return this.to('SendWhatsAppMessage');
    }
    /**
     * Grants permission to add a tag to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove a tag from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Adds a resource of type phone-number-id to the statement
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_WhatsAppPhoneNumberDetail.html
     *
     * @param originationPhoneNumberId - Identifier for the originationPhoneNumberId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPhoneNumberId(originationPhoneNumberId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:social-messaging:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:phone-number-id/${originationPhoneNumberId}`);
    }
    /**
     * Adds a resource of type waba to the statement
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_LinkedWhatsAppBusinessAccountSummary.html
     *
     * @param wabaId - Identifier for the wabaId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWaba(wabaId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:social-messaging:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:waba/${wabaId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toAssociateWhatsAppBusinessAccount()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toAssociateWhatsAppBusinessAccount()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * Applies to resource types:
     * - phone-number-id
     * - waba
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toAssociateWhatsAppBusinessAccount()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [social-messaging](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsendusermessagingsocial.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'social-messaging';
        this.accessLevelList = {
            Write: [
                'AssociateWhatsAppBusinessAccount',
                'DeleteWhatsAppMessageMedia',
                'DisassociateWhatsAppBusinessAccount',
                'GetWhatsAppMessageMedia',
                'PostWhatsAppMessageMedia',
                'PutWhatsAppBusinessAccountEventDestinations',
                'SendWhatsAppMessage'
            ],
            Read: [
                'GetLinkedWhatsAppBusinessAccount',
                'GetLinkedWhatsAppBusinessAccountPhoneNumber',
                'ListTagsForResource'
            ],
            List: [
                'ListLinkedWhatsAppBusinessAccounts'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.SocialMessaging = SocialMessaging;
//# sourceMappingURL=data:application/json;base64,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