"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Finspace = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [finspace](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonfinspace.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Finspace extends shared_1.PolicyStatement {
    /**
     * Grants permission to connect to a kdb cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/finspace/latest/userguide/interacting-with-kdb-clusters.html
     */
    toConnectKxCluster() {
        return this.to('ConnectKxCluster');
    }
    /**
     * Grants permission to create a FinSpace environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_CreateEnvironment.html
     */
    toCreateEnvironment() {
        return this.to('CreateEnvironment');
    }
    /**
     * Grants permission to create a changeset for a kdb database
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_CreateKxChangeset.html
     */
    toCreateKxChangeset() {
        return this.to('CreateKxChangeset');
    }
    /**
     * Grants permission to create a cluster in a managed kdb environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - ec2:DescribeSubnets
     * - finspace:MountKxDatabase
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_CreateKxCluster.html
     */
    toCreateKxCluster() {
        return this.to('CreateKxCluster');
    }
    /**
     * Grants permission to create a kdb database in a managed kdb environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_CreateKxDatabase.html
     */
    toCreateKxDatabase() {
        return this.to('CreateKxDatabase');
    }
    /**
     * Grants permission to create a dataview in a managed kdb environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_CreateKxDataview.html
     */
    toCreateKxDataview() {
        return this.to('CreateKxDataview');
    }
    /**
     * Grants permission to create a managed kdb environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_CreateKxEnvironment.html
     */
    toCreateKxEnvironment() {
        return this.to('CreateKxEnvironment');
    }
    /**
     * Grants permission to create a scaling group in a managed kdb environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_CreateKxScalingGroup.html
     */
    toCreateKxScalingGroup() {
        return this.to('CreateKxScalingGroup');
    }
    /**
     * Grants permission to create a user in a managed kdb environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_CreateKxUser.html
     */
    toCreateKxUser() {
        return this.to('CreateKxUser');
    }
    /**
     * Grants permission to create a volume in a managed kdb environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_CreateKxVolume.html
     */
    toCreateKxVolume() {
        return this.to('CreateKxVolume');
    }
    /**
     * Grants permission to create a FinSpace user
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/finspace/latest/userguide/finspace-what-is.html
     */
    toCreateUser() {
        return this.to('CreateUser');
    }
    /**
     * Grants permission to delete a FinSpace environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_DeleteEnvironment.html
     */
    toDeleteEnvironment() {
        return this.to('DeleteEnvironment');
    }
    /**
     * Grants permission to delete a kdb cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_DeleteKxCluster.html
     */
    toDeleteKxCluster() {
        return this.to('DeleteKxCluster');
    }
    /**
     * Grants permission to delete a node from a kdb cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_DeleteKxClusterNode.html
     */
    toDeleteKxClusterNode() {
        return this.to('DeleteKxClusterNode');
    }
    /**
     * Grants permission to delete a kdb database
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_DeleteKxDatabase.html
     */
    toDeleteKxDatabase() {
        return this.to('DeleteKxDatabase');
    }
    /**
     * Grants permission to delete a dataview in a managed kdb environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_DeleteKxDataview.html
     */
    toDeleteKxDataview() {
        return this.to('DeleteKxDataview');
    }
    /**
     * Grants permission to delete a managed kdb environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_DeleteKxEnvironment.html
     */
    toDeleteKxEnvironment() {
        return this.to('DeleteKxEnvironment');
    }
    /**
     * Grants permission to delete a scaling group in a managed kdb environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_DeleteKxScalingGroup.html
     */
    toDeleteKxScalingGroup() {
        return this.to('DeleteKxScalingGroup');
    }
    /**
     * Grants permission to delete a kdb user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_DeleteKxUser.html
     */
    toDeleteKxUser() {
        return this.to('DeleteKxUser');
    }
    /**
     * Grants permission to delete a volume in a managed kdb environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_DeleteKxVolume.html
     */
    toDeleteKxVolume() {
        return this.to('DeleteKxVolume');
    }
    /**
     * Grants permission to describe a FinSpace environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_GetEnvironment.html
     */
    toGetEnvironment() {
        return this.to('GetEnvironment');
    }
    /**
     * Grants permission to describe a changeset for a kdb database
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_GetKxChangeset.html
     */
    toGetKxChangeset() {
        return this.to('GetKxChangeset');
    }
    /**
     * Grants permission to describe a cluster in a managed kdb environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_GetKxCluster.html
     */
    toGetKxCluster() {
        return this.to('GetKxCluster');
    }
    /**
     * Grants permission to retrieve a connection string for kdb clusters
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - finspace:ConnectKxCluster
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_GetKxConnectionString.html
     */
    toGetKxConnectionString() {
        return this.to('GetKxConnectionString');
    }
    /**
     * Grants permission to describe a kdb database
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_GetKxDatabase.html
     */
    toGetKxDatabase() {
        return this.to('GetKxDatabase');
    }
    /**
     * Grants permission to describe a databiew in a managed kdb environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_GetKxDataview.html
     */
    toGetKxDataview() {
        return this.to('GetKxDataview');
    }
    /**
     * Grants permission to describe a managed kdb environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_GetKxEnvironment.html
     */
    toGetKxEnvironment() {
        return this.to('GetKxEnvironment');
    }
    /**
     * Grants permission to describe a scaling group in a managed kdb environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_GetKxScalingGroup.html
     */
    toGetKxScalingGroup() {
        return this.to('GetKxScalingGroup');
    }
    /**
     * Grants permission to describe a kdb user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_GetKxUser.html
     */
    toGetKxUser() {
        return this.to('GetKxUser');
    }
    /**
     * Grants permission to describe a volume in a managed kdb environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_GetKxVolume.html
     */
    toGetKxVolume() {
        return this.to('GetKxVolume');
    }
    /**
     * Grants permission to request status of the loading of sample data bundle
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/finspace/latest/userguide/finspace-what-is.html
     */
    toGetLoadSampleDataSetGroupIntoEnvironmentStatus() {
        return this.to('GetLoadSampleDataSetGroupIntoEnvironmentStatus');
    }
    /**
     * Grants permission to describe a FinSpace user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/finspace/latest/userguide/finspace-what-is.html
     */
    toGetUser() {
        return this.to('GetUser');
    }
    /**
     * Grants permission to list FinSpace environments in the AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_ListEnvironments.html
     */
    toListEnvironments() {
        return this.to('ListEnvironments');
    }
    /**
     * Grants permission to list changesets for a kdb database
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_ListKxChangesets.html
     */
    toListKxChangesets() {
        return this.to('ListKxChangesets');
    }
    /**
     * Grants permission to list cluster nodes in a managed kdb environment
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_ListKxClusterNodes.html
     */
    toListKxClusterNodes() {
        return this.to('ListKxClusterNodes');
    }
    /**
     * Grants permission to list clusters in a managed kdb environment
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_ListKxClusters.html
     */
    toListKxClusters() {
        return this.to('ListKxClusters');
    }
    /**
     * Grants permission to list kdb databases in a managed kdb environment
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_ListKxDatabases.html
     */
    toListKxDatabases() {
        return this.to('ListKxDatabases');
    }
    /**
     * Grants permission to list dataviews in a database
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_ListKxDataviews.html
     */
    toListKxDataviews() {
        return this.to('ListKxDataviews');
    }
    /**
     * Grants permission to list managed kdb environments
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_ListKxEnvironments.html
     */
    toListKxEnvironments() {
        return this.to('ListKxEnvironments');
    }
    /**
     * Grants permission to list scaling groups in a managed kdb environment
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_ListKxScalingGroups.html
     */
    toListKxScalingGroups() {
        return this.to('ListKxScalingGroups');
    }
    /**
     * Grants permission to list users in a managed kdb environment
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_ListKxUsers.html
     */
    toListKxUsers() {
        return this.to('ListKxUsers');
    }
    /**
     * Grants permission to list volumes in a managed kdb environment
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_ListKxVolumes.html
     */
    toListKxVolumes() {
        return this.to('ListKxVolumes');
    }
    /**
     * Grants permission to return a list of tags for a resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list FinSpace users in an environment
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/finspace/latest/userguide/finspace-what-is.html
     */
    toListUsers() {
        return this.to('ListUsers');
    }
    /**
     * Grants permission to load sample data bundle into your FinSpace environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/finspace/latest/userguide/finspace-what-is.html
     */
    toLoadSampleDataSetGroupIntoEnvironment() {
        return this.to('LoadSampleDataSetGroupIntoEnvironment');
    }
    /**
     * Grants permission to mount a database to a kdb cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/finspace/latest/userguide/finspace-managed-kdb-db.html
     */
    toMountKxDatabase() {
        return this.to('MountKxDatabase');
    }
    /**
     * Grants permission to reset the password for a FinSpace user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/finspace/latest/userguide/finspace-what-is.html
     */
    toResetUserPassword() {
        return this.to('ResetUserPassword');
    }
    /**
     * Grants permission to tag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a FinSpace environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_UpdateEnvironment.html
     */
    toUpdateEnvironment() {
        return this.to('UpdateEnvironment');
    }
    /**
     * Grants permission to update code configuration for a cluster in a managed kdb environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_UpdateKxClusterCodeConfiguration.html
     */
    toUpdateKxClusterCodeConfiguration() {
        return this.to('UpdateKxClusterCodeConfiguration');
    }
    /**
     * Grants permission to update databases for a cluster in a managed kdb environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_UpdateKxClusterDatabases.html
     */
    toUpdateKxClusterDatabases() {
        return this.to('UpdateKxClusterDatabases');
    }
    /**
     * Grants permission to update a kdb database
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_UpdateKxDatabase.html
     */
    toUpdateKxDatabase() {
        return this.to('UpdateKxDatabase');
    }
    /**
     * Grants permission to update a dataview in a managed kdb environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_UpdateKxDataview.html
     */
    toUpdateKxDataview() {
        return this.to('UpdateKxDataview');
    }
    /**
     * Grants permission to update a managed kdb environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_UpdateKxEnvironment.html
     */
    toUpdateKxEnvironment() {
        return this.to('UpdateKxEnvironment');
    }
    /**
     * Grants permission to update the network for a managed kdb environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_UpdateKxEnvironmentNetwork.html
     */
    toUpdateKxEnvironmentNetwork() {
        return this.to('UpdateKxEnvironmentNetwork');
    }
    /**
     * Grants permission to update a kdb user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_UpdateKxUser.html
     */
    toUpdateKxUser() {
        return this.to('UpdateKxUser');
    }
    /**
     * Grants permission to update a volume in a managed kdb environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/finspace/latest/management-api/API_UpdateKxVolume.html
     */
    toUpdateKxVolume() {
        return this.to('UpdateKxVolume');
    }
    /**
     * Grants permission to update a FinSpace user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/finspace/latest/userguide/finspace-what-is.html
     */
    toUpdateUser() {
        return this.to('UpdateUser');
    }
    /**
     * Adds a resource of type environment to the statement
     *
     * https://docs.aws.amazon.com/finspace/latest/userguide/finspace-example-policies.html
     *
     * @param environmentId - Identifier for the environmentId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEnvironment(environmentId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:finspace:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:environment/${environmentId}`);
    }
    /**
     * Adds a resource of type user to the statement
     *
     * https://docs.aws.amazon.com/finspace/latest/userguide/finspace-example-policies.html
     *
     * @param userId - Identifier for the userId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onUser(userId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:finspace:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:user/${userId}`);
    }
    /**
     * Adds a resource of type kxEnvironment to the statement
     *
     * https://docs.aws.amazon.com/finspace/latest/userguide/finspace-example-policies.html
     *
     * @param environmentId - Identifier for the environmentId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onKxEnvironment(environmentId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:finspace:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:kxEnvironment/${environmentId}`);
    }
    /**
     * Adds a resource of type kxUser to the statement
     *
     * https://docs.aws.amazon.com/finspace/latest/userguide/finspace-example-policies.html
     *
     * @param environmentId - Identifier for the environmentId.
     * @param userName - Identifier for the userName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onKxUser(environmentId, userName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:finspace:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:kxEnvironment/${environmentId}/kxUser/${userName}`);
    }
    /**
     * Adds a resource of type kxCluster to the statement
     *
     * https://docs.aws.amazon.com/finspace/latest/userguide/finspace-example-policies.html
     *
     * @param environmentId - Identifier for the environmentId.
     * @param kxCluster - Identifier for the kxCluster.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onKxCluster(environmentId, kxCluster, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:finspace:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:kxEnvironment/${environmentId}/kxCluster/${kxCluster}`);
    }
    /**
     * Adds a resource of type kxDatabase to the statement
     *
     * https://docs.aws.amazon.com/finspace/latest/userguide/finspace-example-policies.html
     *
     * @param environmentId - Identifier for the environmentId.
     * @param kxDatabase - Identifier for the kxDatabase.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onKxDatabase(environmentId, kxDatabase, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:finspace:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:kxEnvironment/${environmentId}/kxDatabase/${kxDatabase}`);
    }
    /**
     * Adds a resource of type kxScalingGroup to the statement
     *
     * https://docs.aws.amazon.com/finspace/latest/userguide/finspace-example-policies.html
     *
     * @param environmentId - Identifier for the environmentId.
     * @param kxScalingGroup - Identifier for the kxScalingGroup.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onKxScalingGroup(environmentId, kxScalingGroup, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:finspace:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:kxEnvironment/${environmentId}/kxScalingGroup/${kxScalingGroup}`);
    }
    /**
     * Adds a resource of type kxDataview to the statement
     *
     * https://docs.aws.amazon.com/finspace/latest/userguide/finspace-example-policies.html
     *
     * @param environmentId - Identifier for the environmentId.
     * @param kxDatabase - Identifier for the kxDatabase.
     * @param kxDataview - Identifier for the kxDataview.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onKxDataview(environmentId, kxDatabase, kxDataview, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:finspace:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:kxEnvironment/${environmentId}/kxDatabase/${kxDatabase}/kxDataview/${kxDataview}`);
    }
    /**
     * Adds a resource of type kxVolume to the statement
     *
     * https://docs.aws.amazon.com/finspace/latest/userguide/finspace-example-policies.html
     *
     * @param environmentId - Identifier for the environmentId.
     * @param kxVolume - Identifier for the kxVolume.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onKxVolume(environmentId, kxVolume, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:finspace:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:kxEnvironment/${environmentId}/kxVolume/${kxVolume}`);
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateEnvironment()
     * - .toCreateKxCluster()
     * - .toCreateKxDatabase()
     * - .toCreateKxDataview()
     * - .toCreateKxEnvironment()
     * - .toCreateKxScalingGroup()
     * - .toCreateKxUser()
     * - .toCreateKxVolume()
     * - .toCreateUser()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - environment
     * - user
     * - kxEnvironment
     * - kxUser
     * - kxCluster
     * - kxDatabase
     * - kxScalingGroup
     * - kxDataview
     * - kxVolume
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateEnvironment()
     * - .toCreateKxCluster()
     * - .toCreateKxDatabase()
     * - .toCreateKxDataview()
     * - .toCreateKxEnvironment()
     * - .toCreateKxScalingGroup()
     * - .toCreateKxUser()
     * - .toCreateKxVolume()
     * - .toCreateUser()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [finspace](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonfinspace.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'finspace';
        this.accessLevelList = {
            Write: [
                'ConnectKxCluster',
                'CreateEnvironment',
                'CreateKxChangeset',
                'CreateKxCluster',
                'CreateKxDatabase',
                'CreateKxDataview',
                'CreateKxEnvironment',
                'CreateKxScalingGroup',
                'CreateKxUser',
                'CreateKxVolume',
                'CreateUser',
                'DeleteEnvironment',
                'DeleteKxCluster',
                'DeleteKxClusterNode',
                'DeleteKxDatabase',
                'DeleteKxDataview',
                'DeleteKxEnvironment',
                'DeleteKxScalingGroup',
                'DeleteKxUser',
                'DeleteKxVolume',
                'LoadSampleDataSetGroupIntoEnvironment',
                'MountKxDatabase',
                'ResetUserPassword',
                'UpdateEnvironment',
                'UpdateKxClusterCodeConfiguration',
                'UpdateKxClusterDatabases',
                'UpdateKxDatabase',
                'UpdateKxDataview',
                'UpdateKxEnvironment',
                'UpdateKxEnvironmentNetwork',
                'UpdateKxUser',
                'UpdateKxVolume',
                'UpdateUser'
            ],
            Read: [
                'GetEnvironment',
                'GetKxChangeset',
                'GetKxCluster',
                'GetKxConnectionString',
                'GetKxDatabase',
                'GetKxDataview',
                'GetKxEnvironment',
                'GetKxScalingGroup',
                'GetKxUser',
                'GetKxVolume',
                'GetLoadSampleDataSetGroupIntoEnvironmentStatus',
                'GetUser'
            ],
            List: [
                'ListEnvironments',
                'ListKxChangesets',
                'ListKxClusterNodes',
                'ListKxClusters',
                'ListKxDatabases',
                'ListKxDataviews',
                'ListKxEnvironments',
                'ListKxScalingGroups',
                'ListKxUsers',
                'ListKxVolumes',
                'ListTagsForResource',
                'ListUsers'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Finspace = Finspace;
//# sourceMappingURL=data:application/json;base64,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