"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Fms = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [fms](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsfirewallmanager.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Fms extends shared_1.PolicyStatement {
    /**
     * Grants permission to set the AWS Firewall Manager administrator account and enables the service in all organization accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_AssociateAdminAccount.html
     */
    toAssociateAdminAccount() {
        return this.to('AssociateAdminAccount');
    }
    /**
     * Grants permission to set the Firewall Manager administrator as a tenant administrator of a third-party firewall service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_AssociateThirdPartyFirewall.html
     */
    toAssociateThirdPartyFirewall() {
        return this.to('AssociateThirdPartyFirewall');
    }
    /**
     * Grants permission to associate resources to an AWS Firewall Manager resource set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_BatchAssociateResource.html
     */
    toBatchAssociateResource() {
        return this.to('BatchAssociateResource');
    }
    /**
     * Grants permission to disassociate resources from an AWS Firewall Manager resource set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_BatchDisassociateResource.html
     */
    toBatchDisassociateResource() {
        return this.to('BatchDisassociateResource');
    }
    /**
     * Grants permission to permanently deletes an AWS Firewall Manager applications list
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_DeleteAppsList.html
     */
    toDeleteAppsList() {
        return this.to('DeleteAppsList');
    }
    /**
     * Grants permission to delete an AWS Firewall Manager association with the IAM role and the Amazon Simple Notification Service (SNS) topic that is used to notify the FM administrator about major FM events and errors across the organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_DeleteNotificationChannel.html
     */
    toDeleteNotificationChannel() {
        return this.to('DeleteNotificationChannel');
    }
    /**
     * Grants permission to permanently delete an AWS Firewall Manager policy
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_DeletePolicy.html
     */
    toDeletePolicy() {
        return this.to('DeletePolicy');
    }
    /**
     * Grants permission to permanently deletes an AWS Firewall Manager protocols list
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_DeleteProtocolsList.html
     */
    toDeleteProtocolsList() {
        return this.to('DeleteProtocolsList');
    }
    /**
     * Grants permission to permanently delete an AWS Firewall Manager resource set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_DeleteResourceSet.html
     */
    toDeleteResourceSet() {
        return this.to('DeleteResourceSet');
    }
    /**
     * Grants permission to disassociate the account that has been set as the AWS Firewall Manager administrator account and and disables the service in all organization accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_DisassociateAdminAccount.html
     */
    toDisassociateAdminAccount() {
        return this.to('DisassociateAdminAccount');
    }
    /**
     * Grants permission to disassociate a Firewall Manager administrator from a third-party firewall tenant
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_DisassociateThirdPartyFirewall.html
     */
    toDisassociateThirdPartyFirewall() {
        return this.to('DisassociateThirdPartyFirewall');
    }
    /**
     * Grants permission to return the AWS Organizations account that is associated with AWS Firewall Manager as the AWS Firewall Manager administrator
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetAdminAccount.html
     */
    toGetAdminAccount() {
        return this.to('GetAdminAccount');
    }
    /**
     * Grants permission to return information about the specified account's administrative scope
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetAdminScope.html
     */
    toGetAdminScope() {
        return this.to('GetAdminScope');
    }
    /**
     * Grants permission to return information about the specified AWS Firewall Manager applications list
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetAppsList.html
     */
    toGetAppsList() {
        return this.to('GetAppsList');
    }
    /**
     * Grants permission to retrieve detailed compliance information about the specified member account. Details include resources that are in and out of compliance with the specified policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetComplianceDetail.html
     */
    toGetComplianceDetail() {
        return this.to('GetComplianceDetail');
    }
    /**
     * Grants permission to retrieve information about the Amazon Simple Notification Service (SNS) topic that is used to record AWS Firewall Manager SNS logs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetNotificationChannel.html
     */
    toGetNotificationChannel() {
        return this.to('GetNotificationChannel');
    }
    /**
     * Grants permission to retrieve information about the specified AWS Firewall Manager policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetPolicy.html
     */
    toGetPolicy() {
        return this.to('GetPolicy');
    }
    /**
     * Grants permission to retrieve policy-level attack summary information in the event of a potential DDoS attack
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetProtectionStatus.html
     */
    toGetProtectionStatus() {
        return this.to('GetProtectionStatus');
    }
    /**
     * Grants permission to return information about the specified AWS Firewall Manager protocols list
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetProtocolsList.html
     */
    toGetProtocolsList() {
        return this.to('GetProtocolsList');
    }
    /**
     * Grants permission to retrieve information about the specified AWS Firewall Manager resource set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetResourceSet.html
     */
    toGetResourceSet() {
        return this.to('GetResourceSet');
    }
    /**
     * Grants permission to retrieve the onboarding status of a Firewall Manager administrator account to third-party firewall vendor tenant
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetThirdPartyFirewallAssociationStatus.html
     */
    toGetThirdPartyFirewallAssociationStatus() {
        return this.to('GetThirdPartyFirewallAssociationStatus');
    }
    /**
     * Grants permission to retrieve violations for a resource based on the specified AWS Firewall Manager policy and AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetViolationDetails.html
     */
    toGetViolationDetails() {
        return this.to('GetViolationDetails');
    }
    /**
     * Grants permission to return a AdminAccounts object that lists the Firewall Manager administrators within the organization that are onboarded to Firewall Manager by AssociateAdminAccount
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListAdminAccountsForOrganization.html
     */
    toListAdminAccountsForOrganization() {
        return this.to('ListAdminAccountsForOrganization');
    }
    /**
     * Grants permission to list the accounts that are managing the specified AWS Organizations member account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListAdminsManagingAccount.html
     */
    toListAdminsManagingAccount() {
        return this.to('ListAdminsManagingAccount');
    }
    /**
     * Grants permission to return an array of AppsListDataSummary objects
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListAppsLists.html
     */
    toListAppsLists() {
        return this.to('ListAppsLists');
    }
    /**
     * Grants permission to retrieve an array of PolicyComplianceStatus objects in the response. Use PolicyComplianceStatus to get a summary of which member accounts are protected by the specified policy
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListComplianceStatus.html
     */
    toListComplianceStatus() {
        return this.to('ListComplianceStatus');
    }
    /**
     * Grants permission to retrieve an array of resources in the organization's accounts that are available to be associated with a resource set
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListDiscoveredResources.html
     */
    toListDiscoveredResources() {
        return this.to('ListDiscoveredResources');
    }
    /**
     * Grants permission to retrieve an array of member account ids if the caller is FMS admin account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListMemberAccounts.html
     */
    toListMemberAccounts() {
        return this.to('ListMemberAccounts');
    }
    /**
     * Grants permission to retrieve an array of PolicySummary objects in the response
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListPolicies.html
     */
    toListPolicies() {
        return this.to('ListPolicies');
    }
    /**
     * Grants permission to return an array of ProtocolsListDataSummary objects
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListProtocolsLists.html
     */
    toListProtocolsLists() {
        return this.to('ListProtocolsLists');
    }
    /**
     * Grants permission to retrieve an array of resources that are currently associated to a resource set
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListResourceSetResources.html
     */
    toListResourceSetResources() {
        return this.to('ListResourceSetResources');
    }
    /**
     * Grants permission to retrieve an array of ResourceSetSummary objects
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListResourceSets.html
     */
    toListResourceSets() {
        return this.to('ListResourceSets');
    }
    /**
     * Grants permission to list Tags for a given resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to retrieve a list of all of the third-party firewall policies that are associated with the third-party firewall administrator's account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListThirdPartyFirewallFirewallPolicies.html
     */
    toListThirdPartyFirewallFirewallPolicies() {
        return this.to('ListThirdPartyFirewallFirewallPolicies');
    }
    /**
     * Grants permission to create or update an Firewall Manager administrator account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_PutAdminAccount.html
     */
    toPutAdminAccount() {
        return this.to('PutAdminAccount');
    }
    /**
     * Grants permission to create an AWS Firewall Manager applications list
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_PutAppsList.html
     */
    toPutAppsList() {
        return this.to('PutAppsList');
    }
    /**
     * Grants permission to designate the IAM role and Amazon Simple Notification Service (SNS) topic that AWS Firewall Manager (FM) could use to notify the FM administrator about major FM events and errors across the organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_PutNotificationChannel.html
     */
    toPutNotificationChannel() {
        return this.to('PutNotificationChannel');
    }
    /**
     * Grants permission to create an AWS Firewall Manager policy
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_PutPolicy.html
     */
    toPutPolicy() {
        return this.to('PutPolicy');
    }
    /**
     * Grants permission to creates an AWS Firewall Manager protocols list
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_PutProtocolsList.html
     */
    toPutProtocolsList() {
        return this.to('PutProtocolsList');
    }
    /**
     * Grants permission to create an AWS Firewall Manager resource set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_PutResourceSet.html
     */
    toPutResourceSet() {
        return this.to('PutResourceSet');
    }
    /**
     * Grants permission to add a Tag to a given resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove a Tag from a given resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Adds a resource of type policy to the statement
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_Policy.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPolicy(id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:fms:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:policy/${id}`);
    }
    /**
     * Adds a resource of type applications-list to the statement
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_AppsListData.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onApplicationsList(id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:fms:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:applications-list/${id}`);
    }
    /**
     * Adds a resource of type protocols-list to the statement
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ProtocolsListData.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProtocolsList(id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:fms:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:protocols-list/${id}`);
    }
    /**
     * Adds a resource of type resource-set to the statement
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ResourceSet.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onResourceSet(id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:fms:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:resource-set/${id}`);
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toPutAppsList()
     * - .toPutPolicy()
     * - .toPutProtocolsList()
     * - .toPutResourceSet()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toDeletePolicy()
     * - .toDeleteResourceSet()
     *
     * Applies to resource types:
     * - policy
     * - applications-list
     * - protocols-list
     * - resource-set
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toPutAppsList()
     * - .toPutPolicy()
     * - .toPutProtocolsList()
     * - .toPutResourceSet()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [fms](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsfirewallmanager.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'fms';
        this.accessLevelList = {
            Write: [
                'AssociateAdminAccount',
                'AssociateThirdPartyFirewall',
                'BatchAssociateResource',
                'BatchDisassociateResource',
                'DeleteAppsList',
                'DeleteNotificationChannel',
                'DeletePolicy',
                'DeleteProtocolsList',
                'DeleteResourceSet',
                'DisassociateAdminAccount',
                'DisassociateThirdPartyFirewall',
                'PutAdminAccount',
                'PutAppsList',
                'PutNotificationChannel',
                'PutPolicy',
                'PutProtocolsList',
                'PutResourceSet'
            ],
            Read: [
                'GetAdminAccount',
                'GetAdminScope',
                'GetAppsList',
                'GetComplianceDetail',
                'GetNotificationChannel',
                'GetPolicy',
                'GetProtectionStatus',
                'GetProtocolsList',
                'GetResourceSet',
                'GetThirdPartyFirewallAssociationStatus',
                'GetViolationDetails',
                'ListTagsForResource'
            ],
            List: [
                'ListAdminAccountsForOrganization',
                'ListAdminsManagingAccount',
                'ListAppsLists',
                'ListComplianceStatus',
                'ListDiscoveredResources',
                'ListMemberAccounts',
                'ListPolicies',
                'ListProtocolsLists',
                'ListResourceSetResources',
                'ListResourceSets',
                'ListThirdPartyFirewallFirewallPolicies'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Fms = Fms;
//# sourceMappingURL=data:application/json;base64,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