"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Freertos = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [freertos](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonfreertos.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Freertos extends shared_1.PolicyStatement {
    /**
     * Grants permission to create a software configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toCreateSoftwareConfiguration() {
        return this.to('CreateSoftwareConfiguration');
    }
    /**
     * Grants permission to create a subscription for FreeRTOS extended maintenance plan (EMP)
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-getting-started-emp.html
     */
    toCreateSubscription() {
        return this.to('CreateSubscription');
    }
    /**
     * Grants permission to delete the software configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toDeleteSoftwareConfiguration() {
        return this.to('DeleteSoftwareConfiguration');
    }
    /**
     * Grants permission to describe the hardware platform
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toDescribeHardwarePlatform() {
        return this.to('DescribeHardwarePlatform');
    }
    /**
     * Grants permission to describe the software configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toDescribeSoftwareConfiguration() {
        return this.to('DescribeSoftwareConfiguration');
    }
    /**
     * Grants permission to describes the subscription for FreeRTOS extended maintenance plan (EMP)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-getting-started-emp.html
     */
    toDescribeSubscription() {
        return this.to('DescribeSubscription');
    }
    /**
     * Grants permission to get URL for sotware patch-release, patch-diff and release notes under FreeRTOS extended maintenance plan (EMP)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-getting-started-emp.html
     */
    toGetEmpPatchUrl() {
        return this.to('GetEmpPatchUrl');
    }
    /**
     * Grants permission to get the URL for Amazon FreeRTOS software download
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toGetSoftwareURL() {
        return this.to('GetSoftwareURL');
    }
    /**
     * Grants permission to get the URL for Amazon FreeRTOS software download based on the configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toGetSoftwareURLForConfiguration() {
        return this.to('GetSoftwareURLForConfiguration');
    }
    /**
     * Grants permission to fetch the subscription billing amount for FreeRTOS extended maintenance plan (EMP)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-getting-started-emp.html
     */
    toGetSubscriptionBillingAmount() {
        return this.to('GetSubscriptionBillingAmount');
    }
    /**
     * Grants permission to lists versions of AmazonFreeRTOS
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toListFreeRTOSVersions() {
        return this.to('ListFreeRTOSVersions');
    }
    /**
     * Grants permission to list the hardware platforms
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toListHardwarePlatforms() {
        return this.to('ListHardwarePlatforms');
    }
    /**
     * Grants permission to list the hardware vendors
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toListHardwareVendors() {
        return this.to('ListHardwareVendors');
    }
    /**
     * Grants permission to lists the software configurations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toListSoftwareConfigurations() {
        return this.to('ListSoftwareConfigurations');
    }
    /**
     * Grants permission to list software patches of subscription for FreeRTOS extended maintenance plan (EMP)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-getting-started-emp.html
     */
    toListSoftwarePatches() {
        return this.to('ListSoftwarePatches');
    }
    /**
     * Grants permission to list the subscription emails for FreeRTOS extended maintenance plan (EMP)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-getting-started-emp.html
     */
    toListSubscriptionEmails() {
        return this.to('ListSubscriptionEmails');
    }
    /**
     * Grants permission to list the subscriptions for FreeRTOS extended maintenance plan (EMP)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-getting-started-emp.html
     */
    toListSubscriptions() {
        return this.to('ListSubscriptions');
    }
    /**
     * Grants permission to update list of subscription email address for FreeRTOS extended maintenance plan (EMP)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-getting-started-emp.html
     */
    toUpdateEmailRecipients() {
        return this.to('UpdateEmailRecipients');
    }
    /**
     * Grants permission to update the software configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toUpdateSoftwareConfiguration() {
        return this.to('UpdateSoftwareConfiguration');
    }
    /**
     * Grants permission to verify the email for FreeRTOS extended maintenance plan (EMP)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-getting-started-emp.html
     */
    toVerifyEmail() {
        return this.to('VerifyEmail');
    }
    /**
     * Adds a resource of type configuration to the statement
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     *
     * @param configurationName - Identifier for the configurationName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConfiguration(configurationName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:freertos:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:configuration/${configurationName}`);
    }
    /**
     * Adds a resource of type subscription to the statement
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-getting-started-emp.html
     *
     * @param subscriptionID - Identifier for the subscriptionID.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSubscription(subscriptionID, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:freertos:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:subscription/${subscriptionID}`);
    }
    /**
     * Filters access by tag key present in the request that the user makes to Amazon FreeRTOS
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateSoftwareConfiguration()
     * - .toCreateSubscription()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key component attached to an Amazon FreeRTOS resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - configuration
     * - subscription
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the list of all the tag key names associated with the resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateSoftwareConfiguration()
     * - .toCreateSubscription()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [freertos](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonfreertos.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'freertos';
        this.accessLevelList = {
            Write: [
                'CreateSoftwareConfiguration',
                'CreateSubscription',
                'DeleteSoftwareConfiguration',
                'UpdateEmailRecipients',
                'UpdateSoftwareConfiguration',
                'VerifyEmail'
            ],
            Read: [
                'DescribeHardwarePlatform',
                'DescribeSoftwareConfiguration',
                'DescribeSubscription',
                'GetEmpPatchUrl',
                'GetSoftwareURL',
                'GetSoftwareURLForConfiguration',
                'GetSubscriptionBillingAmount'
            ],
            List: [
                'ListFreeRTOSVersions',
                'ListHardwarePlatforms',
                'ListHardwareVendors',
                'ListSoftwareConfigurations',
                'ListSoftwarePatches',
                'ListSubscriptionEmails',
                'ListSubscriptions'
            ]
        };
    }
}
exports.Freertos = Freertos;
//# sourceMappingURL=data:application/json;base64,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