"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Guardduty = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [guardduty](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonguardduty.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Guardduty extends shared_1.PolicyStatement {
    /**
     * Grants permission to accept invitations to become a GuardDuty member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_AcceptAdministratorInvitation.html
     */
    toAcceptAdministratorInvitation() {
        return this.to('AcceptAdministratorInvitation');
    }
    /**
     * Grants permission to accept invitations to become a GuardDuty member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_AcceptInvitation.html
     */
    toAcceptInvitation() {
        return this.to('AcceptInvitation');
    }
    /**
     * Grants permission to archive GuardDuty findings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ArchiveFindings.html
     */
    toArchiveFindings() {
        return this.to('ArchiveFindings');
    }
    /**
     * Grants permission to create a detector
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateDetector.html
     */
    toCreateDetector() {
        return this.to('CreateDetector');
    }
    /**
     * Grants permission to create GuardDuty filters. A filters defines finding attributes and conditions used to filter findings
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateFilter.html
     */
    toCreateFilter() {
        return this.to('CreateFilter');
    }
    /**
     * Grants permission to create an IPSet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:DeleteRolePolicy
     * - iam:PutRolePolicy
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateIPSet.html
     */
    toCreateIPSet() {
        return this.to('CreateIPSet');
    }
    /**
     * Grants permission to create a new Malware Protection plan
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateMalwareProtectionPlan.html
     */
    toCreateMalwareProtectionPlan() {
        return this.to('CreateMalwareProtectionPlan');
    }
    /**
     * Grants permission to create GuardDuty member accounts, where the account used to create a member becomes the GuardDuty administrator account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateMembers.html
     */
    toCreateMembers() {
        return this.to('CreateMembers');
    }
    /**
     * Grants permission to create a publishing destination
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     * - s3:ListBucket
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreatePublishingDestination.html
     */
    toCreatePublishingDestination() {
        return this.to('CreatePublishingDestination');
    }
    /**
     * Grants permission to create sample findings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateSampleFindings.html
     */
    toCreateSampleFindings() {
        return this.to('CreateSampleFindings');
    }
    /**
     * Grants permission to create GuardDuty ThreatIntelSets, where a ThreatIntelSet consists of known malicious IP addresses used by GuardDuty to generate findings
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateThreatIntelSet.html
     */
    toCreateThreatIntelSet() {
        return this.to('CreateThreatIntelSet');
    }
    /**
     * Grants permission to decline invitations to become a GuardDuty member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeclineInvitations.html
     */
    toDeclineInvitations() {
        return this.to('DeclineInvitations');
    }
    /**
     * Grants permission to delete GuardDuty detectors
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteDetector.html
     */
    toDeleteDetector() {
        return this.to('DeleteDetector');
    }
    /**
     * Grants permission to delete GuardDuty filters
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteFilter.html
     */
    toDeleteFilter() {
        return this.to('DeleteFilter');
    }
    /**
     * Grants permission to delete GuardDuty IPSets
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteIPSet.html
     */
    toDeleteIPSet() {
        return this.to('DeleteIPSet');
    }
    /**
     * Grants permission to delete invitations to become a GuardDuty member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteInvitations.html
     */
    toDeleteInvitations() {
        return this.to('DeleteInvitations');
    }
    /**
     * Grants permission to delete a Malware Protection plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteMalwareProtectionPlan.html
     */
    toDeleteMalwareProtectionPlan() {
        return this.to('DeleteMalwareProtectionPlan');
    }
    /**
     * Grants permission to delete GuardDuty member accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteMembers.html
     */
    toDeleteMembers() {
        return this.to('DeleteMembers');
    }
    /**
     * Grants permission to delete a publishing destination
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeletePublishingDestination.html
     */
    toDeletePublishingDestination() {
        return this.to('DeletePublishingDestination');
    }
    /**
     * Grants permission to delete GuardDuty ThreatIntelSets
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteThreatIntelSet.html
     */
    toDeleteThreatIntelSet() {
        return this.to('DeleteThreatIntelSet');
    }
    /**
     * Grants permission to retrieve details about malware scans
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DescribeMalwareScans.html
     */
    toDescribeMalwareScans() {
        return this.to('DescribeMalwareScans');
    }
    /**
     * Grants permission to retrieve details about the delegated administrator associated with a GuardDuty detector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DescribeOrganizationConfiguration.html
     */
    toDescribeOrganizationConfiguration() {
        return this.to('DescribeOrganizationConfiguration');
    }
    /**
     * Grants permission to retrieve details about a publishing destination
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DescribePublishingDestination.html
     */
    toDescribePublishingDestination() {
        return this.to('DescribePublishingDestination');
    }
    /**
     * Grants permission to disable the organization delegated administrator for GuardDuty
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DisableOrganizationAdminAccount.html
     */
    toDisableOrganizationAdminAccount() {
        return this.to('DisableOrganizationAdminAccount');
    }
    /**
     * Grants permission to disassociate a GuardDuty member account from its GuardDuty administrator account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DisassociateFromAdministratorAccount.html
     */
    toDisassociateFromAdministratorAccount() {
        return this.to('DisassociateFromAdministratorAccount');
    }
    /**
     * Grants permission to disassociate a GuardDuty member account from its GuardDuty administrator account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DisassociateFromMasterAccount.html
     */
    toDisassociateFromMasterAccount() {
        return this.to('DisassociateFromMasterAccount');
    }
    /**
     * Grants permission to disassociate GuardDuty member accounts from their administrator GuardDuty account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DisassociateMembers.html
     */
    toDisassociateMembers() {
        return this.to('DisassociateMembers');
    }
    /**
     * Grants permission to enable an organization delegated administrator for GuardDuty
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_EnableOrganizationAdminAccount.html
     */
    toEnableOrganizationAdminAccount() {
        return this.to('EnableOrganizationAdminAccount');
    }
    /**
     * Grants permission to retrieve details of the GuardDuty administrator account associated with a member account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetAdministratorAccount.html
     */
    toGetAdministratorAccount() {
        return this.to('GetAdministratorAccount');
    }
    /**
     * Grants permission to list Amazon GuardDuty coverage statistics for the specified GuardDuty account in a Region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetCoverageStatistics.html
     */
    toGetCoverageStatistics() {
        return this.to('GetCoverageStatistics');
    }
    /**
     * Grants permission to retrieve GuardDuty detectors
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetDetector.html
     */
    toGetDetector() {
        return this.to('GetDetector');
    }
    /**
     * Grants permission to retrieve GuardDuty filters
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetFilter.html
     */
    toGetFilter() {
        return this.to('GetFilter');
    }
    /**
     * Grants permission to retrieve GuardDuty findings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetFindings.html
     */
    toGetFindings() {
        return this.to('GetFindings');
    }
    /**
     * Grants permission to retrieve a list of GuardDuty finding statistics
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetFindingsStatistics.html
     */
    toGetFindingsStatistics() {
        return this.to('GetFindingsStatistics');
    }
    /**
     * Grants permission to retrieve GuardDuty IPSets
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetIPSet.html
     */
    toGetIPSet() {
        return this.to('GetIPSet');
    }
    /**
     * Grants permission to retrieve the count of all GuardDuty invitations sent to a specified account, which does not include the accepted invitation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetInvitationsCount.html
     */
    toGetInvitationsCount() {
        return this.to('GetInvitationsCount');
    }
    /**
     * Grants permission to retrieve a Malware Protection plan details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetMalwareProtectionPlan.html
     */
    toGetMalwareProtectionPlan() {
        return this.to('GetMalwareProtectionPlan');
    }
    /**
     * Grants permission to retrieve the malware scan settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetMalwareScanSettings.html
     */
    toGetMalwareScanSettings() {
        return this.to('GetMalwareScanSettings');
    }
    /**
     * Grants permission to retrieve details of the GuardDuty administrator account associated with a member account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetMasterAccount.html
     */
    toGetMasterAccount() {
        return this.to('GetMasterAccount');
    }
    /**
     * Grants permission to describe which data sources are enabled for member accounts detectors
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetMemberDetectors.html
     */
    toGetMemberDetectors() {
        return this.to('GetMemberDetectors');
    }
    /**
     * Grants permission to retrieve the member accounts associated with an administrator account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetMembers.html
     */
    toGetMembers() {
        return this.to('GetMembers');
    }
    /**
     * Grants permission to retrieve GuardDuty protection plan coverage statistics for member accounts in a Region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetOrganizationStatistics.html
     */
    toGetOrganizationStatistics() {
        return this.to('GetOrganizationStatistics');
    }
    /**
     * Grants permission to provide the number of days left for each data source used in the free trial period
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetRemainingFreeTrialDays.html
     */
    toGetRemainingFreeTrialDays() {
        return this.to('GetRemainingFreeTrialDays');
    }
    /**
     * Grants permission to retrieve GuardDuty ThreatIntelSets
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetThreatIntelSet.html
     */
    toGetThreatIntelSet() {
        return this.to('GetThreatIntelSet');
    }
    /**
     * Grants permission to list Amazon GuardDuty usage statistics over the last 30 days for the specified detector ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetUsageStatistics.html
     */
    toGetUsageStatistics() {
        return this.to('GetUsageStatistics');
    }
    /**
     * Grants permission to invite other AWS accounts to enable GuardDuty and become GuardDuty member accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_InviteMembers.html
     */
    toInviteMembers() {
        return this.to('InviteMembers');
    }
    /**
     * Grants permission to list all the resource details for a given account in a Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListCoverage.html
     */
    toListCoverage() {
        return this.to('ListCoverage');
    }
    /**
     * Grants permission to retrieve a list of GuardDuty detectors
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListDetectors.html
     */
    toListDetectors() {
        return this.to('ListDetectors');
    }
    /**
     * Grants permission to retrieve a list of GuardDuty filters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListFilters.html
     */
    toListFilters() {
        return this.to('ListFilters');
    }
    /**
     * Grants permission to retrieve a list of GuardDuty findings
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListFindings.html
     */
    toListFindings() {
        return this.to('ListFindings');
    }
    /**
     * Grants permission to retrieve a list of GuardDuty IPSets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListIPSets.html
     */
    toListIPSets() {
        return this.to('ListIPSets');
    }
    /**
     * Grants permission to retrieve a list of all of the GuardDuty membership invitations that were sent to an AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListInvitations.html
     */
    toListInvitations() {
        return this.to('ListInvitations');
    }
    /**
     * Grants permission to retrieve a list of Malware Protection plans
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListMalwareProtectionPlans.html
     */
    toListMalwareProtectionPlans() {
        return this.to('ListMalwareProtectionPlans');
    }
    /**
     * Grants permission to retrieve a list of GuardDuty member accounts associated with an administrator account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListMembers.html
     */
    toListMembers() {
        return this.to('ListMembers');
    }
    /**
     * Grants permission to list details about the organization delegated administrator for GuardDuty
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListOrganizationAdminAccounts.html
     */
    toListOrganizationAdminAccounts() {
        return this.to('ListOrganizationAdminAccounts');
    }
    /**
     * Grants permission to retrieve a list of publishing destinations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListPublishingDestinations.html
     */
    toListPublishingDestinations() {
        return this.to('ListPublishingDestinations');
    }
    /**
     * Grants permission to retrieve a list of tags associated with a GuardDuty resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to retrieve a list of GuardDuty ThreatIntelSets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListThreatIntelSets.html
     */
    toListThreatIntelSets() {
        return this.to('ListThreatIntelSets');
    }
    /**
     * Grants permission to send security telemetry for a specific GuardDuty account in a Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_SendSecurityTelemetry.html
     */
    toSendSecurityTelemetry() {
        return this.to('SendSecurityTelemetry');
    }
    /**
     * Grants permission to initiate a new malware scan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_StartMalwareScan.html
     */
    toStartMalwareScan() {
        return this.to('StartMalwareScan');
    }
    /**
     * Grants permission to a GuardDuty administrator account to monitor findings from GuardDuty member accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_StartMonitoringMembers.html
     */
    toStartMonitoringMembers() {
        return this.to('StartMonitoringMembers');
    }
    /**
     * Grants permission to disable monitoring findings from member accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_StopMonitoringMembers.html
     */
    toStopMonitoringMembers() {
        return this.to('StopMonitoringMembers');
    }
    /**
     * Grants permission to add tags to a GuardDuty resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to unarchive GuardDuty findings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UnarchiveFindings.html
     */
    toUnarchiveFindings() {
        return this.to('UnarchiveFindings');
    }
    /**
     * Grants permission to remove tags from a GuardDuty resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update GuardDuty detectors
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateDetector.html
     */
    toUpdateDetector() {
        return this.to('UpdateDetector');
    }
    /**
     * Grants permission to updates GuardDuty filters
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateFilter.html
     */
    toUpdateFilter() {
        return this.to('UpdateFilter');
    }
    /**
     * Grants permission to update findings feedback to mark GuardDuty findings as useful or not useful
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateFindingsFeedback.html
     */
    toUpdateFindingsFeedback() {
        return this.to('UpdateFindingsFeedback');
    }
    /**
     * Grants permission to update GuardDuty IPSets
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:DeleteRolePolicy
     * - iam:PutRolePolicy
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateIPSet.html
     */
    toUpdateIPSet() {
        return this.to('UpdateIPSet');
    }
    /**
     * Grants permission to update the Malware Protection plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateMalwareProtectionPlan.html
     */
    toUpdateMalwareProtectionPlan() {
        return this.to('UpdateMalwareProtectionPlan');
    }
    /**
     * Grants permission to update the malware scan settings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateMalwareScanSettings.html
     */
    toUpdateMalwareScanSettings() {
        return this.to('UpdateMalwareScanSettings');
    }
    /**
     * Grants permission to update which data sources are enabled for member accounts detectors
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateMemberDetectors.html
     */
    toUpdateMemberDetectors() {
        return this.to('UpdateMemberDetectors');
    }
    /**
     * Grants permission to update the delegated administrator configuration associated with a GuardDuty detector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateOrganizationConfiguration.html
     */
    toUpdateOrganizationConfiguration() {
        return this.to('UpdateOrganizationConfiguration');
    }
    /**
     * Grants permission to update a publishing destination
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     * - s3:ListBucket
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdatePublishingDestination.html
     */
    toUpdatePublishingDestination() {
        return this.to('UpdatePublishingDestination');
    }
    /**
     * Grants permission to updates the GuardDuty ThreatIntelSets
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:DeleteRolePolicy
     * - iam:PutRolePolicy
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateThreatIntelSet.html
     */
    toUpdateThreatIntelSet() {
        return this.to('UpdateThreatIntelSet');
    }
    /**
     * Adds a resource of type detector to the statement
     *
     * https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources
     *
     * @param detectorId - Identifier for the detectorId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDetector(detectorId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:guardduty:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:detector/${detectorId}`);
    }
    /**
     * Adds a resource of type filter to the statement
     *
     * https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources
     *
     * @param detectorId - Identifier for the detectorId.
     * @param filterName - Identifier for the filterName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFilter(detectorId, filterName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:guardduty:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:detector/${detectorId}/filter/${filterName}`);
    }
    /**
     * Adds a resource of type ipset to the statement
     *
     * https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources
     *
     * @param detectorId - Identifier for the detectorId.
     * @param iPSetId - Identifier for the iPSetId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIpset(detectorId, iPSetId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:guardduty:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:detector/${detectorId}/ipset/${iPSetId}`);
    }
    /**
     * Adds a resource of type threatintelset to the statement
     *
     * https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources
     *
     * @param detectorId - Identifier for the detectorId.
     * @param threatIntelSetId - Identifier for the threatIntelSetId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onThreatintelset(detectorId, threatIntelSetId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:guardduty:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:detector/${detectorId}/threatintelset/${threatIntelSetId}`);
    }
    /**
     * Adds a resource of type publishingDestination to the statement
     *
     * https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources
     *
     * @param detectorId - Identifier for the detectorId.
     * @param publishingDestinationId - Identifier for the publishingDestinationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onPublishingDestination(detectorId, publishingDestinationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:guardduty:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:detector/${detectorId}/publishingDestination/${publishingDestinationId}`);
    }
    /**
     * Adds a resource of type malwareprotectionplan to the statement
     *
     * https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources
     *
     * @param malwareProtectionPlanId - Identifier for the malwareProtectionPlanId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMalwareprotectionplan(malwareProtectionPlanId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:guardduty:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:malware-protection-plan/${malwareProtectionPlanId}`);
    }
    /**
     * Filters access by tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateDetector()
     * - .toCreateFilter()
     * - .toCreateIPSet()
     * - .toCreateMalwareProtectionPlan()
     * - .toCreateThreatIntelSet()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toCreateDetector()
     * - .toCreateIPSet()
     * - .toCreateMalwareProtectionPlan()
     *
     * Applies to resource types:
     * - detector
     * - filter
     * - ipset
     * - threatintelset
     * - malwareprotectionplan
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateDetector()
     * - .toCreateFilter()
     * - .toCreateIPSet()
     * - .toCreateMalwareProtectionPlan()
     * - .toCreateThreatIntelSet()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [guardduty](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonguardduty.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'guardduty';
        this.accessLevelList = {
            Write: [
                'AcceptAdministratorInvitation',
                'AcceptInvitation',
                'ArchiveFindings',
                'CreateDetector',
                'CreateFilter',
                'CreateIPSet',
                'CreateMalwareProtectionPlan',
                'CreateMembers',
                'CreatePublishingDestination',
                'CreateSampleFindings',
                'CreateThreatIntelSet',
                'DeclineInvitations',
                'DeleteDetector',
                'DeleteFilter',
                'DeleteIPSet',
                'DeleteInvitations',
                'DeleteMalwareProtectionPlan',
                'DeleteMembers',
                'DeletePublishingDestination',
                'DeleteThreatIntelSet',
                'DisableOrganizationAdminAccount',
                'DisassociateFromAdministratorAccount',
                'DisassociateFromMasterAccount',
                'DisassociateMembers',
                'EnableOrganizationAdminAccount',
                'InviteMembers',
                'SendSecurityTelemetry',
                'StartMalwareScan',
                'StartMonitoringMembers',
                'StopMonitoringMembers',
                'UnarchiveFindings',
                'UpdateDetector',
                'UpdateFilter',
                'UpdateFindingsFeedback',
                'UpdateIPSet',
                'UpdateMalwareProtectionPlan',
                'UpdateMalwareScanSettings',
                'UpdateMemberDetectors',
                'UpdateOrganizationConfiguration',
                'UpdatePublishingDestination',
                'UpdateThreatIntelSet'
            ],
            Read: [
                'DescribeMalwareScans',
                'DescribeOrganizationConfiguration',
                'DescribePublishingDestination',
                'GetAdministratorAccount',
                'GetCoverageStatistics',
                'GetDetector',
                'GetFilter',
                'GetFindings',
                'GetFindingsStatistics',
                'GetIPSet',
                'GetInvitationsCount',
                'GetMalwareProtectionPlan',
                'GetMalwareScanSettings',
                'GetMasterAccount',
                'GetMemberDetectors',
                'GetMembers',
                'GetOrganizationStatistics',
                'GetRemainingFreeTrialDays',
                'GetThreatIntelSet',
                'GetUsageStatistics',
                'ListTagsForResource'
            ],
            List: [
                'ListCoverage',
                'ListDetectors',
                'ListFilters',
                'ListFindings',
                'ListIPSets',
                'ListInvitations',
                'ListMalwareProtectionPlans',
                'ListMembers',
                'ListOrganizationAdminAccounts',
                'ListPublishingDestinations',
                'ListThreatIntelSets'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Guardduty = Guardduty;
//# sourceMappingURL=data:application/json;base64,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