"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AccessAnalyzer = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [access-analyzer](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiamaccessanalyzer.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class AccessAnalyzer extends shared_1.PolicyStatement {
    /**
     * Grants permission to apply an archive rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_ApplyArchiveRule.html
     */
    toApplyArchiveRule() {
        return this.to('ApplyArchiveRule');
    }
    /**
     * Grants permission to cancel a policy generation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_CancelPolicyGeneration.html
     */
    toCancelPolicyGeneration() {
        return this.to('CancelPolicyGeneration');
    }
    /**
     * Grants permission to check that specified access is not allowed by a policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_CheckAccessNotGranted.html
     */
    toCheckAccessNotGranted() {
        return this.to('CheckAccessNotGranted');
    }
    /**
     * Grants permission to check that no new access is allowed when compared to an existing policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_CheckNoNewAccess.html
     */
    toCheckNoNewAccess() {
        return this.to('CheckNoNewAccess');
    }
    /**
     * Grants permission to check that public access is not allowed by a resource policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_CheckNoPublicAccess.html
     */
    toCheckNoPublicAccess() {
        return this.to('CheckNoPublicAccess');
    }
    /**
     * Grants permission to create an access preview for the specified analyzer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_CreateAccessPreview.html
     */
    toCreateAccessPreview() {
        return this.to('CreateAccessPreview');
    }
    /**
     * Grants permission to create an analyzer
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_CreateAnalyzer.html
     */
    toCreateAnalyzer() {
        return this.to('CreateAnalyzer');
    }
    /**
     * Grants permission to create an archive rule for the specified analyzer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_CreateArchiveRule.html
     */
    toCreateArchiveRule() {
        return this.to('CreateArchiveRule');
    }
    /**
     * Grants permission to delete the specified analyzer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_DeleteAnalyzer.html
     */
    toDeleteAnalyzer() {
        return this.to('DeleteAnalyzer');
    }
    /**
     * Grants permission to delete archive rules for the specified analyzer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_DeleteArchiveRule.html
     */
    toDeleteArchiveRule() {
        return this.to('DeleteArchiveRule');
    }
    /**
     * Grants permission to generate recommendation steps to resolve a finding
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_GenerateFindingRecommendation.html
     */
    toGenerateFindingRecommendation() {
        return this.to('GenerateFindingRecommendation');
    }
    /**
     * Grants permission to retrieve information about an access preview
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_GetAccessPreview.html
     */
    toGetAccessPreview() {
        return this.to('GetAccessPreview');
    }
    /**
     * Grants permission to retrieve information about an analyzed resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_GetAnalyzedResource.html
     */
    toGetAnalyzedResource() {
        return this.to('GetAnalyzedResource');
    }
    /**
     * Grants permission to retrieve information about analyzers
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_GetAnalyzer.html
     */
    toGetAnalyzer() {
        return this.to('GetAnalyzer');
    }
    /**
     * Grants permission to retrieve information about archive rules for the specified analyzer
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_GetArchiveRule.html
     */
    toGetArchiveRule() {
        return this.to('GetArchiveRule');
    }
    /**
     * Grants permission to retrieve findings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_GetFindingV2.html
     */
    toGetFinding() {
        return this.to('GetFinding');
    }
    /**
     * Grants permission to retrieve recommendation steps to resolve a finding
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_GetFindingRecommendation.html
     */
    toGetFindingRecommendation() {
        return this.to('GetFindingRecommendation');
    }
    /**
     * Grants permission to retrieve statistics for findings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/access-analyzer-getting-started.html#access-analyzer-permissions
     */
    toGetFindingsStatistics() {
        return this.to('GetFindingsStatistics');
    }
    /**
     * Grants permission to retrieve a policy that was generated using StartPolicyGeneration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_GetGeneratedPolicy.html
     */
    toGetGeneratedPolicy() {
        return this.to('GetGeneratedPolicy');
    }
    /**
     * Grants permission to retrieve a list of findings from an access preview
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_ListAccessPreviewFindings.html
     */
    toListAccessPreviewFindings() {
        return this.to('ListAccessPreviewFindings');
    }
    /**
     * Grants permission to retrieve a list of access previews
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_ListAccessPreviews.html
     */
    toListAccessPreviews() {
        return this.to('ListAccessPreviews');
    }
    /**
     * Grants permission to retrieve a list of resources that have been analyzed
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_ListAnalyzedResources.html
     */
    toListAnalyzedResources() {
        return this.to('ListAnalyzedResources');
    }
    /**
     * Grants permission to retrieves a list of analyzers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_ListAnalyzers.html
     */
    toListAnalyzers() {
        return this.to('ListAnalyzers');
    }
    /**
     * Grants permission to retrieve a list of archive rules from an analyzer
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_ListArchiveRules.html
     */
    toListArchiveRules() {
        return this.to('ListArchiveRules');
    }
    /**
     * Grants permission to retrieve a list of findings from an analyzer
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_ListFindingsV2.html
     */
    toListFindings() {
        return this.to('ListFindings');
    }
    /**
     * Grants permission to list all the recently started policy generations
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_ListPolicyGenerations.html
     */
    toListPolicyGenerations() {
        return this.to('ListPolicyGenerations');
    }
    /**
     * Grants permission to retrieve a list of tags applied to a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to start a policy generation
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_StartPolicyGeneration.html
     */
    toStartPolicyGeneration() {
        return this.to('StartPolicyGeneration');
    }
    /**
     * Grants permission to start a scan of the policies applied to a resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_StartResourceScan.html
     */
    toStartResourceScan() {
        return this.to('StartResourceScan');
    }
    /**
     * Grants permission to add a tag to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove a tag from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to modify an analyzer's configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_UpdateAnalyzer.html
     */
    toUpdateAnalyzer() {
        return this.to('UpdateAnalyzer');
    }
    /**
     * Grants permission to modify an archive rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_UpdateArchiveRule.html
     */
    toUpdateArchiveRule() {
        return this.to('UpdateArchiveRule');
    }
    /**
     * Grants permission to modify findings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_UpdateFindings.html
     */
    toUpdateFindings() {
        return this.to('UpdateFindings');
    }
    /**
     * Grants permission to validate a policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_ValidatePolicy.html
     */
    toValidatePolicy() {
        return this.to('ValidatePolicy');
    }
    /**
     * Adds a resource of type Analyzer to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/access-analyzer-getting-started.html#permission-resources
     *
     * @param analyzerName - Identifier for the analyzerName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAnalyzer(analyzerName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:access-analyzer:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:analyzer/${analyzerName}`);
    }
    /**
     * Adds a resource of type ArchiveRule to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/access-analyzer-getting-started.html#permission-resources
     *
     * @param analyzerName - Identifier for the analyzerName.
     * @param ruleName - Identifier for the ruleName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onArchiveRule(analyzerName, ruleName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:access-analyzer:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:analyzer/${analyzerName}/archive-rule/${ruleName}`);
    }
    /**
     * Filters actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateAnalyzer()
     * - .toGetAnalyzer()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - Analyzer
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateAnalyzer()
     * - .toGetAnalyzer()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [access-analyzer](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiamaccessanalyzer.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'access-analyzer';
        this.accessLevelList = {
            Write: [
                'ApplyArchiveRule',
                'CancelPolicyGeneration',
                'CreateAccessPreview',
                'CreateAnalyzer',
                'CreateArchiveRule',
                'DeleteAnalyzer',
                'DeleteArchiveRule',
                'GenerateFindingRecommendation',
                'StartPolicyGeneration',
                'StartResourceScan',
                'UpdateAnalyzer',
                'UpdateArchiveRule',
                'UpdateFindings'
            ],
            Read: [
                'CheckAccessNotGranted',
                'CheckNoNewAccess',
                'CheckNoPublicAccess',
                'GetAccessPreview',
                'GetAnalyzedResource',
                'GetAnalyzer',
                'GetArchiveRule',
                'GetFinding',
                'GetFindingRecommendation',
                'GetFindingsStatistics',
                'GetGeneratedPolicy',
                'ListAccessPreviewFindings',
                'ListAnalyzedResources',
                'ListFindings',
                'ListPolicyGenerations',
                'ListTagsForResource',
                'ValidatePolicy'
            ],
            List: [
                'ListAccessPreviews',
                'ListAnalyzers',
                'ListArchiveRules'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.AccessAnalyzer = AccessAnalyzer;
//# sourceMappingURL=data:application/json;base64,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