"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Sso = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [sso](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiamidentitycenter.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Sso extends shared_1.PolicyStatement {
    /**
     * Grants permission to connect a directory to be used by AWS IAM Identity Center
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ds:AuthorizeApplication
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toAssociateDirectory() {
        return this.to('AssociateDirectory');
    }
    /**
     * Grants permission to create an association between a directory user or group and a profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toAssociateProfile() {
        return this.to('AssociateProfile');
    }
    /**
     * Grants permission to attach a customer managed policy reference to a permission set
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_AttachCustomerManagedPolicyReferenceToPermissionSet.html
     */
    toAttachCustomerManagedPolicyReferenceToPermissionSet() {
        return this.to('AttachCustomerManagedPolicyReferenceToPermissionSet');
    }
    /**
     * Grants permission to attach an AWS managed policy to a permission set
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_AttachManagedPolicyToPermissionSet.html
     */
    toAttachManagedPolicyToPermissionSet() {
        return this.to('AttachManagedPolicyToPermissionSet');
    }
    /**
     * Grants permission to assign access to a Principal for a specified AWS account using a specified permission set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_CreateAccountAssignment.html
     */
    toCreateAccountAssignment() {
        return this.to('CreateAccountAssignment');
    }
    /**
     * Grants permission to create an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_CreateApplication.html
     */
    toCreateApplication() {
        return this.to('CreateApplication');
    }
    /**
     * Grants permission to create an application assignment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_CreateApplicationAssignment.html
     */
    toCreateApplicationAssignment() {
        return this.to('CreateApplicationAssignment');
    }
    /**
     * Grants permission to add an application instance to AWS IAM Identity Center
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateApplicationInstance() {
        return this.to('CreateApplicationInstance');
    }
    /**
     * Grants permission to add a new certificate for an application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateApplicationInstanceCertificate() {
        return this.to('CreateApplicationInstanceCertificate');
    }
    /**
     * Grants permission to create an identity center instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - organizations:DescribeOrganization
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_CreateInstance.html
     */
    toCreateInstance() {
        return this.to('CreateInstance');
    }
    /**
     * Grants permission to enable the instance for ABAC and specify the attributes
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:AttachRolePolicy
     * - iam:CreateRole
     * - iam:DeleteRole
     * - iam:DeleteRolePolicy
     * - iam:DetachRolePolicy
     * - iam:GetRole
     * - iam:ListAttachedRolePolicies
     * - iam:ListRolePolicies
     * - iam:PutRolePolicy
     * - iam:UpdateAssumeRolePolicy
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_CreateInstanceAccessControlAttributeConfiguration.html
     */
    toCreateInstanceAccessControlAttributeConfiguration() {
        return this.to('CreateInstanceAccessControlAttributeConfiguration');
    }
    /**
     * Grants permission to add a managed application instance to AWS IAM Identity Center
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateManagedApplicationInstance() {
        return this.to('CreateManagedApplicationInstance');
    }
    /**
     * Grants permission to create a permission set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_CreatePermissionSet.html
     */
    toCreatePermissionSet() {
        return this.to('CreatePermissionSet');
    }
    /**
     * Grants permission to create a profile for an application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateProfile() {
        return this.to('CreateProfile');
    }
    /**
     * Grants permission to create a federation trust in a target account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateTrust() {
        return this.to('CreateTrust');
    }
    /**
     * Grants permission to create a trusted token issuer for an instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_CreateTrustedTokenIssuer.html
     */
    toCreateTrustedTokenIssuer() {
        return this.to('CreateTrustedTokenIssuer');
    }
    /**
     * Grants permission to delete a Principal's access from a specified AWS account using a specified permission set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DeleteAccountAssignment.html
     */
    toDeleteAccountAssignment() {
        return this.to('DeleteAccountAssignment');
    }
    /**
     * Grants permission to delete an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DeleteApplication.html
     */
    toDeleteApplication() {
        return this.to('DeleteApplication');
    }
    /**
     * Grants permission to delete an access scope to an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DeleteApplicationAccessScope.html
     */
    toDeleteApplicationAccessScope() {
        return this.to('DeleteApplicationAccessScope');
    }
    /**
     * Grants permission to delete an application assignment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DeleteApplicationAssignment.html
     */
    toDeleteApplicationAssignment() {
        return this.to('DeleteApplicationAssignment');
    }
    /**
     * Grants permission to delete an authentication method to an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DeleteApplicationAuthenticationMethod.html
     */
    toDeleteApplicationAuthenticationMethod() {
        return this.to('DeleteApplicationAuthenticationMethod');
    }
    /**
     * Grants permission to delete a grant from an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DeleteApplicationGrant.html
     */
    toDeleteApplicationGrant() {
        return this.to('DeleteApplicationGrant');
    }
    /**
     * Grants permission to delete the application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteApplicationInstance() {
        return this.to('DeleteApplicationInstance');
    }
    /**
     * Grants permission to delete an inactive or expired certificate from the application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteApplicationInstanceCertificate() {
        return this.to('DeleteApplicationInstanceCertificate');
    }
    /**
     * Grants permission to delete the inline policy from a specified permission set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DeleteInlinePolicyFromPermissionSet.html
     */
    toDeleteInlinePolicyFromPermissionSet() {
        return this.to('DeleteInlinePolicyFromPermissionSet');
    }
    /**
     * Grants permission to delete an identity center instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DeleteInstance.html
     */
    toDeleteInstance() {
        return this.to('DeleteInstance');
    }
    /**
     * Grants permission to disable ABAC and remove the attributes list for the instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DeleteInstanceAccessControlAttributeConfiguration.html
     */
    toDeleteInstanceAccessControlAttributeConfiguration() {
        return this.to('DeleteInstanceAccessControlAttributeConfiguration');
    }
    /**
     * Grants permission to delete the managed application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteManagedApplicationInstance() {
        return this.to('DeleteManagedApplicationInstance');
    }
    /**
     * Grants permission to delete a permission set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DeletePermissionSet.html
     */
    toDeletePermissionSet() {
        return this.to('DeletePermissionSet');
    }
    /**
     * Grants permission to remove permissions boundary from a permission set
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DeletePermissionsBoundaryFromPermissionSet.html
     */
    toDeletePermissionsBoundaryFromPermissionSet() {
        return this.to('DeletePermissionsBoundaryFromPermissionSet');
    }
    /**
     * Grants permission to delete the permission policy associated with a permission set
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeletePermissionsPolicy() {
        return this.to('DeletePermissionsPolicy');
    }
    /**
     * Grants permission to delete the profile for an application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteProfile() {
        return this.to('DeleteProfile');
    }
    /**
     * Grants permission to delete a trusted token issuer for an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DeleteTrustedTokenIssuer.html
     */
    toDeleteTrustedTokenIssuer() {
        return this.to('DeleteTrustedTokenIssuer');
    }
    /**
     * Grants permission to describe the status of the assignment creation request
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DescribeAccountAssignmentCreationStatus.html
     */
    toDescribeAccountAssignmentCreationStatus() {
        return this.to('DescribeAccountAssignmentCreationStatus');
    }
    /**
     * Grants permission to describe the status of an assignment deletion request
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DescribeAccountAssignmentDeletionStatus.html
     */
    toDescribeAccountAssignmentDeletionStatus() {
        return this.to('DescribeAccountAssignmentDeletionStatus');
    }
    /**
     * Grants permission to obtain information about an application
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DescribeApplication.html
     */
    toDescribeApplication() {
        return this.to('DescribeApplication');
    }
    /**
     * Grants permission to retrieve an application assignment
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DescribeApplicationAssignment.html
     */
    toDescribeApplicationAssignment() {
        return this.to('DescribeApplicationAssignment');
    }
    /**
     * Grants permission to describe an application provider
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DescribeApplicationProvider.html
     */
    toDescribeApplicationProvider() {
        return this.to('DescribeApplicationProvider');
    }
    /**
     * Grants permission to obtain information about the directories for this account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDescribeDirectories() {
        return this.to('DescribeDirectories');
    }
    /**
     * Grants permission to obtain information about an identity center instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DescribeInstance.html
     */
    toDescribeInstance() {
        return this.to('DescribeInstance');
    }
    /**
     * Grants permission to get the list of attributes used by the instance for ABAC
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DescribeInstanceAccessControlAttributeConfiguration.html
     */
    toDescribeInstanceAccessControlAttributeConfiguration() {
        return this.to('DescribeInstanceAccessControlAttributeConfiguration');
    }
    /**
     * Grants permission to describe a permission set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DescribePermissionSet.html
     */
    toDescribePermissionSet() {
        return this.to('DescribePermissionSet');
    }
    /**
     * Grants permission to describe the status for the given Permission Set Provisioning request
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DescribePermissionSetProvisioningStatus.html
     */
    toDescribePermissionSetProvisioningStatus() {
        return this.to('DescribePermissionSetProvisioningStatus');
    }
    /**
     * Grants permission to retrieve all the permissions policies associated with a permission set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDescribePermissionsPolicies() {
        return this.to('DescribePermissionsPolicies');
    }
    /**
     * Grants permission to obtain the regions where your organization has enabled AWS IAM Identity Center
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDescribeRegisteredRegions() {
        return this.to('DescribeRegisteredRegions');
    }
    /**
     * Grants permission to describe a trusted token issuer for an instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DescribeTrustedTokenIssuer.html
     */
    toDescribeTrustedTokenIssuer() {
        return this.to('DescribeTrustedTokenIssuer');
    }
    /**
     * Grants permission to obtain information about the trust relationships for this account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDescribeTrusts() {
        return this.to('DescribeTrusts');
    }
    /**
     * Grants permission to detach a customer managed policy reference from a permission set
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DetachCustomerManagedPolicyReferenceFromPermissionSet.html
     */
    toDetachCustomerManagedPolicyReferenceFromPermissionSet() {
        return this.to('DetachCustomerManagedPolicyReferenceFromPermissionSet');
    }
    /**
     * Grants permission to detach the attached AWS managed policy from the specified permission set
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DetachManagedPolicyFromPermissionSet.html
     */
    toDetachManagedPolicyFromPermissionSet() {
        return this.to('DetachManagedPolicyFromPermissionSet');
    }
    /**
     * Grants permission to disassociate a directory to be used by AWS IAM Identity Center
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ds:UnauthorizeApplication
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDisassociateDirectory() {
        return this.to('DisassociateDirectory');
    }
    /**
     * Grants permission to disassociate a directory user or group from a profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDisassociateProfile() {
        return this.to('DisassociateProfile');
    }
    /**
     * Grants permission to get an access scope to an application
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_GetApplicationAccessScope.html
     */
    toGetApplicationAccessScope() {
        return this.to('GetApplicationAccessScope');
    }
    /**
     * Grants permission to read assignment configurations for an application
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_GetApplicationAssignmentConfiguration.html
     */
    toGetApplicationAssignmentConfiguration() {
        return this.to('GetApplicationAssignmentConfiguration');
    }
    /**
     * Grants permission to get an authentication method to an application
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_GetApplicationAuthenticationMethod.html
     */
    toGetApplicationAuthenticationMethod() {
        return this.to('GetApplicationAuthenticationMethod');
    }
    /**
     * Grants permission to obtain details about a grant belonging to an application
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_GetApplicationGrant.html
     */
    toGetApplicationGrant() {
        return this.to('GetApplicationGrant');
    }
    /**
     * Grants permission to retrieve details for an application instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetApplicationInstance() {
        return this.to('GetApplicationInstance');
    }
    /**
     * Grants permission to retrieve application template details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetApplicationTemplate() {
        return this.to('GetApplicationTemplate');
    }
    /**
     * Grants permission to obtain the inline policy assigned to the permission set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_GetInlinePolicyForPermissionSet.html
     */
    toGetInlinePolicyForPermissionSet() {
        return this.to('GetInlinePolicyForPermissionSet');
    }
    /**
     * Grants permission to retrieve details for an application instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetManagedApplicationInstance() {
        return this.to('GetManagedApplicationInstance');
    }
    /**
     * Grants permission to retrieve Mfa Device Management settings for the directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetMfaDeviceManagementForDirectory() {
        return this.to('GetMfaDeviceManagementForDirectory');
    }
    /**
     * Grants permission to retrieve details of a permission set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetPermissionSet() {
        return this.to('GetPermissionSet');
    }
    /**
     * Grants permission to get permissions boundary for a permission set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_GetPermissionsBoundaryForPermissionSet.html
     */
    toGetPermissionsBoundaryForPermissionSet() {
        return this.to('GetPermissionsBoundaryForPermissionSet');
    }
    /**
     * Grants permission to retrieve all permission policies associated with a permission set
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - sso:DescribePermissionsPolicies
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetPermissionsPolicy() {
        return this.to('GetPermissionsPolicy');
    }
    /**
     * Grants permission to retrieve a profile for an application instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetProfile() {
        return this.to('GetProfile');
    }
    /**
     * Grants permission to check if AWS IAM Identity Center is enabled
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetSSOStatus() {
        return this.to('GetSSOStatus');
    }
    /**
     * Grants permission to retrieve shared configuration for the current SSO instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetSharedSsoConfiguration() {
        return this.to('GetSharedSsoConfiguration');
    }
    /**
     * Grants permission to retrieve configuration for the current SSO instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetSsoConfiguration() {
        return this.to('GetSsoConfiguration');
    }
    /**
     * Grants permission to retrieve the federation trust in a target account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetTrust() {
        return this.to('GetTrust');
    }
    /**
     * Grants permission to update the application instance by uploading an application SAML metadata file provided by the service provider
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toImportApplicationInstanceServiceProviderMetadata() {
        return this.to('ImportApplicationInstanceServiceProviderMetadata');
    }
    /**
     * Grants permission to list the status of the AWS account assignment creation requests for a specified SSO instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListAccountAssignmentCreationStatus.html
     */
    toListAccountAssignmentCreationStatus() {
        return this.to('ListAccountAssignmentCreationStatus');
    }
    /**
     * Grants permission to list the status of the AWS account assignment deletion requests for a specified SSO instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListAccountAssignmentDeletionStatus.html
     */
    toListAccountAssignmentDeletionStatus() {
        return this.to('ListAccountAssignmentDeletionStatus');
    }
    /**
     * Grants permission to list the assignee of the specified AWS account with the specified permission set
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListAccountAssignments.html
     */
    toListAccountAssignments() {
        return this.to('ListAccountAssignments');
    }
    /**
     * Grants permission to list accounts assigned to user or group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListAccountAssignmentsForPrincipal.html
     */
    toListAccountAssignmentsForPrincipal() {
        return this.to('ListAccountAssignmentsForPrincipal');
    }
    /**
     * Grants permission to list all the AWS accounts where the specified permission set is provisioned
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListAccountsForProvisionedPermissionSet.html
     */
    toListAccountsForProvisionedPermissionSet() {
        return this.to('ListAccountsForProvisionedPermissionSet');
    }
    /**
     * Grants permission to list access scopes to an application
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListApplicationAccessScopes.html
     */
    toListApplicationAccessScopes() {
        return this.to('ListApplicationAccessScopes');
    }
    /**
     * Grants permission to list application assignments
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListApplicationAssignments.html
     */
    toListApplicationAssignments() {
        return this.to('ListApplicationAssignments');
    }
    /**
     * Grants permission to list applications assigned to user or group
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListApplicationAssignmentsForPrincipal.html
     */
    toListApplicationAssignmentsForPrincipal() {
        return this.to('ListApplicationAssignmentsForPrincipal');
    }
    /**
     * Grants permission to list authentication methods to an application
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListApplicationAuthenticationMethods.html
     */
    toListApplicationAuthenticationMethods() {
        return this.to('ListApplicationAuthenticationMethods');
    }
    /**
     * Grants permission to list grants from an application
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListApplicationGrants.html
     */
    toListApplicationGrants() {
        return this.to('ListApplicationGrants');
    }
    /**
     * Grants permission to retrieve all of the certificates for a given application instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListApplicationInstanceCertificates() {
        return this.to('ListApplicationInstanceCertificates');
    }
    /**
     * Grants permission to retrieve all application instances
     *
     * Access Level: List
     *
     * Dependent actions:
     * - sso:GetApplicationInstance
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListApplicationInstances() {
        return this.to('ListApplicationInstances');
    }
    /**
     * Grants permission to list application providers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListApplicationProviders.html
     */
    toListApplicationProviders() {
        return this.to('ListApplicationProviders');
    }
    /**
     * Grants permission to retrieve all supported application templates
     *
     * Access Level: List
     *
     * Dependent actions:
     * - sso:GetApplicationTemplate
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListApplicationTemplates() {
        return this.to('ListApplicationTemplates');
    }
    /**
     * Grants permission to retrieve all applications associated with the instance of IAM Identity Center
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListApplications.html
     */
    toListApplications() {
        return this.to('ListApplications');
    }
    /**
     * Grants permission to list the customer managed policy references that are attached to a permission set
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListCustomerManagedPolicyReferencesInPermissionSet.html
     */
    toListCustomerManagedPolicyReferencesInPermissionSet() {
        return this.to('ListCustomerManagedPolicyReferencesInPermissionSet');
    }
    /**
     * Grants permission to retrieve details about the directory connected to AWS IAM Identity Center
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListDirectoryAssociations() {
        return this.to('ListDirectoryAssociations');
    }
    /**
     * Grants permission to list the SSO Instances that the caller has access to
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListInstances.html
     */
    toListInstances() {
        return this.to('ListInstances');
    }
    /**
     * Grants permission to list the AWS managed policies that are attached to a specified permission set
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListManagedPoliciesInPermissionSet.html
     */
    toListManagedPoliciesInPermissionSet() {
        return this.to('ListManagedPoliciesInPermissionSet');
    }
    /**
     * Grants permission to list the status of the Permission Set Provisioning requests for a specified SSO instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListPermissionSetProvisioningStatus.html
     */
    toListPermissionSetProvisioningStatus() {
        return this.to('ListPermissionSetProvisioningStatus');
    }
    /**
     * Grants permission to retrieve all permission sets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListPermissionSets.html
     */
    toListPermissionSets() {
        return this.to('ListPermissionSets');
    }
    /**
     * Grants permission to list all the permission sets that are provisioned to a specified AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListPermissionSetsProvisionedToAccount.html
     */
    toListPermissionSetsProvisionedToAccount() {
        return this.to('ListPermissionSetsProvisionedToAccount');
    }
    /**
     * Grants permission to retrieve the directory user or group associated with the profile
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListProfileAssociations() {
        return this.to('ListProfileAssociations');
    }
    /**
     * Grants permission to retrieve all profiles for an application instance
     *
     * Access Level: List
     *
     * Dependent actions:
     * - sso:GetProfile
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListProfiles() {
        return this.to('ListProfiles');
    }
    /**
     * Grants permission to list the tags that are attached to a specified resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list trusted token issuers for an instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListTrustedTokenIssuers.html
     */
    toListTrustedTokenIssuers() {
        return this.to('ListTrustedTokenIssuers');
    }
    /**
     * Grants permission to provision a specified permission set to the specified target
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ProvisionPermissionSet.html
     */
    toProvisionPermissionSet() {
        return this.to('ProvisionPermissionSet');
    }
    /**
     * Grants permission to create/update an access scope to an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_PutApplicationAccessScope.html
     */
    toPutApplicationAccessScope() {
        return this.to('PutApplicationAccessScope');
    }
    /**
     * Grants permission to add assignment configurations to an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_PutApplicationAssignmentConfiguration.html
     */
    toPutApplicationAssignmentConfiguration() {
        return this.to('PutApplicationAssignmentConfiguration');
    }
    /**
     * Grants permission to create/update an authentication method to an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_PutApplicationAuthenticationMethod.html
     */
    toPutApplicationAuthenticationMethod() {
        return this.to('PutApplicationAuthenticationMethod');
    }
    /**
     * Grants permission to create/update a grant to an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_PutApplicationGrant.html
     */
    toPutApplicationGrant() {
        return this.to('PutApplicationGrant');
    }
    /**
     * Grants permission to attach an IAM inline policy to a permission set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_PutInlinePolicyToPermissionSet.html
     */
    toPutInlinePolicyToPermissionSet() {
        return this.to('PutInlinePolicyToPermissionSet');
    }
    /**
     * Grants permission to put Mfa Device Management settings for the directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toPutMfaDeviceManagementForDirectory() {
        return this.to('PutMfaDeviceManagementForDirectory');
    }
    /**
     * Grants permission to add permissions boundary to a permission set
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_PutPermissionsBoundaryToPermissionSet.html
     */
    toPutPermissionsBoundaryToPermissionSet() {
        return this.to('PutPermissionsBoundaryToPermissionSet');
    }
    /**
     * Grants permission to add a policy to a permission set
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toPutPermissionsPolicy() {
        return this.to('PutPermissionsPolicy');
    }
    /**
     * Grants permission to search for groups within the associated directory
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - ds:DescribeDirectories
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toSearchGroups() {
        return this.to('SearchGroups');
    }
    /**
     * Grants permission to search for users within the associated directory
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - ds:DescribeDirectories
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toSearchUsers() {
        return this.to('SearchUsers');
    }
    /**
     * Grants permission to initialize AWS IAM Identity Center
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - organizations:DescribeOrganization
     * - organizations:EnableAWSServiceAccess
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toStartSSO() {
        return this.to('StartSSO');
    }
    /**
     * Grants permission to associate a set of tags with a specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to disassociate a set of tags from a specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_UpdateApplication.html
     */
    toUpdateApplication() {
        return this.to('UpdateApplication');
    }
    /**
     * Grants permission to set a certificate as the active one for this application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateApplicationInstanceActiveCertificate() {
        return this.to('UpdateApplicationInstanceActiveCertificate');
    }
    /**
     * Grants permission to update display data of an application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateApplicationInstanceDisplayData() {
        return this.to('UpdateApplicationInstanceDisplayData');
    }
    /**
     * Grants permission to update federation response configuration for the application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateApplicationInstanceResponseConfiguration() {
        return this.to('UpdateApplicationInstanceResponseConfiguration');
    }
    /**
     * Grants permission to update federation response schema configuration for the application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateApplicationInstanceResponseSchemaConfiguration() {
        return this.to('UpdateApplicationInstanceResponseSchemaConfiguration');
    }
    /**
     * Grants permission to update security details for the application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateApplicationInstanceSecurityConfiguration() {
        return this.to('UpdateApplicationInstanceSecurityConfiguration');
    }
    /**
     * Grants permission to update service provider related configuration for the application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateApplicationInstanceServiceProviderConfiguration() {
        return this.to('UpdateApplicationInstanceServiceProviderConfiguration');
    }
    /**
     * Grants permission to update the status of an application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateApplicationInstanceStatus() {
        return this.to('UpdateApplicationInstanceStatus');
    }
    /**
     * Grants permission to update the user attribute mappings for your connected directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateDirectoryAssociation() {
        return this.to('UpdateDirectoryAssociation');
    }
    /**
     * Grants permission to update an identity center instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_UpdateInstance.html
     */
    toUpdateInstance() {
        return this.to('UpdateInstance');
    }
    /**
     * Grants permission to update the attributes to use with the instance for ABAC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_UpdateInstanceAccessControlAttributeConfiguration.html
     */
    toUpdateInstanceAccessControlAttributeConfiguration() {
        return this.to('UpdateInstanceAccessControlAttributeConfiguration');
    }
    /**
     * Grants permission to update the status of a managed application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateManagedApplicationInstanceStatus() {
        return this.to('UpdateManagedApplicationInstanceStatus');
    }
    /**
     * Grants permission to update the permission set
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_UpdatePermissionSet.html
     */
    toUpdatePermissionSet() {
        return this.to('UpdatePermissionSet');
    }
    /**
     * Grants permission to update the profile for an application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateProfile() {
        return this.to('UpdateProfile');
    }
    /**
     * Grants permission to update the configuration for the current SSO instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateSSOConfiguration() {
        return this.to('UpdateSSOConfiguration');
    }
    /**
     * Grants permission to update the federation trust in a target account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateTrust() {
        return this.to('UpdateTrust');
    }
    /**
     * Grants permission to update a trusted token issuer for an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_UpdateTrustedTokenIssuer.html
     */
    toUpdateTrustedTokenIssuer() {
        return this.to('UpdateTrustedTokenIssuer');
    }
    /**
     * Adds a resource of type PermissionSet to the statement
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/permissionsetsconcept.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param permissionSetId - Identifier for the permissionSetId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPermissionSet(instanceId, permissionSetId, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sso:::permissionSet/${instanceId}/${permissionSetId}`);
    }
    /**
     * Adds a resource of type Account to the statement
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/manage-your-accounts.html
     *
     * @param accountId - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAccount(accountId, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sso:::account/${accountId ?? this.defaultAccount}`);
    }
    /**
     * Adds a resource of type Instance to the statement
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_InstanceMetadata.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onInstance(instanceId, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sso:::instance/${instanceId}`);
    }
    /**
     * Adds a resource of type Application to the statement
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_Application.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param applicationId - Identifier for the applicationId.
     * @param accountId - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifApplicationAccount()
     */
    onApplication(instanceId, applicationId, accountId, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sso::${accountId ?? this.defaultAccount}:application/${instanceId}/${applicationId}`);
    }
    /**
     * Adds a resource of type TrustedTokenIssuer to the statement
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_TrustedTokenIssuerMetadata.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param trustedTokenIssuerId - Identifier for the trustedTokenIssuerId.
     * @param accountId - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTrustedTokenIssuer(instanceId, trustedTokenIssuerId, accountId, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sso::${accountId ?? this.defaultAccount}:trustedTokenIssuer/${instanceId}/${trustedTokenIssuerId}`);
    }
    /**
     * Adds a resource of type ApplicationProvider to the statement
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ApplicationProvider.html
     *
     * @param applicationProviderId - Identifier for the applicationProviderId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onApplicationProvider(applicationProviderId, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sso::aws:applicationProvider/${applicationProviderId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/tagging.html
     *
     * Applies to actions:
     * - .toCreateApplication()
     * - .toCreateInstance()
     * - .toCreatePermissionSet()
     * - .toCreateTrustedTokenIssuer()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/tagging.html
     *
     * Applies to resource types:
     * - PermissionSet
     * - Instance
     * - Application
     * - TrustedTokenIssuer
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/tagging.html
     *
     * Applies to actions:
     * - .toCreateApplication()
     * - .toCreateInstance()
     * - .toCreatePermissionSet()
     * - .toCreateTrustedTokenIssuer()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the account which creates the application. This condition key is not supported for customer managed SAML applications
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/API_Application.html
     *
     * Applies to actions:
     * - .toCreateApplicationAssignment()
     * - .toDeleteApplication()
     * - .toDeleteApplicationAccessScope()
     * - .toDeleteApplicationAssignment()
     * - .toDeleteApplicationAuthenticationMethod()
     * - .toDeleteApplicationGrant()
     * - .toDescribeApplication()
     * - .toDescribeApplicationAssignment()
     * - .toGetApplicationAccessScope()
     * - .toGetApplicationAssignmentConfiguration()
     * - .toGetApplicationAuthenticationMethod()
     * - .toGetApplicationGrant()
     * - .toListApplicationAccessScopes()
     * - .toListApplicationAssignments()
     * - .toListApplicationAssignmentsForPrincipal()
     * - .toListApplicationAuthenticationMethods()
     * - .toListApplicationGrants()
     * - .toPutApplicationAccessScope()
     * - .toPutApplicationAssignmentConfiguration()
     * - .toPutApplicationAuthenticationMethod()
     * - .toPutApplicationGrant()
     * - .toUpdateApplication()
     *
     * Applies to resource types:
     * - Application
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifApplicationAccount(value, operator) {
        return this.if(`ApplicationAccount`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [sso](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiamidentitycenter.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'sso';
        this.accessLevelList = {
            Write: [
                'AssociateDirectory',
                'AssociateProfile',
                'CreateAccountAssignment',
                'CreateApplication',
                'CreateApplicationAssignment',
                'CreateApplicationInstance',
                'CreateApplicationInstanceCertificate',
                'CreateInstance',
                'CreateInstanceAccessControlAttributeConfiguration',
                'CreateManagedApplicationInstance',
                'CreatePermissionSet',
                'CreateProfile',
                'CreateTrust',
                'CreateTrustedTokenIssuer',
                'DeleteAccountAssignment',
                'DeleteApplication',
                'DeleteApplicationAccessScope',
                'DeleteApplicationAssignment',
                'DeleteApplicationAuthenticationMethod',
                'DeleteApplicationGrant',
                'DeleteApplicationInstance',
                'DeleteApplicationInstanceCertificate',
                'DeleteInlinePolicyFromPermissionSet',
                'DeleteInstance',
                'DeleteInstanceAccessControlAttributeConfiguration',
                'DeleteManagedApplicationInstance',
                'DeletePermissionSet',
                'DeleteProfile',
                'DeleteTrustedTokenIssuer',
                'DisassociateDirectory',
                'DisassociateProfile',
                'ImportApplicationInstanceServiceProviderMetadata',
                'ProvisionPermissionSet',
                'PutApplicationAccessScope',
                'PutApplicationAssignmentConfiguration',
                'PutApplicationAuthenticationMethod',
                'PutApplicationGrant',
                'PutInlinePolicyToPermissionSet',
                'PutMfaDeviceManagementForDirectory',
                'StartSSO',
                'UpdateApplication',
                'UpdateApplicationInstanceActiveCertificate',
                'UpdateApplicationInstanceDisplayData',
                'UpdateApplicationInstanceResponseConfiguration',
                'UpdateApplicationInstanceResponseSchemaConfiguration',
                'UpdateApplicationInstanceSecurityConfiguration',
                'UpdateApplicationInstanceServiceProviderConfiguration',
                'UpdateApplicationInstanceStatus',
                'UpdateDirectoryAssociation',
                'UpdateInstance',
                'UpdateInstanceAccessControlAttributeConfiguration',
                'UpdateManagedApplicationInstanceStatus',
                'UpdateProfile',
                'UpdateSSOConfiguration',
                'UpdateTrust',
                'UpdateTrustedTokenIssuer'
            ],
            'Permissions management': [
                'AttachCustomerManagedPolicyReferenceToPermissionSet',
                'AttachManagedPolicyToPermissionSet',
                'DeletePermissionsBoundaryFromPermissionSet',
                'DeletePermissionsPolicy',
                'DetachCustomerManagedPolicyReferenceFromPermissionSet',
                'DetachManagedPolicyFromPermissionSet',
                'PutPermissionsBoundaryToPermissionSet',
                'PutPermissionsPolicy',
                'UpdatePermissionSet'
            ],
            Read: [
                'DescribeAccountAssignmentCreationStatus',
                'DescribeAccountAssignmentDeletionStatus',
                'DescribeApplication',
                'DescribeApplicationAssignment',
                'DescribeApplicationProvider',
                'DescribeDirectories',
                'DescribeInstance',
                'DescribeInstanceAccessControlAttributeConfiguration',
                'DescribePermissionSet',
                'DescribePermissionSetProvisioningStatus',
                'DescribePermissionsPolicies',
                'DescribeRegisteredRegions',
                'DescribeTrustedTokenIssuer',
                'DescribeTrusts',
                'GetApplicationAccessScope',
                'GetApplicationAssignmentConfiguration',
                'GetApplicationAuthenticationMethod',
                'GetApplicationGrant',
                'GetApplicationInstance',
                'GetApplicationTemplate',
                'GetInlinePolicyForPermissionSet',
                'GetManagedApplicationInstance',
                'GetMfaDeviceManagementForDirectory',
                'GetPermissionSet',
                'GetPermissionsBoundaryForPermissionSet',
                'GetPermissionsPolicy',
                'GetProfile',
                'GetSSOStatus',
                'GetSharedSsoConfiguration',
                'GetSsoConfiguration',
                'GetTrust',
                'ListApplicationInstanceCertificates',
                'ListDirectoryAssociations',
                'ListProfileAssociations',
                'ListTagsForResource',
                'SearchGroups',
                'SearchUsers'
            ],
            List: [
                'ListAccountAssignmentCreationStatus',
                'ListAccountAssignmentDeletionStatus',
                'ListAccountAssignments',
                'ListAccountAssignmentsForPrincipal',
                'ListAccountsForProvisionedPermissionSet',
                'ListApplicationAccessScopes',
                'ListApplicationAssignments',
                'ListApplicationAssignmentsForPrincipal',
                'ListApplicationAuthenticationMethods',
                'ListApplicationGrants',
                'ListApplicationInstances',
                'ListApplicationProviders',
                'ListApplicationTemplates',
                'ListApplications',
                'ListCustomerManagedPolicyReferencesInPermissionSet',
                'ListInstances',
                'ListManagedPoliciesInPermissionSet',
                'ListPermissionSetProvisioningStatus',
                'ListPermissionSets',
                'ListPermissionSetsProvisionedToAccount',
                'ListProfiles',
                'ListTrustedTokenIssuers'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Sso = Sso;
//# sourceMappingURL=data:application/json;base64,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