"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Identitystore = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [identitystore](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsidentitystore.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Identitystore extends shared_1.PolicyStatement {
    /**
     * Grants permission to create a group in the specified IdentityStore
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_CreateGroup.html
     */
    toCreateGroup() {
        return this.to('CreateGroup');
    }
    /**
     * Grants permission to create a member to a group in the specified IdentityStore
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_CreateGroupMembership.html
     */
    toCreateGroupMembership() {
        return this.to('CreateGroupMembership');
    }
    /**
     * Grants permission to create a user in the specified IdentityStore
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_CreateUser.html
     */
    toCreateUser() {
        return this.to('CreateUser');
    }
    /**
     * Grants permission to delete a group in the specified IdentityStore
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_DeleteGroup.html
     */
    toDeleteGroup() {
        return this.to('DeleteGroup');
    }
    /**
     * Grants permission to remove a member that is part of a group in the specified IdentityStore
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_DeleteGroupMembership.html
     */
    toDeleteGroupMembership() {
        return this.to('DeleteGroupMembership');
    }
    /**
     * Grants permission to delete a user in the specified IdentityStore
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_DeleteUser.html
     */
    toDeleteUser() {
        return this.to('DeleteUser');
    }
    /**
     * Grants permission to retrieve information about a group in the specified IdentityStore
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_DescribeGroup.html
     */
    toDescribeGroup() {
        return this.to('DescribeGroup');
    }
    /**
     * Grants permission to retrieve information about a member that is part of a group in the specified IdentityStore
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_DescribeGroupMembership.html
     */
    toDescribeGroupMembership() {
        return this.to('DescribeGroupMembership');
    }
    /**
     * Grants permission to retrieve information about user in the specified IdentityStore
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_DescribeUser.html
     */
    toDescribeUser() {
        return this.to('DescribeUser');
    }
    /**
     * Grants permission to retrieve ID information about group in the specified IdentityStore
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_GetGroupId.html
     */
    toGetGroupId() {
        return this.to('GetGroupId');
    }
    /**
     * Grants permission to retrieve ID information of a member which is part of a group in the specified IdentityStore
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_GetGroupMembershipId.html
     */
    toGetGroupMembershipId() {
        return this.to('GetGroupMembershipId');
    }
    /**
     * Grants permission to retrieves ID information about user in the specified IdentityStore
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_GetUserId.html
     */
    toGetUserId() {
        return this.to('GetUserId');
    }
    /**
     * Grants permission to check if a member is a part of groups in the specified IdentityStore
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_IsMemberInGroups.html
     */
    toIsMemberInGroups() {
        return this.to('IsMemberInGroups');
    }
    /**
     * Grants permission to retrieve all members that are part of a group in the specified IdentityStore
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_ListGroupMemberships.html
     */
    toListGroupMemberships() {
        return this.to('ListGroupMemberships');
    }
    /**
     * Grants permission to list groups of the target member in the specified IdentityStore
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_ListGroupMembershipsForMember.html
     */
    toListGroupMembershipsForMember() {
        return this.to('ListGroupMembershipsForMember');
    }
    /**
     * Grants permission to search for groups within the specified IdentityStore
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_ListGroups.html
     */
    toListGroups() {
        return this.to('ListGroups');
    }
    /**
     * Grants permission to search for users in the specified IdentityStore
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_ListUsers.html
     */
    toListUsers() {
        return this.to('ListUsers');
    }
    /**
     * Grants permission to update information about a group in the specified IdentityStore
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_UpdateGroup.html
     */
    toUpdateGroup() {
        return this.to('UpdateGroup');
    }
    /**
     * Grants permission to update user information in the specified IdentityStore
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_UpdateUser.html
     */
    toUpdateUser() {
        return this.to('UpdateUser');
    }
    /**
     * Adds a resource of type Identitystore to the statement
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/
     *
     * @param identityStoreId - Identifier for the identityStoreId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onIdentitystore(identityStoreId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:identitystore::${account ?? this.defaultAccount}:identitystore/${identityStoreId}`);
    }
    /**
     * Adds a resource of type User to the statement
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_User.html
     *
     * @param userId - Identifier for the userId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onUser(userId, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:identitystore:::user/${userId}`);
    }
    /**
     * Adds a resource of type Group to the statement
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_Group.html
     *
     * @param groupId - Identifier for the groupId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onGroup(groupId, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:identitystore:::group/${groupId}`);
    }
    /**
     * Adds a resource of type GroupMembership to the statement
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_GroupMembership.html
     *
     * @param membershipId - Identifier for the membershipId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onGroupMembership(membershipId, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:identitystore:::membership/${membershipId}`);
    }
    /**
     * Adds a resource of type AllUsers to the statement
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_User.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAllUsers(resourceName, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:identitystore:::user/${resourceName}`);
    }
    /**
     * Adds a resource of type AllGroups to the statement
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_Group.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAllGroups(resourceName, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:identitystore:::group/${resourceName}`);
    }
    /**
     * Adds a resource of type AllGroupMemberships to the statement
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_GroupMembership.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAllGroupMemberships(resourceName, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:identitystore:::membership/${resourceName}`);
    }
    /**
     * Filters access by IAM Identity Center User ID
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/condition-context-keys-sts-idc.html#condition-keys-identity-store-user-id
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifUserId(value, operator) {
        return this.if(`UserId`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [identitystore](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsidentitystore.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'identitystore';
        this.accessLevelList = {
            Write: [
                'CreateGroup',
                'CreateGroupMembership',
                'CreateUser',
                'DeleteGroup',
                'DeleteGroupMembership',
                'DeleteUser',
                'UpdateGroup',
                'UpdateUser'
            ],
            Read: [
                'DescribeGroup',
                'DescribeGroupMembership',
                'DescribeUser',
                'GetGroupId',
                'GetGroupMembershipId',
                'GetUserId',
                'IsMemberInGroups'
            ],
            List: [
                'ListGroupMemberships',
                'ListGroupMembershipsForMember',
                'ListGroups',
                'ListUsers'
            ]
        };
    }
}
exports.Identitystore = Identitystore;
//# sourceMappingURL=data:application/json;base64,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