"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Iotfleetwise = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [iotfleetwise](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiotfleetwise.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Iotfleetwise extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate the given vehicle to a fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_AssociateVehicleFleet.html
     */
    toAssociateVehicleFleet() {
        return this.to('AssociateVehicleFleet');
    }
    /**
     * Grants permission to create a batch of vehicles
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iot:CreateThing
     * - iot:DescribeThing
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_BatchCreateVehicle.html
     */
    toBatchCreateVehicle() {
        return this.to('BatchCreateVehicle');
    }
    /**
     * Grants permission to update a batch of vehicles
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUpdateToModelManifestArn()
     * - .ifUpdateToDecoderManifestArn()
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_BatchUpdateVehicle.html
     */
    toBatchUpdateVehicle() {
        return this.to('BatchUpdateVehicle');
    }
    /**
     * Grants permission to create a campaign
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifDestinationArn()
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_CreateCampaign.html
     */
    toCreateCampaign() {
        return this.to('CreateCampaign');
    }
    /**
     * Grants permission to create a decoder manifest for an existing model
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_CreateDecoderManifest.html
     */
    toCreateDecoderManifest() {
        return this.to('CreateDecoderManifest');
    }
    /**
     * Grants permission to create a fleet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_CreateFleet.html
     */
    toCreateFleet() {
        return this.to('CreateFleet');
    }
    /**
     * Grants permission to create a model manifest definition
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_CreateModelManifest.html
     */
    toCreateModelManifest() {
        return this.to('CreateModelManifest');
    }
    /**
     * Grants permission to create a signal catalog
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_CreateSignalCatalog.html
     */
    toCreateSignalCatalog() {
        return this.to('CreateSignalCatalog');
    }
    /**
     * Grants permission to create a state template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_CreateStateTemplate.html
     */
    toCreateStateTemplate() {
        return this.to('CreateStateTemplate');
    }
    /**
     * Grants permission to create a vehicle
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iot:CreateThing
     * - iot:DescribeThing
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_CreateVehicle.html
     */
    toCreateVehicle() {
        return this.to('CreateVehicle');
    }
    /**
     * Grants permission to delete a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_DeleteCampaign.html
     */
    toDeleteCampaign() {
        return this.to('DeleteCampaign');
    }
    /**
     * Grants permission to delete the given decoder manifest
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_DeleteDecoderManifest.html
     */
    toDeleteDecoderManifest() {
        return this.to('DeleteDecoderManifest');
    }
    /**
     * Grants permission to delete a fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_DeleteFleet.html
     */
    toDeleteFleet() {
        return this.to('DeleteFleet');
    }
    /**
     * Grants permission to delete the given model manifest
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_DeleteModelManifest.html
     */
    toDeleteModelManifest() {
        return this.to('DeleteModelManifest');
    }
    /**
     * Grants permission to delete a specific signal catalog
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_DeleteSignalCatalog.html
     */
    toDeleteSignalCatalog() {
        return this.to('DeleteSignalCatalog');
    }
    /**
     * Grants permission to delete a state template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_DeleteStateTemplate.html
     */
    toDeleteStateTemplate() {
        return this.to('DeleteStateTemplate');
    }
    /**
     * Grants permission to delete a vehicle
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_DeleteVehicle.html
     */
    toDeleteVehicle() {
        return this.to('DeleteVehicle');
    }
    /**
     * Grants permission to disassociate a vehicle from an existing fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_DisassociateVehicleFleet.html
     */
    toDisassociateVehicleFleet() {
        return this.to('DisassociateVehicleFleet');
    }
    /**
     * Grants permission to generate the payload for running a command on a vehicle
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifSignals()
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/developerguide/controlling-access.html#generate-command-payload
     */
    toGenerateCommandPayload() {
        return this.to('GenerateCommandPayload');
    }
    /**
     * Grants permission to get summary information for a given campaign
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_GetCampaign.html
     */
    toGetCampaign() {
        return this.to('GetCampaign');
    }
    /**
     * Grants permission to get summary information for a given decoder manifest definition
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_GetDecoderManifest.html
     */
    toGetDecoderManifest() {
        return this.to('GetDecoderManifest');
    }
    /**
     * Grants permission to get KMS-based encryption status for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_GetEncryptionConfiguration.html
     */
    toGetEncryptionConfiguration() {
        return this.to('GetEncryptionConfiguration');
    }
    /**
     * Grants permission to get summary information for a fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_GetFleet.html
     */
    toGetFleet() {
        return this.to('GetFleet');
    }
    /**
     * Grants permission to get the logging options for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_GetLoggingOptions.html
     */
    toGetLoggingOptions() {
        return this.to('GetLoggingOptions');
    }
    /**
     * Grants permission to get summary information for a given model manifest definition
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_GetModelManifest.html
     */
    toGetModelManifest() {
        return this.to('GetModelManifest');
    }
    /**
     * Grants permission to get the account registration status with IoT FleetWise
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_GetRegisterAccountStatus.html
     */
    toGetRegisterAccountStatus() {
        return this.to('GetRegisterAccountStatus');
    }
    /**
     * Grants permission to get summary information for a specific signal catalog
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_GetSignalCatalog.html
     */
    toGetSignalCatalog() {
        return this.to('GetSignalCatalog');
    }
    /**
     * Grants permission to get summary information for a given state template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_GetStateTemplate.html
     */
    toGetStateTemplate() {
        return this.to('GetStateTemplate');
    }
    /**
     * Grants permission to get summary information for a vehicle
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_GetVehicle.html
     */
    toGetVehicle() {
        return this.to('GetVehicle');
    }
    /**
     * Grants permission to get the status of the campaigns running on a specific vehicle
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_GetVehicleStatus.html
     */
    toGetVehicleStatus() {
        return this.to('GetVehicleStatus');
    }
    /**
     * Grants permission to import an existing decoder manifest
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_ImportDecoderManifest.html
     */
    toImportDecoderManifest() {
        return this.to('ImportDecoderManifest');
    }
    /**
     * Grants permission to create a signal catalog by importing existing definitions
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_ImportSignalCatalog.html
     */
    toImportSignalCatalog() {
        return this.to('ImportSignalCatalog');
    }
    /**
     * Grants permission to list campaigns
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_ListCampaigns.html
     */
    toListCampaigns() {
        return this.to('ListCampaigns');
    }
    /**
     * Grants permission to list network interfaces associated to the existing decoder manifest
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_ListDecoderManifestNetworkInterfaces.html
     */
    toListDecoderManifestNetworkInterfaces() {
        return this.to('ListDecoderManifestNetworkInterfaces');
    }
    /**
     * Grants permission to list decoder manifest signals
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_ListDecoderManifestSignals.html
     */
    toListDecoderManifestSignals() {
        return this.to('ListDecoderManifestSignals');
    }
    /**
     * Grants permission to list all decoder manifests, with an optional filter on model manifest
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_ListDecoderManifests.html
     */
    toListDecoderManifests() {
        return this.to('ListDecoderManifests');
    }
    /**
     * Grants permission to list all fleets
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_ListFleets.html
     */
    toListFleets() {
        return this.to('ListFleets');
    }
    /**
     * Grants permission to list all the fleets that the given vehicle is associated with
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_ListFleetsForVehicle.html
     */
    toListFleetsForVehicle() {
        return this.to('ListFleetsForVehicle');
    }
    /**
     * Grants permission to list all nodes for the given model manifest
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_ListModelManifestNodes.html
     */
    toListModelManifestNodes() {
        return this.to('ListModelManifestNodes');
    }
    /**
     * Grants permission to list all model manifests, with an optional filter on signal catalog
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_ListModelManifests.html
     */
    toListModelManifests() {
        return this.to('ListModelManifests');
    }
    /**
     * Grants permission to list all nodes for a given signal catalog
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_ListSignalCatalogNodes.html
     */
    toListSignalCatalogNodes() {
        return this.to('ListSignalCatalogNodes');
    }
    /**
     * Grants permission to list all signal catalogs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_ListSignalCatalogs.html
     */
    toListSignalCatalogs() {
        return this.to('ListSignalCatalogs');
    }
    /**
     * Grants permission to list state templates
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_ListStateTemplates.html
     */
    toListStateTemplates() {
        return this.to('ListStateTemplates');
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list all vehicles, with an optional filter on model manifest
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_ListVehicles.html
     */
    toListVehicles() {
        return this.to('ListVehicles');
    }
    /**
     * Grants permission to list vehicles in the given fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_ListVehiclesInFleet.html
     */
    toListVehiclesInFleet() {
        return this.to('ListVehiclesInFleet');
    }
    /**
     * Grants permission to enable or disable KMS-based encryption for the AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_PutEncryptionConfiguration.html
     */
    toPutEncryptionConfiguration() {
        return this.to('PutEncryptionConfiguration');
    }
    /**
     * Grants permission to put the logging options for the AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_PutLoggingOptions.html
     */
    toPutLoggingOptions() {
        return this.to('PutLoggingOptions');
    }
    /**
     * Grants permission to register an AWS account to IoT FleetWise
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_RegisterAccount.html
     */
    toRegisterAccount() {
        return this.to('RegisterAccount');
    }
    /**
     * Grants permission to add tags to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update the given campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_UpdateCampaign.html
     */
    toUpdateCampaign() {
        return this.to('UpdateCampaign');
    }
    /**
     * Grants permission to update a decoder manifest defnition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_UpdateDecoderManifest.html
     */
    toUpdateDecoderManifest() {
        return this.to('UpdateDecoderManifest');
    }
    /**
     * Grants permission to update the fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_UpdateFleet.html
     */
    toUpdateFleet() {
        return this.to('UpdateFleet');
    }
    /**
     * Grants permission to update the given model manifest definition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_UpdateModelManifest.html
     */
    toUpdateModelManifest() {
        return this.to('UpdateModelManifest');
    }
    /**
     * Grants permission to update a specific signal catalog definition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_UpdateSignalCatalog.html
     */
    toUpdateSignalCatalog() {
        return this.to('UpdateSignalCatalog');
    }
    /**
     * Grants permission to update the given state template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_UpdateStateTemplate.html
     */
    toUpdateStateTemplate() {
        return this.to('UpdateStateTemplate');
    }
    /**
     * Grants permission to update the vehicle
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUpdateToModelManifestArn()
     * - .ifUpdateToDecoderManifestArn()
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_UpdateVehicle.html
     */
    toUpdateVehicle() {
        return this.to('UpdateVehicle');
    }
    /**
     * Adds a resource of type campaign to the statement
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/developerguide/campaigns.html
     *
     * @param campaignName - Identifier for the campaignName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCampaign(campaignName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotfleetwise:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:campaign/${campaignName}`);
    }
    /**
     * Adds a resource of type decodermanifest to the statement
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/developerguide/decoder-manifests.html
     *
     * @param name - Identifier for the name.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDecodermanifest(name, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotfleetwise:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:decoder-manifest/${name}`);
    }
    /**
     * Adds a resource of type fleet to the statement
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/developerguide/fleets.html
     *
     * @param fleetId - Identifier for the fleetId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFleet(fleetId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotfleetwise:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:fleet/${fleetId}`);
    }
    /**
     * Adds a resource of type modelmanifest to the statement
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/developerguide/vehicle-models.html
     *
     * @param name - Identifier for the name.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onModelmanifest(name, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotfleetwise:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:model-manifest/${name}`);
    }
    /**
     * Adds a resource of type signalcatalog to the statement
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/developerguide/signal-catalogs.html
     *
     * @param name - Identifier for the name.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSignalcatalog(name, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotfleetwise:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:signal-catalog/${name}`);
    }
    /**
     * Adds a resource of type vehicle to the statement
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/developerguide/vehicles.html
     *
     * @param vehicleId - Identifier for the vehicleId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVehicle(vehicleId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotfleetwise:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:vehicle/${vehicleId}`);
    }
    /**
     * Adds a resource of type statetemplate to the statement
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/developerguide/last-known-state.html
     *
     * @param stateTemplateId - Identifier for the stateTemplateId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStatetemplate(stateTemplateId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotfleetwise:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:state-template/${stateTemplateId}`);
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toBatchCreateVehicle()
     * - .toCreateCampaign()
     * - .toCreateDecoderManifest()
     * - .toCreateFleet()
     * - .toCreateModelManifest()
     * - .toCreateSignalCatalog()
     * - .toCreateStateTemplate()
     * - .toCreateVehicle()
     * - .toImportSignalCatalog()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - campaign
     * - decodermanifest
     * - fleet
     * - modelmanifest
     * - signalcatalog
     * - vehicle
     * - statetemplate
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toBatchCreateVehicle()
     * - .toCreateCampaign()
     * - .toCreateDecoderManifest()
     * - .toCreateFleet()
     * - .toCreateModelManifest()
     * - .toCreateSignalCatalog()
     * - .toCreateStateTemplate()
     * - .toCreateVehicle()
     * - .toImportSignalCatalog()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by campaign destination ARN, eg. an S3 bucket ARN or a Timestream ARN
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiotfleetwise.html
     *
     * Applies to actions:
     * - .toCreateCampaign()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifDestinationArn(value, operator) {
        return this.if(`DestinationArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by fully qualified signal names
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiotfleetwise.html
     *
     * Applies to actions:
     * - .toGenerateCommandPayload()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSignals(value, operator) {
        return this.if(`Signals`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a list of IoT FleetWise Decoder Manifest ARNs
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiotfleetwise.html
     *
     * Applies to actions:
     * - .toBatchUpdateVehicle()
     * - .toUpdateVehicle()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifUpdateToDecoderManifestArn(value, operator) {
        return this.if(`UpdateToDecoderManifestArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by a list of IoT FleetWise Model Manifest ARNs
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiotfleetwise.html
     *
     * Applies to actions:
     * - .toBatchUpdateVehicle()
     * - .toUpdateVehicle()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifUpdateToModelManifestArn(value, operator) {
        return this.if(`UpdateToModelManifestArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Statement provider for service [iotfleetwise](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiotfleetwise.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'iotfleetwise';
        this.accessLevelList = {
            Write: [
                'AssociateVehicleFleet',
                'BatchCreateVehicle',
                'BatchUpdateVehicle',
                'CreateCampaign',
                'CreateDecoderManifest',
                'CreateFleet',
                'CreateModelManifest',
                'CreateSignalCatalog',
                'CreateStateTemplate',
                'CreateVehicle',
                'DeleteCampaign',
                'DeleteDecoderManifest',
                'DeleteFleet',
                'DeleteModelManifest',
                'DeleteSignalCatalog',
                'DeleteStateTemplate',
                'DeleteVehicle',
                'DisassociateVehicleFleet',
                'ImportDecoderManifest',
                'ImportSignalCatalog',
                'PutEncryptionConfiguration',
                'PutLoggingOptions',
                'RegisterAccount',
                'UpdateCampaign',
                'UpdateDecoderManifest',
                'UpdateFleet',
                'UpdateModelManifest',
                'UpdateSignalCatalog',
                'UpdateStateTemplate',
                'UpdateVehicle'
            ],
            'Permissions management': [
                'GenerateCommandPayload'
            ],
            Read: [
                'GetCampaign',
                'GetDecoderManifest',
                'GetEncryptionConfiguration',
                'GetFleet',
                'GetLoggingOptions',
                'GetModelManifest',
                'GetRegisterAccountStatus',
                'GetSignalCatalog',
                'GetStateTemplate',
                'GetVehicle',
                'GetVehicleStatus',
                'ListCampaigns',
                'ListDecoderManifests',
                'ListFleets',
                'ListFleetsForVehicle',
                'ListModelManifests',
                'ListSignalCatalogNodes',
                'ListSignalCatalogs',
                'ListStateTemplates',
                'ListTagsForResource',
                'ListVehicles',
                'ListVehiclesInFleet'
            ],
            List: [
                'ListDecoderManifestNetworkInterfaces',
                'ListDecoderManifestSignals',
                'ListModelManifestNodes'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Iotfleetwise = Iotfleetwise;
//# sourceMappingURL=data:application/json;base64,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