"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GreengrassV2 = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [greengrass-v2](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiotgreengrassv2.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class GreengrassV2 extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate a role with your account. AWS IoT Greengrass uses this role to access your Lambda functions and AWS IoT resources
     *
     * Access Level: Permissions management
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/greengrass/v2/APIReference/API_AssociateServiceRoleToAccount.html
     */
    toAssociateServiceRoleToAccount() {
        return this.to('AssociateServiceRoleToAccount');
    }
    /**
     * Grants permission to associate a list of client devices with a core device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/v2/APIReference/API_BatchAssociateClientDeviceWithCoreDevice.html
     */
    toBatchAssociateClientDeviceWithCoreDevice() {
        return this.to('BatchAssociateClientDeviceWithCoreDevice');
    }
    /**
     * Grants permission to disassociate a list of client devices from a core device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/v2/APIReference/API_BatchDisassociateClientDeviceFromCoreDevice.html
     */
    toBatchDisassociateClientDeviceFromCoreDevice() {
        return this.to('BatchDisassociateClientDeviceFromCoreDevice');
    }
    /**
     * Grants permission to cancel a deployment
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iot:CancelJob
     * - iot:DeleteThingShadow
     * - iot:DescribeJob
     * - iot:DescribeThing
     * - iot:DescribeThingGroup
     * - iot:GetThingShadow
     * - iot:UpdateJob
     * - iot:UpdateThingShadow
     *
     * https://docs.aws.amazon.com/greengrass/v2/APIReference/API_CancelDeployment.html
     */
    toCancelDeployment() {
        return this.to('CancelDeployment');
    }
    /**
     * Grants permission to create a component
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/v2/APIReference/API_CreateComponentVersion.html
     */
    toCreateComponentVersion() {
        return this.to('CreateComponentVersion');
    }
    /**
     * Grants permission to create a deployment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iot:CancelJob
     * - iot:CreateJob
     * - iot:DeleteThingShadow
     * - iot:DescribeJob
     * - iot:DescribeThing
     * - iot:DescribeThingGroup
     * - iot:GetThingShadow
     * - iot:UpdateJob
     * - iot:UpdateThingShadow
     *
     * https://docs.aws.amazon.com/greengrass/v2/APIReference/API_CreateDeployment.html
     */
    toCreateDeployment() {
        return this.to('CreateDeployment');
    }
    /**
     * Grants permission to delete a component
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/v2/APIReference/API_DeleteComponent.html
     */
    toDeleteComponent() {
        return this.to('DeleteComponent');
    }
    /**
     * Grants permission to delete a AWS IoT Greengrass core device, which is an AWS IoT thing. This operation removes the core device from the list of core devices. This operation doesn't delete the AWS IoT thing
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iot:DescribeJobExecution
     *
     * https://docs.aws.amazon.com/greengrass/v2/APIReference/API_DeleteCoreDevice.html
     */
    toDeleteCoreDevice() {
        return this.to('DeleteCoreDevice');
    }
    /**
     * Grants permission to delete a deployment. To delete an active deployment, it needs to be cancelled first
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iot:DeleteJob
     *
     * https://docs.aws.amazon.com/greengrass/v2/APIReference/API_DeleteDeployment.html
     */
    toDeleteDeployment() {
        return this.to('DeleteDeployment');
    }
    /**
     * Grants permission to retrieve metadata for a version of a component
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/v2/APIReference/API_DescribeComponent.html
     */
    toDescribeComponent() {
        return this.to('DescribeComponent');
    }
    /**
     * Grants permission to disassociate the service role from an account. Without a service role, deployments will not work
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/greengrass/v2/APIReference/API_DisassociateServiceRoleFromAccount.html
     */
    toDisassociateServiceRoleFromAccount() {
        return this.to('DisassociateServiceRoleFromAccount');
    }
    /**
     * Grants permission to get the recipe for a version of a component
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/v2/APIReference/API_GetComponent.html
     */
    toGetComponent() {
        return this.to('GetComponent');
    }
    /**
     * Grants permission to get the pre-signed URL to download a public component artifact
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/v2/APIReference/API_GetComponentVersionArtifact.html
     */
    toGetComponentVersionArtifact() {
        return this.to('GetComponentVersionArtifact');
    }
    /**
     * Grants permission to retrieve the connectivity information for a Greengrass core device
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - iot:GetThingShadow
     *
     * https://docs.aws.amazon.com/greengrass/v2/APIReference/API_GetConnectivityInfo.html
     */
    toGetConnectivityInfo() {
        return this.to('GetConnectivityInfo');
    }
    /**
     * Grants permission to retrieves metadata for a AWS IoT Greengrass core device
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/v2/APIReference/API_GetCoreDevice.html
     */
    toGetCoreDevice() {
        return this.to('GetCoreDevice');
    }
    /**
     * Grants permission to get a deployment
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - iot:DescribeJob
     * - iot:DescribeThing
     * - iot:DescribeThingGroup
     * - iot:GetThingShadow
     *
     * https://docs.aws.amazon.com/greengrass/v2/APIReference/API_GetDeployment.html
     */
    toGetDeployment() {
        return this.to('GetDeployment');
    }
    /**
     * Grants permission to retrieve the service role that is attached to an account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/greengrass/v2/APIReference/API_GetServiceRoleForAccount.html
     */
    toGetServiceRoleForAccount() {
        return this.to('GetServiceRoleForAccount');
    }
    /**
     * Grants permission to retrieve a paginated list of client devices associated to a AWS IoT Greengrass core device
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/v2/APIReference/API_ListClientDevicesAssociatedWithCoreDevice.html
     */
    toListClientDevicesAssociatedWithCoreDevice() {
        return this.to('ListClientDevicesAssociatedWithCoreDevice');
    }
    /**
     * Grants permission to retrieve a paginated list of all versions for a component
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/v2/APIReference/API_ListComponentVersions.html
     */
    toListComponentVersions() {
        return this.to('ListComponentVersions');
    }
    /**
     * Grants permission to retrieve a paginated list of component summaries
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/v2/APIReference/API_ListComponents.html
     */
    toListComponents() {
        return this.to('ListComponents');
    }
    /**
     * Grants permission to retrieve a paginated list of AWS IoT Greengrass core devices
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/v2/APIReference/API_ListCoreDevices.html
     */
    toListCoreDevices() {
        return this.to('ListCoreDevices');
    }
    /**
     * Grants permission to retrieves a paginated list of deployments
     *
     * Access Level: List
     *
     * Dependent actions:
     * - iot:DescribeJob
     * - iot:DescribeThing
     * - iot:DescribeThingGroup
     * - iot:GetThingShadow
     *
     * https://docs.aws.amazon.com/greengrass/v2/APIReference/API_ListDeployments.html
     */
    toListDeployments() {
        return this.to('ListDeployments');
    }
    /**
     * Grants permission to retrieves a paginated list of deployment jobs that AWS IoT Greengrass sends to AWS IoT Greengrass core devices
     *
     * Access Level: List
     *
     * Dependent actions:
     * - iot:DescribeJob
     * - iot:DescribeJobExecution
     * - iot:DescribeThing
     * - iot:DescribeThingGroup
     * - iot:GetThingShadow
     *
     * https://docs.aws.amazon.com/greengrass/v2/APIReference/API_ListEffectiveDeployments.html
     */
    toListEffectiveDeployments() {
        return this.to('ListEffectiveDeployments');
    }
    /**
     * Grants permission to retrieve a paginated list of the components that a AWS IoT Greengrass core device runs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/v2/APIReference/API_ListInstalledComponents.html
     */
    toListInstalledComponents() {
        return this.to('ListInstalledComponents');
    }
    /**
     * Grants permission to list the tags for a resource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/v2/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list components that meet the component, version, and platform requirements of a deployment
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/greengrass/v2/APIReference/API_ResolveComponentCandidates.html
     */
    toResolveComponentCandidates() {
        return this.to('ResolveComponentCandidates');
    }
    /**
     * Grants permission to add tags to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/v2/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/greengrass/v2/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update the connectivity information for a Greengrass core. Any devices that belong to the group that has this core will receive this information in order to find the location of the core and connect to it
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iot:GetThingShadow
     * - iot:UpdateThingShadow
     *
     * https://docs.aws.amazon.com/greengrass/v2/APIReference/API_UpdateConnectivityInfo.html
     */
    toUpdateConnectivityInfo() {
        return this.to('UpdateConnectivityInfo');
    }
    /**
     * Adds a resource of type connectivityInfo to the statement
     *
     * https://docs.aws.amazon.com/greengrass/v2/APIReference/API_ConnectivityInfo.html
     *
     * @param thingName - Identifier for the thingName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onConnectivityInfo(thingName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:greengrass:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:/greengrass/things/${thingName}/connectivityInfo`);
    }
    /**
     * Adds a resource of type component to the statement
     *
     * https://docs.aws.amazon.com/greengrass/v2/APIReference/API_Component.html
     *
     * @param componentName - Identifier for the componentName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onComponent(componentName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:greengrass:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:components:${componentName}`);
    }
    /**
     * Adds a resource of type componentVersion to the statement
     *
     * https://docs.aws.amazon.com/greengrass/v2/APIReference/API_Component.html
     *
     * @param componentName - Identifier for the componentName.
     * @param componentVersion - Identifier for the componentVersion.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onComponentVersion(componentName, componentVersion, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:greengrass:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:components:${componentName}:versions:${componentVersion}`);
    }
    /**
     * Adds a resource of type coreDevice to the statement
     *
     * https://docs.aws.amazon.com/greengrass/v2/APIReference/API_CoreDevice.html
     *
     * @param coreDeviceThingName - Identifier for the coreDeviceThingName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCoreDevice(coreDeviceThingName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:greengrass:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:coreDevices:${coreDeviceThingName}`);
    }
    /**
     * Adds a resource of type deployment to the statement
     *
     * https://docs.aws.amazon.com/greengrass/v2/APIReference/API_Deployment.html
     *
     * @param deploymentId - Identifier for the deploymentId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDeployment(deploymentId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:greengrass:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:deployments:${deploymentId}`);
    }
    /**
     * Filters access by checking tag key/value pairs included in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateComponentVersion()
     * - .toCreateDeployment()
     * - .toListTagsForResource()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by checking tag key/value pairs associated with a specific resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - component
     * - componentVersion
     * - coreDevice
     * - deployment
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by checking tag keys passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateComponentVersion()
     * - .toCreateDeployment()
     * - .toListTagsForResource()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [greengrass-v2](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiotgreengrassv2.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'greengrass';
        this.accessLevelList = {
            'Permissions management': [
                'AssociateServiceRoleToAccount'
            ],
            Write: [
                'BatchAssociateClientDeviceWithCoreDevice',
                'BatchDisassociateClientDeviceFromCoreDevice',
                'CancelDeployment',
                'CreateComponentVersion',
                'CreateDeployment',
                'DeleteComponent',
                'DeleteCoreDevice',
                'DeleteDeployment',
                'DisassociateServiceRoleFromAccount',
                'UpdateConnectivityInfo'
            ],
            Read: [
                'DescribeComponent',
                'GetComponent',
                'GetComponentVersionArtifact',
                'GetConnectivityInfo',
                'GetCoreDevice',
                'GetDeployment',
                'GetServiceRoleForAccount',
                'ListTagsForResource'
            ],
            List: [
                'ListClientDevicesAssociatedWithCoreDevice',
                'ListComponentVersions',
                'ListComponents',
                'ListCoreDevices',
                'ListDeployments',
                'ListEffectiveDeployments',
                'ListInstalledComponents',
                'ResolveComponentCandidates'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.GreengrassV2 = GreengrassV2;
//# sourceMappingURL=data:application/json;base64,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