"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cassandra = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [cassandra](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonkeyspacesforapachecassandra.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Cassandra extends shared_1.PolicyStatement {
    /**
     * Grants permission to alter a keyspace or table
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toAlter() {
        return this.to('Alter');
    }
    /**
     * Grants permission to alter a multiregion keyspace or table
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toAlterMultiRegionResource() {
        return this.to('AlterMultiRegionResource');
    }
    /**
     * Grants permission to create a keyspace or table
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toCreate() {
        return this.to('Create');
    }
    /**
     * Grants permission to create a multiregion keyspace or table
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toCreateMultiRegionResource() {
        return this.to('CreateMultiRegionResource');
    }
    /**
     * Grants permission to drop a keyspace or table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toDrop() {
        return this.to('Drop');
    }
    /**
     * Grants permission to drop a multiregion keyspace or table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toDropMultiRegionResource() {
        return this.to('DropMultiRegionResource');
    }
    /**
     * Grants permission to INSERT, UPDATE or DELETE data in a table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toModify() {
        return this.to('Modify');
    }
    /**
     * Grants permission to INSERT, UPDATE or DELETE data in a multiregion table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toModifyMultiRegionResource() {
        return this.to('ModifyMultiRegionResource');
    }
    /**
     * Grants permission to restore table from a backup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toRestore() {
        return this.to('Restore');
    }
    /**
     * Grants permission to restore multiregion table from a backup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toRestoreMultiRegionTable() {
        return this.to('RestoreMultiRegionTable');
    }
    /**
     * Grants permission to SELECT data from a table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toSelect() {
        return this.to('Select');
    }
    /**
     * Grants permission to SELECT data from a multiregion table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toSelectMultiRegionResource() {
        return this.to('SelectMultiRegionResource');
    }
    /**
     * Grants permission to tag a multiregion keyspace or table
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toTagMultiRegionResource() {
        return this.to('TagMultiRegionResource');
    }
    /**
     * Grants permission to tag a keyspace or table
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag a multiregion keyspace or table
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toUnTagMultiRegionResource() {
        return this.to('UnTagMultiRegionResource');
    }
    /**
     * Grants permission to untag a keyspace or table
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to UPDATE the partitioner in a system table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toUpdatePartitioner() {
        return this.to('UpdatePartitioner');
    }
    /**
     * Adds a resource of type keyspace to the statement
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/what-is.html
     *
     * @param keyspaceName - Identifier for the keyspaceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onKeyspace(keyspaceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cassandra:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:/keyspace/${keyspaceName}`);
    }
    /**
     * Adds a resource of type table to the statement
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/what-is.html
     *
     * @param keyspaceName - Identifier for the keyspaceName.
     * @param tableName - Identifier for the tableName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTable(keyspaceName, tableName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cassandra:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:/keyspace/${keyspaceName}/table/${tableName}`);
    }
    /**
     * Filters actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toAlter()
     * - .toAlterMultiRegionResource()
     * - .toCreate()
     * - .toCreateMultiRegionResource()
     * - .toTagMultiRegionResource()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - keyspace
     * - table
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toAlter()
     * - .toAlterMultiRegionResource()
     * - .toCreate()
     * - .toCreateMultiRegionResource()
     * - .toTagMultiRegionResource()
     * - .toTagResource()
     * - .toUnTagMultiRegionResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [cassandra](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonkeyspacesforapachecassandra.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'cassandra';
        this.accessLevelList = {
            Write: [
                'Alter',
                'AlterMultiRegionResource',
                'Create',
                'CreateMultiRegionResource',
                'Drop',
                'DropMultiRegionResource',
                'Modify',
                'ModifyMultiRegionResource',
                'Restore',
                'RestoreMultiRegionTable',
                'UpdatePartitioner'
            ],
            Read: [
                'Select',
                'SelectMultiRegionResource'
            ],
            Tagging: [
                'TagMultiRegionResource',
                'TagResource',
                'UnTagMultiRegionResource',
                'UntagResource'
            ]
        };
    }
}
exports.Cassandra = Cassandra;
//# sourceMappingURL=data:application/json;base64,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