"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Firehose = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [firehose](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonkinesisfirehose.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Firehose extends shared_1.PolicyStatement {
    /**
     * Grants permission to create a delivery stream
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_CreateDeliveryStream.html
     */
    toCreateDeliveryStream() {
        return this.to('CreateDeliveryStream');
    }
    /**
     * Grants permission to delete a delivery stream and its data
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_DeleteDeliveryStream.html
     */
    toDeleteDeliveryStream() {
        return this.to('DeleteDeliveryStream');
    }
    /**
     * Grants permission to describe the specified delivery stream and gets the status
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_DescribeDeliveryStream.html
     */
    toDescribeDeliveryStream() {
        return this.to('DescribeDeliveryStream');
    }
    /**
     * Grants permission to list your delivery streams
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_ListDeliveryStreams.html
     */
    toListDeliveryStreams() {
        return this.to('ListDeliveryStreams');
    }
    /**
     * Grants permission to list the tags for the specified delivery stream
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_ListTagsForDeliveryStream.html
     */
    toListTagsForDeliveryStream() {
        return this.to('ListTagsForDeliveryStream');
    }
    /**
     * Grants permission to write a single data record into an Amazon Kinesis Firehose delivery stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_PutRecord.html
     */
    toPutRecord() {
        return this.to('PutRecord');
    }
    /**
     * Grants permission to write multiple data records into a delivery stream in a single call, which can achieve higher throughput per producer than when writing single records
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_PutRecordBatch.html
     */
    toPutRecordBatch() {
        return this.to('PutRecordBatch');
    }
    /**
     * Grants permission to enable server-side encryption (SSE) for the delivery stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_StartDeliveryStreamEncryption.html
     */
    toStartDeliveryStreamEncryption() {
        return this.to('StartDeliveryStreamEncryption');
    }
    /**
     * Grants permission to disable the specified destination of the specified delivery stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_StopDeliveryStreamEncryption.html
     */
    toStopDeliveryStreamEncryption() {
        return this.to('StopDeliveryStreamEncryption');
    }
    /**
     * Grants permission to add or update tags for the specified delivery stream
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_TagDeliveryStream.html
     */
    toTagDeliveryStream() {
        return this.to('TagDeliveryStream');
    }
    /**
     * Grants permission to remove tags from the specified delivery stream
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_UntagDeliveryStream.html
     */
    toUntagDeliveryStream() {
        return this.to('UntagDeliveryStream');
    }
    /**
     * Grants permission to update the specified destination of the specified delivery stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_UpdateDestination.html
     */
    toUpdateDestination() {
        return this.to('UpdateDestination');
    }
    /**
     * Adds a resource of type deliverystream to the statement
     *
     * https://docs.aws.amazon.com/firehose/latest/dev/basic-create.html
     *
     * @param deliveryStreamName - Identifier for the deliveryStreamName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDeliverystream(deliveryStreamName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:firehose:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:deliverystream/${deliveryStreamName}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateDeliveryStream()
     * - .toTagDeliveryStream()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - deliverystream
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateDeliveryStream()
     * - .toTagDeliveryStream()
     * - .toUntagDeliveryStream()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [firehose](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonkinesisfirehose.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'firehose';
        this.accessLevelList = {
            Write: [
                'CreateDeliveryStream',
                'DeleteDeliveryStream',
                'PutRecord',
                'PutRecordBatch',
                'StartDeliveryStreamEncryption',
                'StopDeliveryStreamEncryption',
                'UpdateDestination'
            ],
            Read: [
                'DescribeDeliveryStream'
            ],
            List: [
                'ListDeliveryStreams',
                'ListTagsForDeliveryStream'
            ],
            Tagging: [
                'TagDeliveryStream',
                'UntagDeliveryStream'
            ]
        };
    }
}
exports.Firehose = Firehose;
//# sourceMappingURL=data:application/json;base64,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