"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Kafka = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [kafka](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmanagedstreamingforapachekafka.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Kafka extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate one or more Scram Secrets with an Amazon MSK cluster
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:CreateGrant
     * - kms:RetireGrant
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-scram-secrets.html#BatchAssociateScramSecret
     */
    toBatchAssociateScramSecret() {
        return this.to('BatchAssociateScramSecret');
    }
    /**
     * Grants permission to disassociate one or more Scram Secrets from an Amazon MSK cluster
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:RetireGrant
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-scram-secrets.html#BatchDisassociateScramSecret
     */
    toBatchDisassociateScramSecret() {
        return this.to('BatchDisassociateScramSecret');
    }
    /**
     * Grants permission to create an MSK cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - iam:AttachRolePolicy
     * - iam:CreateServiceLinkedRole
     * - iam:PutRolePolicy
     * - kms:CreateGrant
     * - kms:DescribeKey
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters.html#CreateCluster
     */
    toCreateCluster() {
        return this.to('CreateCluster');
    }
    /**
     * Grants permission to create an MSK cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:CreateTags
     * - ec2:CreateVpcEndpoint
     * - ec2:DeleteVpcEndpoints
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcAttribute
     * - ec2:DescribeVpcEndpoints
     * - ec2:DescribeVpcs
     * - iam:AttachRolePolicy
     * - iam:CreateServiceLinkedRole
     * - iam:PutRolePolicy
     * - kms:CreateGrant
     * - kms:DescribeKey
     *
     * https://docs.aws.amazon.com/MSK/2.0/APIReference/v2-clusters.html#CreateClusterV2
     */
    toCreateClusterV2() {
        return this.to('CreateClusterV2');
    }
    /**
     * Grants permission to create an MSK configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/configurations.html#CreateConfiguration
     */
    toCreateConfiguration() {
        return this.to('CreateConfiguration');
    }
    /**
     * Grants permission to create a MSK replicator
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - iam:AttachRolePolicy
     * - iam:CreateServiceLinkedRole
     * - iam:PassRole
     * - iam:PutRolePolicy
     * - kafka:DescribeClusterV2
     * - kafka:GetBootstrapBrokers
     *
     * https://docs.aws.amazon.com/msk/latest/developerguide/v1-replicators.html#CreateReplicator
     */
    toCreateReplicator() {
        return this.to('CreateReplicator');
    }
    /**
     * Grants permission to create a MSK VPC connection
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:CreateTags
     * - ec2:CreateVpcEndpoint
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcAttribute
     * - ec2:DescribeVpcEndpoints
     * - ec2:DescribeVpcs
     * - iam:AttachRolePolicy
     * - iam:CreateServiceLinkedRole
     * - iam:PutRolePolicy
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/vpc-connection.html#CreateVpcConnection
     */
    toCreateVpcConnection() {
        return this.to('CreateVpcConnection');
    }
    /**
     * Grants permission to delete an MSK cluster
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DeleteVpcEndpoints
     * - ec2:DescribeVpcAttribute
     * - ec2:DescribeVpcEndpoints
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn.html#DeleteCluster
     */
    toDeleteCluster() {
        return this.to('DeleteCluster');
    }
    /**
     * Grants permission to delete a cluster resource-based policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-policy.html#DeleteClusterPolicy
     */
    toDeleteClusterPolicy() {
        return this.to('DeleteClusterPolicy');
    }
    /**
     * Grants permission to delete the specified MSK configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/configurations-arn.html#DeleteConfiguration
     */
    toDeleteConfiguration() {
        return this.to('DeleteConfiguration');
    }
    /**
     * Grants permission to delete a MSK replicator
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/latest/developerguide/v1-replicators.html#DeleteReplicator
     */
    toDeleteReplicator() {
        return this.to('DeleteReplicator');
    }
    /**
     * Grants permission to delete a MSK VPC connection
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DeleteVpcEndpoints
     * - ec2:DescribeVpcEndpoints
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/vpc-connection-arn.html#DeleteVpcConnection
     */
    toDeleteVpcConnection() {
        return this.to('DeleteVpcConnection');
    }
    /**
     * Grants permission to describe an MSK cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn.html#DescribeCluster
     */
    toDescribeCluster() {
        return this.to('DescribeCluster');
    }
    /**
     * Grants permission to describe the cluster operation that is specified by the given ARN
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/operations-clusteroperationarn.html#DescribeClusterOperation
     */
    toDescribeClusterOperation() {
        return this.to('DescribeClusterOperation');
    }
    /**
     * Grants permission to describe the cluster operation that is specified by the given ARN
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/MSK/2.0/APIReference/v2-operations-clusteroperationarn.html#DescribeClusterOperationV2
     */
    toDescribeClusterOperationV2() {
        return this.to('DescribeClusterOperationV2');
    }
    /**
     * Grants permission to describe an MSK cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/MSK/2.0/APIReference/v2-clusters-clusterarn.html#DescribeClusterV2
     */
    toDescribeClusterV2() {
        return this.to('DescribeClusterV2');
    }
    /**
     * Grants permission to describe an MSK configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/configurations-arn.html#DescribeConfiguration
     */
    toDescribeConfiguration() {
        return this.to('DescribeConfiguration');
    }
    /**
     * Grants permission to describe an MSK configuration revision
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/configurations-arn-revisions-revision.html#DescribeConfigurationRevision
     */
    toDescribeConfigurationRevision() {
        return this.to('DescribeConfigurationRevision');
    }
    /**
     * Grants permission to describe a MSK replicator
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/msk/latest/developerguide/v1-replicators.html#DescribeReplicator
     */
    toDescribeReplicator() {
        return this.to('DescribeReplicator');
    }
    /**
     * Grants permission to describe a MSK VPC connection
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/vpc-connection-arn.html#DescribeVpcConnection
     */
    toDescribeVpcConnection() {
        return this.to('DescribeVpcConnection');
    }
    /**
     * Grants permission to get connection details for the brokers in an MSK cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-bootstrap-brokers.html#GetBootstrapBrokers
     */
    toGetBootstrapBrokers() {
        return this.to('GetBootstrapBrokers');
    }
    /**
     * Grants permission to describe a cluster resource-based policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-policy.html#GetClusterPolicy
     */
    toGetClusterPolicy() {
        return this.to('GetClusterPolicy');
    }
    /**
     * Grants permission to get a list of the Apache Kafka versions to which you can update an MSK cluster
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/compatible-kafka-versions.html#GetCompatibleKafkaVersions
     */
    toGetCompatibleKafkaVersions() {
        return this.to('GetCompatibleKafkaVersions');
    }
    /**
     * Grants permission to list all MSK VPC connections created for a cluster
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-client-vpc-connections.html#ListClientVpcConnections
     */
    toListClientVpcConnections() {
        return this.to('ListClientVpcConnections');
    }
    /**
     * Grants permission to return a list of all the operations that have been performed on the specified MSK cluster
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-operations.html#ListClusterOperations
     */
    toListClusterOperations() {
        return this.to('ListClusterOperations');
    }
    /**
     * Grants permission to return a list of all the operations that have been performed on the specified MSK cluster
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/MSK/2.0/APIReference/v2-clusters-clusterarn-operations.html#ListClusterOperationsV2
     */
    toListClusterOperationsV2() {
        return this.to('ListClusterOperationsV2');
    }
    /**
     * Grants permission to list all MSK clusters in this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters.html#ListClusters
     */
    toListClusters() {
        return this.to('ListClusters');
    }
    /**
     * Grants permission to list all MSK clusters in this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/MSK/2.0/APIReference/v2-clusters.html#ListClustersV2
     */
    toListClustersV2() {
        return this.to('ListClustersV2');
    }
    /**
     * Grants permission to list all revisions for an MSK configuration in this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/configurations-arn-revisions.html#ListConfigurationRevisions
     */
    toListConfigurationRevisions() {
        return this.to('ListConfigurationRevisions');
    }
    /**
     * Grants permission to list all MSK configurations in this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/configurations.html#ListConfigurations
     */
    toListConfigurations() {
        return this.to('ListConfigurations');
    }
    /**
     * Grants permission to list all Apache Kafka versions supported by Amazon MSK
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/kafka-versions.html#ListKafkaVersions
     */
    toListKafkaVersions() {
        return this.to('ListKafkaVersions');
    }
    /**
     * Grants permission to list brokers in an MSK cluster
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-nodes.html#ListNodes
     */
    toListNodes() {
        return this.to('ListNodes');
    }
    /**
     * Grants permission to list all MSK replicators in this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/msk/latest/developerguide/v1-replicators.html#ListReplicators
     */
    toListReplicators() {
        return this.to('ListReplicators');
    }
    /**
     * Grants permission to list the Scram Secrets associated with an Amazon MSK cluster
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-scram-secrets.html#ListScramSecrets
     */
    toListScramSecrets() {
        return this.to('ListScramSecrets');
    }
    /**
     * Grants permission to list tags of an MSK resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/tags-resourcearn.html#ListTagsForResource
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list all MSK VPC connections that this account uses
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/vpc-connections.html#ListVpcConnections
     */
    toListVpcConnections() {
        return this.to('ListVpcConnections');
    }
    /**
     * Grants permission to create or update the resource-based policy for a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-policy.html#PutClusterPolicy
     */
    toPutClusterPolicy() {
        return this.to('PutClusterPolicy');
    }
    /**
     * Grants permission to reboot broker
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-reboot-broker.html#RebootBroker
     */
    toRebootBroker() {
        return this.to('RebootBroker');
    }
    /**
     * Grants permission to reject a MSK VPC connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-client-vpc-connection.html#RejectClientVpcConnection
     */
    toRejectClientVpcConnection() {
        return this.to('RejectClientVpcConnection');
    }
    /**
     * Grants permission to tag an MSK resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/tags-resourcearn.html#TagResource
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from an MSK resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/tags-resourcearn.html#UntagResource
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update the number of brokers of the MSK cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-nodes-count.html#UpdateBrokerCount
     */
    toUpdateBrokerCount() {
        return this.to('UpdateBrokerCount');
    }
    /**
     * Grants permission to update the storage size of the brokers of the MSK cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-nodes-storage.html#UpdateBrokerStorage
     */
    toUpdateBrokerStorage() {
        return this.to('UpdateBrokerStorage');
    }
    /**
     * Grants permission to update the broker type of an Amazon MSK cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-nodes-type.html#UpdateBrokerType
     */
    toUpdateBrokerType() {
        return this.to('UpdateBrokerType');
    }
    /**
     * Grants permission to update the configuration of the MSK cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-configuration.html#UpdateClusterConfiguration
     */
    toUpdateClusterConfiguration() {
        return this.to('UpdateClusterConfiguration');
    }
    /**
     * Grants permission to update the MSK cluster to the specified Apache Kafka version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-version.html#UpdateClusterKafkaVersion
     */
    toUpdateClusterKafkaVersion() {
        return this.to('UpdateClusterKafkaVersion');
    }
    /**
     * Grants permission to create a new revision of the MSK configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/configurations-arn.html#UpdateConfiguration
     */
    toUpdateConfiguration() {
        return this.to('UpdateConfiguration');
    }
    /**
     * Grants permission to update the connectivity settings for the MSK cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPublicAccessEnabled()
     *
     * Dependent actions:
     * - ec2:DescribeRouteTables
     * - ec2:DescribeSubnets
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-connectivity.html#UpdateConnectivity
     */
    toUpdateConnectivity() {
        return this.to('UpdateConnectivity');
    }
    /**
     * Grants permission to update the monitoring settings for the MSK cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-monitoring.html#UpdateMonitoring
     */
    toUpdateMonitoring() {
        return this.to('UpdateMonitoring');
    }
    /**
     * Grants permission to update the replication info of the MSK replicator
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/latest/developerguide/v1-replicators.html#UpdateReplicationInfo
     */
    toUpdateReplicationInfo() {
        return this.to('UpdateReplicationInfo');
    }
    /**
     * Grants permission to update the security settings for the MSK cluster
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:RetireGrant
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-security.html#UpdateSecurity
     */
    toUpdateSecurity() {
        return this.to('UpdateSecurity');
    }
    /**
     * Grants permission to update the EBS storage (size or provisioned throughput) associated with MSK brokers or set cluster storage mode to TIERED
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-storage.html#UpdateStorage
     */
    toUpdateStorage() {
        return this.to('UpdateStorage');
    }
    /**
     * Adds a resource of type cluster to the statement
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param uuid - Identifier for the uuid.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCluster(clusterName, uuid, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:kafka:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:cluster/${clusterName}/${uuid}`);
    }
    /**
     * Adds a resource of type configuration to the statement
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/configurations-arn.html
     *
     * @param configurationName - Identifier for the configurationName.
     * @param uuid - Identifier for the uuid.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onConfiguration(configurationName, uuid, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:kafka:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:configuration/${configurationName}/${uuid}`);
    }
    /**
     * Adds a resource of type vpc-connection to the statement
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/vpc-connections-arn.html
     *
     * @param vpcOwnerAccount - Identifier for the vpcOwnerAccount.
     * @param clusterOwnerAccount - Identifier for the clusterOwnerAccount.
     * @param clusterName - Identifier for the clusterName.
     * @param uuid - Identifier for the uuid.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVpcConnection(vpcOwnerAccount, clusterOwnerAccount, clusterName, uuid, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:kafka:${region ?? this.defaultRegion}:${vpcOwnerAccount}:vpc-connection/${clusterOwnerAccount}/${clusterName}/${uuid}`);
    }
    /**
     * Adds a resource of type replicator to the statement
     *
     * https://docs.aws.amazon.com/msk/latest/developerguide/v1-replicators.html
     *
     * @param replicatorName - Identifier for the replicatorName.
     * @param uuid - Identifier for the uuid.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onReplicator(replicatorName, uuid, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:kafka:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:replicator/${replicatorName}/${uuid}`);
    }
    /**
     * Adds a resource of type topic to the statement
     *
     * https://docs.aws.amazon.com/msk/latest/developerguide/iam-access-control.html#msk-iam-resources
     *
     * @param clusterName - Identifier for the clusterName.
     * @param clusterUuid - Identifier for the clusterUuid.
     * @param topicName - Identifier for the topicName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onTopic(clusterName, clusterUuid, topicName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:kafka:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:topic/${clusterName}/${clusterUuid}/${topicName}`);
    }
    /**
     * Adds a resource of type group to the statement
     *
     * https://docs.aws.amazon.com/msk/latest/developerguide/iam-access-control.html#msk-iam-resources
     *
     * @param clusterName - Identifier for the clusterName.
     * @param clusterUuid - Identifier for the clusterUuid.
     * @param groupName - Identifier for the groupName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onGroup(clusterName, clusterUuid, groupName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:kafka:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:group/${clusterName}/${clusterUuid}/${groupName}`);
    }
    /**
     * Adds a resource of type transactional-id to the statement
     *
     * https://docs.aws.amazon.com/msk/latest/developerguide/iam-access-control.html#msk-iam-resources
     *
     * @param clusterName - Identifier for the clusterName.
     * @param clusterUuid - Identifier for the clusterUuid.
     * @param transactionalId - Identifier for the transactionalId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onTransactionalId(clusterName, clusterUuid, transactionalId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:kafka:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:transactional-id/${clusterName}/${clusterUuid}/${transactionalId}`);
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateCluster()
     * - .toCreateClusterV2()
     * - .toCreateReplicator()
     * - .toCreateVpcConnection()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - cluster
     * - vpc-connection
     * - replicator
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateCluster()
     * - .toCreateClusterV2()
     * - .toCreateReplicator()
     * - .toCreateVpcConnection()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of public access enabled in the request
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_apachekafkaapisforamazonmskclusters.html#apachekafkaapisforamazonmskclusters-policy-keys
     *
     * Applies to actions:
     * - .toUpdateConnectivity()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifPublicAccessEnabled(value) {
        return this.if(`publicAccessEnabled`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Statement provider for service [kafka](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmanagedstreamingforapachekafka.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'kafka';
        this.accessLevelList = {
            Write: [
                'BatchAssociateScramSecret',
                'BatchDisassociateScramSecret',
                'CreateCluster',
                'CreateClusterV2',
                'CreateConfiguration',
                'CreateReplicator',
                'CreateVpcConnection',
                'DeleteCluster',
                'DeleteClusterPolicy',
                'DeleteConfiguration',
                'DeleteReplicator',
                'DeleteVpcConnection',
                'PutClusterPolicy',
                'RebootBroker',
                'RejectClientVpcConnection',
                'UpdateBrokerCount',
                'UpdateBrokerStorage',
                'UpdateBrokerType',
                'UpdateClusterConfiguration',
                'UpdateClusterKafkaVersion',
                'UpdateConfiguration',
                'UpdateConnectivity',
                'UpdateMonitoring',
                'UpdateReplicationInfo',
                'UpdateSecurity',
                'UpdateStorage'
            ],
            Read: [
                'DescribeCluster',
                'DescribeClusterOperation',
                'DescribeClusterOperationV2',
                'DescribeClusterV2',
                'DescribeConfiguration',
                'DescribeConfigurationRevision',
                'DescribeReplicator',
                'DescribeVpcConnection',
                'GetBootstrapBrokers',
                'GetClusterPolicy',
                'ListTagsForResource'
            ],
            List: [
                'GetCompatibleKafkaVersions',
                'ListClientVpcConnections',
                'ListClusterOperations',
                'ListClusterOperationsV2',
                'ListClusters',
                'ListClustersV2',
                'ListConfigurationRevisions',
                'ListConfigurations',
                'ListKafkaVersions',
                'ListNodes',
                'ListReplicators',
                'ListScramSecrets',
                'ListVpcConnections'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Kafka = Kafka;
//# sourceMappingURL=data:application/json;base64,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