"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Memorydb = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [memorydb](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmemorydb.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Memorydb extends shared_1.PolicyStatement {
    /**
     * Grants permissions to apply service updates
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_BatchUpdateCluster.html
     */
    toBatchUpdateCluster() {
        return this.to('BatchUpdateCluster');
    }
    /**
     * Allows an IAM user or role to connect as a specified MemoryDB user to a node in a cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/devguide/auth-iam.html
     */
    toConnect() {
        return this.to('Connect');
    }
    /**
     * Grants permissions to make a copy of an existing snapshot
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - memorydb:TagResource
     * - s3:DeleteObject
     * - s3:GetBucketAcl
     * - s3:PutObject
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_CopySnapshot.html
     */
    toCopySnapshot() {
        return this.to('CopySnapshot');
    }
    /**
     * Grants permissions to create a new access control list
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - memorydb:TagResource
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_CreateAcl.html
     */
    toCreateAcl() {
        return this.to('CreateAcl');
    }
    /**
     * Grants permissions to create a cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifTLSEnabled()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - memorydb:TagResource
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_CreateCluster.html
     */
    toCreateCluster() {
        return this.to('CreateCluster');
    }
    /**
     * Grants permissions to create a Multi-Region cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifTLSEnabled()
     *
     * Dependent actions:
     * - memorydb:TagResource
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_CreateMultiRegionCluster.html
     */
    toCreateMultiRegionCluster() {
        return this.to('CreateMultiRegionCluster');
    }
    /**
     * Grants permissions to create a new parameter group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - memorydb:TagResource
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_CreateParameterGroup.html
     */
    toCreateParameterGroup() {
        return this.to('CreateParameterGroup');
    }
    /**
     * Grants permissions to create a backup of a cluster at the current point in time
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - memorydb:TagResource
     * - s3:DeleteObject
     * - s3:GetBucketAcl
     * - s3:PutObject
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_CreateSnapshot.html
     */
    toCreateSnapshot() {
        return this.to('CreateSnapshot');
    }
    /**
     * Grants permissions to create a new subnet group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - memorydb:TagResource
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_CreateSubnetGroup.html
     */
    toCreateSubnetGroup() {
        return this.to('CreateSubnetGroup');
    }
    /**
     * Grants permissions to create a new user
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifUserAuthenticationMode()
     *
     * Dependent actions:
     * - memorydb:TagResource
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_CreateUser.html
     */
    toCreateUser() {
        return this.to('CreateUser');
    }
    /**
     * Grants permissions to delete an access control list
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DeleteAcl.html
     */
    toDeleteAcl() {
        return this.to('DeleteAcl');
    }
    /**
     * Grants permissions to delete a previously provisioned cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DeleteCluster.html
     */
    toDeleteCluster() {
        return this.to('DeleteCluster');
    }
    /**
     * Grants permissions to delete a Multi-Region cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DeleteMultiRegionCluster.html
     */
    toDeleteMultiRegionCluster() {
        return this.to('DeleteMultiRegionCluster');
    }
    /**
     * Grants permissions to delete a parameter group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DeleteParameterGroup.html
     */
    toDeleteParameterGroup() {
        return this.to('DeleteParameterGroup');
    }
    /**
     * Grants permissions to delete a snapshot
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DeleteSnapshot.html
     */
    toDeleteSnapshot() {
        return this.to('DeleteSnapshot');
    }
    /**
     * Grants permissions to delete a subnet group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DeleteSubnetGroup.html
     */
    toDeleteSubnetGroup() {
        return this.to('DeleteSubnetGroup');
    }
    /**
     * Grants permissions to delete a user
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DeleteUser.html
     */
    toDeleteUser() {
        return this.to('DeleteUser');
    }
    /**
     * Grants permissions to retrieve information about access control lists
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DescribeAcls.html
     */
    toDescribeAcls() {
        return this.to('DescribeAcls');
    }
    /**
     * Grants permissions to retrieve information about all provisioned clusters if no cluster identifier is specified, or about a specific cluster if a cluster identifier is supplied
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DescribeClusters.html
     */
    toDescribeClusters() {
        return this.to('DescribeClusters');
    }
    /**
     * Grants permissions to list of the available engines and their versions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DescribeEngineVersions.html
     */
    toDescribeEngineVersions() {
        return this.to('DescribeEngineVersions');
    }
    /**
     * Grants permissions to retrieve events related to clusters, subnet groups, and parameter groups
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DescribeEvents.html
     */
    toDescribeEvents() {
        return this.to('DescribeEvents');
    }
    /**
     * Grants permissions to retrieve information about all Multi-Region clusters if no cluster identifier is specified, or about a specific Multi-Region cluster if a cluster identifier is supplied
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DescribeMultiRegionClusters.html
     */
    toDescribeMultiRegionClusters() {
        return this.to('DescribeMultiRegionClusters');
    }
    /**
     * Grants permissions to retrieve information about Multi-Region parameter groups
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DescribeMultiRegionParameterGroups.html
     */
    toDescribeMultiRegionParameterGroups() {
        return this.to('DescribeMultiRegionParameterGroups');
    }
    /**
     * Grants permissions to retrieve a detailed parameter list for a particular Multi-Region parameter group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DescribeMultiRegionParameters.html
     */
    toDescribeMultiRegionParameters() {
        return this.to('DescribeMultiRegionParameters');
    }
    /**
     * Grants permissions to retrieve information about parameter groups
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DescribeParameterGroups.html
     */
    toDescribeParameterGroups() {
        return this.to('DescribeParameterGroups');
    }
    /**
     * Grants permissions to retrieve a detailed parameter list for a particular parameter group
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DescribeParameters.html
     */
    toDescribeParameters() {
        return this.to('DescribeParameters');
    }
    /**
     * Grants permissions to retrieve reserved nodes
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DescribeReservedNodes.html
     */
    toDescribeReservedNodes() {
        return this.to('DescribeReservedNodes');
    }
    /**
     * Grants permissions to retrieve reserved nodes offerings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DescribeReservedNodesOfferings.html
     */
    toDescribeReservedNodesOfferings() {
        return this.to('DescribeReservedNodesOfferings');
    }
    /**
     * Grants permissions to retrieve details of the service updates
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DescribeServiceUpdates.html
     */
    toDescribeServiceUpdates() {
        return this.to('DescribeServiceUpdates');
    }
    /**
     * Grants permissions to retrieve information about cluster snapshots
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DescribeSnapshots.html
     */
    toDescribeSnapshots() {
        return this.to('DescribeSnapshots');
    }
    /**
     * Grants permissions to retrieve a list of subnet group
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DescribeSubnetGroups.html
     */
    toDescribeSubnetGroups() {
        return this.to('DescribeSubnetGroups');
    }
    /**
     * Grants permissions to retrieve information about users
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DescribeUsers.html
     */
    toDescribeUsers() {
        return this.to('DescribeUsers');
    }
    /**
     * Grants permissions to test automatic failover on a specified shard in a cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_FailoverShard.html
     */
    toFailoverShard() {
        return this.to('FailoverShard');
    }
    /**
     * Grants permissions to list available Multi-Region cluster updates
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_ListAllowedMultiRegionClusterUpdates.html
     */
    toListAllowedMultiRegionClusterUpdates() {
        return this.to('ListAllowedMultiRegionClusterUpdates');
    }
    /**
     * Grants permissions to list available node type updates
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_ListAllowedNodeTypeUpdates.html
     */
    toListAllowedNodeTypeUpdates() {
        return this.to('ListAllowedNodeTypeUpdates');
    }
    /**
     * Grants permissions to list cost allocation tags
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_ListTags.html
     */
    toListTags() {
        return this.to('ListTags');
    }
    /**
     * Grants permissions to purchase a new reserved node
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - memorydb:TagResource
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_PurchaseReservedNodesOffering.html
     */
    toPurchaseReservedNodesOffering() {
        return this.to('PurchaseReservedNodesOffering');
    }
    /**
     * Grants permissions to modify the parameters of a parameter group to the engine or system default value
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_ResetParameterGroup.html
     */
    toResetParameterGroup() {
        return this.to('ResetParameterGroup');
    }
    /**
     * Grants permissions to add up to 10 cost allocation tags to the named resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permissions to remove the tags identified by the TagKeys list from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permissions to update an access control list
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_UpdateAcl.html
     */
    toUpdateAcl() {
        return this.to('UpdateAcl');
    }
    /**
     * Grants permissions to update the settings for a cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_UpdateCluster.html
     */
    toUpdateCluster() {
        return this.to('UpdateCluster');
    }
    /**
     * Grants permissions to update the settings for a Multi-Region cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_UpdateMultiRegionCluster.html
     */
    toUpdateMultiRegionCluster() {
        return this.to('UpdateMultiRegionCluster');
    }
    /**
     * Grants permissions to update parameters in a parameter group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_UpdateParameterGroup.html
     */
    toUpdateParameterGroup() {
        return this.to('UpdateParameterGroup');
    }
    /**
     * Grants permissions to update a subnet group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_UpdateSubnetGroup.html
     */
    toUpdateSubnetGroup() {
        return this.to('UpdateSubnetGroup');
    }
    /**
     * Grants permissions to update a user
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifUserAuthenticationMode()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_UpdateUser.html
     */
    toUpdateUser() {
        return this.to('UpdateUser');
    }
    /**
     * Adds a resource of type multiregionparametergroup to the statement
     *
     * https://docs.aws.amazon.com/memorydb/latest/devguide/WhatIs.Components.html
     *
     * @param multiRegionParameterGroupName - Identifier for the multiRegionParameterGroupName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onMultiregionparametergroup(multiRegionParameterGroupName, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:memorydb::${account ?? this.defaultAccount}:multiregionparametergroup/${multiRegionParameterGroupName}`);
    }
    /**
     * Adds a resource of type parametergroup to the statement
     *
     * https://docs.aws.amazon.com/memorydb/latest/devguide/WhatIs.Components.html
     *
     * @param parameterGroupName - Identifier for the parameterGroupName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onParametergroup(parameterGroupName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:memorydb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:parametergroup/${parameterGroupName}`);
    }
    /**
     * Adds a resource of type subnetgroup to the statement
     *
     * https://docs.aws.amazon.com/memorydb/latest/devguide/WhatIs.Components.html
     *
     * @param subnetGroupName - Identifier for the subnetGroupName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSubnetgroup(subnetGroupName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:memorydb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:subnetgroup/${subnetGroupName}`);
    }
    /**
     * Adds a resource of type multiregioncluster to the statement
     *
     * https://docs.aws.amazon.com/memorydb/latest/devguide/WhatIs.Components.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifTLSEnabled()
     */
    onMultiregioncluster(clusterName, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:memorydb::${account ?? this.defaultAccount}:multiregioncluster/${clusterName}`);
    }
    /**
     * Adds a resource of type cluster to the statement
     *
     * https://docs.aws.amazon.com/memorydb/latest/devguide/WhatIs.Components.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCluster(clusterName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:memorydb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:cluster/${clusterName}`);
    }
    /**
     * Adds a resource of type snapshot to the statement
     *
     * https://docs.aws.amazon.com/memorydb/latest/devguide/WhatIs.Components.html
     *
     * @param snapshotName - Identifier for the snapshotName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSnapshot(snapshotName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:memorydb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:snapshot/${snapshotName}`);
    }
    /**
     * Adds a resource of type user to the statement
     *
     * https://docs.aws.amazon.com/memorydb/latest/devguide/WhatIs.Components.html
     *
     * @param userName - Identifier for the userName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onUser(userName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:memorydb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:user/${userName}`);
    }
    /**
     * Adds a resource of type acl to the statement
     *
     * https://docs.aws.amazon.com/memorydb/latest/devguide/WhatIs.Components.html
     *
     * @param aclName - Identifier for the aclName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAcl(aclName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:memorydb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:acl/${aclName}`);
    }
    /**
     * Adds a resource of type reservednode to the statement
     *
     * https://docs.aws.amazon.com/memorydb/latest/devguide/WhatIs.Components.html
     *
     * @param reservationID - Identifier for the reservationID.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onReservednode(reservationID, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:memorydb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:reservednode/${reservationID}`);
    }
    /**
     * Filters actions based on the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCopySnapshot()
     * - .toCreateAcl()
     * - .toCreateCluster()
     * - .toCreateMultiRegionCluster()
     * - .toCreateParameterGroup()
     * - .toCreateSnapshot()
     * - .toCreateSubnetGroup()
     * - .toCreateUser()
     * - .toPurchaseReservedNodesOffering()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toBatchUpdateCluster()
     * - .toConnect()
     * - .toCopySnapshot()
     * - .toCreateAcl()
     * - .toCreateCluster()
     * - .toCreateMultiRegionCluster()
     * - .toCreateSnapshot()
     * - .toDeleteAcl()
     * - .toDeleteCluster()
     * - .toDeleteMultiRegionCluster()
     * - .toDeleteParameterGroup()
     * - .toDeleteSnapshot()
     * - .toDeleteSubnetGroup()
     * - .toDeleteUser()
     * - .toDescribeAcls()
     * - .toDescribeClusters()
     * - .toDescribeMultiRegionClusters()
     * - .toDescribeParameterGroups()
     * - .toDescribeParameters()
     * - .toDescribeReservedNodes()
     * - .toDescribeSnapshots()
     * - .toDescribeSubnetGroups()
     * - .toDescribeUsers()
     * - .toFailoverShard()
     * - .toListAllowedMultiRegionClusterUpdates()
     * - .toListAllowedNodeTypeUpdates()
     * - .toListTags()
     * - .toPurchaseReservedNodesOffering()
     * - .toResetParameterGroup()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateAcl()
     * - .toUpdateCluster()
     * - .toUpdateMultiRegionCluster()
     * - .toUpdateParameterGroup()
     * - .toUpdateSubnetGroup()
     * - .toUpdateUser()
     *
     * Applies to resource types:
     * - parametergroup
     * - subnetgroup
     * - multiregioncluster
     * - cluster
     * - snapshot
     * - user
     * - acl
     * - reservednode
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCopySnapshot()
     * - .toCreateAcl()
     * - .toCreateCluster()
     * - .toCreateMultiRegionCluster()
     * - .toCreateParameterGroup()
     * - .toCreateSnapshot()
     * - .toCreateSubnetGroup()
     * - .toCreateUser()
     * - .toPurchaseReservedNodesOffering()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the TLSEnabled parameter present in the request or defaults to true value if parameter is not present
     *
     * https://docs.aws.amazon.com/memorydb/latest/devguide/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateCluster()
     * - .toCreateMultiRegionCluster()
     *
     * Applies to resource types:
     * - multiregioncluster
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifTLSEnabled(value) {
        return this.if(`TLSEnabled`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the UserAuthenticationMode.Type parameter in the request
     *
     * https://docs.aws.amazon.com/memorydb/latest/devguide/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateUser()
     * - .toUpdateUser()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifUserAuthenticationMode(value, operator) {
        return this.if(`UserAuthenticationMode`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [memorydb](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmemorydb.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'memorydb';
        this.accessLevelList = {
            Write: [
                'BatchUpdateCluster',
                'Connect',
                'CopySnapshot',
                'CreateAcl',
                'CreateCluster',
                'CreateMultiRegionCluster',
                'CreateParameterGroup',
                'CreateSnapshot',
                'CreateSubnetGroup',
                'CreateUser',
                'DeleteAcl',
                'DeleteCluster',
                'DeleteMultiRegionCluster',
                'DeleteParameterGroup',
                'DeleteSnapshot',
                'DeleteSubnetGroup',
                'DeleteUser',
                'FailoverShard',
                'PurchaseReservedNodesOffering',
                'ResetParameterGroup',
                'UpdateAcl',
                'UpdateCluster',
                'UpdateMultiRegionCluster',
                'UpdateParameterGroup',
                'UpdateSubnetGroup',
                'UpdateUser'
            ],
            Read: [
                'DescribeAcls',
                'DescribeClusters',
                'DescribeEngineVersions',
                'DescribeEvents',
                'DescribeMultiRegionClusters',
                'DescribeMultiRegionParameterGroups',
                'DescribeMultiRegionParameters',
                'DescribeParameterGroups',
                'DescribeParameters',
                'DescribeReservedNodes',
                'DescribeReservedNodesOfferings',
                'DescribeServiceUpdates',
                'DescribeSnapshots',
                'DescribeSubnetGroups',
                'DescribeUsers',
                'ListAllowedMultiRegionClusterUpdates',
                'ListAllowedNodeTypeUpdates',
                'ListTags'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Memorydb = Memorydb;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibWVtb3J5ZGIuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJtZW1vcnlkYi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFDQSx5Q0FBeUQ7QUFHekQ7Ozs7R0FJRztBQUNILE1BQWEsUUFBUyxTQUFRLHdCQUFlO0lBRzNDOzs7Ozs7Ozs7Ozs7Ozs7OztPQWlCRztJQUNJLG9CQUFvQjtRQUN6QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksU0FBUztRQUNkLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxTQUFTLENBQUMsQ0FBQztJQUM1QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7O09BaUJHO0lBQ0ksY0FBYztRQUNuQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsY0FBYyxDQUFDLENBQUM7SUFDakMsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7OztPQWNHO0lBQ0ksV0FBVztRQUNoQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsV0FBVyxDQUFDLENBQUM7SUFDOUIsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FxQkc7SUFDSSxlQUFlO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7OztPQWVHO0lBQ0ksMEJBQTBCO1FBQy9CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO0lBQzdDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksc0JBQXNCO1FBQzNCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FpQkc7SUFDSSxnQkFBZ0I7UUFDckIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxtQkFBbUI7UUFDeEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7OztPQWNHO0lBQ0ksWUFBWTtRQUNqQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsWUFBWSxDQUFDLENBQUM7SUFDL0IsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLFdBQVc7UUFDaEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFdBQVcsQ0FBQyxDQUFDO0lBQzlCLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7OztPQWdCRztJQUNJLGVBQWU7UUFDcEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSwwQkFBMEI7UUFDL0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLHNCQUFzQjtRQUMzQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksZ0JBQWdCO1FBQ3JCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO0lBQ25DLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7OztPQWdCRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksWUFBWTtRQUNqQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsWUFBWSxDQUFDLENBQUM7SUFDL0IsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLGNBQWM7UUFDbkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGNBQWMsQ0FBQyxDQUFDO0lBQ2pDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxrQkFBa0I7UUFDdkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFrQixDQUFDLENBQUM7SUFDckMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdCQUF3QjtRQUM3QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO0lBQ25DLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSw2QkFBNkI7UUFDbEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDZCQUE2QixDQUFDLENBQUM7SUFDaEQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9DQUFvQztRQUN6QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0NBQW9DLENBQUMsQ0FBQztJQUN2RCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksK0JBQStCO1FBQ3BDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO0lBQ2xELENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSx5QkFBeUI7UUFDOUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7SUFDNUMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLG9CQUFvQjtRQUN6QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksdUJBQXVCO1FBQzVCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO0lBQzFDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQ0FBZ0M7UUFDckMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdDQUFnQyxDQUFDLENBQUM7SUFDbkQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdCQUF3QjtRQUM3QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksbUJBQW1CO1FBQ3hCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxzQkFBc0I7UUFDM0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUM7SUFDekMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLGVBQWU7UUFDcEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7OztPQWdCRztJQUNJLGVBQWU7UUFDcEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxzQ0FBc0M7UUFDM0MsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHNDQUFzQyxDQUFDLENBQUM7SUFDekQsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLDRCQUE0QjtRQUNqQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNEJBQTRCLENBQUMsQ0FBQztJQUMvQyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksVUFBVTtRQUNmLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxVQUFVLENBQUMsQ0FBQztJQUM3QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7O09BY0c7SUFDSSwrQkFBK0I7UUFDcEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLCtCQUErQixDQUFDLENBQUM7SUFDbEQsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxhQUFhO1FBQ2xCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUNoQyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGVBQWU7UUFDcEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxXQUFXO1FBQ2hCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxXQUFXLENBQUMsQ0FBQztJQUM5QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7T0FnQkc7SUFDSSxlQUFlO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7T0FnQkc7SUFDSSwwQkFBMEI7UUFDL0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLHNCQUFzQjtRQUMzQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksbUJBQW1CO1FBQ3hCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksWUFBWTtRQUNqQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsWUFBWSxDQUFDLENBQUM7SUFDL0IsQ0FBQztJQXlERDs7Ozs7Ozs7T0FRRztJQUNJLDJCQUEyQixDQUFDLDZCQUFxQyxFQUFFLE9BQWdCLEVBQUUsU0FBa0I7UUFDNUcsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsY0FBZSxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsOEJBQStCLDZCQUE4QixFQUFFLENBQUMsQ0FBQztJQUMzSyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksZ0JBQWdCLENBQUMsa0JBQTBCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDdkcsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsYUFBYyxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsbUJBQW9CLGtCQUFtQixFQUFFLENBQUMsQ0FBQztJQUN0TCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksYUFBYSxDQUFDLGVBQXVCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDakcsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsYUFBYyxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsZ0JBQWlCLGVBQWdCLEVBQUUsQ0FBQyxDQUFDO0lBQ2hMLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxvQkFBb0IsQ0FBQyxXQUFtQixFQUFFLE9BQWdCLEVBQUUsU0FBa0I7UUFDbkYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsY0FBZSxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsdUJBQXdCLFdBQVksRUFBRSxDQUFDLENBQUM7SUFDbEosQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLFNBQVMsQ0FBQyxXQUFtQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3pGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGFBQWMsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLFlBQWEsV0FBWSxFQUFFLENBQUMsQ0FBQztJQUN4SyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksVUFBVSxDQUFDLFlBQW9CLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDM0YsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsYUFBYyxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsYUFBYyxZQUFhLEVBQUUsQ0FBQyxDQUFDO0lBQzFLLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxNQUFNLENBQUMsUUFBZ0IsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNuRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixhQUFjLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxTQUFVLFFBQVMsRUFBRSxDQUFDLENBQUM7SUFDbEssQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLEtBQUssQ0FBQyxPQUFlLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDakYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsYUFBYyxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsUUFBUyxPQUFRLEVBQUUsQ0FBQyxDQUFDO0lBQ2hLLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxjQUFjLENBQUMsYUFBcUIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNoRyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixhQUFjLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxpQkFBa0IsYUFBYyxFQUFFLENBQUMsQ0FBQztJQUMvSyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09Bb0JHO0lBQ0ksZUFBZSxDQUFDLE1BQWMsRUFBRSxLQUF3QixFQUFFLFFBQTRCO1FBQzNGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBbUIsTUFBTyxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNoRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQXlERztJQUNJLGdCQUFnQixDQUFDLE1BQWMsRUFBRSxLQUF3QixFQUFFLFFBQTRCO1FBQzVGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBb0IsTUFBTyxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNqRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09Bb0JHO0lBQ0ksWUFBWSxDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDeEUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2pFLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksWUFBWSxDQUFDLEtBQWU7UUFDakMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFlBQVksRUFBRSxDQUFDLE9BQU8sS0FBSyxLQUFLLFdBQVcsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsRUFBRSxNQUFNLENBQUMsQ0FBQztJQUN0RixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSx3QkFBd0IsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ3BGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQzVFLENBQUM7SUFFRDs7O09BR0c7SUFDSCxZQUFZLEtBQWdDO1FBQzFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztRQS9qQ1Isa0JBQWEsR0FBRyxVQUFVLENBQUM7UUE0dEJ4QixvQkFBZSxHQUFvQjtZQUMzQyxLQUFLLEVBQUU7Z0JBQ0wsb0JBQW9CO2dCQUNwQixTQUFTO2dCQUNULGNBQWM7Z0JBQ2QsV0FBVztnQkFDWCxlQUFlO2dCQUNmLDBCQUEwQjtnQkFDMUIsc0JBQXNCO2dCQUN0QixnQkFBZ0I7Z0JBQ2hCLG1CQUFtQjtnQkFDbkIsWUFBWTtnQkFDWixXQUFXO2dCQUNYLGVBQWU7Z0JBQ2YsMEJBQTBCO2dCQUMxQixzQkFBc0I7Z0JBQ3RCLGdCQUFnQjtnQkFDaEIsbUJBQW1CO2dCQUNuQixZQUFZO2dCQUNaLGVBQWU7Z0JBQ2YsK0JBQStCO2dCQUMvQixxQkFBcUI7Z0JBQ3JCLFdBQVc7Z0JBQ1gsZUFBZTtnQkFDZiwwQkFBMEI7Z0JBQzFCLHNCQUFzQjtnQkFDdEIsbUJBQW1CO2dCQUNuQixZQUFZO2FBQ2I7WUFDRCxJQUFJLEVBQUU7Z0JBQ0osY0FBYztnQkFDZCxrQkFBa0I7Z0JBQ2xCLHdCQUF3QjtnQkFDeEIsZ0JBQWdCO2dCQUNoQiw2QkFBNkI7Z0JBQzdCLG9DQUFvQztnQkFDcEMsK0JBQStCO2dCQUMvQix5QkFBeUI7Z0JBQ3pCLG9CQUFvQjtnQkFDcEIsdUJBQXVCO2dCQUN2QixnQ0FBZ0M7Z0JBQ2hDLHdCQUF3QjtnQkFDeEIsbUJBQW1CO2dCQUNuQixzQkFBc0I7Z0JBQ3RCLGVBQWU7Z0JBQ2Ysc0NBQXNDO2dCQUN0Qyw0QkFBNEI7Z0JBQzVCLFVBQVU7YUFDWDtZQUNELE9BQU8sRUFBRTtnQkFDUCxhQUFhO2dCQUNiLGVBQWU7YUFDaEI7U0FDRixDQUFDO0lBK1NGLENBQUM7Q0FDRjtBQWxrQ0QsNEJBa2tDQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEFjY2Vzc0xldmVsTGlzdCB9IGZyb20gJy4uLy4uL3NoYXJlZC9hY2Nlc3MtbGV2ZWwnO1xuaW1wb3J0IHsgUG9saWN5U3RhdGVtZW50LCBPcGVyYXRvciB9IGZyb20gJy4uLy4uL3NoYXJlZCc7XG5pbXBvcnQgeyBhd3NfaWFtIGFzIGlhbSB9IGZyb20gXCJhd3MtY2RrLWxpYlwiO1xuXG4vKipcbiAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbbWVtb3J5ZGJdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXJ2aWNlLWF1dGhvcml6YXRpb24vbGF0ZXN0L3JlZmVyZW5jZS9saXN0X2FtYXpvbm1lbW9yeWRiLmh0bWwpLlxuICpcbiAqIEBwYXJhbSBzaWQgW1NJRF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19zaWQuaHRtbCkgb2YgdGhlIHN0YXRlbWVudFxuICovXG5leHBvcnQgY2xhc3MgTWVtb3J5ZGIgZXh0ZW5kcyBQb2xpY3lTdGF0ZW1lbnQge1xuICBwdWJsaWMgc2VydmljZVByZWZpeCA9ICdtZW1vcnlkYic7XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9ucyB0byBhcHBseSBzZXJ2aWNlIHVwZGF0ZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gZWMyOkNyZWF0ZU5ldHdvcmtJbnRlcmZhY2VcbiAgICogLSBlYzI6RGVsZXRlTmV0d29ya0ludGVyZmFjZVxuICAgKiAtIGVjMjpEZXNjcmliZU5ldHdvcmtJbnRlcmZhY2VzXG4gICAqIC0gZWMyOkRlc2NyaWJlU3VibmV0c1xuICAgKiAtIGVjMjpEZXNjcmliZVZwY3NcbiAgICogLSBzMzpHZXRPYmplY3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lbW9yeWRiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0JhdGNoVXBkYXRlQ2x1c3Rlci5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9CYXRjaFVwZGF0ZUNsdXN0ZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0JhdGNoVXBkYXRlQ2x1c3RlcicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFsbG93cyBhbiBJQU0gdXNlciBvciByb2xlIHRvIGNvbm5lY3QgYXMgYSBzcGVjaWZpZWQgTWVtb3J5REIgdXNlciB0byBhIG5vZGUgaW4gYSBjbHVzdGVyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZW1vcnlkYi9sYXRlc3QvZGV2Z3VpZGUvYXV0aC1pYW0uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ29ubmVjdCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ29ubmVjdCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9ucyB0byBtYWtlIGEgY29weSBvZiBhbiBleGlzdGluZyBzbmFwc2hvdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBtZW1vcnlkYjpUYWdSZXNvdXJjZVxuICAgKiAtIHMzOkRlbGV0ZU9iamVjdFxuICAgKiAtIHMzOkdldEJ1Y2tldEFjbFxuICAgKiAtIHMzOlB1dE9iamVjdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVtb3J5ZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ29weVNuYXBzaG90Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NvcHlTbmFwc2hvdCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ29weVNuYXBzaG90Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb25zIHRvIGNyZWF0ZSBhIG5ldyBhY2Nlc3MgY29udHJvbCBsaXN0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIG1lbW9yeWRiOlRhZ1Jlc291cmNlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZW1vcnlkYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVBY2wuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlQWNsKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVBY2wnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbnMgdG8gY3JlYXRlIGEgY2x1c3RlclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKiAtIC5pZlRMU0VuYWJsZWQoKVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBlYzI6Q3JlYXRlTmV0d29ya0ludGVyZmFjZVxuICAgKiAtIGVjMjpEZWxldGVOZXR3b3JrSW50ZXJmYWNlXG4gICAqIC0gZWMyOkRlc2NyaWJlTmV0d29ya0ludGVyZmFjZXNcbiAgICogLSBlYzI6RGVzY3JpYmVTdWJuZXRzXG4gICAqIC0gZWMyOkRlc2NyaWJlVnBjc1xuICAgKiAtIG1lbW9yeWRiOlRhZ1Jlc291cmNlXG4gICAqIC0gczM6R2V0T2JqZWN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZW1vcnlkYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVDbHVzdGVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUNsdXN0ZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZUNsdXN0ZXInKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbnMgdG8gY3JlYXRlIGEgTXVsdGktUmVnaW9uIGNsdXN0ZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICogLSAuaWZUTFNFbmFibGVkKClcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gbWVtb3J5ZGI6VGFnUmVzb3VyY2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lbW9yeWRiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZU11bHRpUmVnaW9uQ2x1c3Rlci5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVNdWx0aVJlZ2lvbkNsdXN0ZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZU11bHRpUmVnaW9uQ2x1c3RlcicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9ucyB0byBjcmVhdGUgYSBuZXcgcGFyYW1ldGVyIGdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gbWVtb3J5ZGI6VGFnUmVzb3VyY2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lbW9yeWRiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZVBhcmFtZXRlckdyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVBhcmFtZXRlckdyb3VwKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVQYXJhbWV0ZXJHcm91cCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9ucyB0byBjcmVhdGUgYSBiYWNrdXAgb2YgYSBjbHVzdGVyIGF0IHRoZSBjdXJyZW50IHBvaW50IGluIHRpbWVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gbWVtb3J5ZGI6VGFnUmVzb3VyY2VcbiAgICogLSBzMzpEZWxldGVPYmplY3RcbiAgICogLSBzMzpHZXRCdWNrZXRBY2xcbiAgICogLSBzMzpQdXRPYmplY3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lbW9yeWRiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZVNuYXBzaG90Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVNuYXBzaG90KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVTbmFwc2hvdCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9ucyB0byBjcmVhdGUgYSBuZXcgc3VibmV0IGdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gbWVtb3J5ZGI6VGFnUmVzb3VyY2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lbW9yeWRiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZVN1Ym5ldEdyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVN1Ym5ldEdyb3VwKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVTdWJuZXRHcm91cCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9ucyB0byBjcmVhdGUgYSBuZXcgdXNlclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqIC0gLmlmVXNlckF1dGhlbnRpY2F0aW9uTW9kZSgpXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIG1lbW9yeWRiOlRhZ1Jlc291cmNlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZW1vcnlkYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVVc2VyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVVzZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZVVzZXInKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbnMgdG8gZGVsZXRlIGFuIGFjY2VzcyBjb250cm9sIGxpc3RcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lbW9yeWRiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUFjbC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVBY2woKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZUFjbCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9ucyB0byBkZWxldGUgYSBwcmV2aW91c2x5IHByb3Zpc2lvbmVkIGNsdXN0ZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gZWMyOkNyZWF0ZU5ldHdvcmtJbnRlcmZhY2VcbiAgICogLSBlYzI6RGVsZXRlTmV0d29ya0ludGVyZmFjZVxuICAgKiAtIGVjMjpEZXNjcmliZU5ldHdvcmtJbnRlcmZhY2VzXG4gICAqIC0gZWMyOkRlc2NyaWJlU3VibmV0c1xuICAgKiAtIGVjMjpEZXNjcmliZVZwY3NcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lbW9yeWRiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUNsdXN0ZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlQ2x1c3RlcigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlQ2x1c3RlcicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9ucyB0byBkZWxldGUgYSBNdWx0aS1SZWdpb24gY2x1c3RlclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVtb3J5ZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlTXVsdGlSZWdpb25DbHVzdGVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZU11bHRpUmVnaW9uQ2x1c3RlcigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlTXVsdGlSZWdpb25DbHVzdGVyJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb25zIHRvIGRlbGV0ZSBhIHBhcmFtZXRlciBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVtb3J5ZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlUGFyYW1ldGVyR3JvdXAuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlUGFyYW1ldGVyR3JvdXAoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZVBhcmFtZXRlckdyb3VwJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb25zIHRvIGRlbGV0ZSBhIHNuYXBzaG90XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZW1vcnlkYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVTbmFwc2hvdC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVTbmFwc2hvdCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlU25hcHNob3QnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbnMgdG8gZGVsZXRlIGEgc3VibmV0IGdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIGVjMjpDcmVhdGVOZXR3b3JrSW50ZXJmYWNlXG4gICAqIC0gZWMyOkRlbGV0ZU5ldHdvcmtJbnRlcmZhY2VcbiAgICogLSBlYzI6RGVzY3JpYmVOZXR3b3JrSW50ZXJmYWNlc1xuICAgKiAtIGVjMjpEZXNjcmliZVN1Ym5ldHNcbiAgICogLSBlYzI6RGVzY3JpYmVWcGNzXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZW1vcnlkYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVTdWJuZXRHcm91cC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVTdWJuZXRHcm91cCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlU3VibmV0R3JvdXAnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbnMgdG8gZGVsZXRlIGEgdXNlclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVtb3J5ZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlVXNlci5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVVc2VyKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVVc2VyJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb25zIHRvIHJldHJpZXZlIGluZm9ybWF0aW9uIGFib3V0IGFjY2VzcyBjb250cm9sIGxpc3RzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lbW9yeWRiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlQWNscy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZUFjbHMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlQWNscycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9ucyB0byByZXRyaWV2ZSBpbmZvcm1hdGlvbiBhYm91dCBhbGwgcHJvdmlzaW9uZWQgY2x1c3RlcnMgaWYgbm8gY2x1c3RlciBpZGVudGlmaWVyIGlzIHNwZWNpZmllZCwgb3IgYWJvdXQgYSBzcGVjaWZpYyBjbHVzdGVyIGlmIGEgY2x1c3RlciBpZGVudGlmaWVyIGlzIHN1cHBsaWVkXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lbW9yeWRiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlQ2x1c3RlcnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVDbHVzdGVycygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVDbHVzdGVycycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9ucyB0byBsaXN0IG9mIHRoZSBhdmFpbGFibGUgZW5naW5lcyBhbmQgdGhlaXIgdmVyc2lvbnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZW1vcnlkYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUVuZ2luZVZlcnNpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlRW5naW5lVmVyc2lvbnMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlRW5naW5lVmVyc2lvbnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbnMgdG8gcmV0cmlldmUgZXZlbnRzIHJlbGF0ZWQgdG8gY2x1c3RlcnMsIHN1Ym5ldCBncm91cHMsIGFuZCBwYXJhbWV0ZXIgZ3JvdXBzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVtb3J5ZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVFdmVudHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVFdmVudHMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlRXZlbnRzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb25zIHRvIHJldHJpZXZlIGluZm9ybWF0aW9uIGFib3V0IGFsbCBNdWx0aS1SZWdpb24gY2x1c3RlcnMgaWYgbm8gY2x1c3RlciBpZGVudGlmaWVyIGlzIHNwZWNpZmllZCwgb3IgYWJvdXQgYSBzcGVjaWZpYyBNdWx0aS1SZWdpb24gY2x1c3RlciBpZiBhIGNsdXN0ZXIgaWRlbnRpZmllciBpcyBzdXBwbGllZFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZW1vcnlkYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZU11bHRpUmVnaW9uQ2x1c3RlcnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVNdWx0aVJlZ2lvbkNsdXN0ZXJzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZU11bHRpUmVnaW9uQ2x1c3RlcnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbnMgdG8gcmV0cmlldmUgaW5mb3JtYXRpb24gYWJvdXQgTXVsdGktUmVnaW9uIHBhcmFtZXRlciBncm91cHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZW1vcnlkYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZU11bHRpUmVnaW9uUGFyYW1ldGVyR3JvdXBzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlTXVsdGlSZWdpb25QYXJhbWV0ZXJHcm91cHMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlTXVsdGlSZWdpb25QYXJhbWV0ZXJHcm91cHMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbnMgdG8gcmV0cmlldmUgYSBkZXRhaWxlZCBwYXJhbWV0ZXIgbGlzdCBmb3IgYSBwYXJ0aWN1bGFyIE11bHRpLVJlZ2lvbiBwYXJhbWV0ZXIgZ3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZW1vcnlkYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZU11bHRpUmVnaW9uUGFyYW1ldGVycy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZU11bHRpUmVnaW9uUGFyYW1ldGVycygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVNdWx0aVJlZ2lvblBhcmFtZXRlcnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbnMgdG8gcmV0cmlldmUgaW5mb3JtYXRpb24gYWJvdXQgcGFyYW1ldGVyIGdyb3Vwc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZW1vcnlkYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZVBhcmFtZXRlckdyb3Vwcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZVBhcmFtZXRlckdyb3VwcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVQYXJhbWV0ZXJHcm91cHMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbnMgdG8gcmV0cmlldmUgYSBkZXRhaWxlZCBwYXJhbWV0ZXIgbGlzdCBmb3IgYSBwYXJ0aWN1bGFyIHBhcmFtZXRlciBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZW1vcnlkYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZVBhcmFtZXRlcnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVQYXJhbWV0ZXJzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZVBhcmFtZXRlcnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbnMgdG8gcmV0cmlldmUgcmVzZXJ2ZWQgbm9kZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVtb3J5ZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVSZXNlcnZlZE5vZGVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlUmVzZXJ2ZWROb2RlcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVSZXNlcnZlZE5vZGVzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb25zIHRvIHJldHJpZXZlIHJlc2VydmVkIG5vZGVzIG9mZmVyaW5nc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lbW9yeWRiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlUmVzZXJ2ZWROb2Rlc09mZmVyaW5ncy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZVJlc2VydmVkTm9kZXNPZmZlcmluZ3MoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlUmVzZXJ2ZWROb2Rlc09mZmVyaW5ncycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9ucyB0byByZXRyaWV2ZSBkZXRhaWxzIG9mIHRoZSBzZXJ2aWNlIHVwZGF0ZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZW1vcnlkYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZVNlcnZpY2VVcGRhdGVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlU2VydmljZVVwZGF0ZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlU2VydmljZVVwZGF0ZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbnMgdG8gcmV0cmlldmUgaW5mb3JtYXRpb24gYWJvdXQgY2x1c3RlciBzbmFwc2hvdHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVtb3J5ZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVTbmFwc2hvdHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVTbmFwc2hvdHMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlU25hcHNob3RzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb25zIHRvIHJldHJpZXZlIGEgbGlzdCBvZiBzdWJuZXQgZ3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVtb3J5ZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVTdWJuZXRHcm91cHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVTdWJuZXRHcm91cHMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlU3VibmV0R3JvdXBzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb25zIHRvIHJldHJpZXZlIGluZm9ybWF0aW9uIGFib3V0IHVzZXJzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lbW9yeWRiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlVXNlcnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVVc2VycygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVVc2VycycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9ucyB0byB0ZXN0IGF1dG9tYXRpYyBmYWlsb3ZlciBvbiBhIHNwZWNpZmllZCBzaGFyZCBpbiBhIGNsdXN0ZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gZWMyOkNyZWF0ZU5ldHdvcmtJbnRlcmZhY2VcbiAgICogLSBlYzI6RGVsZXRlTmV0d29ya0ludGVyZmFjZVxuICAgKiAtIGVjMjpEZXNjcmliZU5ldHdvcmtJbnRlcmZhY2VzXG4gICAqIC0gZWMyOkRlc2NyaWJlU3VibmV0c1xuICAgKiAtIGVjMjpEZXNjcmliZVZwY3NcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lbW9yeWRiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0ZhaWxvdmVyU2hhcmQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRmFpbG92ZXJTaGFyZCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRmFpbG92ZXJTaGFyZCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9ucyB0byBsaXN0IGF2YWlsYWJsZSBNdWx0aS1SZWdpb24gY2x1c3RlciB1cGRhdGVzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lbW9yeWRiL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RBbGxvd2VkTXVsdGlSZWdpb25DbHVzdGVyVXBkYXRlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0QWxsb3dlZE11bHRpUmVnaW9uQ2x1c3RlclVwZGF0ZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RBbGxvd2VkTXVsdGlSZWdpb25DbHVzdGVyVXBkYXRlcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9ucyB0byBsaXN0IGF2YWlsYWJsZSBub2RlIHR5cGUgdXBkYXRlc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZW1vcnlkYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0QWxsb3dlZE5vZGVUeXBlVXBkYXRlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0QWxsb3dlZE5vZGVUeXBlVXBkYXRlcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdEFsbG93ZWROb2RlVHlwZVVwZGF0ZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbnMgdG8gbGlzdCBjb3N0IGFsbG9jYXRpb24gdGFnc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZW1vcnlkYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0VGFncy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0VGFncygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdFRhZ3MnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbnMgdG8gcHVyY2hhc2UgYSBuZXcgcmVzZXJ2ZWQgbm9kZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBtZW1vcnlkYjpUYWdSZXNvdXJjZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVtb3J5ZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUHVyY2hhc2VSZXNlcnZlZE5vZGVzT2ZmZXJpbmcuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUHVyY2hhc2VSZXNlcnZlZE5vZGVzT2ZmZXJpbmcoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1B1cmNoYXNlUmVzZXJ2ZWROb2Rlc09mZmVyaW5nJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb25zIHRvIG1vZGlmeSB0aGUgcGFyYW1ldGVycyBvZiBhIHBhcmFtZXRlciBncm91cCB0byB0aGUgZW5naW5lIG9yIHN5c3RlbSBkZWZhdWx0IHZhbHVlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZW1vcnlkYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9SZXNldFBhcmFtZXRlckdyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1Jlc2V0UGFyYW1ldGVyR3JvdXAoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1Jlc2V0UGFyYW1ldGVyR3JvdXAnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbnMgdG8gYWRkIHVwIHRvIDEwIGNvc3QgYWxsb2NhdGlvbiB0YWdzIHRvIHRoZSBuYW1lZCByZXNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFRhZ2dpbmdcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZW1vcnlkYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9UYWdSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9UYWdSZXNvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVGFnUmVzb3VyY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbnMgdG8gcmVtb3ZlIHRoZSB0YWdzIGlkZW50aWZpZWQgYnkgdGhlIFRhZ0tleXMgbGlzdCBmcm9tIGEgcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVtb3J5ZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVW50YWdSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VbnRhZ1Jlc291cmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVbnRhZ1Jlc291cmNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb25zIHRvIHVwZGF0ZSBhbiBhY2Nlc3MgY29udHJvbCBsaXN0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZW1vcnlkYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVBY2wuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlQWNsKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVBY2wnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbnMgdG8gdXBkYXRlIHRoZSBzZXR0aW5ncyBmb3IgYSBjbHVzdGVyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIGVjMjpDcmVhdGVOZXR3b3JrSW50ZXJmYWNlXG4gICAqIC0gZWMyOkRlbGV0ZU5ldHdvcmtJbnRlcmZhY2VcbiAgICogLSBlYzI6RGVzY3JpYmVOZXR3b3JrSW50ZXJmYWNlc1xuICAgKiAtIGVjMjpEZXNjcmliZVN1Ym5ldHNcbiAgICogLSBlYzI6RGVzY3JpYmVWcGNzXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZW1vcnlkYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVDbHVzdGVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUNsdXN0ZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZUNsdXN0ZXInKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbnMgdG8gdXBkYXRlIHRoZSBzZXR0aW5ncyBmb3IgYSBNdWx0aS1SZWdpb24gY2x1c3RlclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBlYzI6Q3JlYXRlTmV0d29ya0ludGVyZmFjZVxuICAgKiAtIGVjMjpEZWxldGVOZXR3b3JrSW50ZXJmYWNlXG4gICAqIC0gZWMyOkRlc2NyaWJlTmV0d29ya0ludGVyZmFjZXNcbiAgICogLSBlYzI6RGVzY3JpYmVTdWJuZXRzXG4gICAqIC0gZWMyOkRlc2NyaWJlVnBjc1xuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVtb3J5ZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlTXVsdGlSZWdpb25DbHVzdGVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZU11bHRpUmVnaW9uQ2x1c3RlcigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlTXVsdGlSZWdpb25DbHVzdGVyJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb25zIHRvIHVwZGF0ZSBwYXJhbWV0ZXJzIGluIGEgcGFyYW1ldGVyIGdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZW1vcnlkYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVQYXJhbWV0ZXJHcm91cC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVQYXJhbWV0ZXJHcm91cCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlUGFyYW1ldGVyR3JvdXAnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbnMgdG8gdXBkYXRlIGEgc3VibmV0IGdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZW1vcnlkYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVTdWJuZXRHcm91cC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVTdWJuZXRHcm91cCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlU3VibmV0R3JvdXAnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbnMgdG8gdXBkYXRlIGEgdXNlclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKiAtIC5pZlVzZXJBdXRoZW50aWNhdGlvbk1vZGUoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVtb3J5ZGIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlVXNlci5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVVc2VyKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVVc2VyJyk7XG4gIH1cblxuICBwcm90ZWN0ZWQgYWNjZXNzTGV2ZWxMaXN0OiBBY2Nlc3NMZXZlbExpc3QgPSB7XG4gICAgV3JpdGU6IFtcbiAgICAgICdCYXRjaFVwZGF0ZUNsdXN0ZXInLFxuICAgICAgJ0Nvbm5lY3QnLFxuICAgICAgJ0NvcHlTbmFwc2hvdCcsXG4gICAgICAnQ3JlYXRlQWNsJyxcbiAgICAgICdDcmVhdGVDbHVzdGVyJyxcbiAgICAgICdDcmVhdGVNdWx0aVJlZ2lvbkNsdXN0ZXInLFxuICAgICAgJ0NyZWF0ZVBhcmFtZXRlckdyb3VwJyxcbiAgICAgICdDcmVhdGVTbmFwc2hvdCcsXG4gICAgICAnQ3JlYXRlU3VibmV0R3JvdXAnLFxuICAgICAgJ0NyZWF0ZVVzZXInLFxuICAgICAgJ0RlbGV0ZUFjbCcsXG4gICAgICAnRGVsZXRlQ2x1c3RlcicsXG4gICAgICAnRGVsZXRlTXVsdGlSZWdpb25DbHVzdGVyJyxcbiAgICAgICdEZWxldGVQYXJhbWV0ZXJHcm91cCcsXG4gICAgICAnRGVsZXRlU25hcHNob3QnLFxuICAgICAgJ0RlbGV0ZVN1Ym5ldEdyb3VwJyxcbiAgICAgICdEZWxldGVVc2VyJyxcbiAgICAgICdGYWlsb3ZlclNoYXJkJyxcbiAgICAgICdQdXJjaGFzZVJlc2VydmVkTm9kZXNPZmZlcmluZycsXG4gICAgICAnUmVzZXRQYXJhbWV0ZXJHcm91cCcsXG4gICAgICAnVXBkYXRlQWNsJyxcbiAgICAgICdVcGRhdGVDbHVzdGVyJyxcbiAgICAgICdVcGRhdGVNdWx0aVJlZ2lvbkNsdXN0ZXInLFxuICAgICAgJ1VwZGF0ZVBhcmFtZXRlckdyb3VwJyxcbiAgICAgICdVcGRhdGVTdWJuZXRHcm91cCcsXG4gICAgICAnVXBkYXRlVXNlcidcbiAgICBdLFxuICAgIFJlYWQ6IFtcbiAgICAgICdEZXNjcmliZUFjbHMnLFxuICAgICAgJ0Rlc2NyaWJlQ2x1c3RlcnMnLFxuICAgICAgJ0Rlc2NyaWJlRW5naW5lVmVyc2lvbnMnLFxuICAgICAgJ0Rlc2NyaWJlRXZlbnRzJyxcbiAgICAgICdEZXNjcmliZU11bHRpUmVnaW9uQ2x1c3RlcnMnLFxuICAgICAgJ0Rlc2NyaWJlTXVsdGlSZWdpb25QYXJhbWV0ZXJHcm91cHMnLFxuICAgICAgJ0Rlc2NyaWJlTXVsdGlSZWdpb25QYXJhbWV0ZXJzJyxcbiAgICAgICdEZXNjcmliZVBhcmFtZXRlckdyb3VwcycsXG4gICAgICAnRGVzY3JpYmVQYXJhbWV0ZXJzJyxcbiAgICAgICdEZXNjcmliZVJlc2VydmVkTm9kZXMnLFxuICAgICAgJ0Rlc2NyaWJlUmVzZXJ2ZWROb2Rlc09mZmVyaW5ncycsXG4gICAgICAnRGVzY3JpYmVTZXJ2aWNlVXBkYXRlcycsXG4gICAgICAnRGVzY3JpYmVTbmFwc2hvdHMnLFxuICAgICAgJ0Rlc2NyaWJlU3VibmV0R3JvdXBzJyxcbiAgICAgICdEZXNjcmliZVVzZXJzJyxcbiAgICAgICdMaXN0QWxsb3dlZE11bHRpUmVnaW9uQ2x1c3RlclVwZGF0ZXMnLFxuICAgICAgJ0xpc3RBbGxvd2VkTm9kZVR5cGVVcGRhdGVzJyxcbiAgICAgICdMaXN0VGFncydcbiAgICBdLFxuICAgIFRhZ2dpbmc6IFtcbiAgICAgICdUYWdSZXNvdXJjZScsXG4gICAgICAnVW50YWdSZXNvdXJjZSdcbiAgICBdXG4gIH07XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIG11bHRpcmVnaW9ucGFyYW1ldGVyZ3JvdXAgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVtb3J5ZGIvbGF0ZXN0L2Rldmd1aWRlL1doYXRJcy5Db21wb25lbnRzLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIG11bHRpUmVnaW9uUGFyYW1ldGVyR3JvdXBOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIG11bHRpUmVnaW9uUGFyYW1ldGVyR3JvdXBOYW1lLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqL1xuICBwdWJsaWMgb25NdWx0aXJlZ2lvbnBhcmFtZXRlcmdyb3VwKG11bHRpUmVnaW9uUGFyYW1ldGVyR3JvdXBOYW1lOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9Om1lbW9yeWRiOjokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTptdWx0aXJlZ2lvbnBhcmFtZXRlcmdyb3VwLyR7IG11bHRpUmVnaW9uUGFyYW1ldGVyR3JvdXBOYW1lIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBwYXJhbWV0ZXJncm91cCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZW1vcnlkYi9sYXRlc3QvZGV2Z3VpZGUvV2hhdElzLkNvbXBvbmVudHMuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gcGFyYW1ldGVyR3JvdXBOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIHBhcmFtZXRlckdyb3VwTmFtZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uUGFyYW1ldGVyZ3JvdXAocGFyYW1ldGVyR3JvdXBOYW1lOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06bWVtb3J5ZGI6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OnBhcmFtZXRlcmdyb3VwLyR7IHBhcmFtZXRlckdyb3VwTmFtZSB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgc3VibmV0Z3JvdXAgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVtb3J5ZGIvbGF0ZXN0L2Rldmd1aWRlL1doYXRJcy5Db21wb25lbnRzLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHN1Ym5ldEdyb3VwTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBzdWJuZXRHcm91cE5hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvblN1Ym5ldGdyb3VwKHN1Ym5ldEdyb3VwTmFtZTogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9Om1lbW9yeWRiOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpzdWJuZXRncm91cC8keyBzdWJuZXRHcm91cE5hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIG11bHRpcmVnaW9uY2x1c3RlciB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZW1vcnlkYi9sYXRlc3QvZGV2Z3VpZGUvV2hhdElzLkNvbXBvbmVudHMuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gY2x1c3Rlck5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgY2x1c3Rlck5hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqIC0gLmlmVExTRW5hYmxlZCgpXG4gICAqL1xuICBwdWJsaWMgb25NdWx0aXJlZ2lvbmNsdXN0ZXIoY2x1c3Rlck5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06bWVtb3J5ZGI6OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9Om11bHRpcmVnaW9uY2x1c3Rlci8keyBjbHVzdGVyTmFtZSB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgY2x1c3RlciB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZW1vcnlkYi9sYXRlc3QvZGV2Z3VpZGUvV2hhdElzLkNvbXBvbmVudHMuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gY2x1c3Rlck5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgY2x1c3Rlck5hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkNsdXN0ZXIoY2x1c3Rlck5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTptZW1vcnlkYjokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06Y2x1c3Rlci8keyBjbHVzdGVyTmFtZSB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgc25hcHNob3QgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVtb3J5ZGIvbGF0ZXN0L2Rldmd1aWRlL1doYXRJcy5Db21wb25lbnRzLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHNuYXBzaG90TmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBzbmFwc2hvdE5hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvblNuYXBzaG90KHNuYXBzaG90TmFtZTogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9Om1lbW9yeWRiOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpzbmFwc2hvdC8keyBzbmFwc2hvdE5hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHVzZXIgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVtb3J5ZGIvbGF0ZXN0L2Rldmd1aWRlL1doYXRJcy5Db21wb25lbnRzLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHVzZXJOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIHVzZXJOYW1lLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25Vc2VyKHVzZXJOYW1lOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06bWVtb3J5ZGI6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OnVzZXIvJHsgdXNlck5hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGFjbCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZW1vcnlkYi9sYXRlc3QvZGV2Z3VpZGUvV2hhdElzLkNvbXBvbmVudHMuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gYWNsTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBhY2xOYW1lLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25BY2woYWNsTmFtZTogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9Om1lbW9yeWRiOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTphY2wvJHsgYWNsTmFtZSB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgcmVzZXJ2ZWRub2RlIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lbW9yeWRiL2xhdGVzdC9kZXZndWlkZS9XaGF0SXMuQ29tcG9uZW50cy5odG1sXG4gICAqXG4gICAqIEBwYXJhbSByZXNlcnZhdGlvbklEIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc2VydmF0aW9uSUQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvblJlc2VydmVkbm9kZShyZXNlcnZhdGlvbklEOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06bWVtb3J5ZGI6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OnJlc2VydmVkbm9kZS8keyByZXNlcnZhdGlvbklEIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjdGlvbnMgYmFzZWQgb24gdGhlIHRhZ3MgdGhhdCBhcmUgcGFzc2VkIGluIHRoZSByZXF1ZXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy1yZXF1ZXN0dGFnXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9Db3B5U25hcHNob3QoKVxuICAgKiAtIC50b0NyZWF0ZUFjbCgpXG4gICAqIC0gLnRvQ3JlYXRlQ2x1c3RlcigpXG4gICAqIC0gLnRvQ3JlYXRlTXVsdGlSZWdpb25DbHVzdGVyKClcbiAgICogLSAudG9DcmVhdGVQYXJhbWV0ZXJHcm91cCgpXG4gICAqIC0gLnRvQ3JlYXRlU25hcHNob3QoKVxuICAgKiAtIC50b0NyZWF0ZVN1Ym5ldEdyb3VwKClcbiAgICogLSAudG9DcmVhdGVVc2VyKClcbiAgICogLSAudG9QdXJjaGFzZVJlc2VydmVkTm9kZXNPZmZlcmluZygpXG4gICAqIC0gLnRvVGFnUmVzb3VyY2UoKVxuICAgKlxuICAgKiBAcGFyYW0gdGFnS2V5IFRoZSB0YWcga2V5IHRvIGNoZWNrXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzUmVxdWVzdFRhZyh0YWdLZXk6IHN0cmluZywgdmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpSZXF1ZXN0VGFnLyR7IHRhZ0tleSB9YCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY3Rpb25zIGJhc2VkIG9uIHRoZSB0YWdzIGFzc29jaWF0ZWQgd2l0aCB0aGUgcmVzb3VyY2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXJlc291cmNldGFnXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9CYXRjaFVwZGF0ZUNsdXN0ZXIoKVxuICAgKiAtIC50b0Nvbm5lY3QoKVxuICAgKiAtIC50b0NvcHlTbmFwc2hvdCgpXG4gICAqIC0gLnRvQ3JlYXRlQWNsKClcbiAgICogLSAudG9DcmVhdGVDbHVzdGVyKClcbiAgICogLSAudG9DcmVhdGVNdWx0aVJlZ2lvbkNsdXN0ZXIoKVxuICAgKiAtIC50b0NyZWF0ZVNuYXBzaG90KClcbiAgICogLSAudG9EZWxldGVBY2woKVxuICAgKiAtIC50b0RlbGV0ZUNsdXN0ZXIoKVxuICAgKiAtIC50b0RlbGV0ZU11bHRpUmVnaW9uQ2x1c3RlcigpXG4gICAqIC0gLnRvRGVsZXRlUGFyYW1ldGVyR3JvdXAoKVxuICAgKiAtIC50b0RlbGV0ZVNuYXBzaG90KClcbiAgICogLSAudG9EZWxldGVTdWJuZXRHcm91cCgpXG4gICAqIC0gLnRvRGVsZXRlVXNlcigpXG4gICAqIC0gLnRvRGVzY3JpYmVBY2xzKClcbiAgICogLSAudG9EZXNjcmliZUNsdXN0ZXJzKClcbiAgICogLSAudG9EZXNjcmliZU11bHRpUmVnaW9uQ2x1c3RlcnMoKVxuICAgKiAtIC50b0Rlc2NyaWJlUGFyYW1ldGVyR3JvdXBzKClcbiAgICogLSAudG9EZXNjcmliZVBhcmFtZXRlcnMoKVxuICAgKiAtIC50b0Rlc2NyaWJlUmVzZXJ2ZWROb2RlcygpXG4gICAqIC0gLnRvRGVzY3JpYmVTbmFwc2hvdHMoKVxuICAgKiAtIC50b0Rlc2NyaWJlU3VibmV0R3JvdXBzKClcbiAgICogLSAudG9EZXNjcmliZVVzZXJzKClcbiAgICogLSAudG9GYWlsb3ZlclNoYXJkKClcbiAgICogLSAudG9MaXN0QWxsb3dlZE11bHRpUmVnaW9uQ2x1c3RlclVwZGF0ZXMoKVxuICAgKiAtIC50b0xpc3RBbGxvd2VkTm9kZVR5cGVVcGRhdGVzKClcbiAgICogLSAudG9MaXN0VGFncygpXG4gICAqIC0gLnRvUHVyY2hhc2VSZXNlcnZlZE5vZGVzT2ZmZXJpbmcoKVxuICAgKiAtIC50b1Jlc2V0UGFyYW1ldGVyR3JvdXAoKVxuICAgKiAtIC50b1RhZ1Jlc291cmNlKClcbiAgICogLSAudG9VbnRhZ1Jlc291cmNlKClcbiAgICogLSAudG9VcGRhdGVBY2woKVxuICAgKiAtIC50b1VwZGF0ZUNsdXN0ZXIoKVxuICAgKiAtIC50b1VwZGF0ZU11bHRpUmVnaW9uQ2x1c3RlcigpXG4gICAqIC0gLnRvVXBkYXRlUGFyYW1ldGVyR3JvdXAoKVxuICAgKiAtIC50b1VwZGF0ZVN1Ym5ldEdyb3VwKClcbiAgICogLSAudG9VcGRhdGVVc2VyKClcbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBwYXJhbWV0ZXJncm91cFxuICAgKiAtIHN1Ym5ldGdyb3VwXG4gICAqIC0gbXVsdGlyZWdpb25jbHVzdGVyXG4gICAqIC0gY2x1c3RlclxuICAgKiAtIHNuYXBzaG90XG4gICAqIC0gdXNlclxuICAgKiAtIGFjbFxuICAgKiAtIHJlc2VydmVkbm9kZVxuICAgKlxuICAgKiBAcGFyYW0gdGFnS2V5IFRoZSB0YWcga2V5IHRvIGNoZWNrXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzUmVzb3VyY2VUYWcodGFnS2V5OiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6UmVzb3VyY2VUYWcvJHsgdGFnS2V5IH1gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjdGlvbnMgYmFzZWQgb24gdGhlIHRhZyBrZXlzIHRoYXQgYXJlIHBhc3NlZCBpbiB0aGUgcmVxdWVzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtdGFna2V5c1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQ29weVNuYXBzaG90KClcbiAgICogLSAudG9DcmVhdGVBY2woKVxuICAgKiAtIC50b0NyZWF0ZUNsdXN0ZXIoKVxuICAgKiAtIC50b0NyZWF0ZU11bHRpUmVnaW9uQ2x1c3RlcigpXG4gICAqIC0gLnRvQ3JlYXRlUGFyYW1ldGVyR3JvdXAoKVxuICAgKiAtIC50b0NyZWF0ZVNuYXBzaG90KClcbiAgICogLSAudG9DcmVhdGVTdWJuZXRHcm91cCgpXG4gICAqIC0gLnRvQ3JlYXRlVXNlcigpXG4gICAqIC0gLnRvUHVyY2hhc2VSZXNlcnZlZE5vZGVzT2ZmZXJpbmcoKVxuICAgKiAtIC50b1RhZ1Jlc291cmNlKClcbiAgICogLSAudG9VbnRhZ1Jlc291cmNlKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NUYWdLZXlzKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6VGFnS2V5c2AsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSBUTFNFbmFibGVkIHBhcmFtZXRlciBwcmVzZW50IGluIHRoZSByZXF1ZXN0IG9yIGRlZmF1bHRzIHRvIHRydWUgdmFsdWUgaWYgcGFyYW1ldGVyIGlzIG5vdCBwcmVzZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZW1vcnlkYi9sYXRlc3QvZGV2Z3VpZGUvSUFNLkNvbmRpdGlvbktleXMuaHRtbCNJQU0uU3BlY2lmeWluZ0NvbmRpdGlvbnNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0NyZWF0ZUNsdXN0ZXIoKVxuICAgKiAtIC50b0NyZWF0ZU11bHRpUmVnaW9uQ2x1c3RlcigpXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gbXVsdGlyZWdpb25jbHVzdGVyXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBgdHJ1ZWAgb3IgYGZhbHNlYC4gKipEZWZhdWx0OioqIGB0cnVlYFxuICAgKi9cbiAgcHVibGljIGlmVExTRW5hYmxlZCh2YWx1ZT86IGJvb2xlYW4pIHtcbiAgICByZXR1cm4gdGhpcy5pZihgVExTRW5hYmxlZGAsICh0eXBlb2YgdmFsdWUgIT09ICd1bmRlZmluZWQnID8gdmFsdWUgOiB0cnVlKSwgJ0Jvb2wnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgVXNlckF1dGhlbnRpY2F0aW9uTW9kZS5UeXBlIHBhcmFtZXRlciBpbiB0aGUgcmVxdWVzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVtb3J5ZGIvbGF0ZXN0L2Rldmd1aWRlL0lBTS5Db25kaXRpb25LZXlzLmh0bWwjSUFNLlNwZWNpZnlpbmdDb25kaXRpb25zXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9DcmVhdGVVc2VyKClcbiAgICogLSAudG9VcGRhdGVVc2VyKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZVc2VyQXV0aGVudGljYXRpb25Nb2RlKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBVc2VyQXV0aGVudGljYXRpb25Nb2RlYCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFttZW1vcnlkYl0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYW1hem9ubWVtb3J5ZGIuaHRtbCkuXG4gICAqXG4gICAqL1xuICBjb25zdHJ1Y3Rvcihwcm9wcz86IGlhbS5Qb2xpY3lTdGF0ZW1lbnRQcm9wcykge1xuICAgIHN1cGVyKHByb3BzKTtcbiAgfVxufVxuIl19