"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ec2messages = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [ec2messages](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmessagedeliveryservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Ec2messages extends shared_1.PolicyStatement {
    /**
     * Grants permission to acknowledge a message, ensuring it will not be delivered again
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-setting-up-messageAPIs.html
     */
    toAcknowledgeMessage() {
        return this.to('AcknowledgeMessage');
    }
    /**
     * Grants permission to delete a message
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-setting-up-messageAPIs.html
     */
    toDeleteMessage() {
        return this.to('DeleteMessage');
    }
    /**
     * Grants permission to fail a message, signifying the message could not be processed successfully, ensuring it cannot be replied to or delivered again
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-setting-up-messageAPIs.html
     */
    toFailMessage() {
        return this.to('FailMessage');
    }
    /**
     * Grants permission to route traffic to the correct endpoint based on the given destination for the messages
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-setting-up-messageAPIs.html
     */
    toGetEndpoint() {
        return this.to('GetEndpoint');
    }
    /**
     * Grants permission to deliver messages to clients/instances using long polling
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifSsmSourceInstanceARN()
     * - .ifEc2SourceInstanceARN()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-setting-up-messageAPIs.html
     */
    toGetMessages() {
        return this.to('GetMessages');
    }
    /**
     * Grants permission to send replies from clients/instances to upstream service
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSsmSourceInstanceARN()
     * - .ifEc2SourceInstanceARN()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-setting-up-messageAPIs.html
     */
    toSendReply() {
        return this.to('SendReply');
    }
    /**
     * Filters access by the ARN of the instance from which the request originated
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toGetMessages()
     * - .toSendReply()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifEc2SourceInstanceARN(value, operator) {
        return this.if(`ec2:SourceInstanceARN`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by verifying the Amazon Resource Name (ARN) of the AWS Systems Manager's managed instance from which the request is made. This key is not present when the request comes from the managed instance authenticated with an IAM role associated with EC2 instance profile
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssystemsmanager.html#awssystemsmanager-policy-keys
     *
     * Applies to actions:
     * - .toGetMessages()
     * - .toSendReply()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifSsmSourceInstanceARN(value, operator) {
        return this.if(`ssm:SourceInstanceARN`, value, operator ?? 'ArnLike');
    }
    /**
     * Statement provider for service [ec2messages](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmessagedeliveryservice.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'ec2messages';
        this.accessLevelList = {
            Write: [
                'AcknowledgeMessage',
                'DeleteMessage',
                'FailMessage',
                'SendReply'
            ],
            Read: [
                'GetEndpoint',
                'GetMessages'
            ]
        };
    }
}
exports.Ec2messages = Ec2messages;
//# sourceMappingURL=data:application/json;base64,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