"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mgh = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [mgh](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsmigrationhub.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Mgh extends shared_1.PolicyStatement {
    /**
     * Grants permission to accept a connection
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/mhj/latest/userguide/account-connections.html
     */
    toAcceptConnection() {
        return this.to('AcceptConnection');
    }
    /**
     * Grants permission to associate an IAM role to an automation unit
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_AssociateAutomationUnitRole.html
     */
    toAssociateAutomationUnitRole() {
        return this.to('AssociateAutomationUnitRole');
    }
    /**
     * Grants permission to associate a given AWS artifact to a MigrationTask
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_AssociateCreatedArtifact.html
     */
    toAssociateCreatedArtifact() {
        return this.to('AssociateCreatedArtifact');
    }
    /**
     * Grants permission to associate a given ADS resource to a MigrationTask
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_AssociateDiscoveredResource.html
     */
    toAssociateDiscoveredResource() {
        return this.to('AssociateDiscoveredResource');
    }
    /**
     * Grants permission to associate source resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_AssociateSourceResource.html
     */
    toAssociateSourceResource() {
        return this.to('AssociateSourceResource');
    }
    /**
     * Grants permission to batch-associate IAM roles with a connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mhj/latest/userguide/account-connections.html
     */
    toBatchAssociateIamRoleWithConnection() {
        return this.to('BatchAssociateIamRoleWithConnection');
    }
    /**
     * Grants permission to batch-disassociate IAM roles from a connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mhj/latest/userguide/account-connections.html
     */
    toBatchDisassociateIamRoleFromConnection() {
        return this.to('BatchDisassociateIamRoleFromConnection');
    }
    /**
     * Grants permission to create an automation unit run
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_CreateAutomationRun.html
     */
    toCreateAutomationRun() {
        return this.to('CreateAutomationRun');
    }
    /**
     * Grants permission to create an automation unit
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_CreateAutomationUnit.html
     */
    toCreateAutomationUnit() {
        return this.to('CreateAutomationUnit');
    }
    /**
     * Grants permission to create a Migration Hub Home Region Control
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub-home-region/latest/APIReference/API_CreateHomeRegionControl.html
     */
    toCreateHomeRegionControl() {
        return this.to('CreateHomeRegionControl');
    }
    /**
     * Grants permission to create a ProgressUpdateStream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_CreateProgressUpdateStream.html
     */
    toCreateProgressUpdateStream() {
        return this.to('CreateProgressUpdateStream');
    }
    /**
     * Grants permission to delete an automation unit run
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_DeleteAutomationRun.html
     */
    toDeleteAutomationRun() {
        return this.to('DeleteAutomationRun');
    }
    /**
     * Grants permission to delete an automation unit
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_DeleteAutomationUnit.html
     */
    toDeleteAutomationUnit() {
        return this.to('DeleteAutomationUnit');
    }
    /**
     * Grants permission to delete a connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mhj/latest/userguide/account-connections.html
     */
    toDeleteConnection() {
        return this.to('DeleteConnection');
    }
    /**
     * Grants permission to delete a Migration Hub Home Region Control
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub-home-region/latest/APIReference/API_DeleteHomeRegionControl.html
     */
    toDeleteHomeRegionControl() {
        return this.to('DeleteHomeRegionControl');
    }
    /**
     * Grants permission to delete a ProgressUpdateStream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_DeleteProgressUpdateStream.html
     */
    toDeleteProgressUpdateStream() {
        return this.to('DeleteProgressUpdateStream');
    }
    /**
     * Grants permission to get an Application Discovery Service Application's state
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_DescribeApplicationState.html
     */
    toDescribeApplicationState() {
        return this.to('DescribeApplicationState');
    }
    /**
     * Grants permission to describe an automation unit run
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_DescribeAutomationRun.html
     */
    toDescribeAutomationRun() {
        return this.to('DescribeAutomationRun');
    }
    /**
     * Grants permission to describe an automation unit
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_DescribeAutomationUnit.html
     */
    toDescribeAutomationUnit() {
        return this.to('DescribeAutomationUnit');
    }
    /**
     * Grants permission to list Home Region Controls
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/migrationhub-home-region/latest/APIReference/API_DescribeHomeRegionControls.html
     */
    toDescribeHomeRegionControls() {
        return this.to('DescribeHomeRegionControls');
    }
    /**
     * Grants permission to describe a MigrationTask
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_DescribeMigrationTask.html
     */
    toDescribeMigrationTask() {
        return this.to('DescribeMigrationTask');
    }
    /**
     * Grants permission to disassociate an IAM role from an automation unit
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_DisassociateAutomationUnitRole.html
     */
    toDisassociateAutomationUnitRole() {
        return this.to('DisassociateAutomationUnitRole');
    }
    /**
     * Grants permission to disassociate a given AWS artifact from a MigrationTask
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_DisassociateCreatedArtifact.html
     */
    toDisassociateCreatedArtifact() {
        return this.to('DisassociateCreatedArtifact');
    }
    /**
     * Grants permission to disassociate a given ADS resource from a MigrationTask
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_DisassociateDiscoveredResource.html
     */
    toDisassociateDiscoveredResource() {
        return this.to('DisassociateDiscoveredResource');
    }
    /**
     * Grants permission to diassociate source resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_DisassociateSourceResource.html
     */
    toDisassociateSourceResource() {
        return this.to('DisassociateSourceResource');
    }
    /**
     * Grants permission to get a connection
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mhj/latest/userguide/account-connections.html
     */
    toGetConnection() {
        return this.to('GetConnection');
    }
    /**
     * Grants permission to get the Migration Hub Home Region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub-home-region/latest/APIReference/API_GetHomeRegion.html
     */
    toGetHomeRegion() {
        return this.to('GetHomeRegion');
    }
    /**
     * Grants permission to import a MigrationTask
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_ImportMigrationTask.html
     */
    toImportMigrationTask() {
        return this.to('ImportMigrationTask');
    }
    /**
     * Grants permission to list Application statuses
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_ListApplicationStates.html
     */
    toListApplicationStates() {
        return this.to('ListApplicationStates');
    }
    /**
     * Grants permission to list automation unit runs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_ListAutomationRuns.html
     */
    toListAutomationRuns() {
        return this.to('ListAutomationRuns');
    }
    /**
     * Grants permission to list automation units
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_ListAutomationUnits.html
     */
    toListAutomationUnits() {
        return this.to('ListAutomationUnits');
    }
    /**
     * Grants permission to list connection roles
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mhj/latest/userguide/account-connections.html
     */
    toListConnectionRoles() {
        return this.to('ListConnectionRoles');
    }
    /**
     * Grants permission to list connections
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mhj/latest/userguide/account-connections.html
     */
    toListConnections() {
        return this.to('ListConnections');
    }
    /**
     * Grants permission to list associated created artifacts for a MigrationTask
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_ListCreatedArtifacts.html
     */
    toListCreatedArtifacts() {
        return this.to('ListCreatedArtifacts');
    }
    /**
     * Grants permission to list associated ADS resources from MigrationTask
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_ListDiscoveredResources.html
     */
    toListDiscoveredResources() {
        return this.to('ListDiscoveredResources');
    }
    /**
     * Grants permission to list migration tasks updates
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_ListMigrationTaskUpdates.html
     */
    toListMigrationTaskUpdates() {
        return this.to('ListMigrationTaskUpdates');
    }
    /**
     * Grants permission to list MigrationTasks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_ListMigrationTasks.html
     */
    toListMigrationTasks() {
        return this.to('ListMigrationTasks');
    }
    /**
     * Grants permission to to list ProgressUpdateStreams
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_ListProgressUpdateStreams.html
     */
    toListProgressUpdateStreams() {
        return this.to('ListProgressUpdateStreams');
    }
    /**
     * Grants permission to list source resources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_ListSourceResources.html
     */
    toListSourceResources() {
        return this.to('ListSourceResources');
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to update an Application Discovery Service Application's state
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_NotifyApplicationState.html
     */
    toNotifyApplicationState() {
        return this.to('NotifyApplicationState');
    }
    /**
     * Grants permission to notify latest MigrationTask state
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_NotifyMigrationTaskState.html
     */
    toNotifyMigrationTaskState() {
        return this.to('NotifyMigrationTaskState');
    }
    /**
     * Grants permission to put ResourceAttributes
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_PutResourceAttributes.html
     */
    toPutResourceAttributes() {
        return this.to('PutResourceAttributes');
    }
    /**
     * Grants permission to reject a connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mhj/latest/userguide/account-connections.html
     */
    toRejectConnection() {
        return this.to('RejectConnection');
    }
    /**
     * Grants permission to tag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Adds a resource of type progressUpdateStream to the statement
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_ProgressUpdateStreamSummary.html
     *
     * @param stream - Identifier for the stream.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onProgressUpdateStream(stream, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mgh:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:progressUpdateStream/${stream}`);
    }
    /**
     * Adds a resource of type migrationTask to the statement
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_MigrationTask.html
     *
     * @param stream - Identifier for the stream.
     * @param task - Identifier for the task.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onMigrationTask(stream, task, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mgh:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:progressUpdateStream/${stream}/migrationTask/${task}`);
    }
    /**
     * Adds a resource of type AutomationRunResource to the statement
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_AutomationRunResource.html
     *
     * @param runID - Identifier for the runID.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAutomationRunResourceRunID()
     */
    onAutomationRunResource(runID, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mgh:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:automation-run/${runID}`);
    }
    /**
     * Adds a resource of type AutomationUnitResource to the statement
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_AutomationUnitResource.html
     *
     * @param automationUnitId - Identifier for the automationUnitId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAutomationUnitResourceAutomationUnitArn()
     */
    onAutomationUnitResource(automationUnitId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mgh:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:automation-unit/${automationUnitId}`);
    }
    /**
     * Adds a resource of type ConnectionResource to the statement
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_ConnectionResource.html
     *
     * @param connectionArn - Identifier for the connectionArn.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifConnectionResourceConnectionArn()
     */
    onConnectionResource(connectionArn, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mgh:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:${connectionArn}`);
    }
    /**
     * Filters access based on the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toAcceptConnection()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - ConnectionResource
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toAcceptConnection()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * AutomationRunResource resource runID identifier
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/ContextKeys_AutomationRunResourceRunID.html
     *
     * Applies to resource types:
     * - AutomationRunResource
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAutomationRunResourceRunID(value, operator) {
        return this.if(`AutomationRunResourceRunID`, value, operator ?? 'StringLike');
    }
    /**
     * AutomationUnitResource resource automationUnitArn identifier
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/ContextKeys_AutomationUnitResourceAutomationUnitArn.html
     *
     * Applies to resource types:
     * - AutomationUnitResource
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifAutomationUnitResourceAutomationUnitArn(value, operator) {
        return this.if(`AutomationUnitResourceAutomationUnitArn`, value, operator ?? 'ArnLike');
    }
    /**
     * ConnectionResource resource connectionArn identifier
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/security_iam_service-with-iam-id-based-policies-conditionkeys.html#condition-connectionresourceconnectionarn
     *
     * Applies to resource types:
     * - ConnectionResource
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifConnectionResourceConnectionArn(value, operator) {
        return this.if(`ConnectionResourceConnectionArn`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [mgh](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsmigrationhub.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'mgh';
        this.accessLevelList = {
            Write: [
                'AcceptConnection',
                'AssociateAutomationUnitRole',
                'AssociateCreatedArtifact',
                'AssociateDiscoveredResource',
                'AssociateSourceResource',
                'BatchAssociateIamRoleWithConnection',
                'BatchDisassociateIamRoleFromConnection',
                'CreateAutomationRun',
                'CreateAutomationUnit',
                'CreateHomeRegionControl',
                'CreateProgressUpdateStream',
                'DeleteAutomationRun',
                'DeleteAutomationUnit',
                'DeleteConnection',
                'DeleteHomeRegionControl',
                'DeleteProgressUpdateStream',
                'DisassociateAutomationUnitRole',
                'DisassociateCreatedArtifact',
                'DisassociateDiscoveredResource',
                'DisassociateSourceResource',
                'ImportMigrationTask',
                'NotifyApplicationState',
                'NotifyMigrationTaskState',
                'PutResourceAttributes',
                'RejectConnection'
            ],
            Read: [
                'DescribeApplicationState',
                'DescribeAutomationRun',
                'DescribeAutomationUnit',
                'DescribeMigrationTask',
                'GetConnection',
                'GetHomeRegion'
            ],
            List: [
                'DescribeHomeRegionControls',
                'ListApplicationStates',
                'ListAutomationRuns',
                'ListAutomationUnits',
                'ListConnectionRoles',
                'ListConnections',
                'ListCreatedArtifacts',
                'ListDiscoveredResources',
                'ListMigrationTaskUpdates',
                'ListMigrationTasks',
                'ListProgressUpdateStreams',
                'ListSourceResources',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Mgh = Mgh;
//# sourceMappingURL=data:application/json;base64,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