"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkFirewall = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [network-firewall](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsnetworkfirewall.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class NetworkFirewall extends shared_1.PolicyStatement {
    /**
     * Grants permission to create an association between a firewall policy and a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_AssociateFirewallPolicy.html
     */
    toAssociateFirewallPolicy() {
        return this.to('AssociateFirewallPolicy');
    }
    /**
     * Grants permission to associate VPC subnets to a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_AssociateSubnets.html
     */
    toAssociateSubnets() {
        return this.to('AssociateSubnets');
    }
    /**
     * Grants permission to create an AWS Network Firewall firewall
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_CreateFirewall.html
     */
    toCreateFirewall() {
        return this.to('CreateFirewall');
    }
    /**
     * Grants permission to create an AWS Network Firewall firewall policy
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_CreateFirewallPolicy.html
     */
    toCreateFirewallPolicy() {
        return this.to('CreateFirewallPolicy');
    }
    /**
     * Grants permission to create an AWS Network Firewall rule group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_CreateRuleGroup.html
     */
    toCreateRuleGroup() {
        return this.to('CreateRuleGroup');
    }
    /**
     * Grants permission to create an AWS Network Firewall tls inspection configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_CreateTLSInspectionConfiguration.html
     */
    toCreateTLSInspectionConfiguration() {
        return this.to('CreateTLSInspectionConfiguration');
    }
    /**
     * Grants permission to delete a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DeleteFirewall.html
     */
    toDeleteFirewall() {
        return this.to('DeleteFirewall');
    }
    /**
     * Grants permission to delete a firewall policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DeleteFirewallPolicy.html
     */
    toDeleteFirewallPolicy() {
        return this.to('DeleteFirewallPolicy');
    }
    /**
     * Grants permission to delete a resource policy for a firewall policy or rule group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to delete a rule group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DeleteRuleGroup.html
     */
    toDeleteRuleGroup() {
        return this.to('DeleteRuleGroup');
    }
    /**
     * Grants permission to delete a tls inspection configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DeleteTLSInspectionConfiguration.html
     */
    toDeleteTLSInspectionConfiguration() {
        return this.to('DeleteTLSInspectionConfiguration');
    }
    /**
     * Grants permission to retrieve the data objects that define a firewall
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DescribeFirewall.html
     */
    toDescribeFirewall() {
        return this.to('DescribeFirewall');
    }
    /**
     * Grants permission to retrieve the data objects that define a firewall policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DescribeFirewallPolicy.html
     */
    toDescribeFirewallPolicy() {
        return this.to('DescribeFirewallPolicy');
    }
    /**
     * Grants permission to describe a flow operation performed on a firewall
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DescribeFlowOperation.html
     */
    toDescribeFlowOperation() {
        return this.to('DescribeFlowOperation');
    }
    /**
     * Grants permission to describe the logging configuration of a firewall
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - logs:GetLogDelivery
     * - logs:ListLogDeliveries
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DescribeLoggingConfiguration.html
     */
    toDescribeLoggingConfiguration() {
        return this.to('DescribeLoggingConfiguration');
    }
    /**
     * Grants permission to describe a resource policy for a firewall policy or rule group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DescribeResourcePolicy.html
     */
    toDescribeResourcePolicy() {
        return this.to('DescribeResourcePolicy');
    }
    /**
     * Grants permission to retrieve the data objects that define a rule group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DescribeRuleGroup.html
     */
    toDescribeRuleGroup() {
        return this.to('DescribeRuleGroup');
    }
    /**
     * Grants permission to retrieve the high-level information about a rule group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DescribeRuleGroupMetadata.html
     */
    toDescribeRuleGroupMetadata() {
        return this.to('DescribeRuleGroupMetadata');
    }
    /**
     * Grants permission to retrieve the data objects that define a tls inspection configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DescribeTLSInspectionConfiguration.html
     */
    toDescribeTLSInspectionConfiguration() {
        return this.to('DescribeTLSInspectionConfiguration');
    }
    /**
     * Grants permission to disassociate VPC subnets from a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DisassociateSubnets.html
     */
    toDisassociateSubnets() {
        return this.to('DisassociateSubnets');
    }
    /**
     * Grants permission to retrieve analysis report results of a firewall
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_GetAnalysisReportResults.html
     */
    toGetAnalysisReportResults() {
        return this.to('GetAnalysisReportResults');
    }
    /**
     * Grants permission to list firewall analysis reports
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_ListAnalysisReports.html
     */
    toListAnalysisReports() {
        return this.to('ListAnalysisReports');
    }
    /**
     * Grants permission to retrieve the metadata for firewall policies
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_ListFirewallPolicies.html
     */
    toListFirewallPolicies() {
        return this.to('ListFirewallPolicies');
    }
    /**
     * Grants permission to retrieve the metadata for firewalls
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_ListFirewalls.html
     */
    toListFirewalls() {
        return this.to('ListFirewalls');
    }
    /**
     * Grants permission to list results from a flow operation performed on a firewall
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_ListFlowOperationResults.html
     */
    toListFlowOperationResults() {
        return this.to('ListFlowOperationResults');
    }
    /**
     * Grants permission to list flow operations performed on a firewall
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_ListFlowOperations.html
     */
    toListFlowOperations() {
        return this.to('ListFlowOperations');
    }
    /**
     * Grants permission to retrieve the metadata for rule groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_ListRuleGroups.html
     */
    toListRuleGroups() {
        return this.to('ListRuleGroups');
    }
    /**
     * Grants permission to retrieve the metadata for tls inspection configurations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_ListTLSInspectionConfigurations.html
     */
    toListTLSInspectionConfigurations() {
        return this.to('ListTLSInspectionConfigurations');
    }
    /**
     * Grants permission to retrieve the tags for a resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to put a resource policy for a firewall policy or rule group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_PutResourcePolicy.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to start an analysis report on a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_GetAnalysisReportResults.html
     */
    toStartAnalysisReport() {
        return this.to('StartAnalysisReport');
    }
    /**
     * Grants permission to start capture operation on a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_StartFlowCapture.html
     */
    toStartFlowCapture() {
        return this.to('StartFlowCapture');
    }
    /**
     * Grants permission to start flush operation on a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_StartFlowFlush.html
     */
    toStartFlowFlush() {
        return this.to('StartFlowFlush');
    }
    /**
     * Grants permission to attach tags to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to modify firewall analysis settings of a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_UpdateFirewallAnalysisSettings.html
     */
    toUpdateFirewallAnalysisSettings() {
        return this.to('UpdateFirewallAnalysisSettings');
    }
    /**
     * Grants permission to add or remove delete protection for a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_UpdateFirewallDeleteProtection.html
     */
    toUpdateFirewallDeleteProtection() {
        return this.to('UpdateFirewallDeleteProtection');
    }
    /**
     * Grants permission to modify the description for a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_UpdateFirewallDescription.html
     */
    toUpdateFirewallDescription() {
        return this.to('UpdateFirewallDescription');
    }
    /**
     * Grants permission to modify the encryption configuration of a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_UpdateFirewallEncryptionConfiguration.html
     */
    toUpdateFirewallEncryptionConfiguration() {
        return this.to('UpdateFirewallEncryptionConfiguration');
    }
    /**
     * Grants permission to modify a firewall policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_UpdateFirewallPolicy.html
     */
    toUpdateFirewallPolicy() {
        return this.to('UpdateFirewallPolicy');
    }
    /**
     * Grants permission to add or remove firewall policy change protection for a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_UpdateFirewallPolicyChangeProtection.html
     */
    toUpdateFirewallPolicyChangeProtection() {
        return this.to('UpdateFirewallPolicyChangeProtection');
    }
    /**
     * Grants permission to modify the logging configuration of a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_UpdateLoggingConfiguration.html
     */
    toUpdateLoggingConfiguration() {
        return this.to('UpdateLoggingConfiguration');
    }
    /**
     * Grants permission to modify a rule group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_UpdateRuleGroup.html
     */
    toUpdateRuleGroup() {
        return this.to('UpdateRuleGroup');
    }
    /**
     * Grants permission to add or remove subnet change protection for a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_UpdateSubnetChangeProtection.html
     */
    toUpdateSubnetChangeProtection() {
        return this.to('UpdateSubnetChangeProtection');
    }
    /**
     * Grants permission to modify a tls inspection configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_UpdateTLSInspectionConfiguration.html
     */
    toUpdateTLSInspectionConfiguration() {
        return this.to('UpdateTLSInspectionConfiguration');
    }
    /**
     * Adds a resource of type Firewall to the statement
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_Firewall.html
     *
     * @param name - Identifier for the name.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFirewall(name, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:network-firewall:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:firewall/${name}`);
    }
    /**
     * Adds a resource of type FirewallPolicy to the statement
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_FirewallPolicyResponse.html
     *
     * @param name - Identifier for the name.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFirewallPolicy(name, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:network-firewall:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:firewall-policy/${name}`);
    }
    /**
     * Adds a resource of type StatefulRuleGroup to the statement
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_RuleGroupResponse.html
     *
     * @param name - Identifier for the name.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStatefulRuleGroup(name, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:network-firewall:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:stateful-rulegroup/${name}`);
    }
    /**
     * Adds a resource of type StatelessRuleGroup to the statement
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_RuleGroupResponse.html
     *
     * @param name - Identifier for the name.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStatelessRuleGroup(name, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:network-firewall:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:stateless-rulegroup/${name}`);
    }
    /**
     * Adds a resource of type TLSInspectionConfiguration to the statement
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_TLSInspectionConfigurationResponse.html
     *
     * @param name - Identifier for the name.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTLSInspectionConfiguration(name, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:network-firewall:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:tls-configuration/${name}`);
    }
    /**
     * Filters access by on the allowed set of values for each of the tags
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateFirewall()
     * - .toCreateFirewallPolicy()
     * - .toCreateRuleGroup()
     * - .toCreateTLSInspectionConfiguration()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag value associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - Firewall
     * - FirewallPolicy
     * - StatefulRuleGroup
     * - StatelessRuleGroup
     * - TLSInspectionConfiguration
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of mandatory tags in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateFirewall()
     * - .toCreateFirewallPolicy()
     * - .toCreateRuleGroup()
     * - .toCreateTLSInspectionConfiguration()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [network-firewall](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsnetworkfirewall.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'network-firewall';
        this.accessLevelList = {
            Write: [
                'AssociateFirewallPolicy',
                'AssociateSubnets',
                'CreateFirewall',
                'CreateFirewallPolicy',
                'CreateRuleGroup',
                'CreateTLSInspectionConfiguration',
                'DeleteFirewall',
                'DeleteFirewallPolicy',
                'DeleteResourcePolicy',
                'DeleteRuleGroup',
                'DeleteTLSInspectionConfiguration',
                'DisassociateSubnets',
                'PutResourcePolicy',
                'StartAnalysisReport',
                'StartFlowCapture',
                'StartFlowFlush',
                'UpdateFirewallAnalysisSettings',
                'UpdateFirewallDeleteProtection',
                'UpdateFirewallDescription',
                'UpdateFirewallEncryptionConfiguration',
                'UpdateFirewallPolicy',
                'UpdateFirewallPolicyChangeProtection',
                'UpdateLoggingConfiguration',
                'UpdateRuleGroup',
                'UpdateSubnetChangeProtection',
                'UpdateTLSInspectionConfiguration'
            ],
            Read: [
                'DescribeFirewall',
                'DescribeFirewallPolicy',
                'DescribeFlowOperation',
                'DescribeLoggingConfiguration',
                'DescribeResourcePolicy',
                'DescribeRuleGroup',
                'DescribeRuleGroupMetadata',
                'DescribeTLSInspectionConfiguration',
                'GetAnalysisReportResults',
                'ListFlowOperationResults'
            ],
            List: [
                'ListAnalysisReports',
                'ListFirewallPolicies',
                'ListFirewalls',
                'ListFlowOperations',
                'ListRuleGroups',
                'ListTLSInspectionConfigurations',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.NetworkFirewall = NetworkFirewall;
//# sourceMappingURL=data:application/json;base64,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