"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Opensearch = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [opensearch](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonopensearch.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Opensearch extends shared_1.PolicyStatement {
    /**
     * Grants permission to access OpenSearch Application
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/developerguide/
     */
    toApplicationAccessAll() {
        return this.to('ApplicationAccessAll');
    }
    /**
     * Grants permission to cancel the query that is submitted on the OpenSearch DataSource resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_CancelDirectQuery.html
     */
    toCancelDirectQuery() {
        return this.to('CancelDirectQuery');
    }
    /**
     * Grants permission to get the query status that are performed on the OpenSearch DataSource resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_GetDirectQuery.html
     */
    toGetDirectQuery() {
        return this.to('GetDirectQuery');
    }
    /**
     * Grants permission to get the results of a query that is performed on the OpenSearch DataSource resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_GetDirectQueryResult.html
     */
    toGetDirectQueryResult() {
        return this.to('GetDirectQueryResult');
    }
    /**
     * Grants permission to start a direct query on the provided OpenSearch DataSource arns
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_StartDirectQuery.html
     */
    toStartDirectQuery() {
        return this.to('StartDirectQuery');
    }
    /**
     * Adds a resource of type application to the statement
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/developerguide/ac.html
     *
     * @param appId - Identifier for the appId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onApplication(appId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:opensearch:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:application/${appId}`);
    }
    /**
     * Adds a resource of type datasource to the statement
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/developerguide/datasource.html
     *
     * @param dataSourceName - Identifier for the dataSourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onDatasource(dataSourceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:opensearch:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:datasource/${dataSourceName}`);
    }
    /**
     * Statement provider for service [opensearch](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonopensearch.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'opensearch';
        this.accessLevelList = {
            'Permissions management': [
                'ApplicationAccessAll'
            ],
            Write: [
                'CancelDirectQuery',
                'StartDirectQuery'
            ],
            Read: [
                'GetDirectQuery',
                'GetDirectQueryResult'
            ]
        };
    }
}
exports.Opensearch = Opensearch;
//# sourceMappingURL=data:application/json;base64,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