"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpsworksCm = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [opsworks-cm](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsopsworksconfigurationmanagement.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class OpsworksCm extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate a node to a configuration management server
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks-cm/latest/APIReference/API_AssociateNode.html
     */
    toAssociateNode() {
        return this.to('AssociateNode');
    }
    /**
     * Grants permission to create a backup for the specified server
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks-cm/latest/APIReference/API_CreateBackup.html
     */
    toCreateBackup() {
        return this.to('CreateBackup');
    }
    /**
     * Grants permission to create a new server
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks-cm/latest/APIReference/API_CreateServer.html
     */
    toCreateServer() {
        return this.to('CreateServer');
    }
    /**
     * Grants permission to delete the specified backup and possibly its S3 bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks-cm/latest/APIReference/API_DeleteBackup.html
     */
    toDeleteBackup() {
        return this.to('DeleteBackup');
    }
    /**
     * Grants permission to delete the specified server with its corresponding CloudFormation stack and possibly the S3 bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks-cm/latest/APIReference/API_DeleteServer.html
     */
    toDeleteServer() {
        return this.to('DeleteServer');
    }
    /**
     * Grants permission to describe the service limits for the user's account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks-cm/latest/APIReference/API_DescribeAccountAttributes.html
     */
    toDescribeAccountAttributes() {
        return this.to('DescribeAccountAttributes');
    }
    /**
     * Grants permission to describe a single backup, all backups of a specified server or all backups of the user's account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks-cm/latest/APIReference/API_DescribeBackups.html
     */
    toDescribeBackups() {
        return this.to('DescribeBackups');
    }
    /**
     * Grants permission to describe all events of the specified server
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks-cm/latest/APIReference/API_DescribeEvents.html
     */
    toDescribeEvents() {
        return this.to('DescribeEvents');
    }
    /**
     * Grants permission to describe the association status for the specified node token and the specified server
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks-cm/latest/APIReference/API_DescribeNodeAssociationStatus.html
     */
    toDescribeNodeAssociationStatus() {
        return this.to('DescribeNodeAssociationStatus');
    }
    /**
     * Grants permission to describe the specified server or all servers of the user's account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks-cm/latest/APIReference/API_DescribeServers.html
     */
    toDescribeServers() {
        return this.to('DescribeServers');
    }
    /**
     * Grants permission to disassociate a specified node from a server
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks-cm/latest/APIReference/API_DisassociateNode.html
     */
    toDisassociateNode() {
        return this.to('DisassociateNode');
    }
    /**
     * Grants permission to export an engine attribute from a server
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/opsworks-cm/latest/APIReference/API_ExportServerEngineAttribute.html
     */
    toExportServerEngineAttribute() {
        return this.to('ExportServerEngineAttribute');
    }
    /**
     * Grants permission to list the tags that are applied to the specified server or backup
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/opsworks-cm/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to apply a backup to specified server. Possibly swaps out the ec2-instance if specified
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks-cm/latest/APIReference/API_RestoreServer.html
     */
    toRestoreServer() {
        return this.to('RestoreServer');
    }
    /**
     * Grants permission to start the server maintenance immediately
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks-cm/latest/APIReference/API_StartMaintenance.html
     */
    toStartMaintenance() {
        return this.to('StartMaintenance');
    }
    /**
     * Grants permission to apply tags to the specified server or backup
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/opsworks-cm/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from the specified server or backup
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/opsworks-cm/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update general server settings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks-cm/latest/APIReference/API_UpdateServer.html
     */
    toUpdateServer() {
        return this.to('UpdateServer');
    }
    /**
     * Grants permission to update server settings specific to the configuration management type
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks-cm/latest/APIReference/API_UpdateServerEngineAttributes.html
     */
    toUpdateServerEngineAttributes() {
        return this.to('UpdateServerEngineAttributes');
    }
    /**
     * Adds a resource of type server to the statement
     *
     * @param serverName - Identifier for the serverName.
     * @param uniqueId - Identifier for the uniqueId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onServer(serverName, uniqueId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:opsworks-cm::${account ?? this.defaultAccount}:server/${serverName}/${uniqueId}`);
    }
    /**
     * Adds a resource of type backup to the statement
     *
     * @param serverName - Identifier for the serverName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onBackup(serverName, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:opsworks-cm::${account ?? this.defaultAccount}:backup/${serverName}-{Date-and-Time-Stamp-of-Backup}`);
    }
    /**
     * Statement provider for service [opsworks-cm](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsopsworksconfigurationmanagement.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'opsworks-cm';
        this.accessLevelList = {
            Write: [
                'AssociateNode',
                'CreateBackup',
                'CreateServer',
                'DeleteBackup',
                'DeleteServer',
                'DisassociateNode',
                'RestoreServer',
                'StartMaintenance',
                'UpdateServer',
                'UpdateServerEngineAttributes'
            ],
            List: [
                'DescribeAccountAttributes',
                'DescribeBackups',
                'DescribeEvents',
                'DescribeNodeAssociationStatus',
                'DescribeServers'
            ],
            Read: [
                'ExportServerEngineAttribute',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.OpsworksCm = OpsworksCm;
//# sourceMappingURL=data:application/json;base64,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