"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Organizations = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [organizations](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsorganizations.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Organizations extends shared_1.PolicyStatement {
    /**
     * Grants permission to send a response to the originator of a handshake agreeing to the action proposed by the handshake request
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_AcceptHandshake.html
     */
    toAcceptHandshake() {
        return this.to('AcceptHandshake');
    }
    /**
     * Grants permission to attach a policy to a root, an organizational unit, or an individual account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_AttachPolicy.html
     */
    toAttachPolicy() {
        return this.to('AttachPolicy');
    }
    /**
     * Grants permission to cancel a handshake
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_CancelHandshake.html
     */
    toCancelHandshake() {
        return this.to('CancelHandshake');
    }
    /**
     * Grants permission to close an AWS account that is now a part of an Organizations, either created within the organization, or invited to join the organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_CloseAccount.html
     */
    toCloseAccount() {
        return this.to('CloseAccount');
    }
    /**
     * Grants permission to create an AWS account that is automatically a member of the organization with the credentials that made the request
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreateAccount.html
     */
    toCreateAccount() {
        return this.to('CreateAccount');
    }
    /**
     * Grants permission to create an AWS GovCloud (US) account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreateGovCloudAccount.html
     */
    toCreateGovCloudAccount() {
        return this.to('CreateGovCloudAccount');
    }
    /**
     * Grants permission to create an organization. The account with the credentials that calls the CreateOrganization operation automatically becomes the management account of the new organization
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreateOrganization.html
     */
    toCreateOrganization() {
        return this.to('CreateOrganization');
    }
    /**
     * Grants permission to create an organizational unit (OU) within a root or parent OU
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreateOrganizationalUnit.html
     */
    toCreateOrganizationalUnit() {
        return this.to('CreateOrganizationalUnit');
    }
    /**
     * Grants permission to create a policy that you can attach to a root, an organizational unit (OU), or an individual AWS account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyType()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreatePolicy.html
     */
    toCreatePolicy() {
        return this.to('CreatePolicy');
    }
    /**
     * Grants permission to decline a handshake request. This sets the handshake state to DECLINED and effectively deactivates the request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeclineHandshake.html
     */
    toDeclineHandshake() {
        return this.to('DeclineHandshake');
    }
    /**
     * Grants permission to delete the organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeleteOrganization.html
     */
    toDeleteOrganization() {
        return this.to('DeleteOrganization');
    }
    /**
     * Grants permission to delete an organizational unit from a root or another OU
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeleteOrganizationalUnit.html
     */
    toDeleteOrganizationalUnit() {
        return this.to('DeleteOrganizationalUnit');
    }
    /**
     * Grants permission to delete a policy from your organization
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeletePolicy.html
     */
    toDeletePolicy() {
        return this.to('DeletePolicy');
    }
    /**
     * Grants permission to delete a resource policy from your organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to deregister the specified member AWS account as a delegated administrator for the AWS service that is specified by ServicePrincipal
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifServicePrincipal()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeregisterDelegatedAdministrator.html
     */
    toDeregisterDelegatedAdministrator() {
        return this.to('DeregisterDelegatedAdministrator');
    }
    /**
     * Grants permission to retrieve Organizations-related details about the specified account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeAccount.html
     */
    toDescribeAccount() {
        return this.to('DescribeAccount');
    }
    /**
     * Grants permission to retrieve the current status of an asynchronous request to create an account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeCreateAccountStatus.html
     */
    toDescribeCreateAccountStatus() {
        return this.to('DescribeCreateAccountStatus');
    }
    /**
     * Grants permission to retrieve the effective policy for an account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeEffectivePolicy.html
     */
    toDescribeEffectivePolicy() {
        return this.to('DescribeEffectivePolicy');
    }
    /**
     * Grants permission to retrieve details about a previously requested handshake
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeHandshake.html
     */
    toDescribeHandshake() {
        return this.to('DescribeHandshake');
    }
    /**
     * Grants permission to retrieves details about the organization that the calling credentials belong to
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeOrganization.html
     */
    toDescribeOrganization() {
        return this.to('DescribeOrganization');
    }
    /**
     * Grants permission to retrieve details about an organizational unit (OU)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeOrganizationalUnit.html
     */
    toDescribeOrganizationalUnit() {
        return this.to('DescribeOrganizationalUnit');
    }
    /**
     * Grants permission to retrieves details about a policy
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribePolicy.html
     */
    toDescribePolicy() {
        return this.to('DescribePolicy');
    }
    /**
     * Grants permission to retrieve information about a resource policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeResourcePolicy.html
     */
    toDescribeResourcePolicy() {
        return this.to('DescribeResourcePolicy');
    }
    /**
     * Grants permission to detach a policy from a target root, organizational unit, or account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DetachPolicy.html
     */
    toDetachPolicy() {
        return this.to('DetachPolicy');
    }
    /**
     * Grants permission to disable integration of an AWS service (the service that is specified by ServicePrincipal) with AWS Organizations
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifServicePrincipal()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DisableAWSServiceAccess.html
     */
    toDisableAWSServiceAccess() {
        return this.to('DisableAWSServiceAccess');
    }
    /**
     * Grants permission to disable an organization policy type in a root
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DisablePolicyType.html
     */
    toDisablePolicyType() {
        return this.to('DisablePolicyType');
    }
    /**
     * Grants permission to enable integration of an AWS service (the service that is specified by ServicePrincipal) with AWS Organizations
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifServicePrincipal()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_EnableAWSServiceAccess.html
     */
    toEnableAWSServiceAccess() {
        return this.to('EnableAWSServiceAccess');
    }
    /**
     * Grants permission to start the process to enable all features in an organization, upgrading it from supporting only Consolidated Billing features
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_EnableAllFeatures.html
     */
    toEnableAllFeatures() {
        return this.to('EnableAllFeatures');
    }
    /**
     * Grants permission to enable a policy type in a root
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_EnablePolicyType.html
     */
    toEnablePolicyType() {
        return this.to('EnablePolicyType');
    }
    /**
     * Grants permission to send an invitation to another AWS account, asking it to join your organization as a member account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_InviteAccountToOrganization.html
     */
    toInviteAccountToOrganization() {
        return this.to('InviteAccountToOrganization');
    }
    /**
     * Grants permission to remove a member account from its parent organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_LeaveOrganization.html
     */
    toLeaveOrganization() {
        return this.to('LeaveOrganization');
    }
    /**
     * Grants permission to retrieve the list of the AWS services for which you enabled integration with your organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListAWSServiceAccessForOrganization.html
     */
    toListAWSServiceAccessForOrganization() {
        return this.to('ListAWSServiceAccessForOrganization');
    }
    /**
     * Grants permission to list all of the the accounts in the organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListAccounts.html
     */
    toListAccounts() {
        return this.to('ListAccounts');
    }
    /**
     * Grants permission to list the accounts in an organization that are contained by a root or organizational unit (OU)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListAccountsForParent.html
     */
    toListAccountsForParent() {
        return this.to('ListAccountsForParent');
    }
    /**
     * Grants permission to list all of the OUs or accounts that are contained in a parent OU or root
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListChildren.html
     */
    toListChildren() {
        return this.to('ListChildren');
    }
    /**
     * Grants permission to list the asynchronous account creation requests that are currently being tracked for the organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListCreateAccountStatus.html
     */
    toListCreateAccountStatus() {
        return this.to('ListCreateAccountStatus');
    }
    /**
     * Grants permission to list the AWS accounts that are designated as delegated administrators in this organization
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifServicePrincipal()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListDelegatedAdministrators.html
     */
    toListDelegatedAdministrators() {
        return this.to('ListDelegatedAdministrators');
    }
    /**
     * Grants permission to list the AWS services for which the specified account is a delegated administrator in this organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListDelegatedServicesForAccount.html
     */
    toListDelegatedServicesForAccount() {
        return this.to('ListDelegatedServicesForAccount');
    }
    /**
     * Grants permission to list all of the handshakes that are associated with an account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListHandshakesForAccount.html
     */
    toListHandshakesForAccount() {
        return this.to('ListHandshakesForAccount');
    }
    /**
     * Grants permission to list the handshakes that are associated with the organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListHandshakesForOrganization.html
     */
    toListHandshakesForOrganization() {
        return this.to('ListHandshakesForOrganization');
    }
    /**
     * Grants permission to lists all of the organizational units (OUs) in a parent organizational unit or root
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListOrganizationalUnitsForParent.html
     */
    toListOrganizationalUnitsForParent() {
        return this.to('ListOrganizationalUnitsForParent');
    }
    /**
     * Grants permission to list the root or organizational units (OUs) that serve as the immediate parent of a child OU or account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListParents.html
     */
    toListParents() {
        return this.to('ListParents');
    }
    /**
     * Grants permission to list all of the policies in an organization
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListPolicies.html
     */
    toListPolicies() {
        return this.to('ListPolicies');
    }
    /**
     * Grants permission to list all of the policies that are directly attached to a root, organizational unit (OU), or account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListPoliciesForTarget.html
     */
    toListPoliciesForTarget() {
        return this.to('ListPoliciesForTarget');
    }
    /**
     * Grants permission to list all of the roots that are defined in the organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListRoots.html
     */
    toListRoots() {
        return this.to('ListRoots');
    }
    /**
     * Grants permission to list all tags for the specified resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list all the roots, OUs, and accounts to which a policy is attached
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListTargetsForPolicy.html
     */
    toListTargetsForPolicy() {
        return this.to('ListTargetsForPolicy');
    }
    /**
     * Grants permission to move an account from its current root or OU to another parent root or OU
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_MoveAccount.html
     */
    toMoveAccount() {
        return this.to('MoveAccount');
    }
    /**
     * Grants permission to create or update a resource policy
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_PutResourcePolicy.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to register the specified member account to administer the Organizations features of the AWS service that is specified by ServicePrincipal
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifServicePrincipal()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_RegisterDelegatedAdministrator.html
     */
    toRegisterDelegatedAdministrator() {
        return this.to('RegisterDelegatedAdministrator');
    }
    /**
     * Grants permission to removes the specified account from the organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_RemoveAccountFromOrganization.html
     */
    toRemoveAccountFromOrganization() {
        return this.to('RemoveAccountFromOrganization');
    }
    /**
     * Grants permission to add one or more tags to the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove one or more tags from the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to rename an organizational unit (OU)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_UpdateOrganizationalUnit.html
     */
    toUpdateOrganizationalUnit() {
        return this.to('UpdateOrganizationalUnit');
    }
    /**
     * Grants permission to update an existing policy with a new name, description, or content
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_UpdatePolicy.html
     */
    toUpdatePolicy() {
        return this.to('UpdatePolicy');
    }
    /**
     * Adds a resource of type account to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions_overview.html
     *
     * @param organizationId - Identifier for the organizationId.
     * @param accountId - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAccount(organizationId, accountId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:organizations::${account ?? this.defaultAccount}:account/o-${organizationId}/${accountId ?? this.defaultAccount}`);
    }
    /**
     * Adds a resource of type handshake to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions_overview.html
     *
     * @param organizationId - Identifier for the organizationId.
     * @param handshakeType - Identifier for the handshakeType.
     * @param handshakeId - Identifier for the handshakeId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onHandshake(organizationId, handshakeType, handshakeId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:organizations::${account ?? this.defaultAccount}:handshake/o-${organizationId}/${handshakeType}/h-${handshakeId}`);
    }
    /**
     * Adds a resource of type organization to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions_overview.html
     *
     * @param organizationId - Identifier for the organizationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onOrganization(organizationId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:organizations::${account ?? this.defaultAccount}:organization/o-${organizationId}`);
    }
    /**
     * Adds a resource of type organizationalunit to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions_overview.html
     *
     * @param organizationId - Identifier for the organizationId.
     * @param organizationalUnitId - Identifier for the organizationalUnitId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onOrganizationalunit(organizationId, organizationalUnitId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:organizations::${account ?? this.defaultAccount}:ou/o-${organizationId}/ou-${organizationalUnitId}`);
    }
    /**
     * Adds a resource of type policy to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions_overview.html
     *
     * @param organizationId - Identifier for the organizationId.
     * @param policyType - Identifier for the policyType.
     * @param policyId - Identifier for the policyId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPolicy(organizationId, policyType, policyId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:organizations::${account ?? this.defaultAccount}:policy/o-${organizationId}/${policyType}/p-${policyId}`);
    }
    /**
     * Adds a resource of type resourcepolicy to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions_overview.html
     *
     * @param organizationId - Identifier for the organizationId.
     * @param resourcePolicyId - Identifier for the resourcePolicyId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onResourcepolicy(organizationId, resourcePolicyId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:organizations::${account ?? this.defaultAccount}:resourcepolicy/o-${organizationId}/rp-${resourcePolicyId}`);
    }
    /**
     * Adds a resource of type awspolicy to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions_overview.html
     *
     * @param policyType - Identifier for the policyType.
     * @param policyId - Identifier for the policyId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAwspolicy(policyType, policyId, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:organizations::aws:policy/${policyType}/p-${policyId}`);
    }
    /**
     * Adds a resource of type root to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions_overview.html
     *
     * @param organizationId - Identifier for the organizationId.
     * @param rootId - Identifier for the rootId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRoot(organizationId, rootId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:organizations::${account ?? this.defaultAccount}:root/o-${organizationId}/r-${rootId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateAccount()
     * - .toCreateGovCloudAccount()
     * - .toCreateOrganizationalUnit()
     * - .toCreatePolicy()
     * - .toInviteAccountToOrganization()
     * - .toPutResourcePolicy()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toCreateAccount()
     * - .toCreateGovCloudAccount()
     * - .toCreatePolicy()
     *
     * Applies to resource types:
     * - account
     * - organizationalunit
     * - policy
     * - resourcepolicy
     * - root
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateAccount()
     * - .toCreateGovCloudAccount()
     * - .toCreateOrganizationalUnit()
     * - .toCreatePolicy()
     * - .toInviteAccountToOrganization()
     * - .toPutResourcePolicy()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the specified policy type names
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions_overview.html#orgs_permissions_conditionkeys
     *
     * Applies to actions:
     * - .toAttachPolicy()
     * - .toCreatePolicy()
     * - .toDeletePolicy()
     * - .toDescribeEffectivePolicy()
     * - .toDescribePolicy()
     * - .toDetachPolicy()
     * - .toDisablePolicyType()
     * - .toEnablePolicyType()
     * - .toListPolicies()
     * - .toListPoliciesForTarget()
     * - .toListTargetsForPolicy()
     * - .toTagResource()
     * - .toUpdatePolicy()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPolicyType(value, operator) {
        return this.if(`PolicyType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the specified service principal names
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions_overview.html#orgs_permissions_conditionkeys
     *
     * Applies to actions:
     * - .toDeregisterDelegatedAdministrator()
     * - .toDisableAWSServiceAccess()
     * - .toEnableAWSServiceAccess()
     * - .toListDelegatedAdministrators()
     * - .toRegisterDelegatedAdministrator()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifServicePrincipal(value, operator) {
        return this.if(`ServicePrincipal`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [organizations](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsorganizations.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'organizations';
        this.accessLevelList = {
            Write: [
                'AcceptHandshake',
                'AttachPolicy',
                'CancelHandshake',
                'CloseAccount',
                'CreateAccount',
                'CreateGovCloudAccount',
                'CreateOrganization',
                'CreateOrganizationalUnit',
                'CreatePolicy',
                'DeclineHandshake',
                'DeleteOrganization',
                'DeleteOrganizationalUnit',
                'DeletePolicy',
                'DeleteResourcePolicy',
                'DeregisterDelegatedAdministrator',
                'DetachPolicy',
                'DisableAWSServiceAccess',
                'DisablePolicyType',
                'EnableAWSServiceAccess',
                'EnableAllFeatures',
                'EnablePolicyType',
                'InviteAccountToOrganization',
                'LeaveOrganization',
                'MoveAccount',
                'PutResourcePolicy',
                'RegisterDelegatedAdministrator',
                'RemoveAccountFromOrganization',
                'UpdateOrganizationalUnit',
                'UpdatePolicy'
            ],
            Read: [
                'DescribeAccount',
                'DescribeCreateAccountStatus',
                'DescribeEffectivePolicy',
                'DescribeHandshake',
                'DescribeOrganization',
                'DescribeOrganizationalUnit',
                'DescribePolicy',
                'DescribeResourcePolicy'
            ],
            List: [
                'ListAWSServiceAccessForOrganization',
                'ListAccounts',
                'ListAccountsForParent',
                'ListChildren',
                'ListCreateAccountStatus',
                'ListDelegatedAdministrators',
                'ListDelegatedServicesForAccount',
                'ListHandshakesForAccount',
                'ListHandshakesForOrganization',
                'ListOrganizationalUnitsForParent',
                'ListParents',
                'ListPolicies',
                'ListPoliciesForTarget',
                'ListRoots',
                'ListTagsForResource',
                'ListTargetsForPolicy'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Organizations = Organizations;
//# sourceMappingURL=data:application/json;base64,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