"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mobiletargeting = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [mobiletargeting](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonpinpoint.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Mobiletargeting extends shared_1.PolicyStatement {
    /**
     * Grants permission to create an app
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps.html#CreateApp
     */
    toCreateApp() {
        return this.to('CreateApp');
    }
    /**
     * Grants permission to create a campaign for an app
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-campaigns.html#CreateCampaign
     */
    toCreateCampaign() {
        return this.to('CreateCampaign');
    }
    /**
     * Grants permission to create an email template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-email.html#CreateEmailTemplate
     */
    toCreateEmailTemplate() {
        return this.to('CreateEmailTemplate');
    }
    /**
     * Grants permission to create an export job that exports endpoint definitions to Amazon S3
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-jobs-export.html#CreateExportJob
     */
    toCreateExportJob() {
        return this.to('CreateExportJob');
    }
    /**
     * Grants permission to import endpoint definitions from to create a segment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-jobs-import.html#CreateImportJob
     */
    toCreateImportJob() {
        return this.to('CreateImportJob');
    }
    /**
     * Grants permission to create an in-app message template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-inapp.html#CreateInAppTemplate
     */
    toCreateInAppTemplate() {
        return this.to('CreateInAppTemplate');
    }
    /**
     * Grants permission to create a Journey for an app
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys.html#CreateJourney
     */
    toCreateJourney() {
        return this.to('CreateJourney');
    }
    /**
     * Grants permission to create a push notification template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-push.html#CreatePushTemplate
     */
    toCreatePushTemplate() {
        return this.to('CreatePushTemplate');
    }
    /**
     * Grants permission to create an Amazon Pinpoint configuration for a recommender model
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders.html#CreateRecommenderConfiguration
     */
    toCreateRecommenderConfiguration() {
        return this.to('CreateRecommenderConfiguration');
    }
    /**
     * Grants permission to create a segment that is based on endpoint data reported to Pinpoint by your app. To allow a user to create a segment by importing endpoint data from outside of Pinpoint, allow the mobiletargeting:CreateImportJob action
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-segments.html#CreateSegment
     */
    toCreateSegment() {
        return this.to('CreateSegment');
    }
    /**
     * Grants permission to create an sms message template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-sms.html#CreateSmsTemplate
     */
    toCreateSmsTemplate() {
        return this.to('CreateSmsTemplate');
    }
    /**
     * Grants permission to create a voice message template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-voice.html#CreateVoiceTemplate
     */
    toCreateVoiceTemplate() {
        return this.to('CreateVoiceTemplate');
    }
    /**
     * Grants permission to delete the ADM channel for an app
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-channels-adm.html#DeleteAdmChannel
     */
    toDeleteAdmChannel() {
        return this.to('DeleteAdmChannel');
    }
    /**
     * Grants permission to delete the APNs channel for an app
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-channels-apns.html#DeleteApnsChannel
     */
    toDeleteApnsChannel() {
        return this.to('DeleteApnsChannel');
    }
    /**
     * Grants permission to delete the APNs sandbox channel for an app
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-channels-apns_sandbox.html#DeleteApnsSandboxChannel
     */
    toDeleteApnsSandboxChannel() {
        return this.to('DeleteApnsSandboxChannel');
    }
    /**
     * Grants permission to delete the APNs VoIP channel for an app
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-channels-apns_voip.html#DeleteApnsVoipChannel
     */
    toDeleteApnsVoipChannel() {
        return this.to('DeleteApnsVoipChannel');
    }
    /**
     * Grants permission to delete the APNs VoIP sandbox channel for an app
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-channels-apns_voip_sandbox.html#DeleteApnsVoipSandboxChannel
     */
    toDeleteApnsVoipSandboxChannel() {
        return this.to('DeleteApnsVoipSandboxChannel');
    }
    /**
     * Grants permission to delete a specific campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id.html#DeleteApp
     */
    toDeleteApp() {
        return this.to('DeleteApp');
    }
    /**
     * Grants permission to delete the Baidu channel for an app
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-channels-baidu.html#DeleteBaiduChannel
     */
    toDeleteBaiduChannel() {
        return this.to('DeleteBaiduChannel');
    }
    /**
     * Grants permission to delete a specific campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-campaigns-campaign-id.html#DeleteCampaign
     */
    toDeleteCampaign() {
        return this.to('DeleteCampaign');
    }
    /**
     * Grants permission to delete the email channel for an app
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-channels-email.html#DeleteEmailChannel
     */
    toDeleteEmailChannel() {
        return this.to('DeleteEmailChannel');
    }
    /**
     * Grants permission to delete an email template or an email template version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-email.html#DeleteEmailTemplate
     */
    toDeleteEmailTemplate() {
        return this.to('DeleteEmailTemplate');
    }
    /**
     * Grants permission to delete an endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-endpoints-endpoint-id.html#DeleteEndpoint
     */
    toDeleteEndpoint() {
        return this.to('DeleteEndpoint');
    }
    /**
     * Grants permission to delete the event stream for an app
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-eventstream.html#DeleteEventStream
     */
    toDeleteEventStream() {
        return this.to('DeleteEventStream');
    }
    /**
     * Grants permission to delete the GCM channel for an app
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-channels-gcm.html#DeleteGcmChannel
     */
    toDeleteGcmChannel() {
        return this.to('DeleteGcmChannel');
    }
    /**
     * Grants permission to delete an in-app message template or an in-app message template version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-inapp.html#DeleteInAppTemplate
     */
    toDeleteInAppTemplate() {
        return this.to('DeleteInAppTemplate');
    }
    /**
     * Grants permission to delete a specific journey
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id.html#DeleteJourney
     */
    toDeleteJourney() {
        return this.to('DeleteJourney');
    }
    /**
     * Grants permission to delete a push notification template or a push notification template version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-push.html#DeletePushTemplate
     */
    toDeletePushTemplate() {
        return this.to('DeletePushTemplate');
    }
    /**
     * Grants permission to delete an Amazon Pinpoint configuration for a recommender model
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders-recommender-id.html#DeleteRecommenderConfiguration
     */
    toDeleteRecommenderConfiguration() {
        return this.to('DeleteRecommenderConfiguration');
    }
    /**
     * Grants permission to delete a specific segment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-segments-segment-id.html#DeleteSegment
     */
    toDeleteSegment() {
        return this.to('DeleteSegment');
    }
    /**
     * Grants permission to delete the SMS channel for an app
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-channels-sms.html#DeleteSmsChannel
     */
    toDeleteSmsChannel() {
        return this.to('DeleteSmsChannel');
    }
    /**
     * Grants permission to delete an sms message template or an sms message template version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-sms.html#DeleteSmsTemplate
     */
    toDeleteSmsTemplate() {
        return this.to('DeleteSmsTemplate');
    }
    /**
     * Grants permission to delete all of the endpoints that are associated with a user ID
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-users-user-id.html#DeleteUserEndpoints
     */
    toDeleteUserEndpoints() {
        return this.to('DeleteUserEndpoints');
    }
    /**
     * Grants permission to delete the Voice channel for an app
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-channels-voice.html#DeleteVoiceChannel
     */
    toDeleteVoiceChannel() {
        return this.to('DeleteVoiceChannel');
    }
    /**
     * Grants permission to delete a voice message template or a voice message template version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-voice.html#DeleteVoiceTemplate
     */
    toDeleteVoiceTemplate() {
        return this.to('DeleteVoiceTemplate');
    }
    /**
     * Grants permission to retrieve information about the Amazon Device Messaging (ADM) channel for an app
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-channels-adm.html#GetAdmChannel
     */
    toGetAdmChannel() {
        return this.to('GetAdmChannel');
    }
    /**
     * Grants permission to retrieve information about the APNs channel for an app
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-channels-apns.html#GetApnsChannel
     */
    toGetApnsChannel() {
        return this.to('GetApnsChannel');
    }
    /**
     * Grants permission to retrieve information about the APNs sandbox channel for an app
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-channels-apns_sandbox.html#GetApnsSandboxChannel
     */
    toGetApnsSandboxChannel() {
        return this.to('GetApnsSandboxChannel');
    }
    /**
     * Grants permission to retrieve information about the APNs VoIP channel for an app
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-channels-apns_voip.html#GetApnsVoipChannel
     */
    toGetApnsVoipChannel() {
        return this.to('GetApnsVoipChannel');
    }
    /**
     * Grants permission to retrieve information about the APNs VoIP sandbox channel for an app
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-channels-apns_voip_sandbox.html#GetApnsVoipSandboxChannel
     */
    toGetApnsVoipSandboxChannel() {
        return this.to('GetApnsVoipSandboxChannel');
    }
    /**
     * Grants permission to retrieve information about a specific app in your Amazon Pinpoint account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id.html#GetApp
     */
    toGetApp() {
        return this.to('GetApp');
    }
    /**
     * Grants permission to retrieve (queries) pre-aggregated data for a standard metric that applies to an application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-kpis-daterange-kpi-name.html#GetApplicationDateRangeKpi
     */
    toGetApplicationDateRangeKpi() {
        return this.to('GetApplicationDateRangeKpi');
    }
    /**
     * Grants permission to retrieve the default settings for an app
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-settings.html#GetApplicationSettings
     */
    toGetApplicationSettings() {
        return this.to('GetApplicationSettings');
    }
    /**
     * Grants permission to retrieve a list of apps in your Amazon Pinpoint account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps.html#GetApps
     */
    toGetApps() {
        return this.to('GetApps');
    }
    /**
     * Grants permission to retrieve information about the Baidu channel for an app
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-channels-baidu.html#GetBaiduChannel
     */
    toGetBaiduChannel() {
        return this.to('GetBaiduChannel');
    }
    /**
     * Grants permission to retrieve information about a specific campaign
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-campaigns-campaign-id.html#GetCampaign
     */
    toGetCampaign() {
        return this.to('GetCampaign');
    }
    /**
     * Grants permission to retrieve information about the activities performed by a campaign
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-campaigns-campaign-id-activities.html#GetCampaignActivities
     */
    toGetCampaignActivities() {
        return this.to('GetCampaignActivities');
    }
    /**
     * Grants permission to retrieve (queries) pre-aggregated data for a standard metric that applies to a campaign
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-campaigns-campaign-id-kpis-daterange-kpi-name.html#GetCampaignDateRangeKpi
     */
    toGetCampaignDateRangeKpi() {
        return this.to('GetCampaignDateRangeKpi');
    }
    /**
     * Grants permission to retrieve information about a specific campaign version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-campaigns-campaign-id-versions-version.html#GetCampaignVersion
     */
    toGetCampaignVersion() {
        return this.to('GetCampaignVersion');
    }
    /**
     * Grants permission to retrieve information about the current and prior versions of a campaign
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-campaigns-campaign-id-versions.html#GetCampaignVersions
     */
    toGetCampaignVersions() {
        return this.to('GetCampaignVersions');
    }
    /**
     * Grants permission to retrieve information about all campaigns for an app
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-campaigns.html#GetCampaigns
     */
    toGetCampaigns() {
        return this.to('GetCampaigns');
    }
    /**
     * Grants permission to get all channels information for your app
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-channels.html#GetChannels
     */
    toGetChannels() {
        return this.to('GetChannels');
    }
    /**
     * Grants permission to obtain information about the email channel in an app
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-channels-email.html#GetEmailChannel
     */
    toGetEmailChannel() {
        return this.to('GetEmailChannel');
    }
    /**
     * Grants permission to retrieve information about a specific or the active version of an email template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-email.html#GetEmailTemplate
     */
    toGetEmailTemplate() {
        return this.to('GetEmailTemplate');
    }
    /**
     * Grants permission to retrieve information about a specific endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-endpoints-endpoint-id.html#GetEndpoint
     */
    toGetEndpoint() {
        return this.to('GetEndpoint');
    }
    /**
     * Grants permission to retrieve information about the event stream for an app
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-eventstream.html#GetEventStream
     */
    toGetEventStream() {
        return this.to('GetEventStream');
    }
    /**
     * Grants permission to obtain information about a specific export job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-jobs-export-job-id.html#GetExportJob
     */
    toGetExportJob() {
        return this.to('GetExportJob');
    }
    /**
     * Grants permission to retrieve a list of all of the export jobs for an app
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-jobs-export.html#GetExportJobs
     */
    toGetExportJobs() {
        return this.to('GetExportJobs');
    }
    /**
     * Grants permission to retrieve information about the GCM channel for an app
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-channels-gcm.html#GetGcmChannel
     */
    toGetGcmChannel() {
        return this.to('GetGcmChannel');
    }
    /**
     * Grants permission to retrieve information about a specific import job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-jobs-import-job-id.html#GetImportJob
     */
    toGetImportJob() {
        return this.to('GetImportJob');
    }
    /**
     * Grants permission to retrieve information about all import jobs for an app
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-jobs-import.html#GetImportJobs
     */
    toGetImportJobs() {
        return this.to('GetImportJobs');
    }
    /**
     * Grants permission to retrive in-app messages for the given endpoint id
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-endpoints-endpoint-id-inappmessages.html#GetInAppMessages
     */
    toGetInAppMessages() {
        return this.to('GetInAppMessages');
    }
    /**
     * Grants permission to retrieve information about a specific or the active version of an in-app message template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-inapp.html#GetInAppTemplate
     */
    toGetInAppTemplate() {
        return this.to('GetInAppTemplate');
    }
    /**
     * Grants permission to retrieve information about a specific journey
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id.html#GetJourney
     */
    toGetJourney() {
        return this.to('GetJourney');
    }
    /**
     * Grants permission to retrieve (queries) pre-aggregated data for a standard engagement metric that applies to a journey
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id-kpis-daterange-kpi-name.html#GetJourneyDateRangeKpi
     */
    toGetJourneyDateRangeKpi() {
        return this.to('GetJourneyDateRangeKpi');
    }
    /**
     * Grants permission to retrieve (queries) pre-aggregated data for a standard execution metric that applies to a journey activity
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id-activities-journey-activity-id-execution-metrics.html#GetJourneyExecutionActivityMetrics
     */
    toGetJourneyExecutionActivityMetrics() {
        return this.to('GetJourneyExecutionActivityMetrics');
    }
    /**
     * Grants permission to retrieve (queries) pre-aggregated data for a standard execution metric that applies to a journey
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id-execution-metrics.html#GetJourneyExecutionMetrics
     */
    toGetJourneyExecutionMetrics() {
        return this.to('GetJourneyExecutionMetrics');
    }
    /**
     * Grants permission to retrieve (queries) pre-aggregated data for a standard execution metric that applies to a journey activity for a single journey run
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id-runs-run-id-activities-journey-activity-id-execution-metrics.html#GetJourneyRunExecutionActivityMetrics
     */
    toGetJourneyRunExecutionActivityMetrics() {
        return this.to('GetJourneyRunExecutionActivityMetrics');
    }
    /**
     * Grants permission to retrieve (queries) pre-aggregated data for a standard execution metric that applies to a journey for a single journey run
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id-runs-run-id-execution-metrics.html#GetJourneyRunExecutionMetrics
     */
    toGetJourneyRunExecutionMetrics() {
        return this.to('GetJourneyRunExecutionMetrics');
    }
    /**
     * Grants permission to retrieve information about all journey runs for a journey
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id-runs.html
     */
    toGetJourneyRuns() {
        return this.to('GetJourneyRuns');
    }
    /**
     * Grants permission to retrieve information about a specific or the active version of an push notification template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-push.html#GetPushTemplate
     */
    toGetPushTemplate() {
        return this.to('GetPushTemplate');
    }
    /**
     * Grants permission to retrieve information about an Amazon Pinpoint configuration for a recommender model
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders-recommender-id.html#GetRecommenderConfiguration
     */
    toGetRecommenderConfiguration() {
        return this.to('GetRecommenderConfiguration');
    }
    /**
     * Grants permission to retrieve information about all the recommender model configurations that are associated with an Amazon Pinpoint account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders.html#GetRecommenderConfigurations
     */
    toGetRecommenderConfigurations() {
        return this.to('GetRecommenderConfigurations');
    }
    /**
     * Grants permission to mobiletargeting:GetReports
     *
     * Access Level: Read
     */
    toGetReports() {
        return this.to('GetReports');
    }
    /**
     * Grants permission to retrieve information about a specific segment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-segments-segment-id.html#GetSegment
     */
    toGetSegment() {
        return this.to('GetSegment');
    }
    /**
     * Grants permission to retrieve information about jobs that export endpoint definitions from segments to Amazon S3
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-segments-segment-id-jobs-export.html#GetSegmentExportJobs
     */
    toGetSegmentExportJobs() {
        return this.to('GetSegmentExportJobs');
    }
    /**
     * Grants permission to retrieve information about jobs that create segments by importing endpoint definitions from
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-segments-segment-id-jobs-import.html#GetSegmentImportJobs
     */
    toGetSegmentImportJobs() {
        return this.to('GetSegmentImportJobs');
    }
    /**
     * Grants permission to retrieve information about a specific segment version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-segments-segment-id-versions-version.html#GetSegmentVersion
     */
    toGetSegmentVersion() {
        return this.to('GetSegmentVersion');
    }
    /**
     * Grants permission to retrieve information about the current and prior versions of a segment
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-segments-segment-id-versions.html#GetSegmentVersions
     */
    toGetSegmentVersions() {
        return this.to('GetSegmentVersions');
    }
    /**
     * Grants permission to retrieve information about the segments for an app
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-segments.html#GetSegments
     */
    toGetSegments() {
        return this.to('GetSegments');
    }
    /**
     * Grants permission to obtain information about the SMS channel in an app
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-channels-sms.html#GetSmsChannel
     */
    toGetSmsChannel() {
        return this.to('GetSmsChannel');
    }
    /**
     * Grants permission to retrieve information about a specific or the active version of an sms message template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-sms.html#GetSmsTemplate
     */
    toGetSmsTemplate() {
        return this.to('GetSmsTemplate');
    }
    /**
     * Grants permission to retrieve information about the endpoints that are associated with a user ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-users-user-id.html#GetUserEndpoints
     */
    toGetUserEndpoints() {
        return this.to('GetUserEndpoints');
    }
    /**
     * Grants permission to obtain information about the Voice channel in an app
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-channels-voice.html#GetVoiceChannel
     */
    toGetVoiceChannel() {
        return this.to('GetVoiceChannel');
    }
    /**
     * Grants permission to retrieve information about a specific or the active version of a voice message template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-voice.html#GetVoiceTemplate
     */
    toGetVoiceTemplate() {
        return this.to('GetVoiceTemplate');
    }
    /**
     * Grants permission to retrieve information about all journeys for an app
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys.html#ListJourneys
     */
    toListJourneys() {
        return this.to('ListJourneys');
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/tags-resource-arn.html#ListTagsForResource
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to retrieve all versions about a specific template
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-template-type-versions.html#ListTemplateVersions
     */
    toListTemplateVersions() {
        return this.to('ListTemplateVersions');
    }
    /**
     * Grants permission to retrieve metadata about the queried templates
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates.html#ListTemplates
     */
    toListTemplates() {
        return this.to('ListTemplates');
    }
    /**
     * Grants permission to obtain metadata for a phone number, such as the number type (mobile, landline, or VoIP), location, and provider
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/phone-number-validate.html#PhoneNumberValidate
     */
    toPhoneNumberValidate() {
        return this.to('PhoneNumberValidate');
    }
    /**
     * Grants permission to create or update an event stream for an app
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-eventstream.html#PutEventStream
     */
    toPutEventStream() {
        return this.to('PutEventStream');
    }
    /**
     * Grants permission to create or update events for an app
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-events.html#PutEvents
     */
    toPutEvents() {
        return this.to('PutEvents');
    }
    /**
     * Grants permission to remove the attributes for an app
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-attributes-attribute-type.html#RemoveAttributes
     */
    toRemoveAttributes() {
        return this.to('RemoveAttributes');
    }
    /**
     * Grants permission to send an SMS message or push notification to specific endpoints
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-messages.html#SendMessages
     */
    toSendMessages() {
        return this.to('SendMessages');
    }
    /**
     * Grants permission to send an OTP code to a user of your application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-otp.html#SendOTPMessage
     */
    toSendOTPMessage() {
        return this.to('SendOTPMessage');
    }
    /**
     * Grants permission to send an SMS message or push notification to all endpoints that are associated with a specific user ID
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-users-messages.html#SendUsersMessages
     */
    toSendUsersMessages() {
        return this.to('SendUsersMessages');
    }
    /**
     * Grants permission to add tags to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/tags-resource-arn.html#TagResource
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/tags-resource-arn.html#UntagResource
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update the Amazon Device Messaging (ADM) channel for an app
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-channels-adm.html#UpdateAdmChannel
     */
    toUpdateAdmChannel() {
        return this.to('UpdateAdmChannel');
    }
    /**
     * Grants permission to update the Apple Push Notification service (APNs) channel for an app
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-channels-apns.html#UpdateApnsChannel
     */
    toUpdateApnsChannel() {
        return this.to('UpdateApnsChannel');
    }
    /**
     * Grants permission to update the Apple Push Notification service (APNs) sandbox channel for an app
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-channels-apns_sandbox.html#UpdateApnsSandboxChannel
     */
    toUpdateApnsSandboxChannel() {
        return this.to('UpdateApnsSandboxChannel');
    }
    /**
     * Grants permission to update the Apple Push Notification service (APNs) VoIP channel for an app
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-channels-apns_voip.html#UpdateApnsVoipChannel
     */
    toUpdateApnsVoipChannel() {
        return this.to('UpdateApnsVoipChannel');
    }
    /**
     * Grants permission to update the Apple Push Notification service (APNs) VoIP sandbox channel for an app
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-channels-apns_voip_sandbox.html#UpdateApnsVoipSandboxChannel
     */
    toUpdateApnsVoipSandboxChannel() {
        return this.to('UpdateApnsVoipSandboxChannel');
    }
    /**
     * Grants permission to update the default settings for an app
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-settings.html#UpdateApplicationSettings
     */
    toUpdateApplicationSettings() {
        return this.to('UpdateApplicationSettings');
    }
    /**
     * Grants permission to update the Baidu channel for an app
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-channels-baidu.html#UpdateBaiduChannel
     */
    toUpdateBaiduChannel() {
        return this.to('UpdateBaiduChannel');
    }
    /**
     * Grants permission to update a specific campaign
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-campaigns-campaign-id.html#UpdateCampaign
     */
    toUpdateCampaign() {
        return this.to('UpdateCampaign');
    }
    /**
     * Grants permission to update the email channel for an app
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-channels-email.html#UpdateEmailChannel
     */
    toUpdateEmailChannel() {
        return this.to('UpdateEmailChannel');
    }
    /**
     * Grants permission to update a specific email template under the same version or generate a new version
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-email.html#UpdateEmailTemplate
     */
    toUpdateEmailTemplate() {
        return this.to('UpdateEmailTemplate');
    }
    /**
     * Grants permission to create an endpoint or update the information for an endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-endpoints-endpoint-id.html#UpdateEndpoint
     */
    toUpdateEndpoint() {
        return this.to('UpdateEndpoint');
    }
    /**
     * Grants permission to create or update endpoints as a batch operation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-endpoints.html#UpdateEndpointsBatch
     */
    toUpdateEndpointsBatch() {
        return this.to('UpdateEndpointsBatch');
    }
    /**
     * Grants permission to update the Firebase Cloud Messaging (FCM) or Google Cloud Messaging (GCM) API key that allows to send push notifications to your Android app
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-channels-gcm.html#UpdateGcmChannel
     */
    toUpdateGcmChannel() {
        return this.to('UpdateGcmChannel');
    }
    /**
     * Grants permission to update a specific in-app message template under the same version or generate a new version
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-inapp.html#UpdateInAppTemplate
     */
    toUpdateInAppTemplate() {
        return this.to('UpdateInAppTemplate');
    }
    /**
     * Grants permission to update a specific journey
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id.html#UpdateJourney
     */
    toUpdateJourney() {
        return this.to('UpdateJourney');
    }
    /**
     * Grants permission to update a specific journey state
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id-state.html#UpdateJourneyState
     */
    toUpdateJourneyState() {
        return this.to('UpdateJourneyState');
    }
    /**
     * Grants permission to update a specific push notification template under the same version or generate a new version
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-push.html#UpdatePushTemplate
     */
    toUpdatePushTemplate() {
        return this.to('UpdatePushTemplate');
    }
    /**
     * Grants permission to update an Amazon Pinpoint configuration for a recommender model
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders-recommender-id.html#UpdateRecommenderConfiguration
     */
    toUpdateRecommenderConfiguration() {
        return this.to('UpdateRecommenderConfiguration');
    }
    /**
     * Grants permission to update a specific segment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-segments-segment-id.html#UpdateSegment
     */
    toUpdateSegment() {
        return this.to('UpdateSegment');
    }
    /**
     * Grants permission to update the SMS channel for an app
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-channels-sms.html#UpdateSmsChannel
     */
    toUpdateSmsChannel() {
        return this.to('UpdateSmsChannel');
    }
    /**
     * Grants permission to update a specific sms message template under the same version or generate a new version
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-sms.html#UpdateSmsTemplate
     */
    toUpdateSmsTemplate() {
        return this.to('UpdateSmsTemplate');
    }
    /**
     * Grants permission to update the active version parameter of a specific template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-template-type-active-version.html#UpdateTemplateActiveVersion
     */
    toUpdateTemplateActiveVersion() {
        return this.to('UpdateTemplateActiveVersion');
    }
    /**
     * Grants permission to update the Voice channel for an app
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-channels-voice.html#UpdateVoiceChannel
     */
    toUpdateVoiceChannel() {
        return this.to('UpdateVoiceChannel');
    }
    /**
     * Grants permission to update a specific voice message template under the same version or generate a new version
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-voice.html#UpdateVoiceTemplate
     */
    toUpdateVoiceTemplate() {
        return this.to('UpdateVoiceTemplate');
    }
    /**
     * Grants permission to check the validity of One-Time Passwords (OTPs)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-verify-otp.html#VerifyOTPMessage
     */
    toVerifyOTPMessage() {
        return this.to('VerifyOTPMessage');
    }
    /**
     * Adds a resource of type app to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id.html
     *
     * @param appId - Identifier for the appId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onApp(appId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mobiletargeting:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:apps/${appId}`);
    }
    /**
     * Adds a resource of type apps to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onApps(resourceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mobiletargeting:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:apps/${resourceName}`);
    }
    /**
     * Adds a resource of type campaign to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-campaigns-campaign-id.html
     *
     * @param appId - Identifier for the appId.
     * @param campaignId - Identifier for the campaignId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCampaign(appId, campaignId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mobiletargeting:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:apps/${appId}/campaigns/${campaignId}`);
    }
    /**
     * Adds a resource of type journey to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id.html
     *
     * @param appId - Identifier for the appId.
     * @param journeyId - Identifier for the journeyId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onJourney(appId, journeyId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mobiletargeting:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:apps/${appId}/journeys/${journeyId}`);
    }
    /**
     * Adds a resource of type journeys to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys.html
     *
     * @param appId - Identifier for the appId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onJourneys(appId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mobiletargeting:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:apps/${appId}/journeys`);
    }
    /**
     * Adds a resource of type segment to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-segments-segment-id.html
     *
     * @param appId - Identifier for the appId.
     * @param segmentId - Identifier for the segmentId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSegment(appId, segmentId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mobiletargeting:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:apps/${appId}/segments/${segmentId}`);
    }
    /**
     * Adds a resource of type template to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates.html
     *
     * @param templateName - Identifier for the templateName.
     * @param templateType - Identifier for the templateType.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTemplate(templateName, templateType, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mobiletargeting:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:templates/${templateName}/${templateType}`);
    }
    /**
     * Adds a resource of type templates to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates.html
     *
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onTemplates(account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mobiletargeting:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:templates`);
    }
    /**
     * Adds a resource of type recommender to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders.html
     *
     * @param recommenderId - Identifier for the recommenderId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onRecommender(recommenderId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mobiletargeting:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:recommenders/${recommenderId}`);
    }
    /**
     * Adds a resource of type recommenders to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onRecommenders(resourceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mobiletargeting:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:recommenders/${resourceName}`);
    }
    /**
     * Adds a resource of type phone-number-validate to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/phone-number-validate.html
     *
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onPhoneNumberValidate(account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mobiletargeting:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:phone/number/validate`);
    }
    /**
     * Adds a resource of type channels to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-channels.html
     *
     * @param appId - Identifier for the appId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onChannels(appId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mobiletargeting:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:apps/${appId}/channels`);
    }
    /**
     * Adds a resource of type channel to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-channels.html
     *
     * @param appId - Identifier for the appId.
     * @param channelType - Identifier for the channelType.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onChannel(appId, channelType, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mobiletargeting:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:apps/${appId}/channels/${channelType}`);
    }
    /**
     * Adds a resource of type event-stream to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-eventstream.html
     *
     * @param appId - Identifier for the appId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onEventStream(appId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mobiletargeting:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:apps/${appId}/eventstream`);
    }
    /**
     * Adds a resource of type events to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-events.html
     *
     * @param appId - Identifier for the appId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onEvents(appId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mobiletargeting:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:apps/${appId}/events`);
    }
    /**
     * Adds a resource of type messages to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-messages.html
     *
     * @param appId - Identifier for the appId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onMessages(appId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mobiletargeting:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:apps/${appId}/messages`);
    }
    /**
     * Adds a resource of type verify-otp to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-verify-otp.html
     *
     * @param appId - Identifier for the appId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onVerifyOtp(appId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mobiletargeting:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:apps/${appId}/verify-otp`);
    }
    /**
     * Adds a resource of type otp to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-verify-otp.html
     *
     * @param appId - Identifier for the appId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onOtp(appId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mobiletargeting:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:apps/${appId}/otp`);
    }
    /**
     * Adds a resource of type attribute to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-attributes-attribute-type.html
     *
     * @param appId - Identifier for the appId.
     * @param attributeType - Identifier for the attributeType.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAttribute(appId, attributeType, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mobiletargeting:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:apps/${appId}/attributes/${attributeType}`);
    }
    /**
     * Adds a resource of type user to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-users-user-id.html
     *
     * @param appId - Identifier for the appId.
     * @param userId - Identifier for the userId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onUser(appId, userId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mobiletargeting:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:apps/${appId}/users/${userId}`);
    }
    /**
     * Adds a resource of type endpoint to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-endpoints-endpoint-id.html
     *
     * @param appId - Identifier for the appId.
     * @param endpointId - Identifier for the endpointId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onEndpoint(appId, endpointId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mobiletargeting:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:apps/${appId}/endpoints/${endpointId}`);
    }
    /**
     * Adds a resource of type import-job to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-jobs-import-job-id.html
     *
     * @param appId - Identifier for the appId.
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onImportJob(appId, jobId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mobiletargeting:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:apps/${appId}/jobs/import/${jobId}`);
    }
    /**
     * Adds a resource of type export-job to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-jobs-export-job-id.html
     *
     * @param appId - Identifier for the appId.
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onExportJob(appId, jobId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mobiletargeting:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:apps/${appId}/jobs/export/${jobId}`);
    }
    /**
     * Adds a resource of type application-metrics to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-kpis-daterange-kpi-name.html
     *
     * @param appId - Identifier for the appId.
     * @param kpiName - Identifier for the kpiName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onApplicationMetrics(appId, kpiName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mobiletargeting:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:apps/${appId}/kpis/daterange/${kpiName}`);
    }
    /**
     * Adds a resource of type campaign-metrics to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-campaigns-campaign-id-kpis-daterange-kpi-name.html
     *
     * @param appId - Identifier for the appId.
     * @param campaignId - Identifier for the campaignId.
     * @param kpiName - Identifier for the kpiName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onCampaignMetrics(appId, campaignId, kpiName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mobiletargeting:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:apps/${appId}/campaigns/${campaignId}/kpis/daterange/${kpiName}`);
    }
    /**
     * Adds a resource of type journey-metrics to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id-kpis-daterange-kpi-name.html
     *
     * @param appId - Identifier for the appId.
     * @param journeyId - Identifier for the journeyId.
     * @param kpiName - Identifier for the kpiName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onJourneyMetrics(appId, journeyId, kpiName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mobiletargeting:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:apps/${appId}/journeys/${journeyId}/kpis/daterange/${kpiName}`);
    }
    /**
     * Adds a resource of type journey-execution-metrics to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id-execution-metrics.html
     *
     * @param appId - Identifier for the appId.
     * @param journeyId - Identifier for the journeyId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onJourneyExecutionMetrics(appId, journeyId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mobiletargeting:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:apps/${appId}/journeys/${journeyId}/execution-metrics`);
    }
    /**
     * Adds a resource of type journey-execution-activity-metrics to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id-activities-journey-activity-id-execution-metrics.html
     *
     * @param appId - Identifier for the appId.
     * @param journeyId - Identifier for the journeyId.
     * @param journeyActivityId - Identifier for the journeyActivityId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onJourneyExecutionActivityMetrics(appId, journeyId, journeyActivityId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mobiletargeting:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:apps/${appId}/journeys/${journeyId}/activities/${journeyActivityId}/execution-metrics`);
    }
    /**
     * Adds a resource of type reports to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/reports.html
     *
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onReports(account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mobiletargeting:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:reports`);
    }
    /**
     * Filters access by a key that is present in the request the user makes to the pinpoint service
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toCreateApp()
     * - .toCreateCampaign()
     * - .toCreateEmailTemplate()
     * - .toCreateInAppTemplate()
     * - .toCreateJourney()
     * - .toCreatePushTemplate()
     * - .toCreateSegment()
     * - .toCreateSmsTemplate()
     * - .toCreateVoiceTemplate()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateCampaign()
     * - .toUpdateEmailTemplate()
     * - .toUpdateInAppTemplate()
     * - .toUpdateJourney()
     * - .toUpdateJourneyState()
     * - .toUpdatePushTemplate()
     * - .toUpdateSegment()
     * - .toUpdateSmsTemplate()
     * - .toUpdateVoiceTemplate()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a tag key and value pair
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toCreateApp()
     * - .toCreateCampaign()
     * - .toCreateEmailTemplate()
     * - .toCreateInAppTemplate()
     * - .toCreateJourney()
     * - .toCreatePushTemplate()
     * - .toCreateSegment()
     * - .toCreateSmsTemplate()
     * - .toCreateVoiceTemplate()
     *
     * Applies to resource types:
     * - app
     * - campaign
     * - journey
     * - segment
     * - template
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the list of all the tag key names present in the request the user makes to the pinpoint service
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toCreateApp()
     * - .toCreateCampaign()
     * - .toCreateEmailTemplate()
     * - .toCreateInAppTemplate()
     * - .toCreateJourney()
     * - .toCreatePushTemplate()
     * - .toCreateSegment()
     * - .toCreateSmsTemplate()
     * - .toCreateVoiceTemplate()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateCampaign()
     * - .toUpdateEmailTemplate()
     * - .toUpdateInAppTemplate()
     * - .toUpdateJourney()
     * - .toUpdateJourneyState()
     * - .toUpdatePushTemplate()
     * - .toUpdateSegment()
     * - .toUpdateSmsTemplate()
     * - .toUpdateVoiceTemplate()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [mobiletargeting](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonpinpoint.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'mobiletargeting';
        this.accessLevelList = {
            Write: [
                'CreateApp',
                'CreateCampaign',
                'CreateEmailTemplate',
                'CreateExportJob',
                'CreateImportJob',
                'CreateInAppTemplate',
                'CreateJourney',
                'CreatePushTemplate',
                'CreateRecommenderConfiguration',
                'CreateSegment',
                'CreateSmsTemplate',
                'CreateVoiceTemplate',
                'DeleteAdmChannel',
                'DeleteApnsChannel',
                'DeleteApnsSandboxChannel',
                'DeleteApnsVoipChannel',
                'DeleteApnsVoipSandboxChannel',
                'DeleteApp',
                'DeleteBaiduChannel',
                'DeleteCampaign',
                'DeleteEmailChannel',
                'DeleteEmailTemplate',
                'DeleteEndpoint',
                'DeleteEventStream',
                'DeleteGcmChannel',
                'DeleteInAppTemplate',
                'DeleteJourney',
                'DeletePushTemplate',
                'DeleteRecommenderConfiguration',
                'DeleteSegment',
                'DeleteSmsChannel',
                'DeleteSmsTemplate',
                'DeleteUserEndpoints',
                'DeleteVoiceChannel',
                'DeleteVoiceTemplate',
                'PutEventStream',
                'PutEvents',
                'RemoveAttributes',
                'SendMessages',
                'SendOTPMessage',
                'SendUsersMessages',
                'UpdateAdmChannel',
                'UpdateApnsChannel',
                'UpdateApnsSandboxChannel',
                'UpdateApnsVoipChannel',
                'UpdateApnsVoipSandboxChannel',
                'UpdateApplicationSettings',
                'UpdateBaiduChannel',
                'UpdateCampaign',
                'UpdateEmailChannel',
                'UpdateEmailTemplate',
                'UpdateEndpoint',
                'UpdateEndpointsBatch',
                'UpdateGcmChannel',
                'UpdateInAppTemplate',
                'UpdateJourney',
                'UpdateJourneyState',
                'UpdatePushTemplate',
                'UpdateRecommenderConfiguration',
                'UpdateSegment',
                'UpdateSmsChannel',
                'UpdateSmsTemplate',
                'UpdateTemplateActiveVersion',
                'UpdateVoiceChannel',
                'UpdateVoiceTemplate',
                'VerifyOTPMessage'
            ],
            Read: [
                'GetAdmChannel',
                'GetApnsChannel',
                'GetApnsSandboxChannel',
                'GetApnsVoipChannel',
                'GetApnsVoipSandboxChannel',
                'GetApp',
                'GetApplicationDateRangeKpi',
                'GetApps',
                'GetBaiduChannel',
                'GetCampaign',
                'GetCampaignDateRangeKpi',
                'GetCampaignVersion',
                'GetEmailChannel',
                'GetEmailTemplate',
                'GetEndpoint',
                'GetEventStream',
                'GetExportJob',
                'GetGcmChannel',
                'GetImportJob',
                'GetInAppMessages',
                'GetInAppTemplate',
                'GetJourney',
                'GetJourneyDateRangeKpi',
                'GetJourneyExecutionActivityMetrics',
                'GetJourneyExecutionMetrics',
                'GetJourneyRunExecutionActivityMetrics',
                'GetJourneyRunExecutionMetrics',
                'GetPushTemplate',
                'GetRecommenderConfiguration',
                'GetReports',
                'GetSegment',
                'GetSegmentVersion',
                'GetSmsChannel',
                'GetSmsTemplate',
                'GetUserEndpoints',
                'GetVoiceChannel',
                'GetVoiceTemplate',
                'ListTagsForResource',
                'PhoneNumberValidate'
            ],
            List: [
                'GetApplicationSettings',
                'GetCampaignActivities',
                'GetCampaignVersions',
                'GetCampaigns',
                'GetChannels',
                'GetExportJobs',
                'GetImportJobs',
                'GetJourneyRuns',
                'GetRecommenderConfigurations',
                'GetSegmentExportJobs',
                'GetSegmentImportJobs',
                'GetSegmentVersions',
                'GetSegments',
                'ListJourneys',
                'ListTemplateVersions',
                'ListTemplates'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Mobiletargeting = Mobiletargeting;
//# sourceMappingURL=data:application/json;base64,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