"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Qbusiness = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [qbusiness](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonqbusiness.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Qbusiness extends shared_1.PolicyStatement {
    /**
     * Grants permission to add one or more users for licenses
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_AddUserLicenses.html
     */
    toAddUserLicenses() {
        return this.to('AddUserLicenses');
    }
    /**
     * Grants permission to configure vended log delivery for Amazon Q Business application resource
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    toAllowVendedLogDeliveryForResource() {
        return this.to('AllowVendedLogDeliveryForResource');
    }
    /**
     * Grants permission to associate resource based policy statement to the application
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - qbusiness:PutResourcePolicy
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_AssociatePermission.html
     */
    toAssociatePermission() {
        return this.to('AssociatePermission');
    }
    /**
     * Grants permission to batch delete document
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_BatchDeleteDocument.html
     */
    toBatchDeleteDocument() {
        return this.to('BatchDeleteDocument');
    }
    /**
     * Grants permission to batch put document
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_BatchPutDocument.html
     */
    toBatchPutDocument() {
        return this.to('BatchPutDocument');
    }
    /**
     * Grants permission to cancel a subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_CancelSubscription.html
     */
    toCancelSubscription() {
        return this.to('CancelSubscription');
    }
    /**
     * Grants permission to chat using an application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_Chat.html
     */
    toChat() {
        return this.to('Chat');
    }
    /**
     * Grants permission to chat synchronously using an application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ChatSync.html
     */
    toChatSync() {
        return this.to('ChatSync');
    }
    /**
     * Grants permission to check if a user has access to a document
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_CheckDocumentAccess.html
     */
    toCheckDocumentAccess() {
        return this.to('CheckDocumentAccess');
    }
    /**
     * Grants permission to create a unique URL for anonymous Amazon Q Business web experience
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_CreateAnonymousWebExperienceUrl.html
     */
    toCreateAnonymousWebExperienceUrl() {
        return this.to('CreateAnonymousWebExperienceUrl');
    }
    /**
     * Grants permission to create an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_CreateApplication.html
     */
    toCreateApplication() {
        return this.to('CreateApplication');
    }
    /**
     * Grants permission to create DataAccessor to the application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_CreateDataAccessor.html
     */
    toCreateDataAccessor() {
        return this.to('CreateDataAccessor');
    }
    /**
     * Grants permission to create a data source for a given application and index
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_CreateDataSource.html
     */
    toCreateDataSource() {
        return this.to('CreateDataSource');
    }
    /**
     * Grants permission to create an index for a given application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_CreateIndex.html
     */
    toCreateIndex() {
        return this.to('CreateIndex');
    }
    /**
     * Grants permission to create a new integration for a Q Business application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_CreateIntegration.html
     */
    toCreateIntegration() {
        return this.to('CreateIntegration');
    }
    /**
     * Grants permission to create a license
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_CreateLicense.html
     */
    toCreateLicense() {
        return this.to('CreateLicense');
    }
    /**
     * Grants permission to create a plugin for a given application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_CreatePlugin.html
     */
    toCreatePlugin() {
        return this.to('CreatePlugin');
    }
    /**
     * Grants permission to create a retriever for a given application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_CreateRetriever.html
     */
    toCreateRetriever() {
        return this.to('CreateRetriever');
    }
    /**
     * Grants permission to create a subscription
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifIdentitystoreUserId()
     * - .ifIdentitystoreGroupId()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_CreateSubscription.html
     */
    toCreateSubscription() {
        return this.to('CreateSubscription');
    }
    /**
     * Grants permission to create a user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_CreateUser.html
     */
    toCreateUser() {
        return this.to('CreateUser');
    }
    /**
     * Grants permission to create a web experience for a given application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_CreateWebExperience.html
     */
    toCreateWebExperience() {
        return this.to('CreateWebExperience');
    }
    /**
     * Grants permission to delete an application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_DeleteApplication.html
     */
    toDeleteApplication() {
        return this.to('DeleteApplication');
    }
    /**
     * Grants permission to delete an attachment in the current chat context
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_DeleteAttachment.html
     */
    toDeleteAttachment() {
        return this.to('DeleteAttachment');
    }
    /**
     * Grants permission to delete chat controls configuration for an application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_DeleteChatControlsConfiguration.html
     */
    toDeleteChatControlsConfiguration() {
        return this.to('DeleteChatControlsConfiguration');
    }
    /**
     * Grants permission to delete a conversation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_DeleteConversation.html
     */
    toDeleteConversation() {
        return this.to('DeleteConversation');
    }
    /**
     * Grants permission to delete DataAccessor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_DeleteDataAccessor.html
     */
    toDeleteDataAccessor() {
        return this.to('DeleteDataAccessor');
    }
    /**
     * Grants permission to delete a DataSource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_DeleteDataSource.html
     */
    toDeleteDataSource() {
        return this.to('DeleteDataSource');
    }
    /**
     * Grants permission to delete a group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_DeleteGroup.html
     */
    toDeleteGroup() {
        return this.to('DeleteGroup');
    }
    /**
     * Grants permission to delete an index
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_DeleteIndex.html
     */
    toDeleteIndex() {
        return this.to('DeleteIndex');
    }
    /**
     * Grants permission to delete an integration for a Q Business application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_DeleteIntegration.html
     */
    toDeleteIntegration() {
        return this.to('DeleteIntegration');
    }
    /**
     * Grants permission to delete a plugin
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_DeletePlugin.html
     */
    toDeletePlugin() {
        return this.to('DeletePlugin');
    }
    /**
     * Grants permission to delete a retriever
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_DeleteRetriever.html
     */
    toDeleteRetriever() {
        return this.to('DeleteRetriever');
    }
    /**
     * Grants permission to delete a user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_DeleteUser.html
     */
    toDeleteUser() {
        return this.to('DeleteUser');
    }
    /**
     * Grants permission to delete a web-experience
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_DeleteWebExperience.html
     */
    toDeleteWebExperience() {
        return this.to('DeleteWebExperience');
    }
    /**
     * Grants permission to disable the ACL crawl while creating the Amazon Q Business data source resource
     *
     * Access Level: Write
     */
    toDisableAclOnDataSource() {
        return this.to('DisableAclOnDataSource');
    }
    /**
     * Grants permission to disassociate resource based policy statement to the application
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - qbusiness:PutResourcePolicy
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_DisassociatePermission.html
     */
    toDisassociatePermission() {
        return this.to('DisassociatePermission');
    }
    /**
     * Grants permission to get an application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_GetApplication.html
     */
    toGetApplication() {
        return this.to('GetApplication');
    }
    /**
     * Grants permission to get chat controls configuration for an application
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_GetChatControlsConfiguration.html
     */
    toGetChatControlsConfiguration() {
        return this.to('GetChatControlsConfiguration');
    }
    /**
     * Grants permission to get DataAccessor
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_GetDataAccessor.html
     */
    toGetDataAccessor() {
        return this.to('GetDataAccessor');
    }
    /**
     * Grants permission to get a data source
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_GetDataSource.html
     */
    toGetDataSource() {
        return this.to('GetDataSource');
    }
    /**
     * Grants permission to get a group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_GetGroup.html
     */
    toGetGroup() {
        return this.to('GetGroup');
    }
    /**
     * Grants permission to get an index
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_GetIndex.html
     */
    toGetIndex() {
        return this.to('GetIndex');
    }
    /**
     * Grants permission to get an integration for a Q Business application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_GetIntegration.html
     */
    toGetIntegration() {
        return this.to('GetIntegration');
    }
    /**
     * Grants permission to get a license
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_GetLicense.html
     */
    toGetLicense() {
        return this.to('GetLicense');
    }
    /**
     * Grants permission to get the media associated to a system message
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_GetMedia.html
     */
    toGetMedia() {
        return this.to('GetMedia');
    }
    /**
     * Grants permission to get a plugin
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_GetPlugin.html
     */
    toGetPlugin() {
        return this.to('GetPlugin');
    }
    /**
     * Grants permission to get resource based policy of the application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_GetPolicy.html
     */
    toGetPolicy() {
        return this.to('GetPolicy');
    }
    /**
     * Grants permission to get a retriever
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_GetRetriever.html
     */
    toGetRetriever() {
        return this.to('GetRetriever');
    }
    /**
     * Grants permission to get a user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_GetUser.html
     */
    toGetUser() {
        return this.to('GetUser');
    }
    /**
     * Grants permission to get a web-experience
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_GetWebExperience.html
     */
    toGetWebExperience() {
        return this.to('GetWebExperience');
    }
    /**
     * Grants permission to list the applications
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListApplications.html
     */
    toListApplications() {
        return this.to('ListApplications');
    }
    /**
     * Grants permission to list attachments in the current chat context
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListAttachments.html
     */
    toListAttachments() {
        return this.to('ListAttachments');
    }
    /**
     * Grants permission to list all conversations for an application
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListConversations.html
     */
    toListConversations() {
        return this.to('ListConversations');
    }
    /**
     * Grants permission to list DataAccessors for the application
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListDataAccessors.html
     */
    toListDataAccessors() {
        return this.to('ListDataAccessors');
    }
    /**
     * Grants permission to get Data Source sync job history
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListDataSourceSyncJobs.html
     */
    toListDataSourceSyncJobs() {
        return this.to('ListDataSourceSyncJobs');
    }
    /**
     * Grants permission to list the data sources of an application and an index
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListDataSources.html
     */
    toListDataSources() {
        return this.to('ListDataSources');
    }
    /**
     * Grants permission to list all documents
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListDocuments.html
     */
    toListDocuments() {
        return this.to('ListDocuments');
    }
    /**
     * Grants permission to list groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListGroups.html
     */
    toListGroups() {
        return this.to('ListGroups');
    }
    /**
     * Grants permission to list the indices of an application
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListIndices.html
     */
    toListIndices() {
        return this.to('ListIndices');
    }
    /**
     * Grants permission to list all integrations for a Q Business application
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListIntegrations.html
     */
    toListIntegrations() {
        return this.to('ListIntegrations');
    }
    /**
     * Grants permission to list all messages
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListMessages.html
     */
    toListMessages() {
        return this.to('ListMessages');
    }
    /**
     * Grants permission to list the plugins actions of a plugin within application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListPluginActions.html
     */
    toListPluginActions() {
        return this.to('ListPluginActions');
    }
    /**
     * Grants permission to list all the actions for a plugin type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListPluginTypeActions.html
     */
    toListPluginTypeActions() {
        return this.to('ListPluginTypeActions');
    }
    /**
     * Grants permission to list all the plugin type metadata
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListPluginTypeMetadata.html
     */
    toListPluginTypeMetadata() {
        return this.to('ListPluginTypeMetadata');
    }
    /**
     * Grants permission to list the plugins of an application
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListPlugins.html
     */
    toListPlugins() {
        return this.to('ListPlugins');
    }
    /**
     * Grants permission to list the retrievers of an application
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListRetrievers.html
     */
    toListRetrievers() {
        return this.to('ListRetrievers');
    }
    /**
     * Grants permission to list subscriptions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListSubscriptions.html
     */
    toListSubscriptions() {
        return this.to('ListSubscriptions');
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list licenses
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListUserLicenses.html
     */
    toListUserLicenses() {
        return this.to('ListUserLicenses');
    }
    /**
     * Grants permission to list the web experiences of an application
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListWebExperiences.html
     */
    toListWebExperiences() {
        return this.to('ListWebExperiences');
    }
    /**
     * Grants permission to put feedback about a conversation message
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_PutFeedback.html
     */
    toPutFeedback() {
        return this.to('PutFeedback');
    }
    /**
     * Grants permission to put a group of users
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_PutGroup.html
     */
    toPutGroup() {
        return this.to('PutGroup');
    }
    /**
     * Grants permission to put resource based policy statement to the application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_AssociatePermission.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to remove licenses for one or more users
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_RemoveUserLicenses.html
     */
    toRemoveUserLicenses() {
        return this.to('RemoveUserLicenses');
    }
    /**
     * Grants permission to search relevant content from the Amazon Q Business Application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_SearchRelevantContent.html
     */
    toSearchRelevantContent() {
        return this.to('SearchRelevantContent');
    }
    /**
     * Grants permission to start Data Source sync job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_StartDataSourceSyncJob.html
     */
    toStartDataSourceSyncJob() {
        return this.to('StartDataSourceSyncJob');
    }
    /**
     * Grants permission to start deployment for an integration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_StartDeployment.html
     */
    toStartDeployment() {
        return this.to('StartDeployment');
    }
    /**
     * Grants permission to stop Data Source sync job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_StopDataSourceSyncJob.html
     */
    toStopDataSourceSyncJob() {
        return this.to('StopDataSourceSyncJob');
    }
    /**
     * Grants permission to tag a resource with given key value pairs
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove the tag with the given key from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an Application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_UpdateApplication.html
     */
    toUpdateApplication() {
        return this.to('UpdateApplication');
    }
    /**
     * Grants permission to update chat controls configuration for an application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_UpdateChatControlsConfiguration.html
     */
    toUpdateChatControlsConfiguration() {
        return this.to('UpdateChatControlsConfiguration');
    }
    /**
     * Grants permission to update DataAccessor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_UpdateDataAccessor.html
     */
    toUpdateDataAccessor() {
        return this.to('UpdateDataAccessor');
    }
    /**
     * Grants permission to update a DataSource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_UpdateDataSource.html
     */
    toUpdateDataSource() {
        return this.to('UpdateDataSource');
    }
    /**
     * Grants permission to update an index
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_UpdateIndex.html
     */
    toUpdateIndex() {
        return this.to('UpdateIndex');
    }
    /**
     * Grants permission to update an integration for a Q Business application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_UpdateIntegration.html
     */
    toUpdateIntegration() {
        return this.to('UpdateIntegration');
    }
    /**
     * Grants permission to update a plugin
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_UpdatePlugin.html
     */
    toUpdatePlugin() {
        return this.to('UpdatePlugin');
    }
    /**
     * Grants permission to update a Retriever
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_UpdateRetriever.html
     */
    toUpdateRetriever() {
        return this.to('UpdateRetriever');
    }
    /**
     * Grants permission to update a subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_UpdateSubscription.html
     */
    toUpdateSubscription() {
        return this.to('UpdateSubscription');
    }
    /**
     * Grants permission to update a user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_UpdateUser.html
     */
    toUpdateUser() {
        return this.to('UpdateUser');
    }
    /**
     * Grants permission to update a WebExperience
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_UpdateWebExperience.html
     */
    toUpdateWebExperience() {
        return this.to('UpdateWebExperience');
    }
    /**
     * Adds a resource of type application to the statement
     *
     * https://docs.aws.amazon.com/amazonq/latest/business-use-dg/create-application.html
     *
     * @param applicationId - Identifier for the applicationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onApplication(applicationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:qbusiness:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:application/${applicationId}`);
    }
    /**
     * Adds a resource of type integration to the statement
     *
     * https://docs.aws.amazon.com/amazonq/latest/business-use-dg/create-integration.html
     *
     * @param applicationId - Identifier for the applicationId.
     * @param integrationId - Identifier for the integrationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIntegration(applicationId, integrationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:qbusiness:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:application/${applicationId}/integration/${integrationId}`);
    }
    /**
     * Adds a resource of type retriever to the statement
     *
     * https://docs.aws.amazon.com/amazonq/latest/business-use-dg/select-retriever.html
     *
     * @param applicationId - Identifier for the applicationId.
     * @param retrieverId - Identifier for the retrieverId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRetriever(applicationId, retrieverId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:qbusiness:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:application/${applicationId}/retriever/${retrieverId}`);
    }
    /**
     * Adds a resource of type index to the statement
     *
     * https://docs.aws.amazon.com/amazonq/latest/business-use-dg/select-retriever.html
     *
     * @param applicationId - Identifier for the applicationId.
     * @param indexId - Identifier for the indexId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIndex(applicationId, indexId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:qbusiness:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:application/${applicationId}/index/${indexId}`);
    }
    /**
     * Adds a resource of type data-source to the statement
     *
     * https://docs.aws.amazon.com/amazonq/latest/business-use-dg/connect-data.html
     *
     * @param applicationId - Identifier for the applicationId.
     * @param indexId - Identifier for the indexId.
     * @param dataSourceId - Identifier for the dataSourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDataSource(applicationId, indexId, dataSourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:qbusiness:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:application/${applicationId}/index/${indexId}/data-source/${dataSourceId}`);
    }
    /**
     * Adds a resource of type plugin to the statement
     *
     * https://docs.aws.amazon.com/amazonq/latest/business-use-dg/plugins.html
     *
     * @param applicationId - Identifier for the applicationId.
     * @param pluginId - Identifier for the pluginId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPlugin(applicationId, pluginId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:qbusiness:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:application/${applicationId}/plugin/${pluginId}`);
    }
    /**
     * Adds a resource of type web-experience to the statement
     *
     * https://docs.aws.amazon.com/amazonq/latest/business-use-dg/using-web-experience.html
     *
     * @param applicationId - Identifier for the applicationId.
     * @param webExperienceId - Identifier for the webExperienceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWebExperience(applicationId, webExperienceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:qbusiness:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:application/${applicationId}/web-experience/${webExperienceId}`);
    }
    /**
     * Adds a resource of type user-license to the statement
     *
     * https://docs.aws.amazon.com/amazonq/latest/business-use-dg/provisioning.html
     *
     * @param applicationId - Identifier for the applicationId.
     * @param userLicenseId - Identifier for the userLicenseId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onUserLicense(applicationId, userLicenseId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:qbusiness:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:application/${applicationId}/user-license/${userLicenseId}`);
    }
    /**
     * Adds a resource of type subscription to the statement
     *
     * https://docs.aws.amazon.com/amazonq/latest/business-use-dg/subscriptions.html
     *
     * @param applicationId - Identifier for the applicationId.
     * @param subscriptionId - Identifier for the subscriptionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onSubscription(applicationId, subscriptionId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:qbusiness:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:application/${applicationId}/subscription/${subscriptionId}`);
    }
    /**
     * Adds a resource of type data-accessor to the statement
     *
     * https://docs.aws.amazon.com/amazonq/latest/business-use-dg/data-accessors.html
     *
     * @param applicationId - Identifier for the applicationId.
     * @param dataAccessorId - Identifier for the dataAccessorId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDataAccessor(applicationId, dataAccessorId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:qbusiness:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:application/${applicationId}/data-accessor/${dataAccessorId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateApplication()
     * - .toCreateDataSource()
     * - .toCreateIndex()
     * - .toCreateIntegration()
     * - .toCreatePlugin()
     * - .toCreateRetriever()
     * - .toCreateWebExperience()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toAllowVendedLogDeliveryForResource()
     *
     * Applies to resource types:
     * - application
     * - integration
     * - retriever
     * - index
     * - data-source
     * - plugin
     * - web-experience
     * - data-accessor
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateApplication()
     * - .toCreateDataSource()
     * - .toCreateIndex()
     * - .toCreateIntegration()
     * - .toCreatePlugin()
     * - .toCreateRetriever()
     * - .toCreateWebExperience()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by IAM Identity Center Group ID
     *
     * https://docs.aws.amazon.com/amazonq/latest/qbusiness-ug/security_iam_service-with-iam.html
     *
     * Applies to actions:
     * - .toCreateSubscription()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifIdentitystoreGroupId(value, operator) {
        return this.if(`identitystore:GroupId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by IAM Identity Center User ID
     *
     * https://docs.aws.amazon.com/amazonq/latest/qbusiness-ug/security_iam_service-with-iam.html
     *
     * Applies to actions:
     * - .toCreateSubscription()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifIdentitystoreUserId(value, operator) {
        return this.if(`identitystore:UserId`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [qbusiness](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonqbusiness.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'qbusiness';
        this.accessLevelList = {
            Write: [
                'AddUserLicenses',
                'AssociatePermission',
                'BatchDeleteDocument',
                'BatchPutDocument',
                'CancelSubscription',
                'CreateAnonymousWebExperienceUrl',
                'CreateApplication',
                'CreateDataAccessor',
                'CreateDataSource',
                'CreateIndex',
                'CreateIntegration',
                'CreateLicense',
                'CreatePlugin',
                'CreateRetriever',
                'CreateSubscription',
                'CreateUser',
                'CreateWebExperience',
                'DeleteApplication',
                'DeleteAttachment',
                'DeleteChatControlsConfiguration',
                'DeleteConversation',
                'DeleteDataAccessor',
                'DeleteDataSource',
                'DeleteGroup',
                'DeleteIndex',
                'DeleteIntegration',
                'DeletePlugin',
                'DeleteRetriever',
                'DeleteUser',
                'DeleteWebExperience',
                'DisableAclOnDataSource',
                'DisassociatePermission',
                'PutFeedback',
                'PutGroup',
                'PutResourcePolicy',
                'RemoveUserLicenses',
                'StartDataSourceSyncJob',
                'StartDeployment',
                'StopDataSourceSyncJob',
                'UpdateApplication',
                'UpdateChatControlsConfiguration',
                'UpdateDataAccessor',
                'UpdateDataSource',
                'UpdateIndex',
                'UpdateIntegration',
                'UpdatePlugin',
                'UpdateRetriever',
                'UpdateSubscription',
                'UpdateUser',
                'UpdateWebExperience'
            ],
            'Permissions management': [
                'AllowVendedLogDeliveryForResource'
            ],
            Read: [
                'Chat',
                'ChatSync',
                'CheckDocumentAccess',
                'GetApplication',
                'GetDataAccessor',
                'GetDataSource',
                'GetGroup',
                'GetIndex',
                'GetIntegration',
                'GetLicense',
                'GetMedia',
                'GetPlugin',
                'GetPolicy',
                'GetRetriever',
                'GetUser',
                'GetWebExperience',
                'ListPluginActions',
                'ListPluginTypeActions',
                'ListPluginTypeMetadata',
                'ListTagsForResource',
                'SearchRelevantContent'
            ],
            List: [
                'GetChatControlsConfiguration',
                'ListApplications',
                'ListAttachments',
                'ListConversations',
                'ListDataAccessors',
                'ListDataSourceSyncJobs',
                'ListDataSources',
                'ListDocuments',
                'ListGroups',
                'ListIndices',
                'ListIntegrations',
                'ListMessages',
                'ListPlugins',
                'ListRetrievers',
                'ListSubscriptions',
                'ListUserLicenses',
                'ListWebExperiences'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Qbusiness = Qbusiness;
//# sourceMappingURL=data:application/json;base64,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