"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Qldb = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [qldb](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonqldb.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Qldb extends shared_1.PolicyStatement {
    /**
     * Grants permission to cancel a journal kinesis stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_CancelJournalKinesisStream.html
     */
    toCancelJournalKinesisStream() {
        return this.to('CancelJournalKinesisStream');
    }
    /**
     * Grants permission to create a ledger
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_CreateLedger.html
     */
    toCreateLedger() {
        return this.to('CreateLedger');
    }
    /**
     * Grants permission to delete a ledger
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_DeleteLedger.html
     */
    toDeleteLedger() {
        return this.to('DeleteLedger');
    }
    /**
     * Grants permission to describe information about a journal kinesis stream
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_DescribeJournalKinesisStream.html
     */
    toDescribeJournalKinesisStream() {
        return this.to('DescribeJournalKinesisStream');
    }
    /**
     * Grants permission to describe information about a journal export job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_DescribeJournalS3Export.html
     */
    toDescribeJournalS3Export() {
        return this.to('DescribeJournalS3Export');
    }
    /**
     * Grants permission to describe a ledger
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_DescribeLedger.html
     */
    toDescribeLedger() {
        return this.to('DescribeLedger');
    }
    /**
     * Grants permission to send commands to a ledger via the console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/console_QLDB.html
     */
    toExecuteStatement() {
        return this.to('ExecuteStatement');
    }
    /**
     * Grants permission to export journal contents to an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_ExportJournalToS3.html
     */
    toExportJournalToS3() {
        return this.to('ExportJournalToS3');
    }
    /**
     * Grants permission to retrieve a block from a ledger for a given BlockAddress
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_GetBlock.html
     */
    toGetBlock() {
        return this.to('GetBlock');
    }
    /**
     * Grants permission to retrieve a digest from a ledger for a given BlockAddress
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_GetDigest.html
     */
    toGetDigest() {
        return this.to('GetDigest');
    }
    /**
     * Grants permission to retrieve a revision for a given document ID and a given BlockAddress
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_GetRevision.html
     */
    toGetRevision() {
        return this.to('GetRevision');
    }
    /**
     * Grants permission to insert sample application data via the console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/console_QLDB.html
     */
    toInsertSampleData() {
        return this.to('InsertSampleData');
    }
    /**
     * Grants permission to list journal kinesis streams for a specified ledger
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_ListJournalKinesisStreamsForLedger.html
     */
    toListJournalKinesisStreamsForLedger() {
        return this.to('ListJournalKinesisStreamsForLedger');
    }
    /**
     * Grants permission to list journal export jobs for all ledgers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_ListJournalS3Exports.html
     */
    toListJournalS3Exports() {
        return this.to('ListJournalS3Exports');
    }
    /**
     * Grants permission to list journal export jobs for a specified ledger
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_ListJournalS3ExportsForLedger.html
     */
    toListJournalS3ExportsForLedger() {
        return this.to('ListJournalS3ExportsForLedger');
    }
    /**
     * Grants permission to list existing ledgers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_ListLedgers.html
     */
    toListLedgers() {
        return this.to('ListLedgers');
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to create an index on a table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/ql-reference.create-index.html
     */
    toPartiQLCreateIndex() {
        return this.to('PartiQLCreateIndex');
    }
    /**
     * Grants permission to create a table
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/ql-reference.create-table.html
     */
    toPartiQLCreateTable() {
        return this.to('PartiQLCreateTable');
    }
    /**
     * Grants permission to delete documents from a table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/ql-reference.delete.html
     */
    toPartiQLDelete() {
        return this.to('PartiQLDelete');
    }
    /**
     * Grants permission to drop an index from a table
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPurge()
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/ql-reference.drop-index.html
     */
    toPartiQLDropIndex() {
        return this.to('PartiQLDropIndex');
    }
    /**
     * Grants permission to drop a table
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPurge()
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/ql-reference.drop-table.html
     */
    toPartiQLDropTable() {
        return this.to('PartiQLDropTable');
    }
    /**
     * Grants permission to use the history function on a table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/working.history.html
     */
    toPartiQLHistoryFunction() {
        return this.to('PartiQLHistoryFunction');
    }
    /**
     * Grants permission to insert documents into a table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/ql-reference.insert.html
     */
    toPartiQLInsert() {
        return this.to('PartiQLInsert');
    }
    /**
     * Grants permission to redact historic revisions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/ql-stored-procedures.redact_revision.html
     */
    toPartiQLRedact() {
        return this.to('PartiQLRedact');
    }
    /**
     * Grants permission to select documents from a table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/ql-reference.select.html
     */
    toPartiQLSelect() {
        return this.to('PartiQLSelect');
    }
    /**
     * Grants permission to undrop a table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/ql-reference.undrop-table.html
     */
    toPartiQLUndropTable() {
        return this.to('PartiQLUndropTable');
    }
    /**
     * Grants permission to update existing documents in a table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/ql-reference.update.html
     */
    toPartiQLUpdate() {
        return this.to('PartiQLUpdate');
    }
    /**
     * Grants permission to send commands to a ledger
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_QLDB-Session_SendCommand.html
     */
    toSendCommand() {
        return this.to('SendCommand');
    }
    /**
     * Grants permission to view a ledger's catalog via the console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/console_QLDB.html
     */
    toShowCatalog() {
        return this.to('ShowCatalog');
    }
    /**
     * Grants permission to stream journal contents to a Kinesis Data Stream
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_StreamJournalToKinesis.html
     */
    toStreamJournalToKinesis() {
        return this.to('StreamJournalToKinesis');
    }
    /**
     * Grants permission to add one or more tags to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove one or more tags from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update properties on a ledger
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_UpdateLedger.html
     */
    toUpdateLedger() {
        return this.to('UpdateLedger');
    }
    /**
     * Grants permission to update the permissions mode on a ledger
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_UpdateLedgerPermissionsMode.html
     */
    toUpdateLedgerPermissionsMode() {
        return this.to('UpdateLedgerPermissionsMode');
    }
    /**
     * Adds a resource of type ledger to the statement
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/ledger-structure.html
     *
     * @param ledgerName - Identifier for the ledgerName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onLedger(ledgerName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:qldb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:ledger/${ledgerName}`);
    }
    /**
     * Adds a resource of type stream to the statement
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/streams.html
     *
     * @param ledgerName - Identifier for the ledgerName.
     * @param streamId - Identifier for the streamId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStream(ledgerName, streamId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:qldb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:stream/${ledgerName}/${streamId}`);
    }
    /**
     * Adds a resource of type table to the statement
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/working.manage-tables.html
     *
     * @param ledgerName - Identifier for the ledgerName.
     * @param tableId - Identifier for the tableId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTable(ledgerName, tableId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:qldb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:ledger/${ledgerName}/table/${tableId}`);
    }
    /**
     * Adds a resource of type catalog to the statement
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/working.catalog.html
     *
     * @param ledgerName - Identifier for the ledgerName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCatalog(ledgerName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:qldb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:ledger/${ledgerName}/information_schema/user_tables`);
    }
    /**
     * Filters access by a tag key and value pair that is allowed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateLedger()
     * - .toPartiQLCreateTable()
     * - .toStreamJournalToKinesis()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a tag key and value pair of a resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - ledger
     * - stream
     * - table
     * - catalog
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a list of tag keys that are allowed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateLedger()
     * - .toPartiQLCreateTable()
     * - .toStreamJournalToKinesis()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the value of purge that is specified in a PartiQL DROP statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-purge
     *
     * Applies to actions:
     * - .toPartiQLDropIndex()
     * - .toPartiQLDropTable()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPurge(value, operator) {
        return this.if(`Purge`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [qldb](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonqldb.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'qldb';
        this.accessLevelList = {
            Write: [
                'CancelJournalKinesisStream',
                'CreateLedger',
                'DeleteLedger',
                'ExecuteStatement',
                'ExportJournalToS3',
                'InsertSampleData',
                'PartiQLCreateIndex',
                'PartiQLCreateTable',
                'PartiQLDelete',
                'PartiQLDropIndex',
                'PartiQLDropTable',
                'PartiQLInsert',
                'PartiQLRedact',
                'PartiQLUndropTable',
                'PartiQLUpdate',
                'SendCommand',
                'ShowCatalog',
                'StreamJournalToKinesis',
                'UpdateLedger',
                'UpdateLedgerPermissionsMode'
            ],
            Read: [
                'DescribeJournalKinesisStream',
                'DescribeJournalS3Export',
                'DescribeLedger',
                'GetBlock',
                'GetDigest',
                'GetRevision',
                'ListTagsForResource',
                'PartiQLHistoryFunction',
                'PartiQLSelect'
            ],
            List: [
                'ListJournalKinesisStreamsForLedger',
                'ListJournalS3Exports',
                'ListJournalS3ExportsForLedger',
                'ListLedgers'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Qldb = Qldb;
//# sourceMappingURL=data:application/json;base64,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