"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Rekognition = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [rekognition](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonrekognition.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Rekognition extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate multiple individual faces with a single user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_AssociateFaces.html
     */
    toAssociateFaces() {
        return this.to('AssociateFaces');
    }
    /**
     * Grants permission to compare faces in the source input image with each face detected in the target input image
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_CompareFaces.html
     */
    toCompareFaces() {
        return this.to('CompareFaces');
    }
    /**
     * Grants permission to copy an existing model version to a new model version
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_CopyProjectVersion.html
     */
    toCopyProjectVersion() {
        return this.to('CopyProjectVersion');
    }
    /**
     * Grants permission to create a collection in an AWS Region
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_CreateCollection.html
     */
    toCreateCollection() {
        return this.to('CreateCollection');
    }
    /**
     * Grants permission to create a new Amazon Rekognition Custom Labels dataset
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_CreateDataset.html
     */
    toCreateDataset() {
        return this.to('CreateDataset');
    }
    /**
     * Grants permission to create a face liveness session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_CreateFaceLivenessSession.html
     */
    toCreateFaceLivenessSession() {
        return this.to('CreateFaceLivenessSession');
    }
    /**
     * Grants permission to create an Amazon Rekognition Custom Labels project
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_CreateProject.html
     */
    toCreateProject() {
        return this.to('CreateProject');
    }
    /**
     * Grants permission to begin training a new version of a model
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_CreateProjectVersion.html
     */
    toCreateProjectVersion() {
        return this.to('CreateProjectVersion');
    }
    /**
     * Grants permission to create an Amazon Rekognition stream processor
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_CreateStreamProcessor.html
     */
    toCreateStreamProcessor() {
        return this.to('CreateStreamProcessor');
    }
    /**
     * Grants permission to create a new user in a collection using a unique user ID you provide
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_CreateUser.html
     */
    toCreateUser() {
        return this.to('CreateUser');
    }
    /**
     * Grants permission to delete the specified collection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_DeleteCollection.html
     */
    toDeleteCollection() {
        return this.to('DeleteCollection');
    }
    /**
     * Grants permission to delete an existing Amazon Rekognition Custom Labels dataset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_DeleteDataset.html
     */
    toDeleteDataset() {
        return this.to('DeleteDataset');
    }
    /**
     * Grants permission to delete faces from a collection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_DeleteFaces.html
     */
    toDeleteFaces() {
        return this.to('DeleteFaces');
    }
    /**
     * Grants permission to delete a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_DeleteProject.html
     */
    toDeleteProject() {
        return this.to('DeleteProject');
    }
    /**
     * Grants permission to delete a resource policy attached to a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_DeleteProjectPolicy.html
     */
    toDeleteProjectPolicy() {
        return this.to('DeleteProjectPolicy');
    }
    /**
     * Grants permission to delete a model
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_DeleteProjectVersion.html
     */
    toDeleteProjectVersion() {
        return this.to('DeleteProjectVersion');
    }
    /**
     * Grants permission to delete the specified stream processor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_DeleteStreamProcessor.html
     */
    toDeleteStreamProcessor() {
        return this.to('DeleteStreamProcessor');
    }
    /**
     * Grants permission to delete a user from a collection based on the provided user ID
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_DeleteUser.html
     */
    toDeleteUser() {
        return this.to('DeleteUser');
    }
    /**
     * Grants permission to read details about a collection
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_DescribeCollection.html
     */
    toDescribeCollection() {
        return this.to('DescribeCollection');
    }
    /**
     * Grants permission to describe an Amazon Rekognition Custom Labels dataset
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_DescribeDataset.html
     */
    toDescribeDataset() {
        return this.to('DescribeDataset');
    }
    /**
     * Grants permission to list the versions of a model in an Amazon Rekognition Custom Labels project
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_DescribeProjectVersions.html
     */
    toDescribeProjectVersions() {
        return this.to('DescribeProjectVersions');
    }
    /**
     * Grants permission to list Amazon Rekognition Custom Labels projects
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_DescribeProjects.html
     */
    toDescribeProjects() {
        return this.to('DescribeProjects');
    }
    /**
     * Grants permission to get information about the specified stream processor
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_DescribeStreamProcessor.html
     */
    toDescribeStreamProcessor() {
        return this.to('DescribeStreamProcessor');
    }
    /**
     * Grants permission to detect custom labels in a supplied image
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_DetectCustomLabels.html
     */
    toDetectCustomLabels() {
        return this.to('DetectCustomLabels');
    }
    /**
     * Grants permission to detect human faces within an image provided as input
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_DetectFaces.html
     */
    toDetectFaces() {
        return this.to('DetectFaces');
    }
    /**
     * Grants permission to detect instances of real-world labels within an image provided as input
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_DetectLabels.html
     */
    toDetectLabels() {
        return this.to('DetectLabels');
    }
    /**
     * Grants permission to detect moderation labels within the input image
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_DetectModerationLabels.html
     */
    toDetectModerationLabels() {
        return this.to('DetectModerationLabels');
    }
    /**
     * Grants permission to detect Personal Protective Equipment in the input image
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_DetectProtectiveEquipment.html
     */
    toDetectProtectiveEquipment() {
        return this.to('DetectProtectiveEquipment');
    }
    /**
     * Grants permission to detect text in the input image and convert it into machine-readable text
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_DetectText.html
     */
    toDetectText() {
        return this.to('DetectText');
    }
    /**
     * Grants permission to remove the association between a user ID and a face ID
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_DisassociateFaces.html
     */
    toDisassociateFaces() {
        return this.to('DisassociateFaces');
    }
    /**
     * Grants permission to distribute the entries in a training dataset across the training dataset and the test dataset for a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_DistributeDatasetEntries.html
     */
    toDistributeDatasetEntries() {
        return this.to('DistributeDatasetEntries');
    }
    /**
     * Grants permission to read the name, and additional information, of a celebrity
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_GetCelebrityInfo.html
     */
    toGetCelebrityInfo() {
        return this.to('GetCelebrityInfo');
    }
    /**
     * Grants permission to read the celebrity recognition results found in a stored video by an asynchronous celebrity recognition job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_GetCelebrityRecognition.html
     */
    toGetCelebrityRecognition() {
        return this.to('GetCelebrityRecognition');
    }
    /**
     * Grants permission to read the content moderation analysis results found in a stored video by an asynchronous content moderation job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_GetContentModeration.html
     */
    toGetContentModeration() {
        return this.to('GetContentModeration');
    }
    /**
     * Grants permission to read the faces detection results found in a stored video by an asynchronous face detection job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_GetFaceDetection.html
     */
    toGetFaceDetection() {
        return this.to('GetFaceDetection');
    }
    /**
     * Grants permission to get results of a face liveness session
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_GetFaceLivenessSessionResults.html
     */
    toGetFaceLivenessSessionResults() {
        return this.to('GetFaceLivenessSessionResults');
    }
    /**
     * Grants permission to read the matching collection faces found in a stored video by an asynchronous face search job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_GetFaceSearch.html
     */
    toGetFaceSearch() {
        return this.to('GetFaceSearch');
    }
    /**
     * Grants permission to read the label detected resuls found in a stored video by an asynchronous label detection job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_GetLabelDetection.html
     */
    toGetLabelDetection() {
        return this.to('GetLabelDetection');
    }
    /**
     * Grants permission to read the reference to job results in S3 and additional information about a media analysis job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_GetMediaAnalysisJob.html
     */
    toGetMediaAnalysisJob() {
        return this.to('GetMediaAnalysisJob');
    }
    /**
     * Grants permission to read the list of persons detected in a stored video by an asynchronous person tracking job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_GetPersonTracking.html
     */
    toGetPersonTracking() {
        return this.to('GetPersonTracking');
    }
    /**
     * Grants permission to get the vdeo segments found in a stored video by an asynchronous segment detection job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_GetSegmentDetection.html
     */
    toGetSegmentDetection() {
        return this.to('GetSegmentDetection');
    }
    /**
     * Grants permission to get the text found in a stored video by an asynchronous text detection job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_GetTextDetection.html
     */
    toGetTextDetection() {
        return this.to('GetTextDetection');
    }
    /**
     * Grants permission to update an existing collection with faces detected in the input image
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_IndexFaces.html
     */
    toIndexFaces() {
        return this.to('IndexFaces');
    }
    /**
     * Grants permission to read the collection Id's in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_ListCollections.html
     */
    toListCollections() {
        return this.to('ListCollections');
    }
    /**
     * Grants permission to list the dataset entries in an existing Amazon Rekognition Custom Labels dataset
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_ListDatasetEntries.html
     */
    toListDatasetEntries() {
        return this.to('ListDatasetEntries');
    }
    /**
     * Grants permission to list the labels in a dataset
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_ListDatasetLabels.html
     */
    toListDatasetLabels() {
        return this.to('ListDatasetLabels');
    }
    /**
     * Grants permission to read metadata for faces in the specificed collection
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_ListFaces.html
     */
    toListFaces() {
        return this.to('ListFaces');
    }
    /**
     * Grants permission to read the list of media analysis jobs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_ListMediaAnalysisJobs.html
     */
    toListMediaAnalysisJobs() {
        return this.to('ListMediaAnalysisJobs');
    }
    /**
     * Grants permission to list the resource policies attached to a project
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_ListProjectPolicies.html
     */
    toListProjectPolicies() {
        return this.to('ListProjectPolicies');
    }
    /**
     * Grants permission to get a list of your stream processors
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_ListStreamProcessors.html
     */
    toListStreamProcessors() {
        return this.to('ListStreamProcessors');
    }
    /**
     * Grants permission to return a list of tags associated with a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list UserIds and the UserStatus
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_ListUsers.html
     */
    toListUsers() {
        return this.to('ListUsers');
    }
    /**
     * Grants permission to attach a resource policy to a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_PutProjectPolicy.html
     */
    toPutProjectPolicy() {
        return this.to('PutProjectPolicy');
    }
    /**
     * Grants permission to detect celebrities in the input image
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_RecognizeCelebrities.html
     */
    toRecognizeCelebrities() {
        return this.to('RecognizeCelebrities');
    }
    /**
     * Grants permission to search the specificed collection for the supplied face ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_SearchFaces.html
     */
    toSearchFaces() {
        return this.to('SearchFaces');
    }
    /**
     * Grants permission to search the specificed collection for the largest face in the input image
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_SearchFacesByImage.html
     */
    toSearchFacesByImage() {
        return this.to('SearchFacesByImage');
    }
    /**
     * Grants permission to search the specificed collection for user match result with given either face ID or user ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_SearchUsers.html
     */
    toSearchUsers() {
        return this.to('SearchUsers');
    }
    /**
     * Grants permission to search the specificed collection for user match result by using the largest face in the input image
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_SearchUsersByImage.html
     */
    toSearchUsersByImage() {
        return this.to('SearchUsersByImage');
    }
    /**
     * Grants permission to start the asynchronous recognition of celebrities in a stored video
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_StartCelebrityRecognition.html
     */
    toStartCelebrityRecognition() {
        return this.to('StartCelebrityRecognition');
    }
    /**
     * Grants permission to start asynchronous detection of explicit or suggestive adult content in a stored video
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_StartContentModeration.html
     */
    toStartContentModeration() {
        return this.to('StartContentModeration');
    }
    /**
     * Grants permission to start asynchronous detection of faces in a stored video
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_StartFaceDetection.html
     */
    toStartFaceDetection() {
        return this.to('StartFaceDetection');
    }
    /**
     * Grants permission to start streaming video for a face liveness session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_rekognitionstreaming_StartFaceLivenessSession.html
     */
    toStartFaceLivenessSession() {
        return this.to('StartFaceLivenessSession');
    }
    /**
     * Grants permission to start an asynchronous search for faces in a collection that match the faces of persons detected in a stored video
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_StartFaceSearch.html
     */
    toStartFaceSearch() {
        return this.to('StartFaceSearch');
    }
    /**
     * Grants permission to start asynchronous detection of labels in a stored video
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_StartLabelDetection.html
     */
    toStartLabelDetection() {
        return this.to('StartLabelDetection');
    }
    /**
     * Grants permission to start a media analysis job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_StartMediaAnalysisJob.html
     */
    toStartMediaAnalysisJob() {
        return this.to('StartMediaAnalysisJob');
    }
    /**
     * Grants permission to start the asynchronous tracking of persons in a stored video
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_StartPersonTracking.html
     */
    toStartPersonTracking() {
        return this.to('StartPersonTracking');
    }
    /**
     * Grants permission to start running a model version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_StartProjectVersion.html
     */
    toStartProjectVersion() {
        return this.to('StartProjectVersion');
    }
    /**
     * Grants permission to start the asynchronous detection of segments in a stored video
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_StartSegmentDetection.html
     */
    toStartSegmentDetection() {
        return this.to('StartSegmentDetection');
    }
    /**
     * Grants permission to start running a stream processor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_StartStreamProcessor.html
     */
    toStartStreamProcessor() {
        return this.to('StartStreamProcessor');
    }
    /**
     * Grants permission to start the asynchronous detection of text in a stored video
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_StartTextDetection.html
     */
    toStartTextDetection() {
        return this.to('StartTextDetection');
    }
    /**
     * Grants permission to stop a running model version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_StopProjectVersion.html
     */
    toStopProjectVersion() {
        return this.to('StopProjectVersion');
    }
    /**
     * Grants permission to stop a running stream processor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_StopStreamProcessor.html
     */
    toStopStreamProcessor() {
        return this.to('StopStreamProcessor');
    }
    /**
     * Grants permission to add one or more tags to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove one or more tags from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to add or update one or more JSON Lines (entries) in a dataset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_UpdateDatasetEntries.html
     */
    toUpdateDatasetEntries() {
        return this.to('UpdateDatasetEntries');
    }
    /**
     * Grants permission to modify properties for a stream processor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/APIReference/API_UpdateStreamProcessor.html
     */
    toUpdateStreamProcessor() {
        return this.to('UpdateStreamProcessor');
    }
    /**
     * Adds a resource of type collection to the statement
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/collections.html
     *
     * @param collectionId - Identifier for the collectionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCollection(collectionId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:rekognition:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:collection/${collectionId}`);
    }
    /**
     * Adds a resource of type streamprocessor to the statement
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/streaming-video.html
     *
     * @param streamprocessorId - Identifier for the streamprocessorId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStreamprocessor(streamprocessorId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:rekognition:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:streamprocessor/${streamprocessorId}`);
    }
    /**
     * Adds a resource of type project to the statement
     *
     * https://docs.aws.amazon.com/rekognition/latest/customlabels-dg/mp-create-project.html
     *
     * @param projectName - Identifier for the projectName.
     * @param creationTimestamp - Identifier for the creationTimestamp.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProject(projectName, creationTimestamp, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:rekognition:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:project/${projectName}/${creationTimestamp}`);
    }
    /**
     * Adds a resource of type projectversion to the statement
     *
     * https://docs.aws.amazon.com/rekognition/latest/customlabels-dg/training-model.html
     *
     * @param projectName - Identifier for the projectName.
     * @param versionName - Identifier for the versionName.
     * @param creationTimestamp - Identifier for the creationTimestamp.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProjectversion(projectName, versionName, creationTimestamp, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:rekognition:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:project/${projectName}/version/${versionName}/${creationTimestamp}`);
    }
    /**
     * Adds a resource of type dataset to the statement
     *
     * https://docs.aws.amazon.com/rekognition/latest/customlabels-dg/creating-datasets.html
     *
     * @param projectName - Identifier for the projectName.
     * @param datasetType - Identifier for the datasetType.
     * @param creationTimestamp - Identifier for the creationTimestamp.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDataset(projectName, datasetType, creationTimestamp, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:rekognition:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:project/${projectName}/dataset/${datasetType}/${creationTimestamp}`);
    }
    /**
     * Filters access by tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCopyProjectVersion()
     * - .toCreateCollection()
     * - .toCreateDataset()
     * - .toCreateProject()
     * - .toCreateProjectVersion()
     * - .toCreateStreamProcessor()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - collection
     * - streamprocessor
     * - project
     * - projectversion
     * - dataset
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCopyProjectVersion()
     * - .toCreateCollection()
     * - .toCreateDataset()
     * - .toCreateProject()
     * - .toCreateProjectVersion()
     * - .toCreateStreamProcessor()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [rekognition](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonrekognition.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'rekognition';
        this.accessLevelList = {
            Write: [
                'AssociateFaces',
                'CopyProjectVersion',
                'CreateCollection',
                'CreateDataset',
                'CreateFaceLivenessSession',
                'CreateProject',
                'CreateProjectVersion',
                'CreateStreamProcessor',
                'CreateUser',
                'DeleteCollection',
                'DeleteDataset',
                'DeleteFaces',
                'DeleteProject',
                'DeleteProjectPolicy',
                'DeleteProjectVersion',
                'DeleteStreamProcessor',
                'DeleteUser',
                'DisassociateFaces',
                'DistributeDatasetEntries',
                'IndexFaces',
                'PutProjectPolicy',
                'StartCelebrityRecognition',
                'StartContentModeration',
                'StartFaceDetection',
                'StartFaceLivenessSession',
                'StartFaceSearch',
                'StartLabelDetection',
                'StartMediaAnalysisJob',
                'StartPersonTracking',
                'StartProjectVersion',
                'StartSegmentDetection',
                'StartStreamProcessor',
                'StartTextDetection',
                'StopProjectVersion',
                'StopStreamProcessor',
                'UpdateDatasetEntries',
                'UpdateStreamProcessor'
            ],
            Read: [
                'CompareFaces',
                'DescribeCollection',
                'DescribeDataset',
                'DescribeProjectVersions',
                'DescribeProjects',
                'DescribeStreamProcessor',
                'DetectCustomLabels',
                'DetectFaces',
                'DetectLabels',
                'DetectModerationLabels',
                'DetectProtectiveEquipment',
                'DetectText',
                'GetCelebrityInfo',
                'GetCelebrityRecognition',
                'GetContentModeration',
                'GetFaceDetection',
                'GetFaceLivenessSessionResults',
                'GetFaceSearch',
                'GetLabelDetection',
                'GetMediaAnalysisJob',
                'GetPersonTracking',
                'GetSegmentDetection',
                'GetTextDetection',
                'ListCollections',
                'ListDatasetEntries',
                'ListDatasetLabels',
                'ListFaces',
                'ListMediaAnalysisJobs',
                'ListProjectPolicies',
                'ListTagsForResource',
                'ListUsers',
                'RecognizeCelebrities',
                'SearchFaces',
                'SearchFacesByImage',
                'SearchUsers',
                'SearchUsersByImage'
            ],
            List: [
                'ListStreamProcessors'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Rekognition = Rekognition;
//# sourceMappingURL=data:application/json;base64,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