"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Robomaker = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [robomaker](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsrobomaker.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Robomaker extends shared_1.PolicyStatement {
    /**
     * Delete one or more worlds in a batch operation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_BatchDeleteWorlds.html
     */
    toBatchDeleteWorlds() {
        return this.to('BatchDeleteWorlds');
    }
    /**
     * Describe multiple simulation jobs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_BatchDescribeSimulationJob.html
     */
    toBatchDescribeSimulationJob() {
        return this.to('BatchDescribeSimulationJob');
    }
    /**
     * Cancel a deployment job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CancelDeploymentJob.html
     */
    toCancelDeploymentJob() {
        return this.to('CancelDeploymentJob');
    }
    /**
     * Cancel a simulation job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CancelSimulationJob.html
     */
    toCancelSimulationJob() {
        return this.to('CancelSimulationJob');
    }
    /**
     * Cancel a simulation job batch
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CancelSimulationJobBatch.html
     */
    toCancelSimulationJobBatch() {
        return this.to('CancelSimulationJobBatch');
    }
    /**
     * Cancel a world export job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CancelWorldExportJob.html
     */
    toCancelWorldExportJob() {
        return this.to('CancelWorldExportJob');
    }
    /**
     * Cancel a world generation job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CancelWorldGenerationJob.html
     */
    toCancelWorldGenerationJob() {
        return this.to('CancelWorldGenerationJob');
    }
    /**
     * Create a deployment job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateDeploymentJob.html
     */
    toCreateDeploymentJob() {
        return this.to('CreateDeploymentJob');
    }
    /**
     * Create a deployment fleet that represents a logical group of robots running the same robot application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateFleet.html
     */
    toCreateFleet() {
        return this.to('CreateFleet');
    }
    /**
     * Create a robot that can be registered to a fleet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateRobot.html
     */
    toCreateRobot() {
        return this.to('CreateRobot');
    }
    /**
     * Create a robot application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateRobotApplication.html
     */
    toCreateRobotApplication() {
        return this.to('CreateRobotApplication');
    }
    /**
     * Create a snapshot of a robot application
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateRobotApplicationVersion.html
     */
    toCreateRobotApplicationVersion() {
        return this.to('CreateRobotApplicationVersion');
    }
    /**
     * Create a simulation application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateSimulationApplication.html
     */
    toCreateSimulationApplication() {
        return this.to('CreateSimulationApplication');
    }
    /**
     * Create a snapshot of a simulation application
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateSimulationApplicationVersion.html
     */
    toCreateSimulationApplicationVersion() {
        return this.to('CreateSimulationApplicationVersion');
    }
    /**
     * Create a simulation job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateSimulationJob.html
     */
    toCreateSimulationJob() {
        return this.to('CreateSimulationJob');
    }
    /**
     * Create a world export job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateWorldExportJob.html
     */
    toCreateWorldExportJob() {
        return this.to('CreateWorldExportJob');
    }
    /**
     * Create a world generation job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateWorldGenerationJob.html
     */
    toCreateWorldGenerationJob() {
        return this.to('CreateWorldGenerationJob');
    }
    /**
     * Create a world template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_CreateWorldTemplate.html
     */
    toCreateWorldTemplate() {
        return this.to('CreateWorldTemplate');
    }
    /**
     * Delete a deployment fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DeleteFleet.html
     */
    toDeleteFleet() {
        return this.to('DeleteFleet');
    }
    /**
     * Delete a robot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DeleteRobot.html
     */
    toDeleteRobot() {
        return this.to('DeleteRobot');
    }
    /**
     * Delete a robot application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DeleteRobotApplication.html
     */
    toDeleteRobotApplication() {
        return this.to('DeleteRobotApplication');
    }
    /**
     * Delete a simulation application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DeleteSimulationApplication.html
     */
    toDeleteSimulationApplication() {
        return this.to('DeleteSimulationApplication');
    }
    /**
     * Delete a world template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DeleteWorldTemplate.html
     */
    toDeleteWorldTemplate() {
        return this.to('DeleteWorldTemplate');
    }
    /**
     * Deregister a robot from a fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DeregisterRobot.html
     */
    toDeregisterRobot() {
        return this.to('DeregisterRobot');
    }
    /**
     * Describe a deployment job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeDeploymentJob.html
     */
    toDescribeDeploymentJob() {
        return this.to('DescribeDeploymentJob');
    }
    /**
     * Describe a deployment fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeFleet.html
     */
    toDescribeFleet() {
        return this.to('DescribeFleet');
    }
    /**
     * Describe a robot
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeRobot.html
     */
    toDescribeRobot() {
        return this.to('DescribeRobot');
    }
    /**
     * Describe a robot application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeRobotApplication.html
     */
    toDescribeRobotApplication() {
        return this.to('DescribeRobotApplication');
    }
    /**
     * Describe a simulation application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeSimulationApplication.html
     */
    toDescribeSimulationApplication() {
        return this.to('DescribeSimulationApplication');
    }
    /**
     * Describe a simulation job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeSimulationJob.html
     */
    toDescribeSimulationJob() {
        return this.to('DescribeSimulationJob');
    }
    /**
     * Describe a simulation job batch
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeSimulationJobBatch.html
     */
    toDescribeSimulationJobBatch() {
        return this.to('DescribeSimulationJobBatch');
    }
    /**
     * Describe a world
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeWorld.html
     */
    toDescribeWorld() {
        return this.to('DescribeWorld');
    }
    /**
     * Describe a world export job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeWorldExportJob.html
     */
    toDescribeWorldExportJob() {
        return this.to('DescribeWorldExportJob');
    }
    /**
     * Describe a world generation job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeWorldGenerationJob.html
     */
    toDescribeWorldGenerationJob() {
        return this.to('DescribeWorldGenerationJob');
    }
    /**
     * Describe a world template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_DescribeWorldTemplate.html
     */
    toDescribeWorldTemplate() {
        return this.to('DescribeWorldTemplate');
    }
    /**
     * Get the body of a world template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_GetWorldTemplateBody.html
     */
    toGetWorldTemplateBody() {
        return this.to('GetWorldTemplateBody');
    }
    /**
     * List deployment jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListDeploymentJobs.html
     */
    toListDeploymentJobs() {
        return this.to('ListDeploymentJobs');
    }
    /**
     * List fleets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListFleets.html
     */
    toListFleets() {
        return this.to('ListFleets');
    }
    /**
     * List robot applications
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListRobotApplications.html
     */
    toListRobotApplications() {
        return this.to('ListRobotApplications');
    }
    /**
     * List robots
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListRobots.html
     */
    toListRobots() {
        return this.to('ListRobots');
    }
    /**
     * List simulation applications
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListSimulationApplications.html
     */
    toListSimulationApplications() {
        return this.to('ListSimulationApplications');
    }
    /**
     * List simulation job batches
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListSimulationJobBatches.html
     */
    toListSimulationJobBatches() {
        return this.to('ListSimulationJobBatches');
    }
    /**
     * List simulation jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListSimulationJobs.html
     */
    toListSimulationJobs() {
        return this.to('ListSimulationJobs');
    }
    /**
     * Lists supported availability zones
     *
     * Access Level: List
     */
    toListSupportedAvailabilityZones() {
        return this.to('ListSupportedAvailabilityZones');
    }
    /**
     * List tags for a RoboMaker resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * List world export jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListWorldExportJobs.html
     */
    toListWorldExportJobs() {
        return this.to('ListWorldExportJobs');
    }
    /**
     * List world generation jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListWorldGenerationJobs.html
     */
    toListWorldGenerationJobs() {
        return this.to('ListWorldGenerationJobs');
    }
    /**
     * List world templates
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListWorldTemplates.html
     */
    toListWorldTemplates() {
        return this.to('ListWorldTemplates');
    }
    /**
     * List worlds
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_ListWorlds.html
     */
    toListWorlds() {
        return this.to('ListWorlds');
    }
    /**
     * Register a robot to a fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_RegisterRobot.html
     */
    toRegisterRobot() {
        return this.to('RegisterRobot');
    }
    /**
     * Restart a running simulation job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_RestartSimulationJob.html
     */
    toRestartSimulationJob() {
        return this.to('RestartSimulationJob');
    }
    /**
     * Create a simulation job batch
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_StartSimulationJobBatch.html
     */
    toStartSimulationJobBatch() {
        return this.to('StartSimulationJobBatch');
    }
    /**
     * Ensures the most recently deployed robot application is deployed to all robots in the fleet
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_SyncDeploymentJob.html
     */
    toSyncDeploymentJob() {
        return this.to('SyncDeploymentJob');
    }
    /**
     * Add tags to a RoboMaker resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Remove tags from a RoboMaker resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Update a robot application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_UpdateRobotApplication.html
     */
    toUpdateRobotApplication() {
        return this.to('UpdateRobotApplication');
    }
    /**
     * Report the deployment status for an individual robot
     *
     * Access Level: Write
     */
    toUpdateRobotDeployment() {
        return this.to('UpdateRobotDeployment');
    }
    /**
     * Update a simulation application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_UpdateSimulationApplication.html
     */
    toUpdateSimulationApplication() {
        return this.to('UpdateSimulationApplication');
    }
    /**
     * Update a world template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/API_UpdateWorldTemplate.html
     */
    toUpdateWorldTemplate() {
        return this.to('UpdateWorldTemplate');
    }
    /**
     * Adds a resource of type robotApplication to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/managing-robot-applications.html
     *
     * @param applicationName - Identifier for the applicationName.
     * @param createdOnEpoch - Identifier for the createdOnEpoch.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRobotApplication(applicationName, createdOnEpoch, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:robomaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:robot-application/${applicationName}/${createdOnEpoch}`);
    }
    /**
     * Adds a resource of type simulationApplication to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/managing-simulation-applications.html
     *
     * @param applicationName - Identifier for the applicationName.
     * @param createdOnEpoch - Identifier for the createdOnEpoch.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSimulationApplication(applicationName, createdOnEpoch, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:robomaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:simulation-application/${applicationName}/${createdOnEpoch}`);
    }
    /**
     * Adds a resource of type simulationJob to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/simulation.html
     *
     * @param simulationJobId - Identifier for the simulationJobId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSimulationJob(simulationJobId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:robomaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:simulation-job/${simulationJobId}`);
    }
    /**
     * Adds a resource of type simulationJobBatch to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/simulation-job-batch.html
     *
     * @param simulationJobBatchId - Identifier for the simulationJobBatchId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSimulationJobBatch(simulationJobBatchId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:robomaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:simulation-job-batch/${simulationJobBatchId}`);
    }
    /**
     * Adds a resource of type deploymentJob to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/deployment.html
     *
     * @param deploymentJobId - Identifier for the deploymentJobId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDeploymentJob(deploymentJobId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:robomaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:deployment-job/${deploymentJobId}`);
    }
    /**
     * Adds a resource of type robot to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/fleets.html
     *
     * @param robotName - Identifier for the robotName.
     * @param createdOnEpoch - Identifier for the createdOnEpoch.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRobot(robotName, createdOnEpoch, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:robomaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:robot/${robotName}/${createdOnEpoch}`);
    }
    /**
     * Adds a resource of type deploymentFleet to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/managing-simulation-applications.html
     *
     * @param fleetName - Identifier for the fleetName.
     * @param createdOnEpoch - Identifier for the createdOnEpoch.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDeploymentFleet(fleetName, createdOnEpoch, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:robomaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:deployment-fleet/${fleetName}/${createdOnEpoch}`);
    }
    /**
     * Adds a resource of type worldGenerationJob to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/worlds-managing-generation-jobs.html
     *
     * @param worldGenerationJobId - Identifier for the worldGenerationJobId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorldGenerationJob(worldGenerationJobId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:robomaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:world-generation-job/${worldGenerationJobId}`);
    }
    /**
     * Adds a resource of type worldExportJob to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/worlds-managing-export-jobs.html
     *
     * @param worldExportJobId - Identifier for the worldExportJobId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorldExportJob(worldExportJobId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:robomaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:world-export-job/${worldExportJobId}`);
    }
    /**
     * Adds a resource of type worldTemplate to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/worlds-managing-simworld-templates.html
     *
     * @param worldTemplateJobId - Identifier for the worldTemplateJobId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorldTemplate(worldTemplateJobId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:robomaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:world-template/${worldTemplateJobId}`);
    }
    /**
     * Adds a resource of type world to the statement
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/worlds-managing-generated-worlds.html
     *
     * @param worldId - Identifier for the worldId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorld(worldId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:robomaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:world/${worldId}`);
    }
    /**
     * Filters access based on the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/tagging-resources-iam-policies.html
     *
     * Applies to actions:
     * - .toCreateDeploymentJob()
     * - .toCreateFleet()
     * - .toCreateRobot()
     * - .toCreateRobotApplication()
     * - .toCreateSimulationApplication()
     * - .toCreateSimulationJob()
     * - .toCreateWorldExportJob()
     * - .toCreateWorldGenerationJob()
     * - .toCreateWorldTemplate()
     * - .toStartSimulationJobBatch()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the tags associated with the resource
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/tagging-resources-iam-policies.html
     *
     * Applies to resource types:
     * - robotApplication
     * - simulationApplication
     * - simulationJob
     * - simulationJobBatch
     * - deploymentJob
     * - robot
     * - deploymentFleet
     * - worldGenerationJob
     * - worldExportJob
     * - worldTemplate
     * - world
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/robomaker/latest/dg/tagging-resources-iam-policies.html
     *
     * Applies to actions:
     * - .toCreateDeploymentJob()
     * - .toCreateFleet()
     * - .toCreateRobot()
     * - .toCreateRobotApplication()
     * - .toCreateSimulationApplication()
     * - .toCreateSimulationJob()
     * - .toCreateWorldExportJob()
     * - .toCreateWorldGenerationJob()
     * - .toCreateWorldTemplate()
     * - .toStartSimulationJobBatch()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [robomaker](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsrobomaker.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'robomaker';
        this.accessLevelList = {
            Write: [
                'BatchDeleteWorlds',
                'CancelDeploymentJob',
                'CancelSimulationJob',
                'CancelSimulationJobBatch',
                'CancelWorldExportJob',
                'CancelWorldGenerationJob',
                'CreateDeploymentJob',
                'CreateFleet',
                'CreateRobot',
                'CreateRobotApplication',
                'CreateRobotApplicationVersion',
                'CreateSimulationApplication',
                'CreateSimulationApplicationVersion',
                'CreateSimulationJob',
                'CreateWorldExportJob',
                'CreateWorldGenerationJob',
                'CreateWorldTemplate',
                'DeleteFleet',
                'DeleteRobot',
                'DeleteRobotApplication',
                'DeleteSimulationApplication',
                'DeleteWorldTemplate',
                'DeregisterRobot',
                'RegisterRobot',
                'RestartSimulationJob',
                'StartSimulationJobBatch',
                'SyncDeploymentJob',
                'UpdateRobotApplication',
                'UpdateRobotDeployment',
                'UpdateSimulationApplication',
                'UpdateWorldTemplate'
            ],
            Read: [
                'BatchDescribeSimulationJob',
                'DescribeDeploymentJob',
                'DescribeFleet',
                'DescribeRobot',
                'DescribeRobotApplication',
                'DescribeSimulationApplication',
                'DescribeSimulationJob',
                'DescribeSimulationJobBatch',
                'DescribeWorld',
                'DescribeWorldExportJob',
                'DescribeWorldGenerationJob',
                'DescribeWorldTemplate',
                'GetWorldTemplateBody'
            ],
            List: [
                'ListDeploymentJobs',
                'ListFleets',
                'ListRobotApplications',
                'ListRobots',
                'ListSimulationApplications',
                'ListSimulationJobBatches',
                'ListSimulationJobs',
                'ListSupportedAvailabilityZones',
                'ListTagsForResource',
                'ListWorldExportJobs',
                'ListWorldGenerationJobs',
                'ListWorldTemplates',
                'ListWorlds'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Robomaker = Robomaker;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicm9ib21ha2VyLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsicm9ib21ha2VyLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUNBLHlDQUF5RDtBQUd6RDs7OztHQUlHO0FBQ0gsTUFBYSxTQUFVLFNBQVEsd0JBQWU7SUFHNUM7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw0QkFBNEI7UUFDakMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDRCQUE0QixDQUFDLENBQUM7SUFDL0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwwQkFBMEI7UUFDL0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMEJBQTBCO1FBQy9CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO0lBQzdDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksYUFBYTtRQUNsQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsYUFBYSxDQUFDLENBQUM7SUFDaEMsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxhQUFhO1FBQ2xCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUNoQyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLHdCQUF3QjtRQUM3QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksK0JBQStCO1FBQ3BDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO0lBQ2xELENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksNkJBQTZCO1FBQ2xDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO0lBQ2hELENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxvQ0FBb0M7UUFDekMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG9DQUFvQyxDQUFDLENBQUM7SUFDdkQsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxzQkFBc0I7UUFDM0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUM7SUFDekMsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSwwQkFBMEI7UUFDL0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ2hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxhQUFhO1FBQ2xCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUNoQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksd0JBQXdCO1FBQzdCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO0lBQzNDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw2QkFBNkI7UUFDbEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDZCQUE2QixDQUFDLENBQUM7SUFDaEQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUJBQWlCO1FBQ3RCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO0lBQ3BDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7SUFDMUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGVBQWU7UUFDcEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxlQUFlO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMEJBQTBCO1FBQy9CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO0lBQzdDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwrQkFBK0I7UUFDcEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLCtCQUErQixDQUFDLENBQUM7SUFDbEQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHVCQUF1QjtRQUM1QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsdUJBQXVCLENBQUMsQ0FBQztJQUMxQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksNEJBQTRCO1FBQ2pDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO0lBQy9DLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxlQUFlO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksd0JBQXdCO1FBQzdCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO0lBQzNDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw0QkFBNEI7UUFDakMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDRCQUE0QixDQUFDLENBQUM7SUFDL0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHVCQUF1QjtRQUM1QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsdUJBQXVCLENBQUMsQ0FBQztJQUMxQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0JBQXNCO1FBQzNCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG9CQUFvQixDQUFDLENBQUM7SUFDdkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFlBQVk7UUFDakIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFlBQVksQ0FBQyxDQUFDO0lBQy9CLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7SUFDMUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFlBQVk7UUFDakIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFlBQVksQ0FBQyxDQUFDO0lBQy9CLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw0QkFBNEI7UUFDakMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDRCQUE0QixDQUFDLENBQUM7SUFDL0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDBCQUEwQjtRQUMvQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMEJBQTBCLENBQUMsQ0FBQztJQUM3QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO0lBQ3ZDLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksZ0NBQWdDO1FBQ3JDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO0lBQ25ELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUJBQXlCO1FBQzlCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO0lBQzVDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG9CQUFvQixDQUFDLENBQUM7SUFDdkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFlBQVk7UUFDakIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFlBQVksQ0FBQyxDQUFDO0lBQy9CLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxlQUFlO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0JBQXNCO1FBQzNCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0kseUJBQXlCO1FBQzlCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO0lBQzVDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxtQkFBbUI7UUFDeEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxhQUFhO1FBQ2xCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUNoQyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksZUFBZTtRQUNwQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdCQUF3QjtRQUM3QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLHVCQUF1QjtRQUM1QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsdUJBQXVCLENBQUMsQ0FBQztJQUMxQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksNkJBQTZCO1FBQ2xDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO0lBQ2hELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQXdFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksa0JBQWtCLENBQUMsZUFBdUIsRUFBRSxjQUFzQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQzlILE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGNBQWUsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLHNCQUF1QixlQUFnQixJQUFLLGNBQWUsRUFBRSxDQUFDLENBQUM7SUFDM00sQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSx1QkFBdUIsQ0FBQyxlQUF1QixFQUFFLGNBQXNCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDbkksT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsY0FBZSxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsMkJBQTRCLGVBQWdCLElBQUssY0FBZSxFQUFFLENBQUMsQ0FBQztJQUNoTixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksZUFBZSxDQUFDLGVBQXVCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDbkcsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsY0FBZSxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsbUJBQW9CLGVBQWdCLEVBQUUsQ0FBQyxDQUFDO0lBQ3BMLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxvQkFBb0IsQ0FBQyxvQkFBNEIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUM3RyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixjQUFlLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSx5QkFBMEIsb0JBQXFCLEVBQUUsQ0FBQyxDQUFDO0lBQy9MLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxlQUFlLENBQUMsZUFBdUIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNuRyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixjQUFlLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxtQkFBb0IsZUFBZ0IsRUFBRSxDQUFDLENBQUM7SUFDcEwsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxPQUFPLENBQUMsU0FBaUIsRUFBRSxjQUFzQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQzdHLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGNBQWUsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLFVBQVcsU0FBVSxJQUFLLGNBQWUsRUFBRSxDQUFDLENBQUM7SUFDekwsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxpQkFBaUIsQ0FBQyxTQUFpQixFQUFFLGNBQXNCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDdkgsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsY0FBZSxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUscUJBQXNCLFNBQVUsSUFBSyxjQUFlLEVBQUUsQ0FBQyxDQUFDO0lBQ3BNLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxvQkFBb0IsQ0FBQyxvQkFBNEIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUM3RyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixjQUFlLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSx5QkFBMEIsb0JBQXFCLEVBQUUsQ0FBQyxDQUFDO0lBQy9MLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxnQkFBZ0IsQ0FBQyxnQkFBd0IsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNyRyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixjQUFlLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxxQkFBc0IsZ0JBQWlCLEVBQUUsQ0FBQyxDQUFDO0lBQ3ZMLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxlQUFlLENBQUMsa0JBQTBCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDdEcsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsY0FBZSxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsbUJBQW9CLGtCQUFtQixFQUFFLENBQUMsQ0FBQztJQUN2TCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksT0FBTyxDQUFDLE9BQWUsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNuRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixjQUFlLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxVQUFXLE9BQVEsRUFBRSxDQUFDLENBQUM7SUFDbkssQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FxQkc7SUFDSSxlQUFlLENBQUMsTUFBYyxFQUFFLEtBQXdCLEVBQUUsUUFBNEI7UUFDM0YsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFtQixNQUFPLEVBQUUsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2hGLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09BcUJHO0lBQ0ksZ0JBQWdCLENBQUMsTUFBYyxFQUFFLEtBQXdCLEVBQUUsUUFBNEI7UUFDNUYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFvQixNQUFPLEVBQUUsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2pGLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09BcUJHO0lBQ0ksWUFBWSxDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDeEUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2pFLENBQUM7SUFFRDs7O09BR0c7SUFDSCxZQUFZLEtBQWdDO1FBQzFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztRQW5pQ1Isa0JBQWEsR0FBRyxXQUFXLENBQUM7UUEyc0J6QixvQkFBZSxHQUFvQjtZQUMzQyxLQUFLLEVBQUU7Z0JBQ0wsbUJBQW1CO2dCQUNuQixxQkFBcUI7Z0JBQ3JCLHFCQUFxQjtnQkFDckIsMEJBQTBCO2dCQUMxQixzQkFBc0I7Z0JBQ3RCLDBCQUEwQjtnQkFDMUIscUJBQXFCO2dCQUNyQixhQUFhO2dCQUNiLGFBQWE7Z0JBQ2Isd0JBQXdCO2dCQUN4QiwrQkFBK0I7Z0JBQy9CLDZCQUE2QjtnQkFDN0Isb0NBQW9DO2dCQUNwQyxxQkFBcUI7Z0JBQ3JCLHNCQUFzQjtnQkFDdEIsMEJBQTBCO2dCQUMxQixxQkFBcUI7Z0JBQ3JCLGFBQWE7Z0JBQ2IsYUFBYTtnQkFDYix3QkFBd0I7Z0JBQ3hCLDZCQUE2QjtnQkFDN0IscUJBQXFCO2dCQUNyQixpQkFBaUI7Z0JBQ2pCLGVBQWU7Z0JBQ2Ysc0JBQXNCO2dCQUN0Qix5QkFBeUI7Z0JBQ3pCLG1CQUFtQjtnQkFDbkIsd0JBQXdCO2dCQUN4Qix1QkFBdUI7Z0JBQ3ZCLDZCQUE2QjtnQkFDN0IscUJBQXFCO2FBQ3RCO1lBQ0QsSUFBSSxFQUFFO2dCQUNKLDRCQUE0QjtnQkFDNUIsdUJBQXVCO2dCQUN2QixlQUFlO2dCQUNmLGVBQWU7Z0JBQ2YsMEJBQTBCO2dCQUMxQiwrQkFBK0I7Z0JBQy9CLHVCQUF1QjtnQkFDdkIsNEJBQTRCO2dCQUM1QixlQUFlO2dCQUNmLHdCQUF3QjtnQkFDeEIsNEJBQTRCO2dCQUM1Qix1QkFBdUI7Z0JBQ3ZCLHNCQUFzQjthQUN2QjtZQUNELElBQUksRUFBRTtnQkFDSixvQkFBb0I7Z0JBQ3BCLFlBQVk7Z0JBQ1osdUJBQXVCO2dCQUN2QixZQUFZO2dCQUNaLDRCQUE0QjtnQkFDNUIsMEJBQTBCO2dCQUMxQixvQkFBb0I7Z0JBQ3BCLGdDQUFnQztnQkFDaEMscUJBQXFCO2dCQUNyQixxQkFBcUI7Z0JBQ3JCLHlCQUF5QjtnQkFDekIsb0JBQW9CO2dCQUNwQixZQUFZO2FBQ2I7WUFDRCxPQUFPLEVBQUU7Z0JBQ1AsYUFBYTtnQkFDYixlQUFlO2FBQ2hCO1NBQ0YsQ0FBQztJQXFSRixDQUFDO0NBQ0Y7QUF0aUNELDhCQXNpQ0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBBY2Nlc3NMZXZlbExpc3QgfSBmcm9tICcuLi8uLi9zaGFyZWQvYWNjZXNzLWxldmVsJztcbmltcG9ydCB7IFBvbGljeVN0YXRlbWVudCwgT3BlcmF0b3IgfSBmcm9tICcuLi8uLi9zaGFyZWQnO1xuaW1wb3J0IHsgYXdzX2lhbSBhcyBpYW0gfSBmcm9tIFwiYXdzLWNkay1saWJcIjtcblxuLyoqXG4gKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW3JvYm9tYWtlcl0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYXdzcm9ib21ha2VyLmh0bWwpLlxuICpcbiAqIEBwYXJhbSBzaWQgW1NJRF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19zaWQuaHRtbCkgb2YgdGhlIHN0YXRlbWVudFxuICovXG5leHBvcnQgY2xhc3MgUm9ib21ha2VyIGV4dGVuZHMgUG9saWN5U3RhdGVtZW50IHtcbiAgcHVibGljIHNlcnZpY2VQcmVmaXggPSAncm9ib21ha2VyJztcblxuICAvKipcbiAgICogRGVsZXRlIG9uZSBvciBtb3JlIHdvcmxkcyBpbiBhIGJhdGNoIG9wZXJhdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9yb2JvbWFrZXIvbGF0ZXN0L2RnL0FQSV9CYXRjaERlbGV0ZVdvcmxkcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9CYXRjaERlbGV0ZVdvcmxkcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQmF0Y2hEZWxldGVXb3JsZHMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBEZXNjcmliZSBtdWx0aXBsZSBzaW11bGF0aW9uIGpvYnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9yb2JvbWFrZXIvbGF0ZXN0L2RnL0FQSV9CYXRjaERlc2NyaWJlU2ltdWxhdGlvbkpvYi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9CYXRjaERlc2NyaWJlU2ltdWxhdGlvbkpvYigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQmF0Y2hEZXNjcmliZVNpbXVsYXRpb25Kb2InKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDYW5jZWwgYSBkZXBsb3ltZW50IGpvYlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9yb2JvbWFrZXIvbGF0ZXN0L2RnL0FQSV9DYW5jZWxEZXBsb3ltZW50Sm9iLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NhbmNlbERlcGxveW1lbnRKb2IoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NhbmNlbERlcGxveW1lbnRKb2InKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDYW5jZWwgYSBzaW11bGF0aW9uIGpvYlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9yb2JvbWFrZXIvbGF0ZXN0L2RnL0FQSV9DYW5jZWxTaW11bGF0aW9uSm9iLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NhbmNlbFNpbXVsYXRpb25Kb2IoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NhbmNlbFNpbXVsYXRpb25Kb2InKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDYW5jZWwgYSBzaW11bGF0aW9uIGpvYiBiYXRjaFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9yb2JvbWFrZXIvbGF0ZXN0L2RnL0FQSV9DYW5jZWxTaW11bGF0aW9uSm9iQmF0Y2guaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ2FuY2VsU2ltdWxhdGlvbkpvYkJhdGNoKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDYW5jZWxTaW11bGF0aW9uSm9iQmF0Y2gnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDYW5jZWwgYSB3b3JsZCBleHBvcnQgam9iXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3JvYm9tYWtlci9sYXRlc3QvZGcvQVBJX0NhbmNlbFdvcmxkRXhwb3J0Sm9iLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NhbmNlbFdvcmxkRXhwb3J0Sm9iKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDYW5jZWxXb3JsZEV4cG9ydEpvYicpO1xuICB9XG5cbiAgLyoqXG4gICAqIENhbmNlbCBhIHdvcmxkIGdlbmVyYXRpb24gam9iXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3JvYm9tYWtlci9sYXRlc3QvZGcvQVBJX0NhbmNlbFdvcmxkR2VuZXJhdGlvbkpvYi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DYW5jZWxXb3JsZEdlbmVyYXRpb25Kb2IoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NhbmNlbFdvcmxkR2VuZXJhdGlvbkpvYicpO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZSBhIGRlcGxveW1lbnQgam9iXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gaWFtOkNyZWF0ZVNlcnZpY2VMaW5rZWRSb2xlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9yb2JvbWFrZXIvbGF0ZXN0L2RnL0FQSV9DcmVhdGVEZXBsb3ltZW50Sm9iLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZURlcGxveW1lbnRKb2IoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZURlcGxveW1lbnRKb2InKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGUgYSBkZXBsb3ltZW50IGZsZWV0IHRoYXQgcmVwcmVzZW50cyBhIGxvZ2ljYWwgZ3JvdXAgb2Ygcm9ib3RzIHJ1bm5pbmcgdGhlIHNhbWUgcm9ib3QgYXBwbGljYXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcm9ib21ha2VyL2xhdGVzdC9kZy9BUElfQ3JlYXRlRmxlZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlRmxlZXQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZUZsZWV0Jyk7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlIGEgcm9ib3QgdGhhdCBjYW4gYmUgcmVnaXN0ZXJlZCB0byBhIGZsZWV0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gaWFtOkNyZWF0ZVNlcnZpY2VMaW5rZWRSb2xlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9yb2JvbWFrZXIvbGF0ZXN0L2RnL0FQSV9DcmVhdGVSb2JvdC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVSb2JvdCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlUm9ib3QnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGUgYSByb2JvdCBhcHBsaWNhdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9yb2JvbWFrZXIvbGF0ZXN0L2RnL0FQSV9DcmVhdGVSb2JvdEFwcGxpY2F0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVJvYm90QXBwbGljYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZVJvYm90QXBwbGljYXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGUgYSBzbmFwc2hvdCBvZiBhIHJvYm90IGFwcGxpY2F0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gczM6R2V0T2JqZWN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9yb2JvbWFrZXIvbGF0ZXN0L2RnL0FQSV9DcmVhdGVSb2JvdEFwcGxpY2F0aW9uVmVyc2lvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVSb2JvdEFwcGxpY2F0aW9uVmVyc2lvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlUm9ib3RBcHBsaWNhdGlvblZlcnNpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGUgYSBzaW11bGF0aW9uIGFwcGxpY2F0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3JvYm9tYWtlci9sYXRlc3QvZGcvQVBJX0NyZWF0ZVNpbXVsYXRpb25BcHBsaWNhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVTaW11bGF0aW9uQXBwbGljYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZVNpbXVsYXRpb25BcHBsaWNhdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZSBhIHNuYXBzaG90IG9mIGEgc2ltdWxhdGlvbiBhcHBsaWNhdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIHMzOkdldE9iamVjdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcm9ib21ha2VyL2xhdGVzdC9kZy9BUElfQ3JlYXRlU2ltdWxhdGlvbkFwcGxpY2F0aW9uVmVyc2lvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVTaW11bGF0aW9uQXBwbGljYXRpb25WZXJzaW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVTaW11bGF0aW9uQXBwbGljYXRpb25WZXJzaW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlIGEgc2ltdWxhdGlvbiBqb2JcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBpYW06Q3JlYXRlU2VydmljZUxpbmtlZFJvbGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3JvYm9tYWtlci9sYXRlc3QvZGcvQVBJX0NyZWF0ZVNpbXVsYXRpb25Kb2IuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlU2ltdWxhdGlvbkpvYigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlU2ltdWxhdGlvbkpvYicpO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZSBhIHdvcmxkIGV4cG9ydCBqb2JcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcm9ib21ha2VyL2xhdGVzdC9kZy9BUElfQ3JlYXRlV29ybGRFeHBvcnRKb2IuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlV29ybGRFeHBvcnRKb2IoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZVdvcmxkRXhwb3J0Sm9iJyk7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlIGEgd29ybGQgZ2VuZXJhdGlvbiBqb2JcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcm9ib21ha2VyL2xhdGVzdC9kZy9BUElfQ3JlYXRlV29ybGRHZW5lcmF0aW9uSm9iLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVdvcmxkR2VuZXJhdGlvbkpvYigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlV29ybGRHZW5lcmF0aW9uSm9iJyk7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlIGEgd29ybGQgdGVtcGxhdGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcm9ib21ha2VyL2xhdGVzdC9kZy9BUElfQ3JlYXRlV29ybGRUZW1wbGF0ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVXb3JsZFRlbXBsYXRlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVXb3JsZFRlbXBsYXRlJyk7XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlIGEgZGVwbG95bWVudCBmbGVldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9yb2JvbWFrZXIvbGF0ZXN0L2RnL0FQSV9EZWxldGVGbGVldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVGbGVldCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlRmxlZXQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGUgYSByb2JvdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9yb2JvbWFrZXIvbGF0ZXN0L2RnL0FQSV9EZWxldGVSb2JvdC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVSb2JvdCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlUm9ib3QnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGUgYSByb2JvdCBhcHBsaWNhdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9yb2JvbWFrZXIvbGF0ZXN0L2RnL0FQSV9EZWxldGVSb2JvdEFwcGxpY2F0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVJvYm90QXBwbGljYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZVJvYm90QXBwbGljYXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGUgYSBzaW11bGF0aW9uIGFwcGxpY2F0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3JvYm9tYWtlci9sYXRlc3QvZGcvQVBJX0RlbGV0ZVNpbXVsYXRpb25BcHBsaWNhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVTaW11bGF0aW9uQXBwbGljYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZVNpbXVsYXRpb25BcHBsaWNhdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZSBhIHdvcmxkIHRlbXBsYXRlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3JvYm9tYWtlci9sYXRlc3QvZGcvQVBJX0RlbGV0ZVdvcmxkVGVtcGxhdGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlV29ybGRUZW1wbGF0ZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlV29ybGRUZW1wbGF0ZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIERlcmVnaXN0ZXIgYSByb2JvdCBmcm9tIGEgZmxlZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcm9ib21ha2VyL2xhdGVzdC9kZy9BUElfRGVyZWdpc3RlclJvYm90Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlcmVnaXN0ZXJSb2JvdCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVyZWdpc3RlclJvYm90Jyk7XG4gIH1cblxuICAvKipcbiAgICogRGVzY3JpYmUgYSBkZXBsb3ltZW50IGpvYlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3JvYm9tYWtlci9sYXRlc3QvZGcvQVBJX0Rlc2NyaWJlRGVwbG95bWVudEpvYi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZURlcGxveW1lbnRKb2IoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlRGVwbG95bWVudEpvYicpO1xuICB9XG5cbiAgLyoqXG4gICAqIERlc2NyaWJlIGEgZGVwbG95bWVudCBmbGVldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3JvYm9tYWtlci9sYXRlc3QvZGcvQVBJX0Rlc2NyaWJlRmxlZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVGbGVldCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVGbGVldCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIERlc2NyaWJlIGEgcm9ib3RcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9yb2JvbWFrZXIvbGF0ZXN0L2RnL0FQSV9EZXNjcmliZVJvYm90Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlUm9ib3QoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlUm9ib3QnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBEZXNjcmliZSBhIHJvYm90IGFwcGxpY2F0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcm9ib21ha2VyL2xhdGVzdC9kZy9BUElfRGVzY3JpYmVSb2JvdEFwcGxpY2F0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlUm9ib3RBcHBsaWNhdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVSb2JvdEFwcGxpY2F0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogRGVzY3JpYmUgYSBzaW11bGF0aW9uIGFwcGxpY2F0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcm9ib21ha2VyL2xhdGVzdC9kZy9BUElfRGVzY3JpYmVTaW11bGF0aW9uQXBwbGljYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVTaW11bGF0aW9uQXBwbGljYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlU2ltdWxhdGlvbkFwcGxpY2F0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogRGVzY3JpYmUgYSBzaW11bGF0aW9uIGpvYlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3JvYm9tYWtlci9sYXRlc3QvZGcvQVBJX0Rlc2NyaWJlU2ltdWxhdGlvbkpvYi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZVNpbXVsYXRpb25Kb2IoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlU2ltdWxhdGlvbkpvYicpO1xuICB9XG5cbiAgLyoqXG4gICAqIERlc2NyaWJlIGEgc2ltdWxhdGlvbiBqb2IgYmF0Y2hcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9yb2JvbWFrZXIvbGF0ZXN0L2RnL0FQSV9EZXNjcmliZVNpbXVsYXRpb25Kb2JCYXRjaC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZVNpbXVsYXRpb25Kb2JCYXRjaCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVTaW11bGF0aW9uSm9iQmF0Y2gnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBEZXNjcmliZSBhIHdvcmxkXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcm9ib21ha2VyL2xhdGVzdC9kZy9BUElfRGVzY3JpYmVXb3JsZC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZVdvcmxkKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZVdvcmxkJyk7XG4gIH1cblxuICAvKipcbiAgICogRGVzY3JpYmUgYSB3b3JsZCBleHBvcnQgam9iXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcm9ib21ha2VyL2xhdGVzdC9kZy9BUElfRGVzY3JpYmVXb3JsZEV4cG9ydEpvYi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZVdvcmxkRXhwb3J0Sm9iKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZVdvcmxkRXhwb3J0Sm9iJyk7XG4gIH1cblxuICAvKipcbiAgICogRGVzY3JpYmUgYSB3b3JsZCBnZW5lcmF0aW9uIGpvYlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3JvYm9tYWtlci9sYXRlc3QvZGcvQVBJX0Rlc2NyaWJlV29ybGRHZW5lcmF0aW9uSm9iLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlV29ybGRHZW5lcmF0aW9uSm9iKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZVdvcmxkR2VuZXJhdGlvbkpvYicpO1xuICB9XG5cbiAgLyoqXG4gICAqIERlc2NyaWJlIGEgd29ybGQgdGVtcGxhdGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9yb2JvbWFrZXIvbGF0ZXN0L2RnL0FQSV9EZXNjcmliZVdvcmxkVGVtcGxhdGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVXb3JsZFRlbXBsYXRlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZVdvcmxkVGVtcGxhdGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHZXQgdGhlIGJvZHkgb2YgYSB3b3JsZCB0ZW1wbGF0ZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3JvYm9tYWtlci9sYXRlc3QvZGcvQVBJX0dldFdvcmxkVGVtcGxhdGVCb2R5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFdvcmxkVGVtcGxhdGVCb2R5KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRXb3JsZFRlbXBsYXRlQm9keScpO1xuICB9XG5cbiAgLyoqXG4gICAqIExpc3QgZGVwbG95bWVudCBqb2JzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcm9ib21ha2VyL2xhdGVzdC9kZy9BUElfTGlzdERlcGxveW1lbnRKb2JzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3REZXBsb3ltZW50Sm9icygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdERlcGxveW1lbnRKb2JzJyk7XG4gIH1cblxuICAvKipcbiAgICogTGlzdCBmbGVldHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9yb2JvbWFrZXIvbGF0ZXN0L2RnL0FQSV9MaXN0RmxlZXRzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RGbGVldHMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RGbGVldHMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBMaXN0IHJvYm90IGFwcGxpY2F0aW9uc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3JvYm9tYWtlci9sYXRlc3QvZGcvQVBJX0xpc3RSb2JvdEFwcGxpY2F0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0Um9ib3RBcHBsaWNhdGlvbnMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RSb2JvdEFwcGxpY2F0aW9ucycpO1xuICB9XG5cbiAgLyoqXG4gICAqIExpc3Qgcm9ib3RzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcm9ib21ha2VyL2xhdGVzdC9kZy9BUElfTGlzdFJvYm90cy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0Um9ib3RzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0Um9ib3RzJyk7XG4gIH1cblxuICAvKipcbiAgICogTGlzdCBzaW11bGF0aW9uIGFwcGxpY2F0aW9uc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3JvYm9tYWtlci9sYXRlc3QvZGcvQVBJX0xpc3RTaW11bGF0aW9uQXBwbGljYXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RTaW11bGF0aW9uQXBwbGljYXRpb25zKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0U2ltdWxhdGlvbkFwcGxpY2F0aW9ucycpO1xuICB9XG5cbiAgLyoqXG4gICAqIExpc3Qgc2ltdWxhdGlvbiBqb2IgYmF0Y2hlc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3JvYm9tYWtlci9sYXRlc3QvZGcvQVBJX0xpc3RTaW11bGF0aW9uSm9iQmF0Y2hlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0U2ltdWxhdGlvbkpvYkJhdGNoZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RTaW11bGF0aW9uSm9iQmF0Y2hlcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIExpc3Qgc2ltdWxhdGlvbiBqb2JzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcm9ib21ha2VyL2xhdGVzdC9kZy9BUElfTGlzdFNpbXVsYXRpb25Kb2JzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RTaW11bGF0aW9uSm9icygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdFNpbXVsYXRpb25Kb2JzJyk7XG4gIH1cblxuICAvKipcbiAgICogTGlzdHMgc3VwcG9ydGVkIGF2YWlsYWJpbGl0eSB6b25lc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RTdXBwb3J0ZWRBdmFpbGFiaWxpdHlab25lcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdFN1cHBvcnRlZEF2YWlsYWJpbGl0eVpvbmVzJyk7XG4gIH1cblxuICAvKipcbiAgICogTGlzdCB0YWdzIGZvciBhIFJvYm9NYWtlciByZXNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3JvYm9tYWtlci9sYXRlc3QvZGcvQVBJX0xpc3RUYWdzRm9yUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFRhZ3NGb3JSZXNvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdFRhZ3NGb3JSZXNvdXJjZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIExpc3Qgd29ybGQgZXhwb3J0IGpvYnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9yb2JvbWFrZXIvbGF0ZXN0L2RnL0FQSV9MaXN0V29ybGRFeHBvcnRKb2JzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RXb3JsZEV4cG9ydEpvYnMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RXb3JsZEV4cG9ydEpvYnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBMaXN0IHdvcmxkIGdlbmVyYXRpb24gam9ic1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3JvYm9tYWtlci9sYXRlc3QvZGcvQVBJX0xpc3RXb3JsZEdlbmVyYXRpb25Kb2JzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RXb3JsZEdlbmVyYXRpb25Kb2JzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0V29ybGRHZW5lcmF0aW9uSm9icycpO1xuICB9XG5cbiAgLyoqXG4gICAqIExpc3Qgd29ybGQgdGVtcGxhdGVzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcm9ib21ha2VyL2xhdGVzdC9kZy9BUElfTGlzdFdvcmxkVGVtcGxhdGVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RXb3JsZFRlbXBsYXRlcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdFdvcmxkVGVtcGxhdGVzJyk7XG4gIH1cblxuICAvKipcbiAgICogTGlzdCB3b3JsZHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9yb2JvbWFrZXIvbGF0ZXN0L2RnL0FQSV9MaXN0V29ybGRzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RXb3JsZHMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RXb3JsZHMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZWdpc3RlciBhIHJvYm90IHRvIGEgZmxlZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcm9ib21ha2VyL2xhdGVzdC9kZy9BUElfUmVnaXN0ZXJSb2JvdC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9SZWdpc3RlclJvYm90KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdSZWdpc3RlclJvYm90Jyk7XG4gIH1cblxuICAvKipcbiAgICogUmVzdGFydCBhIHJ1bm5pbmcgc2ltdWxhdGlvbiBqb2JcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcm9ib21ha2VyL2xhdGVzdC9kZy9BUElfUmVzdGFydFNpbXVsYXRpb25Kb2IuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUmVzdGFydFNpbXVsYXRpb25Kb2IoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1Jlc3RhcnRTaW11bGF0aW9uSm9iJyk7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlIGEgc2ltdWxhdGlvbiBqb2IgYmF0Y2hcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBpYW06Q3JlYXRlU2VydmljZUxpbmtlZFJvbGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3JvYm9tYWtlci9sYXRlc3QvZGcvQVBJX1N0YXJ0U2ltdWxhdGlvbkpvYkJhdGNoLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N0YXJ0U2ltdWxhdGlvbkpvYkJhdGNoKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTdGFydFNpbXVsYXRpb25Kb2JCYXRjaCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEVuc3VyZXMgdGhlIG1vc3QgcmVjZW50bHkgZGVwbG95ZWQgcm9ib3QgYXBwbGljYXRpb24gaXMgZGVwbG95ZWQgdG8gYWxsIHJvYm90cyBpbiB0aGUgZmxlZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBpYW06Q3JlYXRlU2VydmljZUxpbmtlZFJvbGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3JvYm9tYWtlci9sYXRlc3QvZGcvQVBJX1N5bmNEZXBsb3ltZW50Sm9iLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N5bmNEZXBsb3ltZW50Sm9iKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTeW5jRGVwbG95bWVudEpvYicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCB0YWdzIHRvIGEgUm9ib01ha2VyIHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogVGFnZ2luZ1xuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcm9ib21ha2VyL2xhdGVzdC9kZy9BUElfVGFnUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVGFnUmVzb3VyY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1RhZ1Jlc291cmNlJyk7XG4gIH1cblxuICAvKipcbiAgICogUmVtb3ZlIHRhZ3MgZnJvbSBhIFJvYm9NYWtlciByZXNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFRhZ2dpbmdcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3JvYm9tYWtlci9sYXRlc3QvZGcvQVBJX1VudGFnUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVW50YWdSZXNvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVW50YWdSZXNvdXJjZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIFVwZGF0ZSBhIHJvYm90IGFwcGxpY2F0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3JvYm9tYWtlci9sYXRlc3QvZGcvQVBJX1VwZGF0ZVJvYm90QXBwbGljYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlUm9ib3RBcHBsaWNhdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlUm9ib3RBcHBsaWNhdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlcG9ydCB0aGUgZGVwbG95bWVudCBzdGF0dXMgZm9yIGFuIGluZGl2aWR1YWwgcm9ib3RcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlUm9ib3REZXBsb3ltZW50KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVSb2JvdERlcGxveW1lbnQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBVcGRhdGUgYSBzaW11bGF0aW9uIGFwcGxpY2F0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3JvYm9tYWtlci9sYXRlc3QvZGcvQVBJX1VwZGF0ZVNpbXVsYXRpb25BcHBsaWNhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVTaW11bGF0aW9uQXBwbGljYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZVNpbXVsYXRpb25BcHBsaWNhdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIFVwZGF0ZSBhIHdvcmxkIHRlbXBsYXRlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3JvYm9tYWtlci9sYXRlc3QvZGcvQVBJX1VwZGF0ZVdvcmxkVGVtcGxhdGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlV29ybGRUZW1wbGF0ZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlV29ybGRUZW1wbGF0ZScpO1xuICB9XG5cbiAgcHJvdGVjdGVkIGFjY2Vzc0xldmVsTGlzdDogQWNjZXNzTGV2ZWxMaXN0ID0ge1xuICAgIFdyaXRlOiBbXG4gICAgICAnQmF0Y2hEZWxldGVXb3JsZHMnLFxuICAgICAgJ0NhbmNlbERlcGxveW1lbnRKb2InLFxuICAgICAgJ0NhbmNlbFNpbXVsYXRpb25Kb2InLFxuICAgICAgJ0NhbmNlbFNpbXVsYXRpb25Kb2JCYXRjaCcsXG4gICAgICAnQ2FuY2VsV29ybGRFeHBvcnRKb2InLFxuICAgICAgJ0NhbmNlbFdvcmxkR2VuZXJhdGlvbkpvYicsXG4gICAgICAnQ3JlYXRlRGVwbG95bWVudEpvYicsXG4gICAgICAnQ3JlYXRlRmxlZXQnLFxuICAgICAgJ0NyZWF0ZVJvYm90JyxcbiAgICAgICdDcmVhdGVSb2JvdEFwcGxpY2F0aW9uJyxcbiAgICAgICdDcmVhdGVSb2JvdEFwcGxpY2F0aW9uVmVyc2lvbicsXG4gICAgICAnQ3JlYXRlU2ltdWxhdGlvbkFwcGxpY2F0aW9uJyxcbiAgICAgICdDcmVhdGVTaW11bGF0aW9uQXBwbGljYXRpb25WZXJzaW9uJyxcbiAgICAgICdDcmVhdGVTaW11bGF0aW9uSm9iJyxcbiAgICAgICdDcmVhdGVXb3JsZEV4cG9ydEpvYicsXG4gICAgICAnQ3JlYXRlV29ybGRHZW5lcmF0aW9uSm9iJyxcbiAgICAgICdDcmVhdGVXb3JsZFRlbXBsYXRlJyxcbiAgICAgICdEZWxldGVGbGVldCcsXG4gICAgICAnRGVsZXRlUm9ib3QnLFxuICAgICAgJ0RlbGV0ZVJvYm90QXBwbGljYXRpb24nLFxuICAgICAgJ0RlbGV0ZVNpbXVsYXRpb25BcHBsaWNhdGlvbicsXG4gICAgICAnRGVsZXRlV29ybGRUZW1wbGF0ZScsXG4gICAgICAnRGVyZWdpc3RlclJvYm90JyxcbiAgICAgICdSZWdpc3RlclJvYm90JyxcbiAgICAgICdSZXN0YXJ0U2ltdWxhdGlvbkpvYicsXG4gICAgICAnU3RhcnRTaW11bGF0aW9uSm9iQmF0Y2gnLFxuICAgICAgJ1N5bmNEZXBsb3ltZW50Sm9iJyxcbiAgICAgICdVcGRhdGVSb2JvdEFwcGxpY2F0aW9uJyxcbiAgICAgICdVcGRhdGVSb2JvdERlcGxveW1lbnQnLFxuICAgICAgJ1VwZGF0ZVNpbXVsYXRpb25BcHBsaWNhdGlvbicsXG4gICAgICAnVXBkYXRlV29ybGRUZW1wbGF0ZSdcbiAgICBdLFxuICAgIFJlYWQ6IFtcbiAgICAgICdCYXRjaERlc2NyaWJlU2ltdWxhdGlvbkpvYicsXG4gICAgICAnRGVzY3JpYmVEZXBsb3ltZW50Sm9iJyxcbiAgICAgICdEZXNjcmliZUZsZWV0JyxcbiAgICAgICdEZXNjcmliZVJvYm90JyxcbiAgICAgICdEZXNjcmliZVJvYm90QXBwbGljYXRpb24nLFxuICAgICAgJ0Rlc2NyaWJlU2ltdWxhdGlvbkFwcGxpY2F0aW9uJyxcbiAgICAgICdEZXNjcmliZVNpbXVsYXRpb25Kb2InLFxuICAgICAgJ0Rlc2NyaWJlU2ltdWxhdGlvbkpvYkJhdGNoJyxcbiAgICAgICdEZXNjcmliZVdvcmxkJyxcbiAgICAgICdEZXNjcmliZVdvcmxkRXhwb3J0Sm9iJyxcbiAgICAgICdEZXNjcmliZVdvcmxkR2VuZXJhdGlvbkpvYicsXG4gICAgICAnRGVzY3JpYmVXb3JsZFRlbXBsYXRlJyxcbiAgICAgICdHZXRXb3JsZFRlbXBsYXRlQm9keSdcbiAgICBdLFxuICAgIExpc3Q6IFtcbiAgICAgICdMaXN0RGVwbG95bWVudEpvYnMnLFxuICAgICAgJ0xpc3RGbGVldHMnLFxuICAgICAgJ0xpc3RSb2JvdEFwcGxpY2F0aW9ucycsXG4gICAgICAnTGlzdFJvYm90cycsXG4gICAgICAnTGlzdFNpbXVsYXRpb25BcHBsaWNhdGlvbnMnLFxuICAgICAgJ0xpc3RTaW11bGF0aW9uSm9iQmF0Y2hlcycsXG4gICAgICAnTGlzdFNpbXVsYXRpb25Kb2JzJyxcbiAgICAgICdMaXN0U3VwcG9ydGVkQXZhaWxhYmlsaXR5Wm9uZXMnLFxuICAgICAgJ0xpc3RUYWdzRm9yUmVzb3VyY2UnLFxuICAgICAgJ0xpc3RXb3JsZEV4cG9ydEpvYnMnLFxuICAgICAgJ0xpc3RXb3JsZEdlbmVyYXRpb25Kb2JzJyxcbiAgICAgICdMaXN0V29ybGRUZW1wbGF0ZXMnLFxuICAgICAgJ0xpc3RXb3JsZHMnXG4gICAgXSxcbiAgICBUYWdnaW5nOiBbXG4gICAgICAnVGFnUmVzb3VyY2UnLFxuICAgICAgJ1VudGFnUmVzb3VyY2UnXG4gICAgXVxuICB9O1xuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSByb2JvdEFwcGxpY2F0aW9uIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3JvYm9tYWtlci9sYXRlc3QvZGcvbWFuYWdpbmctcm9ib3QtYXBwbGljYXRpb25zLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGFwcGxpY2F0aW9uTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBhcHBsaWNhdGlvbk5hbWUuXG4gICAqIEBwYXJhbSBjcmVhdGVkT25FcG9jaCAtIElkZW50aWZpZXIgZm9yIHRoZSBjcmVhdGVkT25FcG9jaC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uUm9ib3RBcHBsaWNhdGlvbihhcHBsaWNhdGlvbk5hbWU6IHN0cmluZywgY3JlYXRlZE9uRXBvY2g6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTpyb2JvbWFrZXI6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OnJvYm90LWFwcGxpY2F0aW9uLyR7IGFwcGxpY2F0aW9uTmFtZSB9LyR7IGNyZWF0ZWRPbkVwb2NoIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBzaW11bGF0aW9uQXBwbGljYXRpb24gdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcm9ib21ha2VyL2xhdGVzdC9kZy9tYW5hZ2luZy1zaW11bGF0aW9uLWFwcGxpY2F0aW9ucy5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBhcHBsaWNhdGlvbk5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgYXBwbGljYXRpb25OYW1lLlxuICAgKiBAcGFyYW0gY3JlYXRlZE9uRXBvY2ggLSBJZGVudGlmaWVyIGZvciB0aGUgY3JlYXRlZE9uRXBvY2guXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvblNpbXVsYXRpb25BcHBsaWNhdGlvbihhcHBsaWNhdGlvbk5hbWU6IHN0cmluZywgY3JlYXRlZE9uRXBvY2g6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTpyb2JvbWFrZXI6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OnNpbXVsYXRpb24tYXBwbGljYXRpb24vJHsgYXBwbGljYXRpb25OYW1lIH0vJHsgY3JlYXRlZE9uRXBvY2ggfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHNpbXVsYXRpb25Kb2IgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcm9ib21ha2VyL2xhdGVzdC9kZy9zaW11bGF0aW9uLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHNpbXVsYXRpb25Kb2JJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBzaW11bGF0aW9uSm9iSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvblNpbXVsYXRpb25Kb2Ioc2ltdWxhdGlvbkpvYklkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06cm9ib21ha2VyOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpzaW11bGF0aW9uLWpvYi8keyBzaW11bGF0aW9uSm9iSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHNpbXVsYXRpb25Kb2JCYXRjaCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9yb2JvbWFrZXIvbGF0ZXN0L2RnL3NpbXVsYXRpb24tam9iLWJhdGNoLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHNpbXVsYXRpb25Kb2JCYXRjaElkIC0gSWRlbnRpZmllciBmb3IgdGhlIHNpbXVsYXRpb25Kb2JCYXRjaElkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25TaW11bGF0aW9uSm9iQmF0Y2goc2ltdWxhdGlvbkpvYkJhdGNoSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTpyb2JvbWFrZXI6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OnNpbXVsYXRpb24tam9iLWJhdGNoLyR7IHNpbXVsYXRpb25Kb2JCYXRjaElkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBkZXBsb3ltZW50Sm9iIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3JvYm9tYWtlci9sYXRlc3QvZGcvZGVwbG95bWVudC5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBkZXBsb3ltZW50Sm9iSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgZGVwbG95bWVudEpvYklkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25EZXBsb3ltZW50Sm9iKGRlcGxveW1lbnRKb2JJZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OnJvYm9tYWtlcjokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06ZGVwbG95bWVudC1qb2IvJHsgZGVwbG95bWVudEpvYklkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSByb2JvdCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9yb2JvbWFrZXIvbGF0ZXN0L2RnL2ZsZWV0cy5odG1sXG4gICAqXG4gICAqIEBwYXJhbSByb2JvdE5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgcm9ib3ROYW1lLlxuICAgKiBAcGFyYW0gY3JlYXRlZE9uRXBvY2ggLSBJZGVudGlmaWVyIGZvciB0aGUgY3JlYXRlZE9uRXBvY2guXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvblJvYm90KHJvYm90TmFtZTogc3RyaW5nLCBjcmVhdGVkT25FcG9jaDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OnJvYm9tYWtlcjokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06cm9ib3QvJHsgcm9ib3ROYW1lIH0vJHsgY3JlYXRlZE9uRXBvY2ggfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGRlcGxveW1lbnRGbGVldCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9yb2JvbWFrZXIvbGF0ZXN0L2RnL21hbmFnaW5nLXNpbXVsYXRpb24tYXBwbGljYXRpb25zLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGZsZWV0TmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBmbGVldE5hbWUuXG4gICAqIEBwYXJhbSBjcmVhdGVkT25FcG9jaCAtIElkZW50aWZpZXIgZm9yIHRoZSBjcmVhdGVkT25FcG9jaC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uRGVwbG95bWVudEZsZWV0KGZsZWV0TmFtZTogc3RyaW5nLCBjcmVhdGVkT25FcG9jaDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OnJvYm9tYWtlcjokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06ZGVwbG95bWVudC1mbGVldC8keyBmbGVldE5hbWUgfS8keyBjcmVhdGVkT25FcG9jaCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgd29ybGRHZW5lcmF0aW9uSm9iIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3JvYm9tYWtlci9sYXRlc3QvZGcvd29ybGRzLW1hbmFnaW5nLWdlbmVyYXRpb24tam9icy5odG1sXG4gICAqXG4gICAqIEBwYXJhbSB3b3JsZEdlbmVyYXRpb25Kb2JJZCAtIElkZW50aWZpZXIgZm9yIHRoZSB3b3JsZEdlbmVyYXRpb25Kb2JJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uV29ybGRHZW5lcmF0aW9uSm9iKHdvcmxkR2VuZXJhdGlvbkpvYklkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06cm9ib21ha2VyOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTp3b3JsZC1nZW5lcmF0aW9uLWpvYi8keyB3b3JsZEdlbmVyYXRpb25Kb2JJZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgd29ybGRFeHBvcnRKb2IgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcm9ib21ha2VyL2xhdGVzdC9kZy93b3JsZHMtbWFuYWdpbmctZXhwb3J0LWpvYnMuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gd29ybGRFeHBvcnRKb2JJZCAtIElkZW50aWZpZXIgZm9yIHRoZSB3b3JsZEV4cG9ydEpvYklkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25Xb3JsZEV4cG9ydEpvYih3b3JsZEV4cG9ydEpvYklkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06cm9ib21ha2VyOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTp3b3JsZC1leHBvcnQtam9iLyR7IHdvcmxkRXhwb3J0Sm9iSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHdvcmxkVGVtcGxhdGUgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcm9ib21ha2VyL2xhdGVzdC9kZy93b3JsZHMtbWFuYWdpbmctc2ltd29ybGQtdGVtcGxhdGVzLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHdvcmxkVGVtcGxhdGVKb2JJZCAtIElkZW50aWZpZXIgZm9yIHRoZSB3b3JsZFRlbXBsYXRlSm9iSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbldvcmxkVGVtcGxhdGUod29ybGRUZW1wbGF0ZUpvYklkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06cm9ib21ha2VyOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTp3b3JsZC10ZW1wbGF0ZS8keyB3b3JsZFRlbXBsYXRlSm9iSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHdvcmxkIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3JvYm9tYWtlci9sYXRlc3QvZGcvd29ybGRzLW1hbmFnaW5nLWdlbmVyYXRlZC13b3JsZHMuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gd29ybGRJZCAtIElkZW50aWZpZXIgZm9yIHRoZSB3b3JsZElkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25Xb3JsZCh3b3JsZElkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06cm9ib21ha2VyOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTp3b3JsZC8keyB3b3JsZElkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBiYXNlZCBvbiB0aGUgdGFncyB0aGF0IGFyZSBwYXNzZWQgaW4gdGhlIHJlcXVlc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3JvYm9tYWtlci9sYXRlc3QvZGcvdGFnZ2luZy1yZXNvdXJjZXMtaWFtLXBvbGljaWVzLmh0bWxcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0NyZWF0ZURlcGxveW1lbnRKb2IoKVxuICAgKiAtIC50b0NyZWF0ZUZsZWV0KClcbiAgICogLSAudG9DcmVhdGVSb2JvdCgpXG4gICAqIC0gLnRvQ3JlYXRlUm9ib3RBcHBsaWNhdGlvbigpXG4gICAqIC0gLnRvQ3JlYXRlU2ltdWxhdGlvbkFwcGxpY2F0aW9uKClcbiAgICogLSAudG9DcmVhdGVTaW11bGF0aW9uSm9iKClcbiAgICogLSAudG9DcmVhdGVXb3JsZEV4cG9ydEpvYigpXG4gICAqIC0gLnRvQ3JlYXRlV29ybGRHZW5lcmF0aW9uSm9iKClcbiAgICogLSAudG9DcmVhdGVXb3JsZFRlbXBsYXRlKClcbiAgICogLSAudG9TdGFydFNpbXVsYXRpb25Kb2JCYXRjaCgpXG4gICAqIC0gLnRvVGFnUmVzb3VyY2UoKVxuICAgKlxuICAgKiBAcGFyYW0gdGFnS2V5IFRoZSB0YWcga2V5IHRvIGNoZWNrXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzUmVxdWVzdFRhZyh0YWdLZXk6IHN0cmluZywgdmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpSZXF1ZXN0VGFnLyR7IHRhZ0tleSB9YCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYmFzZWQgb24gdGhlIHRhZ3MgYXNzb2NpYXRlZCB3aXRoIHRoZSByZXNvdXJjZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcm9ib21ha2VyL2xhdGVzdC9kZy90YWdnaW5nLXJlc291cmNlcy1pYW0tcG9saWNpZXMuaHRtbFxuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIHJvYm90QXBwbGljYXRpb25cbiAgICogLSBzaW11bGF0aW9uQXBwbGljYXRpb25cbiAgICogLSBzaW11bGF0aW9uSm9iXG4gICAqIC0gc2ltdWxhdGlvbkpvYkJhdGNoXG4gICAqIC0gZGVwbG95bWVudEpvYlxuICAgKiAtIHJvYm90XG4gICAqIC0gZGVwbG95bWVudEZsZWV0XG4gICAqIC0gd29ybGRHZW5lcmF0aW9uSm9iXG4gICAqIC0gd29ybGRFeHBvcnRKb2JcbiAgICogLSB3b3JsZFRlbXBsYXRlXG4gICAqIC0gd29ybGRcbiAgICpcbiAgICogQHBhcmFtIHRhZ0tleSBUaGUgdGFnIGtleSB0byBjaGVja1xuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1Jlc291cmNlVGFnKHRhZ0tleTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYXdzOlJlc291cmNlVGFnLyR7IHRhZ0tleSB9YCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYmFzZWQgb24gdGhlIHRhZyBrZXlzIHRoYXQgYXJlIHBhc3NlZCBpbiB0aGUgcmVxdWVzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcm9ib21ha2VyL2xhdGVzdC9kZy90YWdnaW5nLXJlc291cmNlcy1pYW0tcG9saWNpZXMuaHRtbFxuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQ3JlYXRlRGVwbG95bWVudEpvYigpXG4gICAqIC0gLnRvQ3JlYXRlRmxlZXQoKVxuICAgKiAtIC50b0NyZWF0ZVJvYm90KClcbiAgICogLSAudG9DcmVhdGVSb2JvdEFwcGxpY2F0aW9uKClcbiAgICogLSAudG9DcmVhdGVTaW11bGF0aW9uQXBwbGljYXRpb24oKVxuICAgKiAtIC50b0NyZWF0ZVNpbXVsYXRpb25Kb2IoKVxuICAgKiAtIC50b0NyZWF0ZVdvcmxkRXhwb3J0Sm9iKClcbiAgICogLSAudG9DcmVhdGVXb3JsZEdlbmVyYXRpb25Kb2IoKVxuICAgKiAtIC50b0NyZWF0ZVdvcmxkVGVtcGxhdGUoKVxuICAgKiAtIC50b1N0YXJ0U2ltdWxhdGlvbkpvYkJhdGNoKClcbiAgICogLSAudG9UYWdSZXNvdXJjZSgpXG4gICAqIC0gLnRvVW50YWdSZXNvdXJjZSgpXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzVGFnS2V5cyh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYXdzOlRhZ0tleXNgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW3JvYm9tYWtlcl0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYXdzcm9ib21ha2VyLmh0bWwpLlxuICAgKlxuICAgKi9cbiAgY29uc3RydWN0b3IocHJvcHM/OiBpYW0uUG9saWN5U3RhdGVtZW50UHJvcHMpIHtcbiAgICBzdXBlcihwcm9wcyk7XG4gIH1cbn1cbiJdfQ==