import { AccessLevelList } from '../../shared/access-level';
import { PolicyStatement, Operator } from '../../shared';
import { aws_iam as iam } from "aws-cdk-lib";
/**
 * Statement provider for service [s3express](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazons3express.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class S3express extends PolicyStatement {
    servicePrefix: string;
    /**
     * Grants permission to create a new access point
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifLocationName()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateAccessPoint.html
     */
    toCreateAccessPoint(): this;
    /**
     * Grants permission to create a new bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifLocationName()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_CreateBucket.html
     */
    toCreateBucket(): this;
    /**
     * Grants permission to Create Session token which is used for object APIs such as PutObject, GetObject, ect
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSessionMode()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifXAmzServerSideEncryption()
     * - .ifXAmzServerSideEncryptionAwsKmsKeyId()
     * - .ifAllAccessRestrictedToLocalZoneGroup()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_CreateSession.html
     */
    toCreateSession(): this;
    /**
     * Grants permission to delete the access point named in the URI
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPoint.html
     */
    toDeleteAccessPoint(): this;
    /**
     * Grants permission to delete the policy on a specified access point
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPointPolicy.html
     */
    toDeleteAccessPointPolicy(): this;
    /**
     * Grants permission to delete the scope configuration on a specified access point
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPointScope.html
     */
    toDeleteAccessPointScope(): this;
    /**
     * Grants permission to delete the bucket named in the URI
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucket.html
     */
    toDeleteBucket(): this;
    /**
     * Grants permission to delete the policy on a specified bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucketPolicy.html
     */
    toDeleteBucketPolicy(): this;
    /**
     * Grants permission to return configuration information about the specified access point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPoint.html
     */
    toGetAccessPoint(): this;
    /**
     * Grants permission to return the access point policy associated with the specified access point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointPolicy.html
     */
    toGetAccessPointPolicy(): this;
    /**
     * Grants permission to return the scope configuration associated with the specified access point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointScope.html
     */
    toGetAccessPointScope(): this;
    /**
     * Grants permission to return the policy of the specified bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketPolicy.html
     */
    toGetBucketPolicy(): this;
    /**
     * Grants permission to return the default encryption configuration for a directory bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketEncryption.html
     */
    toGetEncryptionConfiguration(): this;
    /**
     * Grants permission to return the lifecycle configuration information set on a directory bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketLifecycleConfiguration.html
     */
    toGetLifecycleConfiguration(): this;
    /**
     * Grants permission to list access points
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListAccessPointsForDirectoryBuckets.html
     */
    toListAccessPointsForDirectoryBuckets(): this;
    /**
     * Grants permission to list all directory buckets owned by the authenticated sender of the request
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListDirectoryBuckets.html
     */
    toListAllMyDirectoryBuckets(): this;
    /**
     * Grants permission to associate an access policy with a specified access point
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutAccessPointPolicy.html
     */
    toPutAccessPointPolicy(): this;
    /**
     * Grants permission to associate an access point with a specified access point scope configuration
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutAccessPointScope.html
     */
    toPutAccessPointScope(): this;
    /**
     * Grants permission to add or replace a bucket policy on a bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketPolicy.html
     */
    toPutBucketPolicy(): this;
    /**
     * Grants permission to set the encryption configuration for a directory bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketEncryption.html
     */
    toPutEncryptionConfiguration(): this;
    /**
     * Grants permission to create a new lifecycle configuration for the directory bucket or replace an existing lifecycle configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketLifecycleConfiguration.html
     */
    toPutLifecycleConfiguration(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type bucket to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-express-security-iam.html
     *
     * @param bucketName - Identifier for the bucketName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onBucket(bucketName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type accesspoint to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/access-points.html
     *
     * @param accessPointName - Identifier for the accessPointName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAccesspoint(accessPointName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters access by the network origin (Internet or VPC)
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/creating-access-points.html#access-points-policies
     *
     * Applies to actions:
     * - .toCreateAccessPoint()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointScope()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointScope()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointScope()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAccessPointNetworkOrigin(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by AWS Local Zone network border group(s) provided in this condition key
     *
     * Applies to actions:
     * - .toCreateSession()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAllAccessRestrictedToLocalZoneGroup(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the AWS Account ID that owns the access point
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/creating-access-points.html#access-points-policies
     *
     * Applies to actions:
     * - .toCreateAccessPoint()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointScope()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointScope()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointScope()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDataAccessPointAccount(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by an access point Amazon Resource Name (ARN)
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/creating-access-points.html#access-points-policies
     *
     * Applies to actions:
     * - .toCreateAccessPoint()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointScope()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointScope()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointScope()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifDataAccessPointArn(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by a specific Availability Zone ID
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/amazon-s3-express-zonal-policy-keys.html#example-location-name
     *
     * Applies to actions:
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLocationName(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the permission requested by Access Point Scope configuration, such as GetObject, PutObject
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPermissions(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the resource owner AWS account ID
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/amazon-s3-express-zonal-policy-keys.html#example-object-resource-account
     *
     * Applies to actions:
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     * - .toCreateSession()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointScope()
     * - .toDeleteBucket()
     * - .toDeleteBucketPolicy()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointScope()
     * - .toGetBucketPolicy()
     * - .toGetEncryptionConfiguration()
     * - .toGetLifecycleConfiguration()
     * - .toListAccessPointsForDirectoryBuckets()
     * - .toListAllMyDirectoryBuckets()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointScope()
     * - .toPutBucketPolicy()
     * - .toPutEncryptionConfiguration()
     * - .toPutLifecycleConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceAccount(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the permission requested by CreateSession API, such as ReadOnly and ReadWrite
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/amazon-s3-express-zonal-policy-keys.html#example-session-mode
     *
     * Applies to actions:
     * - .toCreateSession()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSessionMode(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the TLS version used by the client
     *
     * Applies to actions:
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     * - .toCreateSession()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointScope()
     * - .toDeleteBucket()
     * - .toDeleteBucketPolicy()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointScope()
     * - .toGetBucketPolicy()
     * - .toGetEncryptionConfiguration()
     * - .toGetLifecycleConfiguration()
     * - .toListAccessPointsForDirectoryBuckets()
     * - .toListAllMyDirectoryBuckets()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointScope()
     * - .toPutBucketPolicy()
     * - .toPutEncryptionConfiguration()
     * - .toPutLifecycleConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifTlsVersion(value: number | number[], operator?: Operator | string): this;
    /**
     * Filters access by authentication method
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     * - .toCreateSession()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointScope()
     * - .toDeleteBucket()
     * - .toDeleteBucketPolicy()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointScope()
     * - .toGetBucketPolicy()
     * - .toGetEncryptionConfiguration()
     * - .toGetLifecycleConfiguration()
     * - .toListAccessPointsForDirectoryBuckets()
     * - .toListAllMyDirectoryBuckets()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointScope()
     * - .toPutBucketPolicy()
     * - .toPutEncryptionConfiguration()
     * - .toPutLifecycleConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAuthType(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the age in milliseconds of the request signature
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .toCreateSession()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifSignatureAge(value: number | number[], operator?: Operator | string): this;
    /**
     * Filters access by the AWS Signature Version used on the request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     * - .toCreateSession()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointScope()
     * - .toDeleteBucket()
     * - .toDeleteBucketPolicy()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointScope()
     * - .toGetBucketPolicy()
     * - .toGetEncryptionConfiguration()
     * - .toGetLifecycleConfiguration()
     * - .toListAccessPointsForDirectoryBuckets()
     * - .toListAllMyDirectoryBuckets()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointScope()
     * - .toPutBucketPolicy()
     * - .toPutEncryptionConfiguration()
     * - .toPutLifecycleConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSignatureversion(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by unsigned content in your bucket
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     * - .toCreateSession()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointScope()
     * - .toDeleteBucket()
     * - .toDeleteBucketPolicy()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointScope()
     * - .toGetBucketPolicy()
     * - .toGetEncryptionConfiguration()
     * - .toGetLifecycleConfiguration()
     * - .toListAccessPointsForDirectoryBuckets()
     * - .toListAllMyDirectoryBuckets()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointScope()
     * - .toPutBucketPolicy()
     * - .toPutEncryptionConfiguration()
     * - .toPutLifecycleConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzContentSha256(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by server-side encryption
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-express-data-protection.html
     *
     * Applies to actions:
     * - .toCreateSession()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzServerSideEncryption(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by AWS KMS customer managed key for server-side encryption
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-express-UsingKMSEncryption.html#s3-express-require-sse-kms
     *
     * Applies to actions:
     * - .toCreateSession()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifXAmzServerSideEncryptionAwsKmsKeyId(value: string | string[], operator?: Operator | string): this;
    /**
     * Statement provider for service [s3express](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazons3express.html).
     *
     */
    constructor(props?: iam.PolicyStatementProps);
}
