"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Securityhub = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [securityhub](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssecurityhub.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Securityhub extends shared_1.PolicyStatement {
    /**
     * Grants permission to accept Security Hub invitations to become a member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_AcceptAdministratorInvitation.html
     */
    toAcceptAdministratorInvitation() {
        return this.to('AcceptAdministratorInvitation');
    }
    /**
     * Grants permission to accept Security Hub invitations to become a member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_AcceptInvitation.html
     */
    toAcceptInvitation() {
        return this.to('AcceptInvitation');
    }
    /**
     * Grants permission to delete one or more automation rules in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/automation-rules
     */
    toBatchDeleteAutomationRules() {
        return this.to('BatchDeleteAutomationRules');
    }
    /**
     * Grants permission to disable standards in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchDisableStandards.html
     */
    toBatchDisableStandards() {
        return this.to('BatchDisableStandards');
    }
    /**
     * Grants permission to enable standards in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchEnableStandards.html
     */
    toBatchEnableStandards() {
        return this.to('BatchEnableStandards');
    }
    /**
     * Grants permission to retrieve a list of details for automation rules from Security Hub based on rule Amazon Resource Names (ARNs)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/automation-rules
     */
    toBatchGetAutomationRules() {
        return this.to('BatchGetAutomationRules');
    }
    /**
     * Grants permission to retrieve information about configuration policies associated with a specific list of member accounts and organizational units of the calling account's organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchGetConfigurationPolicyAssociations.html
     */
    toBatchGetConfigurationPolicyAssociations() {
        return this.to('BatchGetConfigurationPolicyAssociations');
    }
    /**
     * Grants permission to get the enablement and compliance status of controls, the findings count for controls, and the overall security score for controls on the Security Hub console
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/iam-permissions-controls-standards.html
     */
    toBatchGetControlEvaluations() {
        return this.to('BatchGetControlEvaluations');
    }
    /**
     * Grants permission to get details about specific security controls identified by ID or ARN
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - securityhub:DescribeStandardsControls
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchGetSecurityControls.html
     */
    toBatchGetSecurityControls() {
        return this.to('BatchGetSecurityControls');
    }
    /**
     * Grants permission to get the enablement status of a batch of security controls in standards
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - securityhub:DescribeStandardsControls
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchGetStandardsControlAssociations.html
     */
    toBatchGetStandardsControlAssociations() {
        return this.to('BatchGetStandardsControlAssociations');
    }
    /**
     * Grants permission to import findings into Security Hub from an integrated product
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTargetAccount()
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchImportFindings.html
     */
    toBatchImportFindings() {
        return this.to('BatchImportFindings');
    }
    /**
     * Grants permission to update one or more automation rules from Security Hub based on rule Amazon Resource Names (ARNs) and input parameters
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/automation-rules
     */
    toBatchUpdateAutomationRules() {
        return this.to('BatchUpdateAutomationRules');
    }
    /**
     * Grants permission to update customer-controlled fields for a selected set of Security Hub findings
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifASFFSyntaxPath()
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchUpdateFindings.html
     */
    toBatchUpdateFindings() {
        return this.to('BatchUpdateFindings');
    }
    /**
     * Grants permission to update the enablement status of a batch of security controls in standards
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - securityhub:UpdateStandardsControl
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchUpdateStandardsControlAssociations.html
     */
    toBatchUpdateStandardsControlAssociations() {
        return this.to('BatchUpdateStandardsControlAssociations');
    }
    /**
     * Grants permission to create custom actions in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_CreateActionTarget.html
     */
    toCreateActionTarget() {
        return this.to('CreateActionTarget');
    }
    /**
     * Grants permission to create an automation rule based on input parameters
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/automation-rules
     */
    toCreateAutomationRule() {
        return this.to('CreateAutomationRule');
    }
    /**
     * Grants permission to create a configuration policy to manage organization member settings in Security Hub
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_CreateConfigurationPolicy.html
     */
    toCreateConfigurationPolicy() {
        return this.to('CreateConfigurationPolicy');
    }
    /**
     * Grants permission to create a finding aggregator, which contains the cross-Region finding aggregation configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateFindingAggregator.html
     */
    toCreateFindingAggregator() {
        return this.to('CreateFindingAggregator');
    }
    /**
     * Grants permission to create insights in Security Hub. Insights are collections of related findings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_CreateInsight.html
     */
    toCreateInsight() {
        return this.to('CreateInsight');
    }
    /**
     * Grants permission to create member accounts in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_CreateMembers.html
     */
    toCreateMembers() {
        return this.to('CreateMembers');
    }
    /**
     * Grants permission to decline Security Hub invitations to become a member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeclineInvitations.html
     */
    toDeclineInvitations() {
        return this.to('DeclineInvitations');
    }
    /**
     * Grants permission to delete custom actions in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeleteActionTarget.html
     */
    toDeleteActionTarget() {
        return this.to('DeleteActionTarget');
    }
    /**
     * Grants permission to delete an existing configuration policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeleteConfigurationPolicy.html
     */
    toDeleteConfigurationPolicy() {
        return this.to('DeleteConfigurationPolicy');
    }
    /**
     * Grants permission to delete a finding aggregator, which disables finding aggregation across Regions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeleteFindingAggregator.html
     */
    toDeleteFindingAggregator() {
        return this.to('DeleteFindingAggregator');
    }
    /**
     * Grants permission to delete insights from Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeleteInsight.html
     */
    toDeleteInsight() {
        return this.to('DeleteInsight');
    }
    /**
     * Grants permission to delete Security Hub invitations to become a member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeleteInvitations.html
     */
    toDeleteInvitations() {
        return this.to('DeleteInvitations');
    }
    /**
     * Grants permission to delete Security Hub member accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeleteMembers.html
     */
    toDeleteMembers() {
        return this.to('DeleteMembers');
    }
    /**
     * Grants permission to retrieve a list of custom actions using the API
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeActionTargets.html
     */
    toDescribeActionTargets() {
        return this.to('DescribeActionTargets');
    }
    /**
     * Grants permission to retrieve information about the hub resource in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeHub.html
     */
    toDescribeHub() {
        return this.to('DescribeHub');
    }
    /**
     * Grants permission to describe the organization configuration for Security Hub
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeOrganizationConfiguration.html
     */
    toDescribeOrganizationConfiguration() {
        return this.to('DescribeOrganizationConfiguration');
    }
    /**
     * Grants permission to retrieve information about the available Security Hub product integrations
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeProducts.html
     */
    toDescribeProducts() {
        return this.to('DescribeProducts');
    }
    /**
     * Grants permission to retrieve information about Security Hub standards
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeStandards.html
     */
    toDescribeStandards() {
        return this.to('DescribeStandards');
    }
    /**
     * Grants permission to retrieve information about Security Hub standards controls
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeStandardsControls.html
     */
    toDescribeStandardsControls() {
        return this.to('DescribeStandardsControls');
    }
    /**
     * Grants permission to disable the findings importing for a Security Hub integrated product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DisableImportFindingsForProduct.html
     */
    toDisableImportFindingsForProduct() {
        return this.to('DisableImportFindingsForProduct');
    }
    /**
     * Grants permission to remove the Security Hub administrator account for your organization
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - organizations:DescribeOrganization
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DisableOrganizationAdminAccount.html
     */
    toDisableOrganizationAdminAccount() {
        return this.to('DisableOrganizationAdminAccount');
    }
    /**
     * Grants permission to disable Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DisableSecurityHub.html
     */
    toDisableSecurityHub() {
        return this.to('DisableSecurityHub');
    }
    /**
     * Grants permission to a Security Hub member account to disassociate from the associated administrator account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DisassociateFromAdministratorAccount.html
     */
    toDisassociateFromAdministratorAccount() {
        return this.to('DisassociateFromAdministratorAccount');
    }
    /**
     * Grants permission to a Security Hub member account to disassociate from the associated master account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DisassociateFromMasterAccount.html
     */
    toDisassociateFromMasterAccount() {
        return this.to('DisassociateFromMasterAccount');
    }
    /**
     * Grants permission to disassociate Security Hub member accounts from the associated administrator account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DisassociateMembers.html
     */
    toDisassociateMembers() {
        return this.to('DisassociateMembers');
    }
    /**
     * Grants permission to enable the findings importing for a Security Hub integrated product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_EnableImportFindingsForProduct.html
     */
    toEnableImportFindingsForProduct() {
        return this.to('EnableImportFindingsForProduct');
    }
    /**
     * Grants permission to designate a Security Hub administrator account for your organization
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - organizations:DescribeOrganization
     * - organizations:EnableAWSServiceAccess
     * - organizations:RegisterDelegatedAdministrator
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_EnableOrganizationAdminAccount.html
     */
    toEnableOrganizationAdminAccount() {
        return this.to('EnableOrganizationAdminAccount');
    }
    /**
     * Grants permission to enable Security Hub
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_EnableSecurityHub.html
     */
    toEnableSecurityHub() {
        return this.to('EnableSecurityHub');
    }
    /**
     * Grants permission to retrieve insight results by providing a set of filters instead of an insight ARN
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetAdhocInsightResults.html
     */
    toGetAdhocInsightResults() {
        return this.to('GetAdhocInsightResults');
    }
    /**
     * Grants permission to retrieve details about the Security Hub administrator account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetAdministratorAccount.html
     */
    toGetAdministratorAccount() {
        return this.to('GetAdministratorAccount');
    }
    /**
     * Grants permission to get a complete overview of one configuration policy created by the calling account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetConfigurationPolicy.html
     */
    toGetConfigurationPolicy() {
        return this.to('GetConfigurationPolicy');
    }
    /**
     * Grants permission to retrieve information about a configuration policy associated with a member account or organizational unit of the calling account's organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetConfigurationPolicyAssociation.html
     */
    toGetConfigurationPolicyAssociation() {
        return this.to('GetConfigurationPolicyAssociation');
    }
    /**
     * Grants permission to retrieve a security score and counts of finding and control statuses for a security standard
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetControlFindingSummary.html
     */
    toGetControlFindingSummary() {
        return this.to('GetControlFindingSummary');
    }
    /**
     * Grants permission to retrieve a list of the standards that are enabled in Security Hub
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetEnabledStandards.html
     */
    toGetEnabledStandards() {
        return this.to('GetEnabledStandards');
    }
    /**
     * Grants permission to retrieve details for a finding aggregator, which configures finding aggregation across Regions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetFindingAggregator.html
     */
    toGetFindingAggregator() {
        return this.to('GetFindingAggregator');
    }
    /**
     * Grants permission to retrieve a list of finding history from Security Hub
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetFindingHistory.html
     */
    toGetFindingHistory() {
        return this.to('GetFindingHistory');
    }
    /**
     * Grants permission to retrieve a list of findings from Security Hub
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetFindings.html
     */
    toGetFindings() {
        return this.to('GetFindings');
    }
    /**
     * Grants permission to retrieve the end date for an account's free trial of Security Hub
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetFreeTrialEndDate.html
     */
    toGetFreeTrialEndDate() {
        return this.to('GetFreeTrialEndDate');
    }
    /**
     * Grants permission to retrieve information about Security Hub usage during the free trial period
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetFreeTrialUsage.html
     */
    toGetFreeTrialUsage() {
        return this.to('GetFreeTrialUsage');
    }
    /**
     * Grants permission to retrieve an insight finding trend from Security Hub in order to generate a graph
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetInsightFindingTrend.html
     */
    toGetInsightFindingTrend() {
        return this.to('GetInsightFindingTrend');
    }
    /**
     * Grants permission to retrieve insight results from Security Hub
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetInsightResults.html
     */
    toGetInsightResults() {
        return this.to('GetInsightResults');
    }
    /**
     * Grants permission to retrieve Security Hub insights
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetInsights.html
     */
    toGetInsights() {
        return this.to('GetInsights');
    }
    /**
     * Grants permission to retrieve the count of Security Hub membership invitations sent to the account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetInvitationsCount.html
     */
    toGetInvitationsCount() {
        return this.to('GetInvitationsCount');
    }
    /**
     * Grants permission to retrieve details about the Security Hub master account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetMasterAccount.html
     */
    toGetMasterAccount() {
        return this.to('GetMasterAccount');
    }
    /**
     * Grants permission to retrieve the details of Security Hub member accounts
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetMembers.html
     */
    toGetMembers() {
        return this.to('GetMembers');
    }
    /**
     * Grants permission to get the definition details of a specific security control identified by ID
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - securityhub:DescribeStandardsControls
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetSecurityControlDefinition.html
     */
    toGetSecurityControlDefinition() {
        return this.to('GetSecurityControlDefinition');
    }
    /**
     * Grants permission to retrieve information about Security Hub usage by accounts
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetUsage.html
     */
    toGetUsage() {
        return this.to('GetUsage');
    }
    /**
     * Grants permission to invite other AWS accounts to become Security Hub member accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_InviteMembers.html
     */
    toInviteMembers() {
        return this.to('InviteMembers');
    }
    /**
     * Grants permission to retrieve a list of automation rules and their metadata for the calling account from Security Hub
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/automation-rules
     */
    toListAutomationRules() {
        return this.to('ListAutomationRules');
    }
    /**
     * Grants permission to list the summaries of all configuration policies created by the calling account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListConfigurationPolicies.html
     */
    toListConfigurationPolicies() {
        return this.to('ListConfigurationPolicies');
    }
    /**
     * Grants permission to retrieve information about all configuration policies associationed with all member accounts and organizational units of the calling account's organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListConfigurationPolicyAssociations.html
     */
    toListConfigurationPolicyAssociations() {
        return this.to('ListConfigurationPolicyAssociations');
    }
    /**
     * Grants permission to retrieve a list of controls for a standard, including the control IDs, statuses and finding counts
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListControlEvaluationSummaries.html
     */
    toListControlEvaluationSummaries() {
        return this.to('ListControlEvaluationSummaries');
    }
    /**
     * Grants permission to retrieve the Security Hub integrated products that are currently enabled
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListEnabledProductsForImport.html
     */
    toListEnabledProductsForImport() {
        return this.to('ListEnabledProductsForImport');
    }
    /**
     * Grants permission to retrieve a list of finding aggregators, which contain the cross-Region finding aggregation configuration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateFindingAggregator.html
     */
    toListFindingAggregators() {
        return this.to('ListFindingAggregators');
    }
    /**
     * Grants permission to retrieve the Security Hub invitations sent to the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListInvitations.html
     */
    toListInvitations() {
        return this.to('ListInvitations');
    }
    /**
     * Grants permission to retrieve details about Security Hub member accounts associated with the administrator account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListMembers.html
     */
    toListMembers() {
        return this.to('ListMembers');
    }
    /**
     * Grants permission to list the Security Hub administrator accounts for your organization
     *
     * Access Level: List
     *
     * Dependent actions:
     * - organizations:DescribeOrganization
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListOrganizationAdminAccounts.html
     */
    toListOrganizationAdminAccounts() {
        return this.to('ListOrganizationAdminAccounts');
    }
    /**
     * Grants permission to retrieve a list of security control definitions, which contain details for security controls in the current region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListSecurityControlDefinitions.html
     */
    toListSecurityControlDefinitions() {
        return this.to('ListSecurityControlDefinitions');
    }
    /**
     * Grants permission to list the enablement status of a security control in standards
     *
     * Access Level: List
     *
     * Dependent actions:
     * - securityhub:DescribeStandardsControls
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListStandardsControlAssociations.html
     */
    toListStandardsControlAssociations() {
        return this.to('ListStandardsControlAssociations');
    }
    /**
     * Grants permission to list of tags associated with a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to use a custom action to send Security Hub findings to Amazon EventBridge
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_SendFindingEvents.html
     */
    toSendFindingEvents() {
        return this.to('SendFindingEvents');
    }
    /**
     * Grants permission to use a custom action to send Security Hub insights to Amazon EventBridge
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_SendInsightEvents.html
     */
    toSendInsightEvents() {
        return this.to('SendInsightEvents');
    }
    /**
     * Grants permission to associate a configuration policy with a member account or organizational unit in the calling account's organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_StartConfigurationPolicyAssociation.html
     */
    toStartConfigurationPolicyAssociation() {
        return this.to('StartConfigurationPolicyAssociation');
    }
    /**
     * Grants permission to remove a configuration policy association from a member account or organizational unit in the calling account's organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_StartConfigurationPolicyDisassociation.html
     */
    toStartConfigurationPolicyDisassociation() {
        return this.to('StartConfigurationPolicyDisassociation');
    }
    /**
     * Grants permission to add tags to a Security Hub resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from a Security Hub resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update custom actions in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateActionTarget.html
     */
    toUpdateActionTarget() {
        return this.to('UpdateActionTarget');
    }
    /**
     * Grants permission to update an existing configuration policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateConfigurationPolicy.html
     */
    toUpdateConfigurationPolicy() {
        return this.to('UpdateConfigurationPolicy');
    }
    /**
     * Grants permission to update a finding aggregator, which contains the cross-Region finding aggregation configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateFindingAggregator.html
     */
    toUpdateFindingAggregator() {
        return this.to('UpdateFindingAggregator');
    }
    /**
     * Grants permission to update Security Hub findings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateFindings.html
     */
    toUpdateFindings() {
        return this.to('UpdateFindings');
    }
    /**
     * Grants permission to update insights in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateInsight.html
     */
    toUpdateInsight() {
        return this.to('UpdateInsight');
    }
    /**
     * Grants permission to update the organization configuration for Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateOrganizationConfiguration.html
     */
    toUpdateOrganizationConfiguration() {
        return this.to('UpdateOrganizationConfiguration');
    }
    /**
     * Grants permission to update properties of a specific security control identified by ID or ARN
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - securityhub:UpdateStandardsControl
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateSecurityControl.html
     */
    toUpdateSecurityControl() {
        return this.to('UpdateSecurityControl');
    }
    /**
     * Grants permission to update Security Hub configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateSecurityHubConfiguration.html
     */
    toUpdateSecurityHubConfiguration() {
        return this.to('UpdateSecurityHubConfiguration');
    }
    /**
     * Grants permission to update Security Hub standards controls
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateStandardsControl.html
     */
    toUpdateStandardsControl() {
        return this.to('UpdateStandardsControl');
    }
    /**
     * Adds a resource of type hub to the statement
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-access.html#resources
     *
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onHub(account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:securityhub:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:hub/default`);
    }
    /**
     * Adds a resource of type product to the statement
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-access.html#resources
     *
     * @param company - Identifier for the company.
     * @param productId - Identifier for the productId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onProduct(company, productId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:securityhub:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:product/${company}/${productId}`);
    }
    /**
     * Adds a resource of type finding-aggregator to the statement
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-access.html#resources
     *
     * @param findingAggregatorId - Identifier for the findingAggregatorId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onFindingAggregator(findingAggregatorId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:securityhub:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:finding-aggregator/${findingAggregatorId}`);
    }
    /**
     * Adds a resource of type automation-rule to the statement
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/automation-rules
     *
     * @param automationRuleId - Identifier for the automationRuleId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAutomationRule(automationRuleId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:securityhub:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:automation-rule/${automationRuleId}`);
    }
    /**
     * Adds a resource of type configuration-policy to the statement
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/central-configuration-intro.html
     *
     * @param configurationPolicyId - Identifier for the configurationPolicyId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onConfigurationPolicy(configurationPolicyId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:securityhub:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:configuration-policy/${configurationPolicyId}`);
    }
    /**
     * Filters access by actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateAutomationRule()
     * - .toCreateConfigurationPolicy()
     * - .toEnableSecurityHub()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - hub
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by actions based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateAutomationRule()
     * - .toCreateConfigurationPolicy()
     * - .toEnableSecurityHub()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the specified fields and values in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-asffsyntaxpath
     *
     * Applies to actions:
     * - .toBatchUpdateFindings()
     *
     * @param aSFFSyntaxPath The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifASFFSyntaxPath(aSFFSyntaxPath, value, operator) {
        return this.if(`ASFFSyntaxPath/${aSFFSyntaxPath}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the AwsAccountId field that is specified in the request
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-access.html#conditions
     *
     * Applies to actions:
     * - .toBatchImportFindings()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTargetAccount(value, operator) {
        return this.if(`TargetAccount`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [securityhub](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssecurityhub.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'securityhub';
        this.accessLevelList = {
            Write: [
                'AcceptAdministratorInvitation',
                'AcceptInvitation',
                'BatchDeleteAutomationRules',
                'BatchDisableStandards',
                'BatchEnableStandards',
                'BatchImportFindings',
                'BatchUpdateAutomationRules',
                'BatchUpdateFindings',
                'BatchUpdateStandardsControlAssociations',
                'CreateActionTarget',
                'CreateAutomationRule',
                'CreateConfigurationPolicy',
                'CreateFindingAggregator',
                'CreateInsight',
                'CreateMembers',
                'DeclineInvitations',
                'DeleteActionTarget',
                'DeleteConfigurationPolicy',
                'DeleteFindingAggregator',
                'DeleteInsight',
                'DeleteInvitations',
                'DeleteMembers',
                'DisableImportFindingsForProduct',
                'DisableOrganizationAdminAccount',
                'DisableSecurityHub',
                'DisassociateFromAdministratorAccount',
                'DisassociateFromMasterAccount',
                'DisassociateMembers',
                'EnableImportFindingsForProduct',
                'EnableOrganizationAdminAccount',
                'EnableSecurityHub',
                'InviteMembers',
                'StartConfigurationPolicyAssociation',
                'StartConfigurationPolicyDisassociation',
                'UpdateActionTarget',
                'UpdateConfigurationPolicy',
                'UpdateFindingAggregator',
                'UpdateFindings',
                'UpdateInsight',
                'UpdateOrganizationConfiguration',
                'UpdateSecurityControl',
                'UpdateSecurityHubConfiguration',
                'UpdateStandardsControl'
            ],
            Read: [
                'BatchGetAutomationRules',
                'BatchGetConfigurationPolicyAssociations',
                'BatchGetControlEvaluations',
                'BatchGetSecurityControls',
                'BatchGetStandardsControlAssociations',
                'DescribeActionTargets',
                'DescribeHub',
                'DescribeOrganizationConfiguration',
                'DescribeProducts',
                'DescribeStandards',
                'DescribeStandardsControls',
                'GetAdhocInsightResults',
                'GetAdministratorAccount',
                'GetConfigurationPolicy',
                'GetConfigurationPolicyAssociation',
                'GetControlFindingSummary',
                'GetFindingAggregator',
                'GetFindingHistory',
                'GetFindings',
                'GetFreeTrialEndDate',
                'GetFreeTrialUsage',
                'GetInsightFindingTrend',
                'GetInsightResults',
                'GetInvitationsCount',
                'GetMasterAccount',
                'GetMembers',
                'GetSecurityControlDefinition',
                'GetUsage',
                'ListControlEvaluationSummaries',
                'ListTagsForResource',
                'SendFindingEvents',
                'SendInsightEvents'
            ],
            List: [
                'GetEnabledStandards',
                'GetInsights',
                'ListAutomationRules',
                'ListConfigurationPolicies',
                'ListConfigurationPolicyAssociations',
                'ListEnabledProductsForImport',
                'ListFindingAggregators',
                'ListInvitations',
                'ListMembers',
                'ListOrganizationAdminAccounts',
                'ListSecurityControlDefinitions',
                'ListStandardsControlAssociations'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Securityhub = Securityhub;
//# sourceMappingURL=data:application/json;base64,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