"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Serverlessrepo = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [serverlessrepo](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsserverlessapplicationrepository.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Serverlessrepo extends shared_1.PolicyStatement {
    /**
     * Grants permission to create an application, optionally including an AWS SAM file to create the first application version in the same call
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/serverlessrepo/latest/devguide/applications.html
     */
    toCreateApplication() {
        return this.to('CreateApplication');
    }
    /**
     * Grants permission to create an application version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/serverlessrepo/latest/devguide/applications-applicationid-versions-semanticversion.html
     */
    toCreateApplicationVersion() {
        return this.to('CreateApplicationVersion');
    }
    /**
     * Grants permission to create an AWS CloudFormation ChangeSet for the given application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationType()
     *
     * https://docs.aws.amazon.com/serverlessrepo/latest/devguide/applications-applicationid-changesets.html
     */
    toCreateCloudFormationChangeSet() {
        return this.to('CreateCloudFormationChangeSet');
    }
    /**
     * Grants permission to create an AWS CloudFormation template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationType()
     *
     * https://docs.aws.amazon.com/serverlessrepo/latest/devguide/applications-applicationid-templates.html
     */
    toCreateCloudFormationTemplate() {
        return this.to('CreateCloudFormationTemplate');
    }
    /**
     * Grants permission to delete the specified application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/serverlessrepo/latest/devguide/applications-applicationid.html
     */
    toDeleteApplication() {
        return this.to('DeleteApplication');
    }
    /**
     * Grants permission to get the specified application
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApplicationType()
     *
     * https://docs.aws.amazon.com/serverlessrepo/latest/devguide/applications-applicationid.html
     */
    toGetApplication() {
        return this.to('GetApplication');
    }
    /**
     * Grants permission to get the policy for the specified application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/serverlessrepo/latest/devguide/applications-applicationid-policy.html
     */
    toGetApplicationPolicy() {
        return this.to('GetApplicationPolicy');
    }
    /**
     * Grants permission to get the specified AWS CloudFormation template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/serverlessrepo/latest/devguide/applications-applicationid-templates-templateid.html
     */
    toGetCloudFormationTemplate() {
        return this.to('GetCloudFormationTemplate');
    }
    /**
     * Grants permission to retrieve the list of applications nested in the containing application
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApplicationType()
     *
     * https://docs.aws.amazon.com/serverlessrepo/latest/devguide/applications-applicationid-dependencies.html
     */
    toListApplicationDependencies() {
        return this.to('ListApplicationDependencies');
    }
    /**
     * Grants permission to list versions for the specified application owned by the requester
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApplicationType()
     *
     * https://docs.aws.amazon.com/serverlessrepo/latest/devguide/applications-applicationid-versions.html
     */
    toListApplicationVersions() {
        return this.to('ListApplicationVersions');
    }
    /**
     * Grants permission to list applications owned by the requester
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/serverlessrepo/latest/devguide/applications.html
     */
    toListApplications() {
        return this.to('ListApplications');
    }
    /**
     * Grants permission to put the policy for the specified application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/serverlessrepo/latest/devguide/applications-applicationid-policy.html
     */
    toPutApplicationPolicy() {
        return this.to('PutApplicationPolicy');
    }
    /**
     * Grants permission to get all applications authorized for this user
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApplicationType()
     *
     * https://docs.aws.amazon.com/serverlessrepo/latest/devguide/applications-applicationid.html
     */
    toSearchApplications() {
        return this.to('SearchApplications');
    }
    /**
     * Grants permission to unshare the specified application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/serverlessrepo/latest/devguide/applications-applicationid.html
     */
    toUnshareApplication() {
        return this.to('UnshareApplication');
    }
    /**
     * Grants permission to update meta-data of the application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/serverlessrepo/latest/devguide/applications-applicationid.html
     */
    toUpdateApplication() {
        return this.to('UpdateApplication');
    }
    /**
     * Adds a resource of type applications to the statement
     *
     * https://docs.aws.amazon.com/serverlessrepo/latest/devguide/applications.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onApplications(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:serverlessrepo:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:applications/${resourceId}`);
    }
    /**
     * Filters access by application type
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/applications.html
     *
     * Applies to actions:
     * - .toCreateCloudFormationChangeSet()
     * - .toCreateCloudFormationTemplate()
     * - .toGetApplication()
     * - .toListApplicationDependencies()
     * - .toListApplicationVersions()
     * - .toSearchApplications()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifApplicationType(value, operator) {
        return this.if(`applicationType`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [serverlessrepo](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsserverlessapplicationrepository.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'serverlessrepo';
        this.accessLevelList = {
            Write: [
                'CreateApplication',
                'CreateApplicationVersion',
                'CreateCloudFormationChangeSet',
                'CreateCloudFormationTemplate',
                'DeleteApplication',
                'PutApplicationPolicy',
                'UnshareApplication',
                'UpdateApplication'
            ],
            Read: [
                'GetApplication',
                'GetApplicationPolicy',
                'GetCloudFormationTemplate',
                'SearchApplications'
            ],
            List: [
                'ListApplicationDependencies',
                'ListApplicationVersions',
                'ListApplications'
            ]
        };
    }
}
exports.Serverlessrepo = Serverlessrepo;
//# sourceMappingURL=data:application/json;base64,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