"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Shield = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [shield](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsshield.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Shield extends shared_1.PolicyStatement {
    /**
     * Grants permission to authorize the DDoS Response team to access the specified Amazon S3 bucket containing your flow logs
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetBucketPolicy
     * - s3:PutBucketPolicy
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_AssociateDRTLogBucket.html
     */
    toAssociateDRTLogBucket() {
        return this.to('AssociateDRTLogBucket');
    }
    /**
     * Grants permission to authorize the DDoS Response team using the specified role, to access your AWS account to assist with DDoS attack mitigation during potential attacks
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:GetRole
     * - iam:ListAttachedRolePolicies
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_AssociateDRTRole.html
     */
    toAssociateDRTRole() {
        return this.to('AssociateDRTRole');
    }
    /**
     * Grants permission to add health-based detection to the Shield Advanced protection for a resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - route53:GetHealthCheck
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_AssociateHealthCheck.html
     */
    toAssociateHealthCheck() {
        return this.to('AssociateHealthCheck');
    }
    /**
     * Grants permission to initialize proactive engagement and set the list of contacts for the DDoS Response Team (DRT) to use
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_AssociateProactiveEngagementDetails.html
     */
    toAssociateProactiveEngagementDetails() {
        return this.to('AssociateProactiveEngagementDetails');
    }
    /**
     * Grants permission to activate DDoS protection service for a given resource ARN
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_CreateProtection.html
     */
    toCreateProtection() {
        return this.to('CreateProtection');
    }
    /**
     * Grants permission to create a grouping of protected resources so they can be handled as a collective
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_CreateProtectionGroup.html
     */
    toCreateProtectionGroup() {
        return this.to('CreateProtectionGroup');
    }
    /**
     * Grants permission to activate subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_CreateSubscription.html
     */
    toCreateSubscription() {
        return this.to('CreateSubscription');
    }
    /**
     * Grants permission to delete an existing protection
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DeleteProtection.html
     */
    toDeleteProtection() {
        return this.to('DeleteProtection');
    }
    /**
     * Grants permission to remove the specified protection group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DeleteProtectionGroup.html
     */
    toDeleteProtectionGroup() {
        return this.to('DeleteProtectionGroup');
    }
    /**
     * Grants permission to deactivate subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DeleteSubscription.html
     */
    toDeleteSubscription() {
        return this.to('DeleteSubscription');
    }
    /**
     * Grants permission to get attack details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DescribeAttack.html
     */
    toDescribeAttack() {
        return this.to('DescribeAttack');
    }
    /**
     * Grants permission to describe information about the number and type of attacks AWS Shield has detected in the last year
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DescribeAttackStatistics.html
     */
    toDescribeAttackStatistics() {
        return this.to('DescribeAttackStatistics');
    }
    /**
     * Grants permission to describe the current role and list of Amazon S3 log buckets used by the DDoS Response team to access your AWS account while assisting with attack mitigation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DescribeDRTAccess.html
     */
    toDescribeDRTAccess() {
        return this.to('DescribeDRTAccess');
    }
    /**
     * Grants permission to list the email addresses that the DRT can use to contact you during a suspected attack
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DescribeEmergencyContactSettings.html
     */
    toDescribeEmergencyContactSettings() {
        return this.to('DescribeEmergencyContactSettings');
    }
    /**
     * Grants permission to get protection details
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DescribeProtection.html
     */
    toDescribeProtection() {
        return this.to('DescribeProtection');
    }
    /**
     * Grants permission to describe the specification for the specified protection group
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DescribeProtectionGroup.html
     */
    toDescribeProtectionGroup() {
        return this.to('DescribeProtectionGroup');
    }
    /**
     * Grants permission to get subscription details, such as start time
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DescribeSubscription.html
     */
    toDescribeSubscription() {
        return this.to('DescribeSubscription');
    }
    /**
     * Grants permission to disable application layer automatic response for Shield Advanced protection for a resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DisableApplicationLayerAutomaticResponse.html
     */
    toDisableApplicationLayerAutomaticResponse() {
        return this.to('DisableApplicationLayerAutomaticResponse');
    }
    /**
     * Grants permission to remove authorization from the DDoS Response Team (DRT) to notify contacts about escalations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DisableProactiveEngagement.html
     */
    toDisableProactiveEngagement() {
        return this.to('DisableProactiveEngagement');
    }
    /**
     * Grants permission to remove the DDoS Response team's access to the specified Amazon S3 bucket containing your flow logs
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:DeleteBucketPolicy
     * - s3:GetBucketPolicy
     * - s3:PutBucketPolicy
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DisassociateDRTLogBucket.html
     */
    toDisassociateDRTLogBucket() {
        return this.to('DisassociateDRTLogBucket');
    }
    /**
     * Grants permission to remove the DDoS Response team's access to your AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DisassociateDRTRole.html
     */
    toDisassociateDRTRole() {
        return this.to('DisassociateDRTRole');
    }
    /**
     * Grants permission to remove health-based detection from the Shield Advanced protection for a resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_DisassociateHealthCheck.html
     */
    toDisassociateHealthCheck() {
        return this.to('DisassociateHealthCheck');
    }
    /**
     * Grants permission to enable application layer automatic response for Shield Advanced protection for a resource
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - apprunner:DescribeWebAclForService
     * - cloudfront:GetDistribution
     * - cognito-idp:GetWebACLForResource
     * - ec2:GetVerifiedAccessInstanceWebAcl
     * - iam:CreateServiceLinkedRole
     * - iam:GetRole
     * - wafv2:GetWebACL
     * - wafv2:GetWebACLForResource
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_EnableApplicationLayerAutomaticResponse.html
     */
    toEnableApplicationLayerAutomaticResponse() {
        return this.to('EnableApplicationLayerAutomaticResponse');
    }
    /**
     * Grants permission to authorize the DDoS Response Team (DRT) to use email and phone to notify contacts about escalations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_EnableProactiveEngagement.html
     */
    toEnableProactiveEngagement() {
        return this.to('EnableProactiveEngagement');
    }
    /**
     * Grants permission to get subscription state
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_GetSubscriptionState.html
     */
    toGetSubscriptionState() {
        return this.to('GetSubscriptionState');
    }
    /**
     * Grants permission to list all existing attacks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_ListAttacks.html
     */
    toListAttacks() {
        return this.to('ListAttacks');
    }
    /**
     * Grants permission to retrieve the protection groups for the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_ListProtectionGroups.html
     */
    toListProtectionGroups() {
        return this.to('ListProtectionGroups');
    }
    /**
     * Grants permission to list all existing protections
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_ListProtections.html
     */
    toListProtections() {
        return this.to('ListProtections');
    }
    /**
     * Grants permission to retrieve the resources that are included in the protection group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_ListResourcesInProtectionGroup.html
     */
    toListResourcesInProtectionGroup() {
        return this.to('ListResourcesInProtectionGroup');
    }
    /**
     * Grants permission to get information about AWS tags for a specified Amazon Resource Name (ARN) in AWS Shield
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to add or updates tags for a resource in AWS Shield
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from a resource in AWS Shield
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update application layer automatic response for Shield Advanced protection for a resource
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - apprunner:DescribeWebAclForService
     * - cognito-idp:GetWebACLForResource
     * - ec2:GetVerifiedAccessInstanceWebAcl
     * - wafv2:GetWebACL
     * - wafv2:GetWebACLForResource
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_UpdateApplicationLayerAutomaticResponse.html
     */
    toUpdateApplicationLayerAutomaticResponse() {
        return this.to('UpdateApplicationLayerAutomaticResponse');
    }
    /**
     * Grants permission to update the details of the list of email addresses that the DRT can use to contact you during a suspected attack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_UpdateEmergencyContactSettings.html
     */
    toUpdateEmergencyContactSettings() {
        return this.to('UpdateEmergencyContactSettings');
    }
    /**
     * Grants permission to update an existing protection group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_UpdateProtectionGroup.html
     */
    toUpdateProtectionGroup() {
        return this.to('UpdateProtectionGroup');
    }
    /**
     * Grants permission to update the details of an existing subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_UpdateSubscription.html
     */
    toUpdateSubscription() {
        return this.to('UpdateSubscription');
    }
    /**
     * Adds a resource of type attack to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_AttackDetail.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAttack(id, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:shield::${account ?? this.defaultAccount}:attack/${id}`);
    }
    /**
     * Adds a resource of type protection to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_Protection.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProtection(id, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:shield::${account ?? this.defaultAccount}:protection/${id}`);
    }
    /**
     * Adds a resource of type protection-group to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_ProtectionGroup.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProtectionGroup(id, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:shield::${account ?? this.defaultAccount}:protection-group/${id}`);
    }
    /**
     * Filters actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateProtection()
     * - .toCreateProtectionGroup()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toAssociateHealthCheck()
     * - .toDeleteProtection()
     * - .toDeleteProtectionGroup()
     * - .toDescribeProtection()
     * - .toDescribeProtectionGroup()
     * - .toDisassociateHealthCheck()
     * - .toUpdateProtectionGroup()
     *
     * Applies to resource types:
     * - protection
     * - protection-group
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateProtection()
     * - .toCreateProtectionGroup()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [shield](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsshield.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'shield';
        this.accessLevelList = {
            Write: [
                'AssociateDRTLogBucket',
                'AssociateDRTRole',
                'AssociateHealthCheck',
                'AssociateProactiveEngagementDetails',
                'CreateProtection',
                'CreateProtectionGroup',
                'CreateSubscription',
                'DeleteProtection',
                'DeleteProtectionGroup',
                'DeleteSubscription',
                'DisableApplicationLayerAutomaticResponse',
                'DisableProactiveEngagement',
                'DisassociateDRTLogBucket',
                'DisassociateDRTRole',
                'DisassociateHealthCheck',
                'EnableApplicationLayerAutomaticResponse',
                'EnableProactiveEngagement',
                'UpdateApplicationLayerAutomaticResponse',
                'UpdateEmergencyContactSettings',
                'UpdateProtectionGroup',
                'UpdateSubscription'
            ],
            Read: [
                'DescribeAttack',
                'DescribeAttackStatistics',
                'DescribeDRTAccess',
                'DescribeEmergencyContactSettings',
                'DescribeProtection',
                'DescribeProtectionGroup',
                'DescribeSubscription',
                'GetSubscriptionState',
                'ListTagsForResource'
            ],
            List: [
                'ListAttacks',
                'ListProtectionGroups',
                'ListProtections',
                'ListResourcesInProtectionGroup'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Shield = Shield;
//# sourceMappingURL=data:application/json;base64,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