"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SesMailmanager = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [ses-mailmanager](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonsimpleemailservice-mailmanager.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class SesMailmanager extends shared_1.PolicyStatement {
    /**
     * Grants permission to configure vended log delivery for Mail Manager resources
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/ses/latest/dg/eb-policies.html
     */
    toAllowVendedLogDeliveryForResource() {
        return this.to('AllowVendedLogDeliveryForResource');
    }
    /**
     * Grants permission to create an addon instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAddonSubscriptionArn()
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_CreateAddonInstance.html
     */
    toCreateAddonInstance() {
        return this.to('CreateAddonInstance');
    }
    /**
     * Grants permission to create an addon subscription
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_CreateAddonSubscription.html
     */
    toCreateAddonSubscription() {
        return this.to('CreateAddonSubscription');
    }
    /**
     * Grants permission to create an address list
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_CreateAddressList.html
     */
    toCreateAddressList() {
        return this.to('CreateAddressList');
    }
    /**
     * Grants permission to create an import job on an address list
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_CreateAddressListImportJob.html
     */
    toCreateAddressListImportJob() {
        return this.to('CreateAddressListImportJob');
    }
    /**
     * Grants permission to create an archive
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_CreateArchive.html
     */
    toCreateArchive() {
        return this.to('CreateArchive');
    }
    /**
     * Grants permission to create an ingress point
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifMailManagerRuleSetArn()
     * - .ifMailManagerTrafficPolicyArn()
     *
     * Dependent actions:
     * - ec2:DescribeVpcEndpoints
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_CreateIngressPoint.html
     */
    toCreateIngressPoint() {
        return this.to('CreateIngressPoint');
    }
    /**
     * Grants permission to create a SMTP relay
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_CreateRelay.html
     */
    toCreateRelay() {
        return this.to('CreateRelay');
    }
    /**
     * Grants permission to create a rule set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_CreateRuleSet.html
     */
    toCreateRuleSet() {
        return this.to('CreateRuleSet');
    }
    /**
     * Grants permission to create a traffic policy
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_CreateTrafficPolicy.html
     */
    toCreateTrafficPolicy() {
        return this.to('CreateTrafficPolicy');
    }
    /**
     * Grants permission to delete an addon instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_DeleteAddonInstance.html
     */
    toDeleteAddonInstance() {
        return this.to('DeleteAddonInstance');
    }
    /**
     * Grants permission to delete an addon subscription
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_DeleteAddonSubscription.html
     */
    toDeleteAddonSubscription() {
        return this.to('DeleteAddonSubscription');
    }
    /**
     * Grants permission to delete an address list
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_DeleteAddressList.html
     */
    toDeleteAddressList() {
        return this.to('DeleteAddressList');
    }
    /**
     * Grants permission to delete an archive
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_DeleteArchive.html
     */
    toDeleteArchive() {
        return this.to('DeleteArchive');
    }
    /**
     * Grants permission to delete an ingress point
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_DeleteIngressPoint.html
     */
    toDeleteIngressPoint() {
        return this.to('DeleteIngressPoint');
    }
    /**
     * Grants permission to delete a SMTP relay
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_DeleteRelay.html
     */
    toDeleteRelay() {
        return this.to('DeleteRelay');
    }
    /**
     * Grants permission to delete a rule set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_DeleteRuleSet.html
     */
    toDeleteRuleSet() {
        return this.to('DeleteRuleSet');
    }
    /**
     * Grants permission to delete a traffic point
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_DeleteTrafficPolicy.html
     */
    toDeleteTrafficPolicy() {
        return this.to('DeleteTrafficPolicy');
    }
    /**
     * Grants permission to remove a member from an address list
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_DeregisterMemberFromAddressList.html
     */
    toDeregisterMemberFromAddressList() {
        return this.to('DeregisterMemberFromAddressList');
    }
    /**
     * Grants permission to get information about an addon instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_GetAddonInstance.html
     */
    toGetAddonInstance() {
        return this.to('GetAddonInstance');
    }
    /**
     * Grants permission to get information about an addon subscription
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_GetAddonSubscription.html
     */
    toGetAddonSubscription() {
        return this.to('GetAddonSubscription');
    }
    /**
     * Grants permission to get information about an address list
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_GetAddressList.html
     */
    toGetAddressList() {
        return this.to('GetAddressList');
    }
    /**
     * Grants permission to get information about an import job on an address list
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_GetAddressListImportJob.html
     */
    toGetAddressListImportJob() {
        return this.to('GetAddressListImportJob');
    }
    /**
     * Grants permission to get information about an archive
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_GetArchive.html
     */
    toGetArchive() {
        return this.to('GetArchive');
    }
    /**
     * Grants permission to get information about an archive export
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_GetArchiveExport.html
     */
    toGetArchiveExport() {
        return this.to('GetArchiveExport');
    }
    /**
     * Grants permission to retrieve archived message
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_GetArchiveMessage.html
     */
    toGetArchiveMessage() {
        return this.to('GetArchiveMessage');
    }
    /**
     * Grants permission to retrieve archived message content
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_GetArchiveMessageContent.html
     */
    toGetArchiveMessageContent() {
        return this.to('GetArchiveMessageContent');
    }
    /**
     * Grants permission to get information about a search
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_GetArchiveSearch.html
     */
    toGetArchiveSearch() {
        return this.to('GetArchiveSearch');
    }
    /**
     * Grants permission to get information about search results
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_GetArchiveSearchResults.html
     */
    toGetArchiveSearchResults() {
        return this.to('GetArchiveSearchResults');
    }
    /**
     * Grants permission to get information about an ingress point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_GetIngressPoint.html
     */
    toGetIngressPoint() {
        return this.to('GetIngressPoint');
    }
    /**
     * Grants permission to get information about a member in an address list
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_GetMemberOfAddressList.html
     */
    toGetMemberOfAddressList() {
        return this.to('GetMemberOfAddressList');
    }
    /**
     * Grants permission to get information about a SMTP relay
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_GetRelay.html
     */
    toGetRelay() {
        return this.to('GetRelay');
    }
    /**
     * Grants permission to get information about a rule set
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_GetRuleSet.html
     */
    toGetRuleSet() {
        return this.to('GetRuleSet');
    }
    /**
     * Grants permission to get information about a traffic policy
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_GetTrafficPolicy.html
     */
    toGetTrafficPolicy() {
        return this.to('GetTrafficPolicy');
    }
    /**
     * Grants permission to list all of the addon instances associated with your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_ListAddonInstances.html
     */
    toListAddonInstances() {
        return this.to('ListAddonInstances');
    }
    /**
     * Grants permission to list all of the addon subscriptions associated with your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_ListAddonSubscriptions.html
     */
    toListAddonSubscriptions() {
        return this.to('ListAddonSubscriptions');
    }
    /**
     * Grants permission to list all of the import jobs associated with an address list
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_ListAddressListImportJobs.html
     */
    toListAddressListImportJobs() {
        return this.to('ListAddressListImportJobs');
    }
    /**
     * Grants permission to list all of the address lists associated with your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_ListAddressLists.html
     */
    toListAddressLists() {
        return this.to('ListAddressLists');
    }
    /**
     * Grants permission to list all of the archive exports associated with your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_ListArchiveExports.html
     */
    toListArchiveExports() {
        return this.to('ListArchiveExports');
    }
    /**
     * Grants permission to list all of the archive searches associated with your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_ListArchiveSearches.html
     */
    toListArchiveSearches() {
        return this.to('ListArchiveSearches');
    }
    /**
     * Grants permission to list all of the archives associated with your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_ListArchives.html
     */
    toListArchives() {
        return this.to('ListArchives');
    }
    /**
     * Grants permission to list all of the ingress points associated with your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_ListIngressPoints.html
     */
    toListIngressPoints() {
        return this.to('ListIngressPoints');
    }
    /**
     * Grants permission to list all of the members associated with an address list
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_ListMembersOfAddressList.html
     */
    toListMembersOfAddressList() {
        return this.to('ListMembersOfAddressList');
    }
    /**
     * Grants permission to list all of the SMTP relays associated with your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_ListRelays.html
     */
    toListRelays() {
        return this.to('ListRelays');
    }
    /**
     * Grants permission to list all of the rule sets associated with your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_ListRuleSets.html
     */
    toListRuleSets() {
        return this.to('ListRuleSets');
    }
    /**
     * Grants permission to list all of the tags associated with the resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list all of the traffic policies associated with your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_ListTrafficPolicies.html
     */
    toListTrafficPolicies() {
        return this.to('ListTrafficPolicies');
    }
    /**
     * Grants permission to add a member to an address list
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_RegisterMemberToAddressList.html
     */
    toRegisterMemberToAddressList() {
        return this.to('RegisterMemberToAddressList');
    }
    /**
     * Grants permission to start an import job on an address list
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_StartAddressListImportJob.html
     */
    toStartAddressListImportJob() {
        return this.to('StartAddressListImportJob');
    }
    /**
     * Grants permission to start an archive export
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_StartArchiveExport.html
     */
    toStartArchiveExport() {
        return this.to('StartArchiveExport');
    }
    /**
     * Grants permission to start an archive search
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_StartArchiveSearch.html
     */
    toStartArchiveSearch() {
        return this.to('StartArchiveSearch');
    }
    /**
     * Grants permission to stop an ongoing import job on an address list
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_StopAddressListImportJob.html
     */
    toStopAddressListImportJob() {
        return this.to('StopAddressListImportJob');
    }
    /**
     * Grants permission to stop archive export
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_StopArchiveExport.html
     */
    toStopArchiveExport() {
        return this.to('StopArchiveExport');
    }
    /**
     * Grants permission to stop archive search
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_StopArchiveSearch.html
     */
    toStopArchiveSearch() {
        return this.to('StopArchiveSearch');
    }
    /**
     * Grants permission to add one or more tags (keys and values) to a specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove one or more tags (keys and values) from a specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an archive
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_UpdateArchive.html
     */
    toUpdateArchive() {
        return this.to('UpdateArchive');
    }
    /**
     * Grants permission to update an ingress point
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifMailManagerTrafficPolicyArn()
     * - .ifMailManagerRuleSetArn()
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_UpdateIngressPoint.html
     */
    toUpdateIngressPoint() {
        return this.to('UpdateIngressPoint');
    }
    /**
     * Grants permission to update a SMTP relay
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_UpdateRelay.html
     */
    toUpdateRelay() {
        return this.to('UpdateRelay');
    }
    /**
     * Grants permission to update a rule set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_UpdateRuleSet.html
     */
    toUpdateRuleSet() {
        return this.to('UpdateRuleSet');
    }
    /**
     * Grants permission to update a traffic policy
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_UpdateTrafficPolicy.html
     */
    toUpdateTrafficPolicy() {
        return this.to('UpdateTrafficPolicy');
    }
    /**
     * Adds a resource of type addon-instance to the statement
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_AddonInstance.html
     *
     * @param addonInstanceId - Identifier for the addonInstanceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAddonInstance(addonInstanceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:addon-instance/${addonInstanceId}`);
    }
    /**
     * Adds a resource of type addon-subscription to the statement
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_AddonSubscription.html
     *
     * @param addonSubscriptionId - Identifier for the addonSubscriptionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAddonSubscription(addonSubscriptionId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:addon-subscription/${addonSubscriptionId}`);
    }
    /**
     * Adds a resource of type mailmanager-archive to the statement
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_Archive.html
     *
     * @param archiveId - Identifier for the archiveId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMailmanagerArchive(archiveId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:mailmanager-archive/${archiveId}`);
    }
    /**
     * Adds a resource of type mailmanager-ingress-point to the statement
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_IngressPoint.html
     *
     * @param ingressPointId - Identifier for the ingressPointId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifMailManagerIngressPointType()
     */
    onMailmanagerIngressPoint(ingressPointId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:mailmanager-ingress-point/${ingressPointId}`);
    }
    /**
     * Adds a resource of type mailmanager-smtp-relay to the statement
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_Relay.html
     *
     * @param relayId - Identifier for the relayId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMailmanagerSmtpRelay(relayId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:mailmanager-smtp-relay/${relayId}`);
    }
    /**
     * Adds a resource of type mailmanager-rule-set to the statement
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_RuleSet.html
     *
     * @param ruleSetId - Identifier for the ruleSetId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMailmanagerRuleSet(ruleSetId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:mailmanager-rule-set/${ruleSetId}`);
    }
    /**
     * Adds a resource of type mailmanager-traffic-policy to the statement
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_TrafficPolicy.html
     *
     * @param trafficPolicyId - Identifier for the trafficPolicyId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMailmanagerTrafficPolicy(trafficPolicyId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:mailmanager-traffic-policy/${trafficPolicyId}`);
    }
    /**
     * Adds a resource of type mailmanager-address-list to the statement
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_AddressList.html
     *
     * @param addressListId - Identifier for the addressListId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMailmanagerAddressList(addressListId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:mailmanager-address-list/${addressListId}`);
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateAddonInstance()
     * - .toCreateAddonSubscription()
     * - .toCreateAddressList()
     * - .toCreateArchive()
     * - .toCreateIngressPoint()
     * - .toCreateRelay()
     * - .toCreateRuleSet()
     * - .toCreateTrafficPolicy()
     * - .toDeleteAddonInstance()
     * - .toDeleteAddonSubscription()
     * - .toDeleteArchive()
     * - .toDeleteIngressPoint()
     * - .toDeleteRelay()
     * - .toDeleteRuleSet()
     * - .toDeleteTrafficPolicy()
     * - .toGetAddonInstance()
     * - .toGetAddonSubscription()
     * - .toGetArchive()
     * - .toGetIngressPoint()
     * - .toGetRelay()
     * - .toGetRuleSet()
     * - .toGetTrafficPolicy()
     * - .toTagResource()
     * - .toUpdateArchive()
     * - .toUpdateIngressPoint()
     * - .toUpdateRelay()
     * - .toUpdateRuleSet()
     * - .toUpdateTrafficPolicy()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toCreateAddressListImportJob()
     * - .toDeleteAddressList()
     * - .toDeregisterMemberFromAddressList()
     * - .toGetAddressList()
     * - .toGetAddressListImportJob()
     * - .toGetMemberOfAddressList()
     * - .toListAddressListImportJobs()
     * - .toListMembersOfAddressList()
     * - .toRegisterMemberToAddressList()
     * - .toStartAddressListImportJob()
     * - .toStopAddressListImportJob()
     *
     * Applies to resource types:
     * - addon-instance
     * - addon-subscription
     * - mailmanager-archive
     * - mailmanager-ingress-point
     * - mailmanager-smtp-relay
     * - mailmanager-rule-set
     * - mailmanager-traffic-policy
     * - mailmanager-address-list
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateAddonInstance()
     * - .toCreateAddonSubscription()
     * - .toCreateAddressList()
     * - .toCreateArchive()
     * - .toCreateIngressPoint()
     * - .toCreateRelay()
     * - .toCreateRuleSet()
     * - .toCreateTrafficPolicy()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by SES Addon Subscription ARN
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsesmailmanager.html#amazonsesmailmanager-policy-keys
     *
     * Applies to actions:
     * - .toCreateAddonInstance()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifAddonSubscriptionArn(value, operator) {
        return this.if(`AddonSubscriptionArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by SES Mail Manager ingress point type, for example OPEN or AUTH
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsesmailmanager.html#amazonsesmailmanager-policy-keys
     *
     * Applies to resource types:
     * - mailmanager-ingress-point
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifMailManagerIngressPointType(value, operator) {
        return this.if(`MailManagerIngressPointType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by SES Mail Manager rule set ARN
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsesmailmanager.html#amazonsesmailmanager-policy-keys
     *
     * Applies to actions:
     * - .toCreateIngressPoint()
     * - .toUpdateIngressPoint()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifMailManagerRuleSetArn(value, operator) {
        return this.if(`MailManagerRuleSetArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by SES Mail Manager traffic policy ARN
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsesmailmanager.html#amazonsesmailmanager-policy-keys
     *
     * Applies to actions:
     * - .toCreateIngressPoint()
     * - .toUpdateIngressPoint()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifMailManagerTrafficPolicyArn(value, operator) {
        return this.if(`MailManagerTrafficPolicyArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Statement provider for service [ses-mailmanager](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonsimpleemailservice-mailmanager.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'ses';
        this.accessLevelList = {
            'Permissions management': [
                'AllowVendedLogDeliveryForResource'
            ],
            Write: [
                'CreateAddonInstance',
                'CreateAddonSubscription',
                'CreateAddressList',
                'CreateAddressListImportJob',
                'CreateArchive',
                'CreateIngressPoint',
                'CreateRelay',
                'CreateRuleSet',
                'CreateTrafficPolicy',
                'DeleteAddonInstance',
                'DeleteAddonSubscription',
                'DeleteAddressList',
                'DeleteArchive',
                'DeleteIngressPoint',
                'DeleteRelay',
                'DeleteRuleSet',
                'DeleteTrafficPolicy',
                'DeregisterMemberFromAddressList',
                'RegisterMemberToAddressList',
                'StartAddressListImportJob',
                'StartArchiveExport',
                'StartArchiveSearch',
                'StopAddressListImportJob',
                'StopArchiveExport',
                'StopArchiveSearch',
                'UpdateArchive',
                'UpdateIngressPoint',
                'UpdateRelay',
                'UpdateRuleSet',
                'UpdateTrafficPolicy'
            ],
            Read: [
                'GetAddonInstance',
                'GetAddonSubscription',
                'GetAddressList',
                'GetAddressListImportJob',
                'GetArchive',
                'GetArchiveExport',
                'GetArchiveMessage',
                'GetArchiveMessageContent',
                'GetArchiveSearch',
                'GetArchiveSearchResults',
                'GetIngressPoint',
                'GetMemberOfAddressList',
                'GetRelay',
                'GetRuleSet',
                'GetTrafficPolicy',
                'ListTagsForResource'
            ],
            List: [
                'ListAddonInstances',
                'ListAddonSubscriptions',
                'ListAddressListImportJobs',
                'ListAddressLists',
                'ListArchiveExports',
                'ListArchiveSearches',
                'ListArchives',
                'ListIngressPoints',
                'ListMembersOfAddressList',
                'ListRelays',
                'ListRuleSets',
                'ListTrafficPolicies'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.SesMailmanager = SesMailmanager;
//# sourceMappingURL=data:application/json;base64,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